--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:          axi_dma_pkg.vhd
-- Description: This package contains various constants and functions for
--              AXI DMA operations.
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

library lib_pkg_v1_0_2;
use lib_pkg_v1_0_2.lib_pkg.clog2;

package axi_dma_pkg is

-------------------------------------------------------------------------------
-- Function declarations
-------------------------------------------------------------------------------
-- Find minimum required btt width
function required_btt_width (dwidth     : integer;
                            burst_size  : integer;
                            btt_width   : integer)
            return  integer;

-- Return correct hertz paramter value
function hertz_prmtr_select(included        : integer;
                            lite_frequency  : integer;
                            sg_frequency    : integer)
    return integer;

-- Return SnF enable or disable
function enable_snf (sf_enabled         : integer;
                     axi_data_width     : integer;
                     axis_tdata_width   : integer)
    return integer;

-------------------------------------------------------------------------------
-- Constant Declarations
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
-- AXI Responce Values
-------------------------------------------------------------------------------
constant OKAY_RESP                  : std_logic_vector(1 downto 0)  := "00";
constant EXOKAY_RESP                : std_logic_vector(1 downto 0)  := "01";
constant SLVERR_RESP                : std_logic_vector(1 downto 0)  := "10";
constant DECERR_RESP                : std_logic_vector(1 downto 0)  := "11";

constant MTBF_STAGES                : integer := 4;
constant C_FIFO_MTBF                : integer := 4;
-------------------------------------------------------------------------------
-- Misc Constants
-------------------------------------------------------------------------------
--constant NUM_REG_TOTAL              : integer := 18;
--constant NUM_REG_TOTAL              : integer := 23;
constant NUM_REG_TOTAL              : integer := 143; -- To accomodate S2MM registers
--constant NUM_REG_PER_CHANNEL        : integer := 6;
constant NUM_REG_PER_CHANNEL        : integer := 12;
constant NUM_REG_PER_S2MM        : integer := 120;
--constant REG_MSB_ADDR_BIT           : integer := clog2(NUM_REG_TOTAL)-1;
constant CMD_BASE_WIDTH             : integer := 40;
constant BUFFER_LENGTH_WIDTH        : integer := 23;

-- Constants Used in Desc Updates
constant DESC_STS_TYPE              : std_logic := '1';
constant DESC_DATA_TYPE             : std_logic := '0';
constant DESC_LAST                  : std_logic := '1';
constant DESC_NOT_LAST              : std_logic := '0';

-- Interrupt Coalescing
constant ZERO_THRESHOLD             : std_logic_vector(7 downto 0) := (others => '0');
constant ONE_THRESHOLD              : std_logic_vector(7 downto 0) := "00000001";
constant ZERO_DELAY                 : std_logic_vector(7 downto 0) := (others => '0');

-------------------------------------------------------------------------------
-- AXI Lite AXI DMA Register Offsets
-------------------------------------------------------------------------------
constant MM2S_DMACR_INDEX           : integer := 0;
constant MM2S_DMASR_INDEX           : integer := 1;
constant MM2S_CURDESC_LSB_INDEX     : integer := 2;
constant MM2S_CURDESC_MSB_INDEX     : integer := 3;
constant MM2S_TAILDESC_LSB_INDEX    : integer := 4;
constant MM2S_TAILDESC_MSB_INDEX    : integer := 5;
constant MM2S_SA_INDEX              : integer := 6;
constant MM2S_SA2_INDEX          : integer := 7;
constant RESERVED_20_INDEX          : integer := 8;
constant RESERVED_24_INDEX          : integer := 9;
constant MM2S_LENGTH_INDEX          : integer := 10;
constant RESERVED_2C_INDEX          : integer := 11;
constant S2MM_DMACR_INDEX           : integer := 12;
constant S2MM_DMASR_INDEX           : integer := 13;
constant S2MM_CURDESC_LSB_INDEX     : integer := 14;
constant S2MM_CURDESC_MSB_INDEX     : integer := 15;
constant S2MM_TAILDESC_LSB_INDEX    : integer := 16;
constant S2MM_TAILDESC_MSB_INDEX    : integer := 17;
constant S2MM_DA_INDEX              : integer := 18;
constant S2MM_DA2_INDEX          : integer := 19;
constant RESERVED_50_INDEX          : integer := 20;
constant RESERVED_54_INDEX          : integer := 21;
--constant S2MM_LENGTH_INDEX          : integer := 22;
constant S2MM_LENGTH_INDEX          : integer := 142; 


constant MM2S_DMACR_OFFSET          : std_logic_vector(9 downto 0) := "0000000000";    -- 0x00
constant MM2S_DMASR_OFFSET          : std_logic_vector(9 downto 0) := "0000000100";    -- 0x04
constant MM2S_CURDESC_LSB_OFFSET    : std_logic_vector(9 downto 0) := "0000001000";    -- 0x08
constant MM2S_CURDESC_MSB_OFFSET    : std_logic_vector(9 downto 0) := "0000001100";    -- 0x0C
constant MM2S_TAILDESC_LSB_OFFSET   : std_logic_vector(9 downto 0) := "0000010000";    -- 0x10
constant MM2S_TAILDESC_MSB_OFFSET   : std_logic_vector(9 downto 0) := "0000010100";    -- 0x14
constant MM2S_SA_OFFSET             : std_logic_vector(9 downto 0) := "0000011000";    -- 0x18
constant MM2S_SA2_OFFSET            : std_logic_vector(9 downto 0) := "0000011100";    -- 0x1C
constant RESERVED_20_OFFSET         : std_logic_vector(9 downto 0) := "0000100000";    -- 0x20
constant RESERVED_24_OFFSET         : std_logic_vector(9 downto 0) := "0000100100";    -- 0x24
constant MM2S_LENGTH_OFFSET         : std_logic_vector(9 downto 0) := "0000101000";    -- 0x28
-- Following was reserved, now is used for SG xCache and xUser
constant SGCTL_OFFSET               : std_logic_vector(9 downto 0) := "0000101100";    -- 0x2C

constant S2MM_DMACR_OFFSET          : std_logic_vector(9 downto 0) := "0000110000";    -- 0x30
constant S2MM_DMASR_OFFSET          : std_logic_vector(9 downto 0) := "0000110100";    -- 0x34
constant S2MM_CURDESC_LSB_OFFSET    : std_logic_vector(9 downto 0) := "0000111000";    -- 0x38
constant S2MM_CURDESC_MSB_OFFSET    : std_logic_vector(9 downto 0) := "0000111100";    -- 0x3C
constant S2MM_TAILDESC_LSB_OFFSET   : std_logic_vector(9 downto 0) := "0001000000";    -- 0x40
constant S2MM_TAILDESC_MSB_OFFSET   : std_logic_vector(9 downto 0) := "0001000100";    -- 0x44
constant S2MM_DA_OFFSET             : std_logic_vector(9 downto 0) := "0001001000";    -- 0x48 --CR603034
constant S2MM_DA2_OFFSET            : std_logic_vector(9 downto 0) := "0001001100";    -- 0x4C
constant RESERVED_50_OFFSET         : std_logic_vector(9 downto 0) := "0001010000";    -- 0x50
constant RESERVED_54_OFFSET         : std_logic_vector(9 downto 0) := "0001010100";    -- 0x54
constant S2MM_LENGTH_OFFSET         : std_logic_vector(9 downto 0) := "0001011000";    -- 0x58

-- New registers for S2MM channels
constant S2MM_CURDESC1_LSB_OFFSET    : std_logic_vector(9 downto 0) := "0001110000";    -- 0x70
constant S2MM_CURDESC1_MSB_OFFSET    : std_logic_vector(9 downto 0) := "0001110100";    -- 0x74
constant S2MM_TAILDESC1_LSB_OFFSET   : std_logic_vector(9 downto 0) := "0001111000";    -- 0x78
constant S2MM_TAILDESC1_MSB_OFFSET   : std_logic_vector(9 downto 0) := "0001111100";    -- 0x7C

constant S2MM_CURDESC2_LSB_OFFSET    : std_logic_vector(9 downto 0) := "0010010000";    -- 0x90
constant S2MM_CURDESC2_MSB_OFFSET    : std_logic_vector(9 downto 0) := "0010010100";    -- 0x94
constant S2MM_TAILDESC2_LSB_OFFSET   : std_logic_vector(9 downto 0) := "0010011000";    -- 0x98
constant S2MM_TAILDESC2_MSB_OFFSET   : std_logic_vector(9 downto 0) := "0010011100";    -- 0x9C

constant S2MM_CURDESC3_LSB_OFFSET    : std_logic_vector(9 downto 0) := "0010110000";    -- 0xB0
constant S2MM_CURDESC3_MSB_OFFSET    : std_logic_vector(9 downto 0) := "0010110100";    -- 0xB4
constant S2MM_TAILDESC3_LSB_OFFSET   : std_logic_vector(9 downto 0) := "0010111000";    -- 0xB8
constant S2MM_TAILDESC3_MSB_OFFSET   : std_logic_vector(9 downto 0) := "0010111100";    -- 0xBC

constant S2MM_CURDESC4_LSB_OFFSET    : std_logic_vector(9 downto 0) := "0011010000";    -- 0xD0
constant S2MM_CURDESC4_MSB_OFFSET    : std_logic_vector(9 downto 0) := "0011010100";    -- 0xD4
constant S2MM_TAILDESC4_LSB_OFFSET   : std_logic_vector(9 downto 0) := "0011011000";    -- 0xD8
constant S2MM_TAILDESC4_MSB_OFFSET   : std_logic_vector(9 downto 0) := "0011011100";    -- 0xDC

constant S2MM_CURDESC5_LSB_OFFSET    : std_logic_vector(9 downto 0) := "0011110000";    -- 0xF0
constant S2MM_CURDESC5_MSB_OFFSET    : std_logic_vector(9 downto 0) := "0011110100";    -- 0xF4
constant S2MM_TAILDESC5_LSB_OFFSET   : std_logic_vector(9 downto 0) := "0011111000";    -- 0xF8
constant S2MM_TAILDESC5_MSB_OFFSET   : std_logic_vector(9 downto 0) := "0011111100";    -- 0xFC

constant S2MM_CURDESC6_LSB_OFFSET    : std_logic_vector(9 downto 0) := "0100010000";    -- 0x110
constant S2MM_CURDESC6_MSB_OFFSET    : std_logic_vector(9 downto 0) := "0100010100";    -- 0x114
constant S2MM_TAILDESC6_LSB_OFFSET   : std_logic_vector(9 downto 0) := "0100011000";    -- 0x118
constant S2MM_TAILDESC6_MSB_OFFSET   : std_logic_vector(9 downto 0) := "0100011100";    -- 0x11C

constant S2MM_CURDESC7_LSB_OFFSET    : std_logic_vector(9 downto 0) := "0100110000";    -- 0x130
constant S2MM_CURDESC7_MSB_OFFSET    : std_logic_vector(9 downto 0) := "0100110100";    -- 0x134
constant S2MM_TAILDESC7_LSB_OFFSET   : std_logic_vector(9 downto 0) := "0100111000";    -- 0x138
constant S2MM_TAILDESC7_MSB_OFFSET   : std_logic_vector(9 downto 0) := "0100111100";    -- 0x13C

constant S2MM_CURDESC8_LSB_OFFSET    : std_logic_vector(9 downto 0) := "0101010000";    -- 0x150
constant S2MM_CURDESC8_MSB_OFFSET    : std_logic_vector(9 downto 0) := "0101010100";    -- 0x154
constant S2MM_TAILDESC8_LSB_OFFSET   : std_logic_vector(9 downto 0) := "0101011000";    -- 0x158
constant S2MM_TAILDESC8_MSB_OFFSET   : std_logic_vector(9 downto 0) := "0101011100";    -- 0x15C

constant S2MM_CURDESC9_LSB_OFFSET    : std_logic_vector(9 downto 0) := "0101110000";    -- 0x170
constant S2MM_CURDESC9_MSB_OFFSET    : std_logic_vector(9 downto 0) := "0101110100";    -- 0x174
constant S2MM_TAILDESC9_LSB_OFFSET   : std_logic_vector(9 downto 0) := "0101111000";    -- 0x178
constant S2MM_TAILDESC9_MSB_OFFSET   : std_logic_vector(9 downto 0) := "0101111100";    -- 0x17C

constant S2MM_CURDESC10_LSB_OFFSET    : std_logic_vector(9 downto 0) := "0110010000";    -- 0x190
constant S2MM_CURDESC10_MSB_OFFSET    : std_logic_vector(9 downto 0) := "0110010100";    -- 0x194
constant S2MM_TAILDESC10_LSB_OFFSET   : std_logic_vector(9 downto 0) := "0110011000";    -- 0x198
constant S2MM_TAILDESC10_MSB_OFFSET   : std_logic_vector(9 downto 0) := "0110011100";    -- 0x19C

constant S2MM_CURDESC11_LSB_OFFSET    : std_logic_vector(9 downto 0) := "0110110000";    -- 0x1B0
constant S2MM_CURDESC11_MSB_OFFSET    : std_logic_vector(9 downto 0) := "0110110100";    -- 0x1B4
constant S2MM_TAILDESC11_LSB_OFFSET   : std_logic_vector(9 downto 0) := "0110111000";    -- 0x1B8
constant S2MM_TAILDESC11_MSB_OFFSET   : std_logic_vector(9 downto 0) := "0110111100";    -- 0x1BC

constant S2MM_CURDESC12_LSB_OFFSET    : std_logic_vector(9 downto 0) := "0111010000";    -- 0x1D0
constant S2MM_CURDESC12_MSB_OFFSET    : std_logic_vector(9 downto 0) := "0111010100";    -- 0x1D4
constant S2MM_TAILDESC12_LSB_OFFSET   : std_logic_vector(9 downto 0) := "0111011000";    -- 0x1D8
constant S2MM_TAILDESC12_MSB_OFFSET   : std_logic_vector(9 downto 0) := "0111011100";    -- 0x1DC

constant S2MM_CURDESC13_LSB_OFFSET    : std_logic_vector(9 downto 0) := "0111110000";    -- 0x1F0
constant S2MM_CURDESC13_MSB_OFFSET    : std_logic_vector(9 downto 0) := "0111110100";    -- 0x1F4
constant S2MM_TAILDESC13_LSB_OFFSET   : std_logic_vector(9 downto 0) := "0111111000";    -- 0x1F8
constant S2MM_TAILDESC13_MSB_OFFSET   : std_logic_vector(9 downto 0) := "0111111100";    -- 0x1FC

constant S2MM_CURDESC14_LSB_OFFSET    : std_logic_vector(9 downto 0) := "1000010000";    -- 0x210
constant S2MM_CURDESC14_MSB_OFFSET    : std_logic_vector(9 downto 0) := "1000010100";    -- 0x214
constant S2MM_TAILDESC14_LSB_OFFSET   : std_logic_vector(9 downto 0) := "1000011000";    -- 0x218
constant S2MM_TAILDESC14_MSB_OFFSET   : std_logic_vector(9 downto 0) := "1000011100";    -- 0x21C

constant S2MM_CURDESC15_LSB_OFFSET    : std_logic_vector(9 downto 0) := "1000110000";    -- 0x230
constant S2MM_CURDESC15_MSB_OFFSET    : std_logic_vector(9 downto 0) := "1000110100";    -- 0x234
constant S2MM_TAILDESC15_LSB_OFFSET   : std_logic_vector(9 downto 0) := "1000111000";    -- 0x238
constant S2MM_TAILDESC15_MSB_OFFSET   : std_logic_vector(9 downto 0) := "1000111100";    -- 0x23C




-------------------------------------------------------------------------------
-- Register Bit Constants
-------------------------------------------------------------------------------
-- DMACR
constant DMACR_RS_BIT               : integer := 0;
constant DMACR_TAILPEN_BIT          : integer := 1;
constant DMACR_RESET_BIT            : integer := 2;
constant DMACR_KH_BIT        : integer := 3;
constant CYCLIC_BIT        : integer := 4;
--constant DMACR_RESERVED3_BIT        : integer := 3;
--constant DMACR_RESERVED4_BIT        : integer := 4;
constant DMACR_RESERVED5_BIT        : integer := 5;
constant DMACR_RESERVED6_BIT        : integer := 6;
constant DMACR_RESERVED7_BIT        : integer := 7;
constant DMACR_RESERVED8_BIT        : integer := 8;
constant DMACR_RESERVED9_BIT        : integer := 9;
constant DMACR_RESERVED10_BIT       : integer := 10;
constant DMACR_RESERVED11_BIT       : integer := 11;
constant DMACR_IOC_IRQEN_BIT        : integer := 12;
constant DMACR_DLY_IRQEN_BIT        : integer := 13;
constant DMACR_ERR_IRQEN_BIT        : integer := 14;
constant DMACR_RESERVED15_BIT       : integer := 15;
constant DMACR_IRQTHRESH_LSB_BIT    : integer := 16;
constant DMACR_IRQTHRESH_MSB_BIT    : integer := 23;
constant DMACR_IRQDELAY_LSB_BIT     : integer := 24;
constant DMACR_IRQDELAY_MSB_BIT     : integer := 31;

-- DMASR
constant DMASR_HALTED_BIT           : integer := 0;
constant DMASR_IDLE_BIT             : integer := 1;
constant DMASR_CMPLT_BIT            : integer := 2;
constant DMASR_ERROR_BIT            : integer := 3;
constant DMASR_DMAINTERR_BIT        : integer := 4;
constant DMASR_DMASLVERR_BIT        : integer := 5;
constant DMASR_DMADECERR_BIT        : integer := 6;
constant DMASR_RESERVED7_BIT        : integer := 7;
constant DMASR_SGINTERR_BIT         : integer := 8;
constant DMASR_SGSLVERR_BIT         : integer := 9;
constant DMASR_SGDECERR_BIT         : integer := 10;
constant DMASR_RESERVED11_BIT       : integer := 11;
constant DMASR_IOCIRQ_BIT           : integer := 12;
constant DMASR_DLYIRQ_BIT           : integer := 13;
constant DMASR_ERRIRQ_BIT           : integer := 14;
constant DMASR_RESERVED15_BIT       : integer := 15;
constant DMASR_IRQTHRESH_LSB_BIT    : integer := 16;
constant DMASR_IRQTHRESH_MSB_BIT    : integer := 23;
constant DMASR_IRQDELAY_LSB_BIT     : integer := 24;
constant DMASR_IRQDELAY_MSB_BIT     : integer := 31;

-- CURDESC
constant CURDESC_LOWER_MSB_BIT      : integer := 31;
constant CURDESC_LOWER_LSB_BIT      : integer := 6;
constant CURDESC_RESERVED_BIT5      : integer := 5;
constant CURDESC_RESERVED_BIT4      : integer := 4;
constant CURDESC_RESERVED_BIT3      : integer := 3;
constant CURDESC_RESERVED_BIT2      : integer := 2;
constant CURDESC_RESERVED_BIT1      : integer := 1;
constant CURDESC_RESERVED_BIT0      : integer := 0;

-- TAILDESC
constant TAILDESC_LOWER_MSB_BIT     : integer := 31;
constant TAILDESC_LOWER_LSB_BIT     : integer := 6;
constant TAILDESC_RESERVED_BIT5     : integer := 5;
constant TAILDESC_RESERVED_BIT4     : integer := 4;
constant TAILDESC_RESERVED_BIT3     : integer := 3;
constant TAILDESC_RESERVED_BIT2     : integer := 2;
constant TAILDESC_RESERVED_BIT1     : integer := 1;
constant TAILDESC_RESERVED_BIT0     : integer := 0;

-- DataMover Command / Status Constants
constant DATAMOVER_CMDDONE_BIT      : integer := 7;
constant DATAMOVER_SLVERR_BIT       : integer := 6;
constant DATAMOVER_DECERR_BIT       : integer := 5;
constant DATAMOVER_INTERR_BIT       : integer := 4;
constant DATAMOVER_TAGMSB_BIT       : integer := 3;
constant DATAMOVER_TAGLSB_BIT       : integer := 0;

-- Descriptor Control Bits
constant DESC_BLENGTH_LSB_BIT       : integer := 0;
constant DESC_BLENGTH_MSB_BIT       : integer := 22;
constant DESC_RSVD23_BIT            : integer := 23;
constant DESC_RSVD24_BIT            : integer := 24;
constant DESC_RSVD25_BIT            : integer := 25;
constant DESC_EOF_BIT               : integer := 26;
constant DESC_SOF_BIT               : integer := 27;
constant DESC_RSVD28_BIT            : integer := 28;
constant DESC_RSVD29_BIT            : integer := 29;
constant DESC_RSVD30_BIT            : integer := 30;
constant DESC_IOC_BIT               : integer := 31;

-- Descriptor Status Bits
constant DESC_STS_CMPLTD_BIT        : integer := 31;
constant DESC_STS_DECERR_BIT        : integer := 30;
constant DESC_STS_SLVERR_BIT        : integer := 29;
constant DESC_STS_INTERR_BIT        : integer := 28;
constant DESC_STS_RXSOF_BIT         : integer := 27;
constant DESC_STS_RXEOF_BIT         : integer := 26;
constant DESC_STS_RSVD25_BIT        : integer := 25;
constant DESC_STS_RSVD24_BIT        : integer := 24;
constant DESC_STS_RSVD23_BIT        : integer := 23;
constant DESC_STS_XFRDBYTS_MSB_BIT  : integer := 22;
constant DESC_STS_XFRDBYTS_LSB_BIT  : integer := 0;


-- DataMover Command / Status Constants
constant DATAMOVER_STS_CMDDONE_BIT  : integer := 7;
constant DATAMOVER_STS_SLVERR_BIT   : integer := 6;
constant DATAMOVER_STS_DECERR_BIT   : integer := 5;
constant DATAMOVER_STS_INTERR_BIT   : integer := 4;
constant DATAMOVER_STS_TAGMSB_BIT   : integer := 3;
constant DATAMOVER_STS_TAGLSB_BIT   : integer := 0;

constant DATAMOVER_STS_TAGEOF_BIT   : integer := 1;
constant DATAMOVER_STS_TLAST_BIT    : integer := 31;

constant DATAMOVER_CMD_BTTLSB_BIT   : integer := 0;
constant DATAMOVER_CMD_BTTMSB_BIT   : integer := 22;
constant DATAMOVER_CMD_TYPE_BIT     : integer := 23;
constant DATAMOVER_CMD_DSALSB_BIT   : integer := 24;
constant DATAMOVER_CMD_DSAMSB_BIT   : integer := 29;
constant DATAMOVER_CMD_EOF_BIT      : integer := 30;
constant DATAMOVER_CMD_DRR_BIT      : integer := 31;
constant DATAMOVER_CMD_ADDRLSB_BIT  : integer := 32;

-- Note: Bit offset require adding ADDR WIDTH to get to actual bit index
constant DATAMOVER_CMD_ADDRMSB_BOFST: integer := 31;
constant DATAMOVER_CMD_TAGLSB_BOFST : integer := 32;
constant DATAMOVER_CMD_TAGMSB_BOFST : integer := 35;
constant DATAMOVER_CMD_RSVLSB_BOFST : integer := 36;
constant DATAMOVER_CMD_RSVMSB_BOFST : integer := 39;


end axi_dma_pkg;

-------------------------------------------------------------------------------
-- PACKAGE BODY
-------------------------------------------------------------------------------
package body axi_dma_pkg is



-------------------------------------------------------------------------------
-- Function to determine minimum bits required for BTT_SIZE field
-------------------------------------------------------------------------------
function required_btt_width ( dwidth    : integer;
                              burst_size: integer;
                              btt_width : integer)
    return integer  is
variable min_width : integer;

begin
    min_width := clog2((dwidth/8)*burst_size)+1;
    if(min_width > btt_width)then
        return min_width;
    else
        return btt_width;
    end if;
end function required_btt_width;

-------------------------------------------------------------------------------
-- function to return Frequency Hertz parameter based on inclusion of sg engine
-------------------------------------------------------------------------------
function hertz_prmtr_select(included        : integer;
                            lite_frequency  : integer;
                            sg_frequency    : integer)
    return integer is
    begin
        -- 1 = Scatter Gather Included
        -- 0 = Scatter Gather Excluded
        if(included = 1)then
            return sg_frequency;
        else
            return lite_frequency;
        end if;
    end;


-------------------------------------------------------------------------------
-- function to enable store and forward based on data width mismatch
-- or directly enabled
-------------------------------------------------------------------------------
function enable_snf (sf_enabled         : integer;
                     axi_data_width     : integer;
                     axis_tdata_width   : integer)
    return integer is
    begin
        -- If store and forward enable or data widths do not
        -- match then return 1 to enable snf
        if( (sf_enabled = 1) or (axi_data_width /= axis_tdata_width))then
            return 1;
        else
-- coverage off
            return 0;
-- coverage on
        end if;
    end;

end package body axi_dma_pkg;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:          axi_dma_reset.vhd
-- Description: This entity encompasses the reset logic (soft and hard) for
--              distribution to the axi_vdma core.
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library lib_cdc_v1_0_2;
library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;


-------------------------------------------------------------------------------
entity  axi_dma_reset is
    generic(
        C_INCLUDE_SG                : integer range 0 to 1          := 1;
            -- Include or Exclude the Scatter Gather Engine
            -- 0 = Exclude SG Engine - Enables Simple DMA Mode
            -- 1 = Include SG Engine - Enables Scatter Gather Mode

        C_SG_INCLUDE_STSCNTRL_STRM  : integer range 0 to 1          := 1;
            -- Include or Exclude AXI Status and AXI Control Streams
            -- 0 = Exclude Status and Control Streams
            -- 1 = Include Status and Control Streams

        C_PRMRY_IS_ACLK_ASYNC           : integer range 0 to 1 := 0;
            -- Primary MM2S/S2MM sync/async mode
            -- 0 = synchronous mode     - all clocks are synchronous
            -- 1 = asynchronous mode    - Primary data path channels (MM2S and S2MM)
            --                            run asynchronous to AXI Lite, DMA Control,
            --                            and SG.
        C_AXI_PRMRY_ACLK_FREQ_HZ        : integer := 100000000;
            -- Primary clock frequency in hertz

        C_AXI_SCNDRY_ACLK_FREQ_HZ       : integer := 100000000
            -- Secondary clock frequency in hertz
    );
    port (
        -- Clock Sources
        m_axi_sg_aclk               : in  std_logic                         ;              --
        axi_prmry_aclk              : in  std_logic                         ;              --
                                                                                           --
        -- Hard Reset                                                                      --
        axi_resetn                  : in  std_logic                         ;              --
                                                                                           --
        -- Soft Reset                                                                      --
        soft_reset                  : in  std_logic                         ;              --
        soft_reset_clr              : out std_logic  := '0'                 ;              --
        soft_reset_done             : in  std_logic                         ;              --
                                                                                           --
                                                                                           --
        all_idle                    : in  std_logic                         ;              --
        stop                        : in  std_logic                         ;              --
        halt                        : out std_logic := '0'                  ;              --
        halt_cmplt                  : in  std_logic                         ;              --
                                                                                           --
        -- Secondary Reset                                                                 --
        scndry_resetn               : out std_logic := '1'                  ;              --
        -- AXI Upsizer and Line Buffer                                                     --
        prmry_resetn                : out std_logic := '0'                  ;              --
        -- AXI DataMover Primary Reset (Raw)                                               --
        dm_prmry_resetn             : out std_logic := '1'                  ;              --
        -- AXI DataMover Secondary Reset (Raw)                                             --
        dm_scndry_resetn            : out std_logic := '1'                  ;              --
        -- AXI Primary Stream Reset Outputs                                                --
        prmry_reset_out_n           : out std_logic := '1'                  ;              --
        -- AXI Alternat Stream Reset Outputs                                               --
        altrnt_reset_out_n          : out std_logic := '1'                                 --
    );

-- Register duplication attribute assignments to control fanout
-- on handshake output signals

Attribute KEEP : string; -- declaration
Attribute EQUIVALENT_REGISTER_REMOVAL : string; -- declaration

Attribute KEEP of scndry_resetn                            : signal is "TRUE";
Attribute KEEP of prmry_resetn                             : signal is "TRUE";
Attribute KEEP of dm_scndry_resetn                         : signal is "TRUE";
Attribute KEEP of dm_prmry_resetn                          : signal is "TRUE";
Attribute KEEP of prmry_reset_out_n                        : signal is "TRUE";
Attribute KEEP of altrnt_reset_out_n                       : signal is "TRUE";

Attribute EQUIVALENT_REGISTER_REMOVAL of scndry_resetn     : signal is "no";
Attribute EQUIVALENT_REGISTER_REMOVAL of prmry_resetn      : signal is "no";
Attribute EQUIVALENT_REGISTER_REMOVAL of dm_scndry_resetn  : signal is "no";
Attribute EQUIVALENT_REGISTER_REMOVAL of dm_prmry_resetn   : signal is "no";
Attribute EQUIVALENT_REGISTER_REMOVAL of prmry_reset_out_n : signal is "no";
Attribute EQUIVALENT_REGISTER_REMOVAL of altrnt_reset_out_n: signal is "no";

end axi_dma_reset;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_dma_reset is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  ATTRIBUTE async_reg                      : STRING;

-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------

-- No Constants Declared

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
-- Soft Reset Support
signal s_soft_reset_i               : std_logic := '0';
signal s_soft_reset_i_d1            : std_logic := '0';
signal s_soft_reset_i_re            : std_logic := '0';
signal assert_sftrst_d1             : std_logic := '0';
signal min_assert_sftrst            : std_logic := '0';
signal min_assert_sftrst_d1_cdc_tig         : std_logic := '0';
  --ATTRIBUTE async_reg OF min_assert_sftrst_d1_cdc_tig  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF min_assert_sftrst  : SIGNAL IS "true";

signal p_min_assert_sftrst          : std_logic := '0';
signal sft_rst_dly1                 : std_logic := '0';
signal sft_rst_dly2                 : std_logic := '0';
signal sft_rst_dly3                 : std_logic := '0';
signal sft_rst_dly4                 : std_logic := '0';
signal sft_rst_dly5                 : std_logic := '0';
signal sft_rst_dly6                 : std_logic := '0';
signal sft_rst_dly7                 : std_logic := '0';
signal sft_rst_dly8                 : std_logic := '0';
signal sft_rst_dly9                 : std_logic := '0';
signal sft_rst_dly10                : std_logic := '0';
signal sft_rst_dly11                : std_logic := '0';
signal sft_rst_dly12                : std_logic := '0';
signal sft_rst_dly13                : std_logic := '0';
signal sft_rst_dly14                : std_logic := '0';
signal sft_rst_dly15                : std_logic := '0';
signal sft_rst_dly16                : std_logic := '0';
signal soft_reset_d1                : std_logic := '0';
signal soft_reset_re                : std_logic := '0';

-- Soft Reset to Primary clock domain signals
signal p_soft_reset                 : std_logic := '0';
signal p_soft_reset_d1_cdc_tig              : std_logic := '0';
signal p_soft_reset_d2              : std_logic := '0';

  --ATTRIBUTE async_reg OF p_soft_reset_d1_cdc_tig  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF p_soft_reset_d2  : SIGNAL IS "true";
signal p_soft_reset_d3              : std_logic := '0';
signal p_soft_reset_re              : std_logic := '0';

-- Qualified soft reset in primary clock domain for
-- generating mimimum reset pulse for soft reset
signal p_soft_reset_i               : std_logic := '0';
signal p_soft_reset_i_d1            : std_logic := '0';
signal p_soft_reset_i_re            : std_logic := '0';


-- Graceful halt control
signal halt_cmplt_d1_cdc_tig                : std_logic := '0';
signal s_halt_cmplt                 : std_logic := '0';

  --ATTRIBUTE async_reg OF halt_cmplt_d1_cdc_tig  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF s_halt_cmplt  : SIGNAL IS "true";
signal p_halt_d1_cdc_tig                    : std_logic := '0';
signal p_halt                       : std_logic := '0';

  --ATTRIBUTE async_reg OF p_halt_d1_cdc_tig  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF p_halt  : SIGNAL IS "true";
signal s_halt                       : std_logic := '0';

-- composite reset (hard and soft)
signal resetn_i                     : std_logic := '1';
signal scndry_resetn_i              : std_logic := '1';
signal axi_resetn_d1_cdc_tig                : std_logic := '1';
signal axi_resetn_d2                : std_logic := '1';

  --ATTRIBUTE async_reg OF axi_resetn_d1_cdc_tig  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF axi_resetn_d2 : SIGNAL IS "true";

signal halt_i                       : std_logic := '0';

signal p_all_idle                   : std_logic := '1';
signal p_all_idle_d1_cdc_tig                : std_logic := '1';

signal halt_cmplt_reg : std_logic;

-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin

-------------------------------------------------------------------------------
-- Internal Hard Reset
-- Generate reset on hardware reset or soft reset
-------------------------------------------------------------------------------
resetn_i    <= '0' when s_soft_reset_i = '1'
                     or min_assert_sftrst = '1'
                     or axi_resetn = '0'
          else '1';

-------------------------------------------------------------------------------
-- Minimum Reset Logic for Soft Reset
-------------------------------------------------------------------------------
-- Register to generate rising edge on soft reset and falling edge
-- on reset assertion.
REG_SFTRST_FOR_RE : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            s_soft_reset_i_d1 <= s_soft_reset_i;
            assert_sftrst_d1  <= min_assert_sftrst;

            -- Register soft reset from DMACR to create
            -- rising edge pulse
            soft_reset_d1     <= soft_reset;

        end if;
    end process REG_SFTRST_FOR_RE;

-- rising edge pulse on internal soft reset
s_soft_reset_i_re <=  s_soft_reset_i and not s_soft_reset_i_d1;

-- CR605883
-- rising edge pulse on DMACR soft reset
REG_SOFT_RE : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            soft_reset_re   <= soft_reset and not soft_reset_d1;
        end if;
    end process REG_SOFT_RE;

-- falling edge detection on min soft rst to clear soft reset
-- bit in register module
soft_reset_clr <= (not min_assert_sftrst and assert_sftrst_d1)
                    or (not axi_resetn);


-------------------------------------------------------------------------------
-- Generate Reset for synchronous configuration
-------------------------------------------------------------------------------
GNE_SYNC_RESET : if C_PRMRY_IS_ACLK_ASYNC = 0 generate
begin

    -- On start of soft reset shift pulse through to assert
    -- 7 clock later.  Used to set minimum 8clk assertion of
    -- reset.  Shift starts when all is idle and internal reset
    -- is asserted.
    MIN_PULSE_GEN : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(s_soft_reset_i_re = '1')then
                    sft_rst_dly1    <= '1';
                    sft_rst_dly2    <= '0';
                    sft_rst_dly3    <= '0';
                    sft_rst_dly4    <= '0';
                    sft_rst_dly5    <= '0';
                    sft_rst_dly6    <= '0';
                    sft_rst_dly7    <= '0';
                elsif(all_idle = '1')then
                    sft_rst_dly1    <= '0';
                    sft_rst_dly2    <= sft_rst_dly1;
                    sft_rst_dly3    <= sft_rst_dly2;
                    sft_rst_dly4    <= sft_rst_dly3;
                    sft_rst_dly5    <= sft_rst_dly4;
                    sft_rst_dly6    <= sft_rst_dly5;
                    sft_rst_dly7    <= sft_rst_dly6;
                end if;
            end if;
        end process MIN_PULSE_GEN;

    -- Drive minimum reset assertion for 8 clocks.
    MIN_RESET_ASSERTION : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then

                if(s_soft_reset_i_re = '1')then
                    min_assert_sftrst <= '1';
                elsif(sft_rst_dly7 = '1')then
                    min_assert_sftrst <= '0';
                end if;
            end if;
        end process MIN_RESET_ASSERTION;

    -------------------------------------------------------------------------------
    -- Soft Reset Support
    -------------------------------------------------------------------------------
    -- Generate reset on hardware reset or soft reset if system is idle
    -- On soft reset or error
    -- mm2s dma controller will idle immediatly
    -- sg fetch engine will complete current task and idle (desc's will flush)
    -- sg update engine will update all completed descriptors then idle
    REG_SOFT_RESET : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(soft_reset = '1'
                and all_idle = '1' and halt_cmplt = '1')then
                    s_soft_reset_i <= '1';

                elsif(soft_reset_done = '1')then
                    s_soft_reset_i <= '0';

                end if;
            end if;
        end process REG_SOFT_RESET;

    -- Halt datamover on soft_reset or on error.  Halt will stay
    -- asserted until s_soft_reset_i assertion which occurs when
    -- halt is complete or hard reset
    REG_DM_HALT : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(resetn_i = '0')then
                    halt_i <= '0';
                elsif(soft_reset_re = '1' or stop = '1')then
                    halt_i <= '1';
                end if;
            end if;
        end process REG_DM_HALT;

    halt <= halt_i;

    -- AXI Stream reset output
    REG_STRM_RESET_OUT : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                prmry_reset_out_n   <= resetn_i and not s_soft_reset_i;
            end if;
        end process REG_STRM_RESET_OUT;

    -- If in Scatter Gather mode and status control stream included
    GEN_ALT_RESET_OUT : if C_INCLUDE_SG = 1 and C_SG_INCLUDE_STSCNTRL_STRM = 1 generate
    begin
        -- AXI Stream reset output
        REG_ALT_RESET_OUT : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    altrnt_reset_out_n  <= resetn_i and not s_soft_reset_i;
                end if;
            end process REG_ALT_RESET_OUT;
    end generate GEN_ALT_RESET_OUT;

    -- If in Simple mode or status control stream excluded
    GEN_NO_ALT_RESET_OUT : if C_INCLUDE_SG = 0 or C_SG_INCLUDE_STSCNTRL_STRM = 0 generate
    begin
        altrnt_reset_out_n <= '1';
    end generate GEN_NO_ALT_RESET_OUT;

    -- Registered primary and secondary resets out
    REG_RESET_OUT : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                prmry_resetn <= resetn_i;
                scndry_resetn <= resetn_i;
            end if;
        end process REG_RESET_OUT;

    -- AXI DataMover Primary Reset (Raw)
    dm_prmry_resetn  <= resetn_i;

    -- AXI DataMover Secondary Reset (Raw)
    dm_scndry_resetn <= resetn_i;

end generate GNE_SYNC_RESET;


-------------------------------------------------------------------------------
-- Generate Reset for asynchronous configuration
-------------------------------------------------------------------------------
GEN_ASYNC_RESET : if C_PRMRY_IS_ACLK_ASYNC = 1 generate
begin

    -- Primary clock is slower or equal to secondary therefore...
    -- For Halt - can simply pass secondary clock version of soft reset
    -- rising edge into p_halt assertion
    -- For Min Rst Assertion - can simply use secondary logic version of min pulse genator
    GEN_PRMRY_GRTR_EQL_SCNDRY : if C_AXI_PRMRY_ACLK_FREQ_HZ >= C_AXI_SCNDRY_ACLK_FREQ_HZ generate
    begin

        -- CR605883 - Register to provide pure register output for synchronizer
        REG_HALT_CONDITIONS : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    s_halt <= soft_reset_re or stop;
                end if;
            end process REG_HALT_CONDITIONS;

        -- Halt data mover on soft reset assertion, error (i.e. stop=1) or
        -- not running
HALT_PROCESS : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => s_halt,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => axi_prmry_aclk,
        scndry_resetn              => '0',
        scndry_out                 => p_halt,
        scndry_vect_out            => open
    );



--        HALT_PROCESS : process(axi_prmry_aclk)
--            begin
--                if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
--                    --p_halt_d1_cdc_tig <= soft_reset_re or stop;       -- CR605883
--                    p_halt_d1_cdc_tig <= s_halt;                        -- CR605883
--                    p_halt    <= p_halt_d1_cdc_tig;
--                end if;
--            end process HALT_PROCESS;

        -- On start of soft reset shift pulse through to assert
        -- 7 clock later.  Used to set minimum 8clk assertion of
        -- reset.  Shift starts when all is idle and internal reset
        -- is asserted.
        -- Adding 5 more flops to make up for 5 stages of Sync flops
        MIN_PULSE_GEN : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(s_soft_reset_i_re = '1')then
                        sft_rst_dly1    <= '1';
                        sft_rst_dly2    <= '0';
                        sft_rst_dly3    <= '0';
                        sft_rst_dly4    <= '0';
                        sft_rst_dly5    <= '0';
                        sft_rst_dly6    <= '0';
                        sft_rst_dly7    <= '0';
                        sft_rst_dly8    <= '0';
                        sft_rst_dly9    <= '0';
                        sft_rst_dly10   <= '0';
                        sft_rst_dly11   <= '0';
                        sft_rst_dly12   <= '0';
                        sft_rst_dly13   <= '0';
                        sft_rst_dly14   <= '0';
                        sft_rst_dly15   <= '0';
                        sft_rst_dly16   <= '0';
                    elsif(all_idle = '1')then
                        sft_rst_dly1    <= '0';
                        sft_rst_dly2    <= sft_rst_dly1;
                        sft_rst_dly3    <= sft_rst_dly2;
                        sft_rst_dly4    <= sft_rst_dly3;
                        sft_rst_dly5    <= sft_rst_dly4;
                        sft_rst_dly6    <= sft_rst_dly5;
                        sft_rst_dly7    <= sft_rst_dly6;
                        sft_rst_dly8    <= sft_rst_dly7;
                        sft_rst_dly9    <= sft_rst_dly8;
                        sft_rst_dly10   <= sft_rst_dly9;
                        sft_rst_dly11   <= sft_rst_dly10;
                        sft_rst_dly12   <= sft_rst_dly11;
                        sft_rst_dly13   <= sft_rst_dly12;
                        sft_rst_dly14   <= sft_rst_dly13;
                        sft_rst_dly15   <= sft_rst_dly14;
                        sft_rst_dly16   <= sft_rst_dly15;
                    end if;
                end if;
            end process MIN_PULSE_GEN;

        -- Drive minimum reset assertion for 8 clocks.
        MIN_RESET_ASSERTION : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then

                    if(s_soft_reset_i_re = '1')then
                        min_assert_sftrst <= '1';
                    elsif(sft_rst_dly16 = '1')then
                        min_assert_sftrst <= '0';
                    end if;
                end if;
            end process MIN_RESET_ASSERTION;

    end generate GEN_PRMRY_GRTR_EQL_SCNDRY;

    -- Primary clock is running slower than secondary therefore need to use a primary clock
    -- based rising edge version of soft_reset for primary halt assertion
    GEN_PRMRY_LESS_SCNDRY :  if C_AXI_PRMRY_ACLK_FREQ_HZ < C_AXI_SCNDRY_ACLK_FREQ_HZ generate
       signal soft_halt_int : std_logic := '0';
    begin

        -- Halt data mover on soft reset assertion, error (i.e. stop=1) or
        -- not running
         soft_halt_int <= p_soft_reset_re or stop;

HALT_PROCESS : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => soft_halt_int,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => axi_prmry_aclk,
        scndry_resetn              => '0',
        scndry_out                 => p_halt,
        scndry_vect_out            => open
    );

--        HALT_PROCESS : process(axi_prmry_aclk)
--            begin
--                if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
--                    p_halt_d1_cdc_tig <= p_soft_reset_re or stop;
--                    p_halt    <= p_halt_d1_cdc_tig;
--                end if;
--            end process HALT_PROCESS;


REG_IDLE2PRMRY : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => all_idle,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => axi_prmry_aclk,
        scndry_resetn              => '0',
        scndry_out                 => p_all_idle,
        scndry_vect_out            => open
    );

--        REG_IDLE2PRMRY : process(axi_prmry_aclk)
--            begin
--                if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
--                    p_all_idle_d1_cdc_tig   <= all_idle;
--                    p_all_idle      <= p_all_idle_d1_cdc_tig;
--                end if;
--            end process REG_IDLE2PRMRY;


        -- On start of soft reset shift pulse through to assert
        -- 7 clock later.  Used to set minimum 8clk assertion of
        -- reset.  Shift starts when all is idle and internal reset
        -- is asserted.
        MIN_PULSE_GEN : process(axi_prmry_aclk)
            begin
                if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
                    -- CR574188 - fixes issue with soft reset terminating too early
                    -- for primary slower than secondary clock
                    --if(p_soft_reset_re = '1')then
                    if(p_soft_reset_i_re = '1')then
                        sft_rst_dly1    <= '1';
                        sft_rst_dly2    <= '0';
                        sft_rst_dly3    <= '0';
                        sft_rst_dly4    <= '0';
                        sft_rst_dly5    <= '0';
                        sft_rst_dly6    <= '0';
                        sft_rst_dly7    <= '0';
                        sft_rst_dly8    <= '0';
                        sft_rst_dly9    <= '0';
                        sft_rst_dly10   <= '0';
                        sft_rst_dly11   <= '0';
                        sft_rst_dly12   <= '0';
                        sft_rst_dly13   <= '0';
                        sft_rst_dly14   <= '0';
                        sft_rst_dly15   <= '0';
                        sft_rst_dly16   <= '0';
                    elsif(p_all_idle = '1')then
                        sft_rst_dly1    <= '0';
                        sft_rst_dly2    <= sft_rst_dly1;
                        sft_rst_dly3    <= sft_rst_dly2;
                        sft_rst_dly4    <= sft_rst_dly3;
                        sft_rst_dly5    <= sft_rst_dly4;
                        sft_rst_dly6    <= sft_rst_dly5;
                        sft_rst_dly7    <= sft_rst_dly6;
                        sft_rst_dly8    <= sft_rst_dly7;
                        sft_rst_dly9    <= sft_rst_dly8;
                        sft_rst_dly10   <= sft_rst_dly9;
                        sft_rst_dly11   <= sft_rst_dly10;
                        sft_rst_dly12   <= sft_rst_dly11;
                        sft_rst_dly13   <= sft_rst_dly12;
                        sft_rst_dly14   <= sft_rst_dly13;
                        sft_rst_dly15   <= sft_rst_dly14;
                        sft_rst_dly16   <= sft_rst_dly15;
                    end if;
                end if;
            end process MIN_PULSE_GEN;

        -- Drive minimum reset assertion for 8 primary clocks.
        MIN_RESET_ASSERTION : process(axi_prmry_aclk)
            begin
                if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then

                    -- CR574188 - fixes issue with soft reset terminating too early
                    -- for primary slower than secondary clock
                    --if(p_soft_reset_re = '1')then
                    if(p_soft_reset_i_re = '1')then
                        p_min_assert_sftrst <= '1';
                    elsif(sft_rst_dly16 = '1')then
                        p_min_assert_sftrst <= '0';
                    end if;
                end if;
            end process MIN_RESET_ASSERTION;

        -- register minimum reset pulse back to secondary domain

REG_MINRST2SCNDRY : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => p_min_assert_sftrst,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => m_axi_sg_aclk,
        scndry_resetn              => '0',
        scndry_out                 => min_assert_sftrst,
        scndry_vect_out            => open
    );

--        REG_MINRST2SCNDRY : process(m_axi_sg_aclk)
--        begin
--            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
--                min_assert_sftrst_d1_cdc_tig <= p_min_assert_sftrst;
--                min_assert_sftrst    <= min_assert_sftrst_d1_cdc_tig;
--            end if;
--        end process REG_MINRST2SCNDRY;

        -- CR574188 - fixes issue with soft reset terminating too early
        -- for primary slower than secondary clock
        -- Generate reset on hardware reset or soft reset if system is idle
        REG_P_SOFT_RESET : process(axi_prmry_aclk)
            begin
                if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
                    if(p_soft_reset = '1'
                    and p_all_idle = '1'
                    and halt_cmplt = '1')then
                        p_soft_reset_i <= '1';
                    else
                        p_soft_reset_i <= '0';
                    end if;
                end if;
            end process REG_P_SOFT_RESET;

        -- CR574188 - fixes issue with soft reset terminating too early
        -- for primary slower than secondary clock
        -- Register qualified soft reset flag for generating rising edge
        -- pulse for starting minimum reset pulse
        REG_SOFT2PRMRY : process(axi_prmry_aclk)
            begin
                if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
                    p_soft_reset_i_d1 <=  p_soft_reset_i;
                end if;
            end process REG_SOFT2PRMRY;

        -- CR574188 - fixes issue with soft reset terminating too early
        -- for primary slower than secondary clock
        -- Generate rising edge pulse on qualified soft reset for min pulse
        -- logic.
        p_soft_reset_i_re <= p_soft_reset_i and not p_soft_reset_i_d1;

    end generate GEN_PRMRY_LESS_SCNDRY;

    -- Double register halt complete flag from primary to secondary
    -- clock domain.
    -- Note: halt complete stays asserted until halt clears therefore
    -- only need to double register from fast to slow clock domain.

process(axi_prmry_aclk)
begin
     if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
         halt_cmplt_reg <=  halt_cmplt;
     end if;
end process;

REG_HALT_CMPLT_IN : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => halt_cmplt_reg,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => m_axi_sg_aclk,
        scndry_resetn              => '0',
        scndry_out                 => s_halt_cmplt,
        scndry_vect_out            => open
    );

--    REG_HALT_CMPLT_IN : process(m_axi_sg_aclk)
--        begin
--            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
--
--                halt_cmplt_d1_cdc_tig   <= halt_cmplt;
--                s_halt_cmplt    <= halt_cmplt_d1_cdc_tig;
--            end if;
--        end process REG_HALT_CMPLT_IN;

    -------------------------------------------------------------------------------
    -- Soft Reset Support
    -------------------------------------------------------------------------------
    -- Generate reset on hardware reset or soft reset if system is idle
    REG_SOFT_RESET : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(soft_reset = '1'
                and all_idle = '1'
                and s_halt_cmplt = '1')then
                    s_soft_reset_i <= '1';
                elsif(soft_reset_done = '1')then
                    s_soft_reset_i <= '0';
                end if;
            end if;
        end process REG_SOFT_RESET;

    -- Register soft reset flag into primary domain to correcly
    -- halt data mover

REG_SOFT2PRMRY : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => soft_reset,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => axi_prmry_aclk,
        scndry_resetn              => '0',
        scndry_out                 => p_soft_reset_d2,
        scndry_vect_out            => open
    );


    REG_SOFT2PRMRY1 : process(axi_prmry_aclk)
        begin
            if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
--                p_soft_reset_d1_cdc_tig <=  soft_reset;
--                p_soft_reset_d2 <=  p_soft_reset_d1_cdc_tig;
                p_soft_reset_d3 <=  p_soft_reset_d2;

            end if;
        end process REG_SOFT2PRMRY1;


    -- Generate rising edge pulse for use with p_halt creation
    p_soft_reset_re <= p_soft_reset_d2 and not p_soft_reset_d3;

    -- used to mask halt reset below
    p_soft_reset    <= p_soft_reset_d2;

    -- Halt datamover on soft_reset or on error.  Halt will stay
    -- asserted until s_soft_reset_i assertion which occurs when
    -- halt is complete or hard reset
    REG_DM_HALT : process(axi_prmry_aclk)
        begin
            if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
                if(axi_resetn_d2 = '0')then
                    halt_i <= '0';
                elsif(p_halt = '1')then
                    halt_i <= '1';
                end if;
            end if;
        end process REG_DM_HALT;

    halt <= halt_i;

    -- CR605883 (CDC) Create pure register out for synchronizer
    REG_CMB_RESET : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                scndry_resetn_i <= resetn_i;
            end if;
        end process REG_CMB_RESET;

    -- Sync to mm2s primary and register resets out

REG_RESET_OUT : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => scndry_resetn_i,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => axi_prmry_aclk,
        scndry_resetn              => '0',
        scndry_out                 => axi_resetn_d2,
        scndry_vect_out            => open
    );

--    REG_RESET_OUT : process(axi_prmry_aclk)
--        begin
--            if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
--                --axi_resetn_d1_cdc_tig  <= resetn_i;   -- CR605883
--                axi_resetn_d1_cdc_tig  <= scndry_resetn_i;
--                axi_resetn_d2  <= axi_resetn_d1_cdc_tig;
--            end if;
--        end process REG_RESET_OUT;

    -- Register resets out to AXI DMA Logic
    REG_SRESET_OUT : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                scndry_resetn <= resetn_i;
            end if;
        end process REG_SRESET_OUT;


    -- AXI Stream reset output
    prmry_reset_out_n   <= axi_resetn_d2;

    -- If in Scatter Gather mode and status control stream included
    GEN_ALT_RESET_OUT : if C_INCLUDE_SG = 1 and C_SG_INCLUDE_STSCNTRL_STRM = 1 generate
    begin
        -- AXI Stream alternate reset output
        altrnt_reset_out_n  <= axi_resetn_d2;
    end generate GEN_ALT_RESET_OUT;

    -- If in Simple Mode or status control stream excluded.
    GEN_NO_ALT_RESET_OUT : if C_INCLUDE_SG = 0 or C_SG_INCLUDE_STSCNTRL_STRM = 0 generate
    begin
        altrnt_reset_out_n  <= '1';
    end generate GEN_NO_ALT_RESET_OUT;

    -- Register primary reset
    prmry_resetn        <= axi_resetn_d2;

    -- AXI DataMover Primary Reset
    dm_prmry_resetn     <= axi_resetn_d2;

    -- AXI DataMover Secondary Reset
    dm_scndry_resetn    <= resetn_i;

end generate GEN_ASYNC_RESET;


end implementation;



--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:          axi_dma_rst_module.vhd
-- Description: This entity is the top level reset module entity for the
--              AXI VDMA core.
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;

library lib_cdc_v1_0_2;


-------------------------------------------------------------------------------
entity  axi_dma_rst_module is
    generic(
        C_INCLUDE_MM2S                  : integer range 0 to 1      := 1;
            -- Include or exclude MM2S primary data path
            -- 0 = Exclude MM2S primary data path
            -- 1 = Include MM2S primary data path
        C_INCLUDE_S2MM                  : integer range 0 to 1      := 1;
            -- Include or exclude S2MM primary data path
            -- 0 = Exclude S2MM primary data path
            -- 1 = Include S2MM primary data path

        C_INCLUDE_SG                : integer range 0 to 1          := 1;
            -- Include or Exclude the Scatter Gather Engine
            -- 0 = Exclude SG Engine - Enables Simple DMA Mode
            -- 1 = Include SG Engine - Enables Scatter Gather Mode

        C_SG_INCLUDE_STSCNTRL_STRM  : integer range 0 to 1          := 1;
            -- Include or Exclude AXI Status and AXI Control Streams
            -- 0 = Exclude Status and Control Streams
            -- 1 = Include Status and Control Streams

        C_PRMRY_IS_ACLK_ASYNC       : integer range 0 to 1          := 0;
            -- Primary MM2S/S2MM sync/async mode
            -- 0 = synchronous mode     - all clocks are synchronous
            -- 1 = asynchronous mode    - Primary data path channels (MM2S and S2MM)
            --                            run asynchronous to AXI Lite, DMA Control,
            --                            and SG.

        C_M_AXI_MM2S_ACLK_FREQ_HZ        : integer := 100000000;
            -- Primary clock frequency in hertz

        C_M_AXI_S2MM_ACLK_FREQ_HZ        : integer := 100000000;
            -- Primary clock frequency in hertz

        C_M_AXI_SG_ACLK_FREQ_HZ          : integer := 100000000
            -- Scatter Gather clock frequency in hertz




    );
    port (
        -----------------------------------------------------------------------
        -- Clock Sources
        -----------------------------------------------------------------------
        s_axi_lite_aclk             : in  std_logic                         ;
        m_axi_sg_aclk               : in  std_logic                         ;           --
        m_axi_mm2s_aclk             : in  std_logic                         ;           --
        m_axi_s2mm_aclk             : in  std_logic                         ;           --
                                                                                        --
        -----------------------------------------------------------------------         --
        -- Hard Reset                                                                   --
        -----------------------------------------------------------------------         --
        axi_resetn                  : in  std_logic                         ;           --

        -----------------------------------------------------------------------         --
        -- Soft Reset                                                                   --
        -----------------------------------------------------------------------         --
        soft_reset                  : in  std_logic                         ;           --
        soft_reset_clr              : out std_logic := '0'                  ;           --
                                                                                        --
        -----------------------------------------------------------------------         --
        -- MM2S Soft Reset Support                                                      --
        -----------------------------------------------------------------------         --
        mm2s_all_idle               : in  std_logic                         ;           --
        mm2s_stop                   : in  std_logic                         ;           --
        mm2s_halt                   : out std_logic := '0'                  ;           --
        mm2s_halt_cmplt             : in  std_logic                         ;           --
                                                                                        --
        -----------------------------------------------------------------------         --
        -- S2MM Soft Reset Support                                                      --
        -----------------------------------------------------------------------         --
        s2mm_all_idle               : in  std_logic                         ;           --
        s2mm_stop                   : in  std_logic                         ;           --
        s2mm_halt                   : out std_logic := '0'                  ;           --
        s2mm_halt_cmplt             : in  std_logic                         ;           --
                                                                                        --
        -----------------------------------------------------------------------         --
        -- MM2S Distributed Reset Out                                                   --
        -----------------------------------------------------------------------         --
        -- AXI DataMover Primary Reset (Raw)                                            --
        dm_mm2s_prmry_resetn        : out std_logic := '1'                  ;           --
        -- AXI DataMover Secondary Reset (Raw)                                          --
        dm_mm2s_scndry_resetn       : out std_logic := '1'                  ;
        -- AXI Stream Primary Reset Outputs                                             --
        mm2s_prmry_reset_out_n      : out std_logic := '1'                  ;           --
        -- AXI Stream Control Reset Outputs                                             --
        mm2s_cntrl_reset_out_n      : out std_logic := '1'                  ;           --
        -- AXI Secondary reset
        mm2s_scndry_resetn          : out std_logic := '1'                  ;           --
        -- AXI Upsizer and Line Buffer                                                  --
        mm2s_prmry_resetn           : out std_logic := '1'                  ;           --
                                                                                        --
                                                                                        --
        -----------------------------------------------------------------------         --
        -- S2MM Distributed Reset Out                                                   --
        -----------------------------------------------------------------------         --
        -- AXI DataMover Primary Reset (Raw)                                            --
        dm_s2mm_prmry_resetn        : out std_logic := '1'                  ;           --
        -- AXI DataMover Secondary Reset (Raw)                                          --
        dm_s2mm_scndry_resetn       : out std_logic := '1'                  ;
        -- AXI Stream Primary Reset Outputs                                             --
        s2mm_prmry_reset_out_n      : out std_logic := '1'                  ;           --
        -- AXI Stream Control Reset Outputs                                             --
        s2mm_sts_reset_out_n        : out std_logic := '1'                  ;           --
        -- AXI Secondary reset
        s2mm_scndry_resetn          : out std_logic := '1'                  ;           --
        -- AXI Upsizer and Line Buffer                                                  --
        s2mm_prmry_resetn           : out std_logic := '1'                  ;           --

        -----------------------------------------------------------------------         --
        -- Scatter Gather Distributed Reset Out
        -----------------------------------------------------------------------         --
        -- AXI Scatter Gather Reset Out
        m_axi_sg_aresetn            : out std_logic := '1'                  ;           --
        -- AXI Scatter Gather Datamover Reset Out
        dm_m_axi_sg_aresetn         : out std_logic := '1'                  ;           --


        -----------------------------------------------------------------------         --
        -- Hard Reset Out                                                               --
        -----------------------------------------------------------------------         --
        m_axi_sg_hrdresetn          : out std_logic := '1'                  ;           --
        s_axi_lite_resetn           : out std_logic := '1'                              --
    );


Attribute KEEP : string; -- declaration
Attribute EQUIVALENT_REGISTER_REMOVAL : string; -- declaration

Attribute KEEP of s_axi_lite_resetn                                 : signal is "TRUE";
Attribute KEEP of m_axi_sg_hrdresetn                                : signal is "TRUE";

Attribute EQUIVALENT_REGISTER_REMOVAL of s_axi_lite_resetn          : signal is "no";
Attribute EQUIVALENT_REGISTER_REMOVAL of m_axi_sg_hrdresetn         : signal is "no";

end axi_dma_rst_module;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_dma_rst_module is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";






-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------

-- No Constants Declared

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
  ATTRIBUTE async_reg                      : STRING;


signal hrd_resetn_i_cdc_tig                     : std_logic := '1';
signal hrd_resetn_i_d1_cdc_tig                  : std_logic := '1';
  --ATTRIBUTE async_reg OF hrd_resetn_i_cdc_tig  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF hrd_resetn_i_d1_cdc_tig : SIGNAL IS "true";

-- Soft reset support
signal mm2s_soft_reset_clr              : std_logic := '0';
signal s2mm_soft_reset_clr              : std_logic := '0';
signal soft_reset_clr_i                 : std_logic := '0';
signal mm2s_soft_reset_done             : std_logic := '0';
signal s2mm_soft_reset_done             : std_logic := '0';

signal mm2s_scndry_resetn_i             : std_logic := '0';
signal s2mm_scndry_resetn_i             : std_logic := '0';

signal dm_mm2s_scndry_resetn_i          : std_logic := '0';
signal dm_s2mm_scndry_resetn_i          : std_logic := '0';

signal sg_hard_reset                    : std_logic := '0';

-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin
-- Register hard reset in

REG_HRD_RST : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => axi_resetn,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => m_axi_sg_aclk,
        scndry_resetn              => '0',
        scndry_out                 => sg_hard_reset, 
        scndry_vect_out            => open
    );

m_axi_sg_hrdresetn <= sg_hard_reset;

--REG_HRD_RST : process(m_axi_sg_aclk)
--    begin
--        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
--            hrd_resetn_i_cdc_tig        <= axi_resetn;
--            m_axi_sg_hrdresetn  <= hrd_resetn_i_cdc_tig;
--        end if;
--    end process REG_HRD_RST;

-- Regsiter hard reset out for axi lite interface

REG_HRD_RST_OUT : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => axi_resetn,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => s_axi_lite_aclk,
        scndry_resetn              => '0',
        scndry_out                 => s_axi_lite_resetn,
        scndry_vect_out            => open
    );


--REG_HRD_RST_OUT : process(s_axi_lite_aclk)
--    begin
--        if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
--            hrd_resetn_i_d1_cdc_tig     <= hrd_resetn_i_cdc_tig;
--            s_axi_lite_resetn   <= hrd_resetn_i_d1_cdc_tig;
--        end if;
--    end process REG_HRD_RST_OUT;

dm_mm2s_scndry_resetn <= dm_mm2s_scndry_resetn_i;
dm_s2mm_scndry_resetn <= dm_s2mm_scndry_resetn_i;


-- mm2s channel included therefore map secondary resets to
-- from mm2s reset module to scatter gather interface (default)
MAP_SG_FOR_BOTH : if C_INCLUDE_MM2S = 1  and C_INCLUDE_S2MM = 1 generate
begin

    -- both must be low before sg reset is asserted.
    m_axi_sg_aresetn       <= mm2s_scndry_resetn_i or s2mm_scndry_resetn_i;
    dm_m_axi_sg_aresetn    <= dm_mm2s_scndry_resetn_i or dm_s2mm_scndry_resetn_i;

end generate MAP_SG_FOR_BOTH;

-- Only s2mm channel included therefore map secondary resets to
-- from s2mm reset module to scatter gather interface
MAP_SG_FOR_S2MM : if C_INCLUDE_MM2S = 0 and C_INCLUDE_S2MM = 1 generate
begin

    m_axi_sg_aresetn       <= s2mm_scndry_resetn_i;
    dm_m_axi_sg_aresetn    <= dm_s2mm_scndry_resetn_i;

end generate MAP_SG_FOR_S2MM;

-- Only mm2s channel included therefore map secondary resets to
-- from mm2s reset module to scatter gather interface
MAP_SG_FOR_MM2S : if C_INCLUDE_MM2S = 1 and C_INCLUDE_S2MM = 0 generate
begin

    m_axi_sg_aresetn       <= mm2s_scndry_resetn_i;
    dm_m_axi_sg_aresetn    <= dm_mm2s_scndry_resetn_i;

end generate MAP_SG_FOR_MM2S;

-- Invalid configuration for axi dma - simply here for completeness
MAP_NO_SG : if C_INCLUDE_MM2S = 0 and C_INCLUDE_S2MM = 0 generate
begin

    m_axi_sg_aresetn       <= '1';
    dm_m_axi_sg_aresetn    <= '1';

end generate MAP_NO_SG;


s2mm_scndry_resetn <= s2mm_scndry_resetn_i;
mm2s_scndry_resetn <= mm2s_scndry_resetn_i;



-- Generate MM2S reset signals
GEN_RESET_FOR_MM2S : if C_INCLUDE_MM2S = 1 generate
begin
    RESET_I : entity  axi_dma_v7_1_13.axi_dma_reset
        generic map(
            C_PRMRY_IS_ACLK_ASYNC       => C_PRMRY_IS_ACLK_ASYNC        ,
            C_AXI_PRMRY_ACLK_FREQ_HZ    => C_M_AXI_MM2S_ACLK_FREQ_HZ    ,
            C_AXI_SCNDRY_ACLK_FREQ_HZ   => C_M_AXI_SG_ACLK_FREQ_HZ      ,
            C_SG_INCLUDE_STSCNTRL_STRM  => C_SG_INCLUDE_STSCNTRL_STRM   ,
            C_INCLUDE_SG                => C_INCLUDE_SG
        )
        port map(
            -- Clock Sources
            m_axi_sg_aclk               => m_axi_sg_aclk                ,
            axi_prmry_aclk              => m_axi_mm2s_aclk              ,

            -- Hard Reset
            axi_resetn                  => sg_hard_reset                 ,

            -- Soft Reset
            soft_reset                  => soft_reset                   ,
            soft_reset_clr              => mm2s_soft_reset_clr          ,
            soft_reset_done             => soft_reset_clr_i             ,

            all_idle                    => mm2s_all_idle                ,
            stop                        => mm2s_stop                    ,
            halt                        => mm2s_halt                    ,
            halt_cmplt                  => mm2s_halt_cmplt              ,


            -- Secondary Reset
            scndry_resetn               => mm2s_scndry_resetn_i         ,
            -- AXI Upsizer and Line Buffer
            prmry_resetn                => mm2s_prmry_resetn            ,
            -- AXI DataMover Primary Reset (Raw)
            dm_prmry_resetn             => dm_mm2s_prmry_resetn         ,
            -- AXI DataMover Secondary Reset (Raw)
            dm_scndry_resetn            => dm_mm2s_scndry_resetn_i      ,
            -- AXI Stream Primary Reset Outputs
            prmry_reset_out_n           => mm2s_prmry_reset_out_n       ,
            -- AXI Stream Alternate Reset Outputs
            altrnt_reset_out_n          => mm2s_cntrl_reset_out_n
        );


    -- Sample an hold mm2s soft reset done to use in
    -- combined reset done to DMACR
    MM2S_SOFT_RST_DONE : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(sg_hard_reset = '0' or soft_reset_clr_i = '1')then
                    mm2s_soft_reset_done <= '0';
                elsif(mm2s_soft_reset_clr = '1')then
                    mm2s_soft_reset_done <= '1';
                end if;
            end if;
        end process MM2S_SOFT_RST_DONE;

end generate GEN_RESET_FOR_MM2S;


-- No MM2S therefore tie off mm2s reset signals
GEN_NO_RESET_FOR_MM2S : if C_INCLUDE_MM2S = 0 generate
begin
    mm2s_prmry_reset_out_n  <= '1';
    mm2s_cntrl_reset_out_n  <= '1';
    dm_mm2s_scndry_resetn_i <= '1';
    dm_mm2s_prmry_resetn    <= '1';
    mm2s_prmry_resetn       <= '1';
    mm2s_scndry_resetn_i    <= '1';
    mm2s_halt               <= '0';
    mm2s_soft_reset_clr     <= '0';
    mm2s_soft_reset_done    <= '1';

end generate GEN_NO_RESET_FOR_MM2S;


-- Generate S2MM reset signals
GEN_RESET_FOR_S2MM : if C_INCLUDE_S2MM = 1 generate
begin
    RESET_I : entity  axi_dma_v7_1_13.axi_dma_reset
        generic map(
            C_PRMRY_IS_ACLK_ASYNC       => C_PRMRY_IS_ACLK_ASYNC        ,
            C_AXI_PRMRY_ACLK_FREQ_HZ    => C_M_AXI_S2MM_ACLK_FREQ_HZ    ,
            C_AXI_SCNDRY_ACLK_FREQ_HZ   => C_M_AXI_SG_ACLK_FREQ_HZ      ,
            C_SG_INCLUDE_STSCNTRL_STRM  => C_SG_INCLUDE_STSCNTRL_STRM   ,
            C_INCLUDE_SG                => C_INCLUDE_SG
        )
        port map(
            -- Clock Sources
            m_axi_sg_aclk               => m_axi_sg_aclk                ,
            axi_prmry_aclk              => m_axi_s2mm_aclk              ,

            -- Hard Reset
            axi_resetn                  => sg_hard_reset                 ,

            -- Soft Reset
            soft_reset                  => soft_reset                   ,
            soft_reset_clr              => s2mm_soft_reset_clr          ,
            soft_reset_done             => soft_reset_clr_i             ,

            all_idle                    => s2mm_all_idle                ,
            stop                        => s2mm_stop                    ,
            halt                        => s2mm_halt                    ,
            halt_cmplt                  => s2mm_halt_cmplt              ,


            -- Secondary Reset
            scndry_resetn               => s2mm_scndry_resetn_i         ,
            -- AXI Upsizer and Line Buffer
            prmry_resetn                => s2mm_prmry_resetn            ,
            -- AXI DataMover Primary Reset (Raw)
            dm_prmry_resetn             => dm_s2mm_prmry_resetn         ,
            -- AXI DataMover Secondary Reset (Raw)
            dm_scndry_resetn            => dm_s2mm_scndry_resetn_i      ,
            -- AXI Stream Primary Reset Outputs
            prmry_reset_out_n           => s2mm_prmry_reset_out_n       ,
            -- AXI Stream Alternate Reset Outputs
            altrnt_reset_out_n          => s2mm_sts_reset_out_n
        );

    -- Sample an hold s2mm soft reset done to use in
    -- combined reset done to DMACR
    S2MM_SOFT_RST_DONE : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(sg_hard_reset = '0' or soft_reset_clr_i = '1')then
                    s2mm_soft_reset_done <= '0';
                elsif(s2mm_soft_reset_clr = '1')then
                    s2mm_soft_reset_done <= '1';
                end if;
            end if;
        end process S2MM_SOFT_RST_DONE;

end generate GEN_RESET_FOR_S2MM;

-- No SsMM therefore tie off mm2s reset signals
GEN_NO_RESET_FOR_S2MM : if C_INCLUDE_S2MM = 0 generate
begin
    s2mm_prmry_reset_out_n  <= '1';
    dm_s2mm_scndry_resetn_i <= '1';
    dm_s2mm_prmry_resetn    <= '1';
    s2mm_prmry_resetn       <= '1';
    s2mm_scndry_resetn_i    <= '1';
    s2mm_halt               <= '0';
    s2mm_soft_reset_clr     <= '0';
    s2mm_soft_reset_done    <= '1';

end generate GEN_NO_RESET_FOR_S2MM;




-- When both mm2s and s2mm are done then drive soft reset clear and
-- also clear s_h registers above
soft_reset_clr_i    <= s2mm_soft_reset_done and mm2s_soft_reset_done;

soft_reset_clr      <= soft_reset_clr_i;


end implementation;



-------------------------------------------------------------------------------
-- axi_dma_lite_if
-------------------------------------------------------------------------------
--
-- *************************************************************************
--
-- (c) Copyright 2010, 2011 Xilinx, Inc. All rights reserved.
--
-- This file contains confidential and proprietary information
-- of Xilinx, Inc. and is protected under U.S. and
-- international copyright and other intellectual property
-- laws.
--
-- DISCLAIMER
-- This disclaimer is not a license and does not grant any
-- rights to the materials distributed herewith. Except as
-- otherwise provided in a valid license issued to you by
-- Xilinx, and to the maximum extent permitted by applicable
-- law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
-- WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
-- AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
-- BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
-- INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
-- (2) Xilinx shall not be liable (whether in contract or tort,
-- including negligence, or under any other theory of
-- liability) for any loss or damage of any kind or nature
-- related to, arising under or in connection with these
-- materials, including for any direct, or any indirect,
-- special, incidental, or consequential loss or damage
-- (including loss of data, profits, goodwill, or any type of
-- loss or damage suffered as a result of any action brought
-- by a third party) even if such damage or loss was
-- reasonably foreseeable or Xilinx had been advised of the
-- possibility of the same.
--
-- CRITICAL APPLICATIONS
-- Xilinx products are not designed or intended to be fail-
-- safe, or for use in any application requiring fail-safe
-- performance, such as life-support or safety devices or
-- systems, Class III medical devices, nuclear facilities,
-- applications related to the deployment of airbags, or any
-- other applications that could lead to death, personal
-- injury, or severe property or environmental damage
-- (individually and collectively, "Critical
-- Applications"). Customer assumes the sole risk and
-- liability of any use of Xilinx products in Critical
-- Applications, subject only to applicable laws and
-- regulations governing limitations on product liability.
--
-- THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
-- PART OF THIS FILE AT ALL TIMES.
--
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:          axi_dma_lite_if.vhd
-- Description: This entity is AXI Lite Interface Module for the AXI DMA
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;

library lib_pkg_v1_0_2;
library lib_cdc_v1_0_2;
use lib_pkg_v1_0_2.lib_pkg.clog2;

-------------------------------------------------------------------------------
entity  axi_dma_lite_if is
    generic(
        C_NUM_CE                    : integer                := 8           ;
        C_AXI_LITE_IS_ASYNC         : integer range 0 to 1   := 0           ;
        C_S_AXI_LITE_ADDR_WIDTH     : integer range 2 to 32 := 32          ;
        C_S_AXI_LITE_DATA_WIDTH     : integer range 32 to 32 := 32
    );
    port (
        -- Async clock input
        ip2axi_aclk                 : in  std_logic                         ;          --
        ip2axi_aresetn              : in  std_logic                         ;          --

        -----------------------------------------------------------------------
        -- AXI Lite Control Interface
        -----------------------------------------------------------------------
        s_axi_lite_aclk             : in  std_logic                         ;          --
        s_axi_lite_aresetn          : in  std_logic                         ;          --
                                                                                       --
        -- AXI Lite Write Address Channel                                              --
        s_axi_lite_awvalid          : in  std_logic                         ;          --
        s_axi_lite_awready          : out std_logic                         ;          --
        s_axi_lite_awaddr           : in  std_logic_vector                             --
                                        (C_S_AXI_LITE_ADDR_WIDTH-1 downto 0);          --
                                                                                       --
        -- AXI Lite Write Data Channel                                                 --
        s_axi_lite_wvalid           : in  std_logic                         ;          --
        s_axi_lite_wready           : out std_logic                         ;          --
        s_axi_lite_wdata            : in  std_logic_vector                             --
                                        (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);          --
                                                                                       --
        -- AXI Lite Write Response Channel                                             --
        s_axi_lite_bresp            : out std_logic_vector(1 downto 0)      ;          --
        s_axi_lite_bvalid           : out std_logic                         ;          --
        s_axi_lite_bready           : in  std_logic                         ;          --
                                                                                       --
        -- AXI Lite Read Address Channel                                               --
        s_axi_lite_arvalid          : in  std_logic                         ;          --
        s_axi_lite_arready          : out std_logic                         ;          --
        s_axi_lite_araddr           : in  std_logic_vector                             --
                                        (C_S_AXI_LITE_ADDR_WIDTH-1 downto 0);          --
        s_axi_lite_rvalid           : out std_logic                         ;          --
        s_axi_lite_rready           : in  std_logic                         ;          --
        s_axi_lite_rdata            : out std_logic_vector                             --
                                        (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);          --
        s_axi_lite_rresp            : out std_logic_vector(1 downto 0)      ;          --
                                                                                       --
        -- User IP Interface                                                           --
        axi2ip_wrce                 : out std_logic_vector                             --
                                        (C_NUM_CE-1 downto 0)               ;          --
        axi2ip_wrdata               : out std_logic_vector                             --
                                        (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);          --
                                                                                       --
        axi2ip_rdce                 : out std_logic_vector                             --
                                        (C_NUM_CE-1 downto 0)               ;          --

        axi2ip_rdaddr               : out std_logic_vector                             --
                                        (C_S_AXI_LITE_ADDR_WIDTH-1 downto 0);          --
        ip2axi_rddata               : in std_logic_vector                              --
                                        (C_S_AXI_LITE_DATA_WIDTH-1 downto 0)           --
    );
end axi_dma_lite_if;


-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_dma_lite_if is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------
-- Register I/F Address offset
constant ADDR_OFFSET    : integer := clog2(C_S_AXI_LITE_DATA_WIDTH/8);
-- Register I/F CE number
constant CE_ADDR_SIZE   : integer := clog2(C_NUM_CE);

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
-- AXI Lite slave interface signals
signal awvalid              : std_logic := '0';
signal awaddr               : std_logic_vector
                                (C_S_AXI_LITE_ADDR_WIDTH-1 downto 0) := (others => '0');
signal wvalid               : std_logic := '0';
signal wdata                : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');


signal arvalid              : std_logic := '0';
signal araddr               : std_logic_vector
                                (C_S_AXI_LITE_ADDR_WIDTH-1 downto 0) := (others => '0');
signal awvalid_d1           : std_logic := '0';
signal awvalid_re           : std_logic := '0';
signal awready_i            : std_logic := '0';
signal wvalid_d1            : std_logic := '0';
signal wvalid_re            : std_logic := '0';
signal wready_i             : std_logic := '0';
signal bvalid_i             : std_logic := '0';

signal wr_addr_cap          : std_logic := '0';
signal wr_data_cap          : std_logic := '0';

-- AXI to IP interface signals
signal axi2ip_wraddr_i      : std_logic_vector
                                (C_S_AXI_LITE_ADDR_WIDTH-1 downto 0) := (others => '0');
signal axi2ip_wrdata_i      : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal axi2ip_wren          : std_logic := '0';
signal wrce                 : std_logic_vector(C_NUM_CE-1 downto 0);

signal rdce                 : std_logic_vector(C_NUM_CE-1 downto 0) := (others => '0');
signal arvalid_d1           : std_logic := '0';
signal arvalid_re           : std_logic := '0';
signal arvalid_re_d1        : std_logic := '0';
signal arvalid_i            : std_logic := '0';
signal arready_i            : std_logic := '0';
signal rvalid               : std_logic := '0';
signal axi2ip_rdaddr_i      : std_logic_vector
                                (C_S_AXI_LITE_ADDR_WIDTH-1 downto 0) := (others => '0');

signal s_axi_lite_rvalid_i  : std_logic := '0';
signal read_in_progress     : std_logic := '0'; -- CR607165
signal rst_rvalid_re        : std_logic := '0'; -- CR576999
signal rst_wvalid_re        : std_logic := '0'; -- CR576999
signal rdy : std_logic := '0';
signal rdy1 : std_logic := '0';
signal wr_in_progress : std_logic := '0';

-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin

--*****************************************************************************
--** AXI LITE READ
--*****************************************************************************

s_axi_lite_wready   <= wready_i;
s_axi_lite_awready  <= awready_i;
s_axi_lite_arready  <= arready_i;

s_axi_lite_bvalid   <= bvalid_i;

-------------------------------------------------------------------------------
-- Register AXI Inputs
-------------------------------------------------------------------------------
REG_INPUTS : process(s_axi_lite_aclk)
    begin
        if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
            if(s_axi_lite_aresetn = '0')then
                awvalid <=  '0'                 ;
                awaddr  <=  (others => '0')     ;
                wvalid  <=  '0'                 ;
                wdata   <=  (others => '0')     ;
                arvalid <=  '0'                 ;
                araddr  <=  (others => '0')     ;
            else
                awvalid <= s_axi_lite_awvalid   ;
                awaddr  <= s_axi_lite_awaddr    ;
                wvalid  <= s_axi_lite_wvalid    ;
                wdata   <= s_axi_lite_wdata     ;
                arvalid <= s_axi_lite_arvalid   ;
                araddr  <= s_axi_lite_araddr    ;
            end if;
        end if;
    end process REG_INPUTS;



-- s_axi_lite_aclk is synchronous to ip clock
GEN_SYNC_WRITE : if C_AXI_LITE_IS_ASYNC = 0 generate
begin


-------------------------------------------------------------------------------
-- Assert Write Adddress Ready Handshake
-- Capture rising edge of valid and register out as ready.  This creates
-- a 3 clock cycle address phase but also registers all inputs and outputs.
-- Note : Single clock cycle address phase can be accomplished using
-- combinatorial logic.
-------------------------------------------------------------------------------
REG_AWVALID : process(s_axi_lite_aclk)
    begin
        if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
            if(s_axi_lite_aresetn = '0' or rst_wvalid_re = '1')then
                awvalid_d1  <= '0';
--                awvalid_re  <= '0';                             -- CR605883
            else
                awvalid_d1  <= awvalid;
--                awvalid_re  <= awvalid and not awvalid_d1;      -- CR605883
            end if;
        end if;
    end process REG_AWVALID;

                awvalid_re  <= awvalid and not awvalid_d1 and (not (wr_in_progress));      -- CR605883
-------------------------------------------------------------------------------
-- Capture assertion of awvalid to indicate that we have captured
-- a valid address
-------------------------------------------------------------------------------

-------------------------------------------------------------------------------
-- Assert Write Data Ready Handshake
-- Capture rising edge of valid and register out as ready.  This creates
-- a 3 clock cycle address phase but also registers all inputs and outputs.
-- Note : Single clock cycle address phase can be accomplished using
-- combinatorial logic.
-------------------------------------------------------------------------------
REG_WVALID : process(s_axi_lite_aclk)
    begin
        if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
            if(s_axi_lite_aresetn = '0' or rst_wvalid_re = '1')then
                wvalid_d1   <= '0';
--                wvalid_re   <= '0';
            else
                wvalid_d1   <= wvalid;
--                wvalid_re   <= wvalid and not wvalid_d1; -- CR605883
            end if;
        end if;
    end process REG_WVALID;

                wvalid_re   <= wvalid and not wvalid_d1; -- CR605883


WRITE_IN_PROGRESS : process(s_axi_lite_aclk)
    begin
        if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
            if(s_axi_lite_aresetn = '0' or rst_wvalid_re = '1')then
                wr_in_progress <= '0';
            elsif(awvalid_re = '1')then
                wr_in_progress <= '1';
            end if;
        end if;
    end process WRITE_IN_PROGRESS;


-- CR605883 (CDC) provide pure register output to synchronizers
--wvalid_re  <= wvalid and not wvalid_d1 and not rst_wvalid_re;

                

-------------------------------------------------------------------------------
-- Capture assertion of wvalid to indicate that we have captured
-- valid data
-------------------------------------------------------------------------------


WRDATA_CAP_FLAG : process(s_axi_lite_aclk)
    begin
        if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
            if(s_axi_lite_aresetn = '0' or rdy = '1')then
                wr_data_cap <= '0';
            elsif(wvalid_re = '1')then
                wr_data_cap <= '1';
            end if;
        end if;
    end process WRDATA_CAP_FLAG;

REG_WREADY : process(s_axi_lite_aclk)
    begin
        if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
            if(s_axi_lite_aresetn = '0' or rdy = '1') then
                rdy <= '0';
            elsif (wr_data_cap = '1' and wr_addr_cap = '1') then
                rdy <= '1';
            end if;
                wready_i <= rdy;
                awready_i <= rdy;
                rdy1 <= rdy; 
        end if;
    end process REG_WREADY;


WRADDR_CAP_FLAG : process(s_axi_lite_aclk)
    begin
        if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
            if(s_axi_lite_aresetn = '0' or rdy = '1')then
                wr_addr_cap <= '0';
            elsif(awvalid_re = '1')then
                wr_addr_cap <= '1';
            end if;
        end if;
    end process WRADDR_CAP_FLAG;
    -------------------------------------------------------------------------------
    -- Capture Write Address
    -------------------------------------------------------------------------------
    REG_WRITE_ADDRESS : process(s_axi_lite_aclk)
        begin
            if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
                if(s_axi_lite_aresetn = '0')then
                 --   axi2ip_wraddr_i   <= (others => '0');

                -- Register address on valid
                elsif(awvalid_re = '1')then
                 --   axi2ip_wraddr_i   <= awaddr;

                end if;
            end if;
        end process REG_WRITE_ADDRESS;

    -------------------------------------------------------------------------------
    -- Capture Write Data
    -------------------------------------------------------------------------------
    REG_WRITE_DATA : process(s_axi_lite_aclk)
        begin
            if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
                if(s_axi_lite_aresetn = '0')then
                    axi2ip_wrdata_i     <= (others => '0');

                -- Register address and assert ready
                elsif(wvalid_re = '1')then
                    axi2ip_wrdata_i     <= wdata;

                end if;
            end if;
        end process REG_WRITE_DATA;

    -------------------------------------------------------------------------------
    -- Must have both a valid address and valid data before updating
    -- a register.  Note in AXI write address can come before or
    -- after AXI write data.
--    axi2ip_wren <= '1' when wr_data_cap = '1' and wr_addr_cap = '1'
--                else '0';
      axi2ip_wren <= rdy; -- or rdy1;
    -------------------------------------------------------------------------------
    -- Decode and assert proper chip enable per captured axi lite write address
    -------------------------------------------------------------------------------
    WRCE_GEN: for j in 0 to C_NUM_CE - 1 generate

    constant BAR    : std_logic_vector(CE_ADDR_SIZE-1 downto 0) :=
                    std_logic_vector(to_unsigned(j,CE_ADDR_SIZE));
    begin

        wrce(j) <= axi2ip_wren when s_axi_lite_awaddr
                                    ((CE_ADDR_SIZE + ADDR_OFFSET) - 1
                                                        downto ADDR_OFFSET)

                                    = BAR(CE_ADDR_SIZE-1 downto 0)
              else '0';

    end generate WRCE_GEN;

    -------------------------------------------------------------------------------
    -- register write ce's and data out to axi dma register module
    -------------------------------------------------------------------------------
    REG_WR_OUT : process(s_axi_lite_aclk)
        begin
        if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
            if(s_axi_lite_aresetn = '0')then
                axi2ip_wrce     <= (others => '0');
        --        axi2ip_wrdata   <= (others => '0');
            else
                axi2ip_wrce     <= wrce;
        --        axi2ip_wrdata   <= axi2ip_wrdata_i;
            end if;
        end if;
    end process REG_WR_OUT;
  
             axi2ip_wrdata <= s_axi_lite_wdata; 

    -------------------------------------------------------------------------------
    -- Write Response
    -------------------------------------------------------------------------------
    s_axi_lite_bresp    <= OKAY_RESP;

    WRESP_PROCESS : process(s_axi_lite_aclk)
        begin
            if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
                if(s_axi_lite_aresetn = '0')then
                    bvalid_i        <= '0';
                    rst_wvalid_re   <= '0';     -- CR576999
                -- If response issued and target indicates ready then
                -- clear response
                elsif(bvalid_i = '1' and s_axi_lite_bready = '1')then
                    bvalid_i        <= '0';
                    rst_wvalid_re   <= '0';     -- CR576999
                -- Issue a resonse on write
                elsif(rdy1 = '1')then
                    bvalid_i        <= '1';
                    rst_wvalid_re   <= '1';     -- CR576999
                end if;
            end if;
        end process WRESP_PROCESS;


end generate GEN_SYNC_WRITE;


-- s_axi_lite_aclk is asynchronous to ip clock
GEN_ASYNC_WRITE : if C_AXI_LITE_IS_ASYNC = 1 generate
-- Data support

 -----------------------------------------------------------------------------
  -- ATTRIBUTE Declarations
  -----------------------------------------------------------------------------
  -- Prevent x-propagation on clock-domain crossing register
  ATTRIBUTE async_reg                      : STRING;
 Attribute KEEP : string; -- declaration
 Attribute EQUIVALENT_REGISTER_REMOVAL : string; -- declaration


signal ip_wvalid_d1_cdc_to     : std_logic := '0';
signal ip_wvalid_d2     : std_logic := '0';
signal ip_wvalid_re     : std_logic := '0';
signal wr_wvalid_re_cdc_from     : std_logic := '0';
signal wr_data_cdc_from          : std_logic_vector                                              -- CR605883
                            (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');    -- CR605883
signal wdata_d1_cdc_to         : std_logic_vector
                            (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal wdata_d2         : std_logic_vector
                            (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal axi2ip_wrdata_cdc_tig         : std_logic_vector
                            (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal ip_data_cap      : std_logic := '0';

-- Address support
signal ip_awvalid_d1_cdc_to    : std_logic := '0';
signal ip_awvalid_d2    : std_logic := '0';
signal ip_awvalid_re    : std_logic := '0';
signal wr_awvalid_re_cdc_from    : std_logic := '0';
signal wr_addr_cdc_from          : std_logic_vector                                              -- CR605883
                            (C_S_AXI_LITE_ADDR_WIDTH-1 downto 0) := (others => '0');    -- CR605883
signal awaddr_d1_cdc_tig        : std_logic_vector
                            (C_S_AXI_LITE_ADDR_WIDTH-1 downto 0) := (others => '0');
signal awaddr_d2        : std_logic_vector
                            (C_S_AXI_LITE_ADDR_WIDTH-1 downto 0) := (others => '0');
signal ip_addr_cap      : std_logic := '0';

-- Bvalid support
signal lite_data_cap_d1 : std_logic := '0';
signal lite_data_cap_d2 : std_logic := '0';
signal lite_addr_cap_d1 : std_logic := '0';
signal lite_addr_cap_d2 : std_logic := '0';
signal lite_axi2ip_wren : std_logic := '0';

signal awvalid_cdc_from : std_logic := '0';
signal awvalid_cdc_to : std_logic := '0';
signal awvalid_to : std_logic := '0';
signal awvalid_to2 : std_logic := '0';
  --ATTRIBUTE async_reg OF awvalid_cdc_to  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF awvalid_to  : SIGNAL IS "true";


signal wvalid_cdc_from : std_logic := '0';
signal wvalid_cdc_to : std_logic := '0';
signal wvalid_to : std_logic := '0';
signal wvalid_to2 : std_logic := '0';
  --ATTRIBUTE async_reg OF wvalid_cdc_to  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF wvalid_to  : SIGNAL IS "true";

signal rdy_cdc_to : std_logic := '0';
signal rdy_cdc_from : std_logic := '0';
signal rdy_to : std_logic := '0';
signal rdy_to2 : std_logic := '0';
signal rdy_to2_cdc_from : std_logic := '0';
signal rdy_out : std_logic := '0';
  --ATTRIBUTE async_reg OF rdy_cdc_to  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF rdy_to  : SIGNAL IS "true";

  Attribute KEEP of rdy_to2_cdc_from       : signal is "TRUE";
  Attribute EQUIVALENT_REGISTER_REMOVAL of rdy_to2_cdc_from : signal is "no";

signal rdy_back_cdc_to : std_logic := '0';
signal rdy_back_to : std_logic :='0';
  --ATTRIBUTE async_reg OF rdy_back_cdc_to  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF rdy_back_to  : SIGNAL IS "true";

signal rdy_back : std_logic := '0';

signal rdy_shut : std_logic := '0';

begin

REG_AWVALID : process(s_axi_lite_aclk)
    begin
        if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
            if(s_axi_lite_aresetn = '0' or rst_wvalid_re = '1')then
                awvalid_d1  <= '0';
            else
                awvalid_d1  <= awvalid;
            end if;
        end if;
    end process REG_AWVALID;

                awvalid_re  <= awvalid and not awvalid_d1 and (not (wr_in_progress));      -- CR605883
-------------------------------------------------------------------------------
-- Capture assertion of awvalid to indicate that we have captured
-- a valid address
-------------------------------------------------------------------------------

-------------------------------------------------------------------------------
-- Assert Write Data Ready Handshake
-- Capture rising edge of valid and register out as ready.  This creates
-- a 3 clock cycle address phase but also registers all inputs and outputs.
-- Note : Single clock cycle address phase can be accomplished using
-- combinatorial logic.
-------------------------------------------------------------------------------
REG_WVALID : process(s_axi_lite_aclk)
    begin
        if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
            if(s_axi_lite_aresetn = '0' or rst_wvalid_re = '1')then
                wvalid_d1   <= '0';
            else
                wvalid_d1   <= wvalid;
            end if;
        end if;
    end process REG_WVALID;

                wvalid_re   <= wvalid and not wvalid_d1; -- CR605883

    --*************************************************************************
    --** Write Address Support
    --*************************************************************************

    AWVLD_CDC_FROM : process(s_axi_lite_aclk)
        begin
            if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
                if(s_axi_lite_aresetn = '0' or rst_wvalid_re = '1')then
                    awvalid_cdc_from <= '0';
                elsif(awvalid_re = '1')then
                    awvalid_cdc_from <= '1';
                end if;
            end if;
        end process AWVLD_CDC_FROM;

AWVLD_CDC_TO : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => s_axi_lite_aclk,
        prmry_resetn               => '0', 
        prmry_in                   => awvalid_cdc_from, 
        prmry_vect_in              => (others => '0'),
                                    
        scndry_aclk                => ip2axi_aclk, 
        scndry_resetn              => '0',
        scndry_out                 => awvalid_to,
        scndry_vect_out            => open
    );


--    AWVLD_CDC_TO : process(ip2axi_aclk)
--        begin
--            if(ip2axi_aclk'EVENT and ip2axi_aclk = '1')then
--                    awvalid_cdc_to <= awvalid_cdc_from;
--                    awvalid_to <= awvalid_cdc_to;
--            end if;
--        end process AWVLD_CDC_TO;

    AWVLD_CDC_TO2 : process(ip2axi_aclk)
        begin
            if(ip2axi_aclk'EVENT and ip2axi_aclk = '1')then
                if(ip2axi_aresetn = '0')then
                    awvalid_to2 <= '0';
                else
                    awvalid_to2 <= awvalid_to;
                end if;
            end if;
        end process AWVLD_CDC_TO2;


               ip_awvalid_re <= awvalid_to and (not awvalid_to2);


    WVLD_CDC_FROM : process(s_axi_lite_aclk)
        begin
            if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
                if(s_axi_lite_aresetn = '0' or rst_wvalid_re = '1')then
                    wvalid_cdc_from <= '0';
                elsif(wvalid_re = '1')then
                    wvalid_cdc_from <= '1';
                end if;
            end if;
        end process WVLD_CDC_FROM;


WVLD_CDC_TO : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => s_axi_lite_aclk,
        prmry_resetn               => '0', 
        prmry_in                   => wvalid_cdc_from, 
        prmry_vect_in              => (others => '0'),
                                    
        scndry_aclk                => ip2axi_aclk, 
        scndry_resetn              => '0',
        scndry_out                 => wvalid_to,
        scndry_vect_out            => open
    );

--    WVLD_CDC_TO : process(ip2axi_aclk)
--        begin
--            if(ip2axi_aclk'EVENT and ip2axi_aclk = '1')then
--                    wvalid_cdc_to <= wvalid_cdc_from;
--                    wvalid_to <= wvalid_cdc_to;
--            end if;
--        end process WVLD_CDC_TO;


    WVLD_CDC_TO2 : process(ip2axi_aclk)
        begin
            if(ip2axi_aclk'EVENT and ip2axi_aclk = '1')then
                if(ip2axi_aresetn = '0')then
                    wvalid_to2 <= '0';
                else
                    wvalid_to2 <= wvalid_to;
                end if;
            end if;
        end process WVLD_CDC_TO2;

               ip_wvalid_re <= wvalid_to and (not wvalid_to2);


REG_WADDR_TO_IPCLK : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 0,
        C_VECTOR_WIDTH             => C_S_AXI_LITE_ADDR_WIDTH,
        C_MTBF_STAGES              => 1
    )
    port map (
        prmry_aclk                 => s_axi_lite_aclk,
        prmry_resetn               => '0', 
        prmry_in                   => '0', 
        prmry_vect_in              => s_axi_lite_awaddr,
                                    
        scndry_aclk                => ip2axi_aclk, 
        scndry_resetn              => '0',
        scndry_out                 => open, 
        scndry_vect_out            => awaddr_d1_cdc_tig
    );


REG_WADDR_TO_IPCLK1 : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 0,
        C_VECTOR_WIDTH             => C_S_AXI_LITE_DATA_WIDTH,
        C_MTBF_STAGES              => 1
    )
    port map (
        prmry_aclk                 => s_axi_lite_aclk,
        prmry_resetn               => '0', 
        prmry_in                   => '0', 
        prmry_vect_in              => s_axi_lite_wdata,
                                    
        scndry_aclk                => ip2axi_aclk, 
        scndry_resetn              => '0',
        scndry_out                 => open, 
        scndry_vect_out            => axi2ip_wrdata_cdc_tig
    );

    -- Double register address in
--    REG_WADDR_TO_IPCLK : process(ip2axi_aclk)
--        begin
--            if(ip2axi_aclk'EVENT and ip2axi_aclk = '1')then
--                if(ip2axi_aresetn = '0')then
--                    awaddr_d1_cdc_tig           <= (others => '0');
--                --    axi2ip_wraddr_i     <= (others => '0');
--                    axi2ip_wrdata_cdc_tig <= (others => '0');
--                else
--                    awaddr_d1_cdc_tig           <= s_axi_lite_awaddr;
--                    axi2ip_wrdata_cdc_tig       <= s_axi_lite_wdata;
--                --    axi2ip_wraddr_i     <= awaddr_d1_cdc_tig;           -- CR605883
--                end if;
--            end if;
--        end process REG_WADDR_TO_IPCLK;

    -- Flag that address has been captured
    REG_IP_ADDR_CAP : process(ip2axi_aclk)
        begin
            if(ip2axi_aclk'EVENT and ip2axi_aclk = '1')then
                if(ip2axi_aresetn = '0' or rdy_shut = '1')then
                    ip_addr_cap <= '0';
                elsif(ip_awvalid_re = '1')then
                    ip_addr_cap <= '1';
                end if;
            end if;
        end process REG_IP_ADDR_CAP;


    REG_WREADY : process(ip2axi_aclk)
    begin
        if(ip2axi_aclk'EVENT and ip2axi_aclk = '1')then
            if(ip2axi_aresetn = '0' or rdy_shut = '1') then -- or rdy = '1') then
                rdy <= '0';
            elsif (ip_data_cap = '1' and ip_addr_cap = '1') then
                rdy <= '1';
            end if;
        end if;
    end process REG_WREADY;

REG3_WREADY : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => s_axi_lite_aclk,
        prmry_resetn               => '0', 
        prmry_in                   => rdy_to2_cdc_from, 
        prmry_vect_in              => (others => '0'),
                                    
        scndry_aclk                => ip2axi_aclk, 
        scndry_resetn              => '0',
        scndry_out                 => rdy_back_to,
        scndry_vect_out            => open
    );

--    REG3_WREADY : process(ip2axi_aclk)
--    begin
--        if(ip2axi_aclk'EVENT and ip2axi_aclk = '1')then
--                rdy_back_cdc_to <= rdy_to2_cdc_from;
--                rdy_back_to <= rdy_back_cdc_to;
--        end if;
--    end process REG3_WREADY;


    REG3_WREADY2 : process(ip2axi_aclk)
    begin
        if(ip2axi_aclk'EVENT and ip2axi_aclk = '1')then
            if(ip2axi_aresetn = '0') then
                rdy_back <= '0';
            else
                rdy_back <= rdy_back_to;
            end if;
        end if;
    end process REG3_WREADY2;

    rdy_shut <= rdy_back_to and (not rdy_back);


    REG1_WREADY : process(ip2axi_aclk)
    begin
        if(ip2axi_aclk'EVENT and ip2axi_aclk = '1')then
            if(ip2axi_aresetn = '0' or rdy_shut = '1') then
                rdy_cdc_from <= '0';
            elsif (rdy = '1') then
                rdy_cdc_from <= '1';
            end if;
        end if;
    end process REG1_WREADY;


REG2_WREADY : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => ip2axi_aclk,
        prmry_resetn               => '0', 
        prmry_in                   => rdy_cdc_from, 
        prmry_vect_in              => (others => '0'),
                                    
        scndry_aclk                => s_axi_lite_aclk, 
        scndry_resetn              => '0',
        scndry_out                 => rdy_to,
        scndry_vect_out            => open
    );


--    REG2_WREADY : process(s_axi_lite_aclk)
--    begin
--        if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
--                rdy_cdc_to <= rdy_cdc_from;
--                rdy_to <= rdy_cdc_to;
--        end if;
--    end process REG2_WREADY;

    REG2_WREADY2 : process(s_axi_lite_aclk)
    begin
        if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
            if(s_axi_lite_aresetn = '0') then
                rdy_to2 <= '0';
                rdy_to2_cdc_from <= '0';
            else
                rdy_to2 <= rdy_to;
                rdy_to2_cdc_from <= rdy_to;
            end if;
        end if;
    end process REG2_WREADY2;


   rdy_out <= not (rdy_to) and rdy_to2;

                wready_i <= rdy_out;
                awready_i <= rdy_out;


    --*************************************************************************
    --** Write Data Support
    --*************************************************************************

    -------------------------------------------------------------------------------
    -- Capture write data
    -------------------------------------------------------------------------------
--    WRDATA_S_H : process(s_axi_lite_aclk)
--        begin
--            if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
--                if(s_axi_lite_aresetn = '0')then
--                    wr_data_cdc_from <= (others => '0');
--                elsif(wvalid_re = '1')then
--                    wr_data_cdc_from <= wdata;
--                end if;
--            end if;
--        end process WRDATA_S_H;


    -- Flag that data has been captured
    REG_IP_DATA_CAP : process(ip2axi_aclk)
        begin
            if(ip2axi_aclk'EVENT and ip2axi_aclk = '1')then
                if(ip2axi_aresetn = '0' or rdy_shut = '1')then
                    ip_data_cap <= '0';
                elsif(ip_wvalid_re = '1')then
                    ip_data_cap <= '1';
                end if;
            end if;
        end process REG_IP_DATA_CAP;

    -- Must have both a valid address and valid data before updating
    -- a register.  Note in AXI write address can come before or
    -- after AXI write data.

      axi2ip_wren <= rdy;
--    axi2ip_wren <= '1' when ip_data_cap = '1' and ip_addr_cap = '1'
--                else '0';

    -------------------------------------------------------------------------------
    -- Decode and assert proper chip enable per captured axi lite write address
    -------------------------------------------------------------------------------
    WRCE_GEN: for j in 0 to C_NUM_CE - 1 generate

    constant BAR    : std_logic_vector(CE_ADDR_SIZE-1 downto 0) :=
                    std_logic_vector(to_unsigned(j,CE_ADDR_SIZE));
    begin

        wrce(j) <= axi2ip_wren when awaddr_d1_cdc_tig
                                    ((CE_ADDR_SIZE + ADDR_OFFSET) - 1
                                                        downto ADDR_OFFSET)

                                    = BAR(CE_ADDR_SIZE-1 downto 0)
              else '0';

    end generate WRCE_GEN;

    -------------------------------------------------------------------------------
    -- register write ce's and data out to axi dma register module
    -------------------------------------------------------------------------------
    REG_WR_OUT : process(ip2axi_aclk)
        begin
        if(ip2axi_aclk'EVENT and ip2axi_aclk = '1')then
            if(ip2axi_aresetn = '0')then
                axi2ip_wrce     <= (others => '0');
            else
                axi2ip_wrce     <= wrce;
            end if;
        end if;
    end process REG_WR_OUT;

     axi2ip_wrdata  <=  axi2ip_wrdata_cdc_tig; --s_axi_lite_wdata;

    --*************************************************************************
    --** Write Response Support
    --*************************************************************************

    -- Minimum of 2 IP clocks for addr and data capture, therefore delaying
    -- Lite clock addr and data capture by 2 Lite clocks will guarenttee bvalid
    -- responce occurs after write data acutally written.
--    REG_ALIGN_CAP : process(s_axi_lite_aclk)
--        begin
--            if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
--                if(s_axi_lite_aresetn = '0')then
--                    lite_data_cap_d1 <= '0';
--                    lite_data_cap_d2 <= '0';

--                    lite_addr_cap_d1 <= '0';
--                    lite_addr_cap_d2 <= '0';
--                else
--                    lite_data_cap_d1 <= rdy; --wr_data_cap;
--                    lite_data_cap_d2 <= lite_data_cap_d1;

--                    lite_addr_cap_d1 <= rdy; --wr_addr_cap;
--                    lite_addr_cap_d2 <= lite_addr_cap_d1;
--                end if;
--            end if;
--        end process REG_ALIGN_CAP;

    -- Pseudo write enable used simply to assert bvalid
  --  lite_axi2ip_wren <= rdy; --'1' when wr_data_cap = '1' and wr_addr_cap = '1'
              --  else '0';

    WRESP_PROCESS : process(s_axi_lite_aclk)
        begin
            if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
                if(s_axi_lite_aresetn = '0')then
                    bvalid_i        <= '0';
                    rst_wvalid_re   <= '0';     -- CR576999
                -- If response issued and target indicates ready then
                -- clear response
                elsif(bvalid_i = '1' and s_axi_lite_bready = '1')then
                    bvalid_i        <= '0';
                    rst_wvalid_re   <= '0';     -- CR576999
                -- Issue a resonse on write
                elsif(rdy_out = '1')then
            --    elsif(lite_axi2ip_wren = '1')then
                    bvalid_i        <= '1';
                    rst_wvalid_re   <= '1';     -- CR576999
                end if;
            end if;
        end process WRESP_PROCESS;

    s_axi_lite_bresp    <= OKAY_RESP;


end generate GEN_ASYNC_WRITE;





--*****************************************************************************
--** AXI LITE READ
--*****************************************************************************

-------------------------------------------------------------------------------
-- Assert Read Adddress Ready Handshake
-- Capture rising edge of valid and register out as ready.  This creates
-- a 3 clock cycle address phase but also registers all inputs and outputs.
-- Note : Single clock cycle address phase can be accomplished using
-- combinatorial logic.
-------------------------------------------------------------------------------
REG_ARVALID : process(s_axi_lite_aclk)
    begin
        if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
            if(s_axi_lite_aresetn = '0' or rst_rvalid_re = '1')then
                arvalid_d1 <= '0';
            else
                arvalid_d1 <= arvalid;
            end if;
        end if;
    end process REG_ARVALID;

arvalid_re  <= arvalid and not arvalid_d1
                and not rst_rvalid_re and not read_in_progress; -- CR607165

-- register for proper alignment
REG_ARREADY : process(s_axi_lite_aclk)
    begin
        if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
            if(s_axi_lite_aresetn = '0')then
                arready_i <= '0';
            else
                arready_i <= arvalid_re;
            end if;
        end if;
    end process REG_ARREADY;

-- Always respond 'okay' axi lite read
s_axi_lite_rresp    <= OKAY_RESP;
s_axi_lite_rvalid   <= s_axi_lite_rvalid_i;


-- s_axi_lite_aclk is synchronous to ip clock
GEN_SYNC_READ : if C_AXI_LITE_IS_ASYNC = 0 generate
begin

    read_in_progress <= '0'; --Not used for sync mode (CR607165)

    -------------------------------------------------------------------------------
    -- Capture Read Address
    -------------------------------------------------------------------------------
    REG_READ_ADDRESS : process(s_axi_lite_aclk)
        begin
            if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
                if(s_axi_lite_aresetn = '0')then
                    axi2ip_rdaddr_i   <= (others => '0');

                -- Register address on valid
                elsif(arvalid_re = '1')then
                    axi2ip_rdaddr_i   <= araddr;

                end if;
            end if;
        end process REG_READ_ADDRESS;



    -------------------------------------------------------------------------------
    -- Generate RdCE based on address match to address bar
    -------------------------------------------------------------------------------
    RDCE_GEN: for j in 0 to C_NUM_CE - 1 generate

    constant BAR    : std_logic_vector(CE_ADDR_SIZE-1 downto 0) :=
                    std_logic_vector(to_unsigned(j,CE_ADDR_SIZE));
    begin

      rdce(j) <= arvalid_re_d1
        when axi2ip_rdaddr_i((CE_ADDR_SIZE + ADDR_OFFSET) - 1
                              downto ADDR_OFFSET)
             = BAR(CE_ADDR_SIZE-1 downto 0)
        else '0';

    end generate RDCE_GEN;

    -------------------------------------------------------------------------------
    -- Register out to IP
    -------------------------------------------------------------------------------
    REG_RDCNTRL_OUT : process(s_axi_lite_aclk)
        begin
            if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
                if(s_axi_lite_aresetn = '0')then
                    --axi2ip_rdce     <= (others => '0');
                    axi2ip_rdaddr   <= (others => '0');
                else
                    --axi2ip_rdce     <= rdce;
                    axi2ip_rdaddr   <= axi2ip_rdaddr_i;
                end if;
            end if;
        end process REG_RDCNTRL_OUT;


    -- Sample and hold rdce value until rvalid assertion
    REG_RDCE_OUT : process(s_axi_lite_aclk)
        begin
            if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
                if(s_axi_lite_aresetn = '0' or rst_rvalid_re = '1')then
                    axi2ip_rdce     <= (others => '0');
                elsif(arvalid_re_d1 = '1')then
                    axi2ip_rdce     <= rdce;
                end if;
            end if;
        end process REG_RDCE_OUT;

    -- Register for proper alignment
    REG_RVALID : process(s_axi_lite_aclk)
        begin
            if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
                if(s_axi_lite_aresetn = '0')then
                    arvalid_re_d1   <= '0';
                    rvalid          <= '0';
                else
                    arvalid_re_d1   <= arvalid_re;
                    rvalid          <= arvalid_re_d1;
                end if;
            end if;
        end process REG_RVALID;

    -------------------------------------------------------------------------------
    -- Drive read data and read data valid out on capture of valid address.
    -------------------------------------------------------------------------------
    REG_RD_OUT : process(s_axi_lite_aclk)
        begin
            if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
                if(s_axi_lite_aresetn = '0')then
                    s_axi_lite_rdata    <= (others => '0');
                    s_axi_lite_rvalid_i <= '0';
                    rst_rvalid_re       <= '0';                 -- CR576999

                -- If rvalid driving out to target and target indicates ready
                -- then de-assert rvalid. (structure guarentees min 1 clock of rvalid)
                elsif(s_axi_lite_rvalid_i = '1' and s_axi_lite_rready = '1')then
                    s_axi_lite_rdata    <= (others => '0');
                    s_axi_lite_rvalid_i <= '0';
                    rst_rvalid_re       <= '0';                 -- CR576999

                -- If read cycle then assert rvalid and rdata out to target
                elsif(rvalid = '1')then
                    s_axi_lite_rdata    <= ip2axi_rddata;
                    s_axi_lite_rvalid_i <= '1';
                    rst_rvalid_re       <= '1';                 -- CR576999

                end if;
            end if;
        end process REG_RD_OUT;


end generate GEN_SYNC_READ;



-- s_axi_lite_aclk is asynchronous to ip clock
GEN_ASYNC_READ : if C_AXI_LITE_IS_ASYNC = 1 generate

  ATTRIBUTE async_reg                      : STRING;

signal ip_arvalid_d1_cdc_tig        : std_logic := '0';
signal ip_arvalid_d2        : std_logic := '0';
signal ip_arvalid_d3        : std_logic := '0';
signal ip_arvalid_re        : std_logic := '0';

signal araddr_d1_cdc_tig            : std_logic_vector(C_S_AXI_LITE_ADDR_WIDTH-1 downto 0) :=(others => '0');
signal araddr_d2            : std_logic_vector(C_S_AXI_LITE_ADDR_WIDTH-1 downto 0) :=(others => '0');
signal araddr_d3            : std_logic_vector(C_S_AXI_LITE_ADDR_WIDTH-1 downto 0) :=(others => '0');

signal lite_rdata_cdc_from           : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) :=(others => '0');
signal lite_rdata_d1_cdc_to        : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) :=(others => '0');
signal lite_rdata_d2        : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) :=(others => '0');

  --ATTRIBUTE async_reg OF ip_arvalid_d1_cdc_tig  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF ip_arvalid_d2  : SIGNAL IS "true";

  --ATTRIBUTE async_reg OF araddr_d1_cdc_tig  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF araddr_d2  : SIGNAL IS "true";

  --ATTRIBUTE async_reg OF lite_rdata_d1_cdc_to : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF lite_rdata_d2  : SIGNAL IS "true";

signal p_pulse_s_h          : std_logic := '0';
signal p_pulse_s_h_clr      : std_logic := '0';
signal s_pulse_d1           : std_logic := '0';
signal s_pulse_d2           : std_logic := '0';
signal s_pulse_d3           : std_logic := '0';
signal s_pulse_re           : std_logic := '0';

signal p_pulse_re_d1        : std_logic := '0';
signal p_pulse_re_d2        : std_logic := '0';
signal p_pulse_re_d3        : std_logic := '0';

signal arready_d1           : std_logic := '0'; -- CR605883
signal arready_d2           : std_logic := '0'; -- CR605883
signal arready_d3           : std_logic := '0'; -- CR605883
signal arready_d4           : std_logic := '0'; -- CR605883
signal arready_d5           : std_logic := '0'; -- CR605883
signal arready_d6           : std_logic := '0'; -- CR605883
signal arready_d7           : std_logic := '0'; -- CR605883
signal arready_d8           : std_logic := '0'; -- CR605883
signal arready_d9           : std_logic := '0'; -- CR605883
signal arready_d10           : std_logic := '0'; -- CR605883
signal arready_d11           : std_logic := '0'; -- CR605883
signal arready_d12           : std_logic := '0'; -- CR605883

begin

    -- CR607165
    -- Flag to prevent overlapping reads
    RD_PROGRESS : process(s_axi_lite_aclk)
        begin
            if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
                if(s_axi_lite_aresetn = '0' or rst_rvalid_re = '1')then
                    read_in_progress <= '0';

                elsif(arvalid_re = '1')then
                    read_in_progress <= '1';
                end if;
            end if;
        end process RD_PROGRESS;


    -- Double register address in
REG_RADDR_TO_IPCLK : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 0,
        C_VECTOR_WIDTH             => C_S_AXI_LITE_ADDR_WIDTH,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => s_axi_lite_aclk,
        prmry_resetn               => '0',
        prmry_in                   => '0',
        prmry_vect_in              => s_axi_lite_araddr,
        
        scndry_aclk                => ip2axi_aclk,
        scndry_resetn              => '0',
        scndry_out                 => open,
        scndry_vect_out            => araddr_d3
    );


--    REG_RADDR_TO_IPCLK : process(ip2axi_aclk)
--        begin
--            if(ip2axi_aclk'EVENT and ip2axi_aclk = '1')then
--                if(ip2axi_aresetn = '0')then
--                    araddr_d1_cdc_tig           <= (others => '0');
--                    araddr_d2           <= (others => '0');
--                    araddr_d3           <= (others => '0');
--                else
--                    araddr_d1_cdc_tig   <= s_axi_lite_araddr;
--                    araddr_d2           <= araddr_d1_cdc_tig;
--                    araddr_d3           <= araddr_d2;
--                end if;
--            end if;
--        end process REG_RADDR_TO_IPCLK;

    -- Latch and hold read address
    REG_ARADDR_PROCESS : process(ip2axi_aclk)
        begin
            if(ip2axi_aclk'EVENT and ip2axi_aclk = '1')then
                if(ip2axi_aresetn = '0')then
                    axi2ip_rdaddr_i <= (others => '0');
                elsif(ip_arvalid_re = '1')then
                    axi2ip_rdaddr_i <= araddr_d3;
                end if;
            end if;
        end process REG_ARADDR_PROCESS;

    axi2ip_rdaddr   <= axi2ip_rdaddr_i;

    -- Register awready into IP clock domain.  awready
    -- is a 1 axi_lite clock delay of the rising edge of
    -- arvalid.  This provides a signal that asserts when
    -- araddr is known to be stable.

REG_ARVALID_TO_IPCLK : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => C_S_AXI_LITE_ADDR_WIDTH,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => s_axi_lite_aclk,
        prmry_resetn               => '0',
        prmry_in                   => arready_i,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => ip2axi_aclk,
        scndry_resetn              => '0',
        scndry_out                 => ip_arvalid_d2,
        scndry_vect_out            => open
    );



    REG_ARVALID_TO_IPCLK1 : process(ip2axi_aclk)
        begin
            if(ip2axi_aclk'EVENT and ip2axi_aclk = '1')then
                if(ip2axi_aresetn = '0')then
--                    ip_arvalid_d1_cdc_tig <= '0';
--                    ip_arvalid_d2 <= '0';
                    ip_arvalid_d3 <= '0';
                else
--                    ip_arvalid_d1_cdc_tig <= arready_i;
--                    ip_arvalid_d2 <= ip_arvalid_d1_cdc_tig;
                    ip_arvalid_d3 <= ip_arvalid_d2;
                end if;
            end if;
        end process REG_ARVALID_TO_IPCLK1;

    ip_arvalid_re <= ip_arvalid_d2 and not ip_arvalid_d3;

    -------------------------------------------------------------------------------
    -- Generate Read CE's
    -------------------------------------------------------------------------------
    RDCE_GEN: for j in 0 to C_NUM_CE - 1 generate

    constant BAR    : std_logic_vector(CE_ADDR_SIZE-1 downto 0) :=
                    std_logic_vector(to_unsigned(j,CE_ADDR_SIZE));
    begin

      rdce(j) <= ip_arvalid_re
        when araddr_d3((CE_ADDR_SIZE + ADDR_OFFSET) - 1
                              downto ADDR_OFFSET)
             = BAR(CE_ADDR_SIZE-1 downto 0)
        else '0';

    end generate RDCE_GEN;

    -------------------------------------------------------------------------------
    -- Register RDCE and RD Data out to IP
    -------------------------------------------------------------------------------
    REG_RDCNTRL_OUT : process(ip2axi_aclk)
        begin
            if(ip2axi_aclk'EVENT and ip2axi_aclk = '1')then
                if(ip2axi_aresetn = '0')then
                    axi2ip_rdce     <= (others => '0');
                elsif(ip_arvalid_re = '1')then
                    axi2ip_rdce     <= rdce;
                else
                    axi2ip_rdce     <= (others => '0');
                end if;
            end if;
        end process REG_RDCNTRL_OUT;

    -- Generate sample and hold pulse to capture read data from IP
    REG_RVALID : process(ip2axi_aclk)
        begin
            if(ip2axi_aclk'EVENT and ip2axi_aclk = '1')then
                if(ip2axi_aresetn = '0')then
                    rvalid          <= '0';
                else
                    rvalid          <= ip_arvalid_re;
                end if;
            end if;
        end process REG_RVALID;

    -------------------------------------------------------------------------------
    -- Sample and hold read data from IP
    -------------------------------------------------------------------------------
    S_H_READ_DATA : process(ip2axi_aclk)
        begin
            if(ip2axi_aclk'EVENT and ip2axi_aclk = '1')then
                if(ip2axi_aresetn = '0')then
                    lite_rdata_cdc_from    <= (others => '0');

                -- If read cycle then assert rvalid and rdata out to target
                elsif(rvalid = '1')then
                    lite_rdata_cdc_from    <= ip2axi_rddata;

                end if;
            end if;
        end process S_H_READ_DATA;

    -- Cross read data to axi_lite clock domain
REG_DATA2LITE_CLOCK : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 0,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => ip2axi_aclk,
        prmry_resetn               => '0',
        prmry_in                   => '0', --lite_rdata_cdc_from,
        prmry_vect_in              => lite_rdata_cdc_from,

        scndry_aclk                => s_axi_lite_aclk,
        scndry_resetn              => '0',
        scndry_out                 => open, --lite_rdata_d2,
        scndry_vect_out            => lite_rdata_d2
    );

--    REG_DATA2LITE_CLOCK : process(s_axi_lite_aclk)
--        begin
--            if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
--                if(s_axi_lite_aresetn = '0')then
--                    lite_rdata_d1_cdc_to   <= (others => '0');
--                    lite_rdata_d2   <= (others => '0');
--                else
--                    lite_rdata_d1_cdc_to   <= lite_rdata_cdc_from;
--                    lite_rdata_d2   <= lite_rdata_d1_cdc_to;
--                end if;
--            end if;
--        end process REG_DATA2LITE_CLOCK;



    -- CR605883 (CDC) modified to remove
    -- Because axi_lite_aclk must be less than or equal to ip2axi_aclk
    -- then read data will appear a maximum 6 clocks from assertion
    -- of arready.
    REG_ALIGN_RDATA_LATCH : process(s_axi_lite_aclk)
        begin
            if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
                if(s_axi_lite_aresetn = '0')then
                    arready_d1 <= '0';
                    arready_d2 <= '0';
                    arready_d3 <= '0';
                    arready_d4 <= '0';
                    arready_d5 <= '0';
                    arready_d6 <= '0';
                    arready_d7 <= '0';
                    arready_d8 <= '0';
                    arready_d9 <= '0';
                    arready_d10 <= '0';
                    arready_d11 <= '0';
                    arready_d12 <= '0';
                else
                    arready_d1 <= arready_i;
                    arready_d2 <= arready_d1;
                    arready_d3 <= arready_d2;
                    arready_d4 <= arready_d3;
                    arready_d5 <= arready_d4;
                    arready_d6 <= arready_d5;
                    arready_d7 <= arready_d6;
                    arready_d8 <= arready_d7;
                    arready_d9 <= arready_d8;
                    arready_d10 <= arready_d9;
                    arready_d11 <= arready_d10;
                    arready_d12 <= arready_d11;
                end if;
            end if;
        end process REG_ALIGN_RDATA_LATCH;

    -------------------------------------------------------------------------------
    -- Drive read data and read data valid out on capture of valid address.
    -------------------------------------------------------------------------------
    REG_RD_OUT : process(s_axi_lite_aclk)
        begin
            if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
                if(s_axi_lite_aresetn = '0')then
                    s_axi_lite_rdata    <= (others => '0');
                    s_axi_lite_rvalid_i <= '0';
                    rst_rvalid_re       <= '0';                 -- CR576999

                -- If rvalid driving out to target and target indicates ready
                -- then de-assert rvalid. (structure guarentees min 1 clock of rvalid)
                elsif(s_axi_lite_rvalid_i = '1' and s_axi_lite_rready = '1')then
                    s_axi_lite_rdata    <= (others => '0');
                    s_axi_lite_rvalid_i <= '0';
                    rst_rvalid_re       <= '0';                 -- CR576999

                -- If read cycle then assert rvalid and rdata out to target
                -- CR605883
                --elsif(s_pulse_re = '1')then
                elsif(arready_d12 = '1')then
                    s_axi_lite_rdata    <= lite_rdata_d2;
                    s_axi_lite_rvalid_i <= '1';
                    rst_rvalid_re       <= '1';                 -- CR576999

                end if;
            end if;
        end process REG_RD_OUT;


end generate GEN_ASYNC_READ;

end implementation;





--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:        axi_dma_register.vhd
--
-- Description:     This entity encompasses the channel register set.
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;

-------------------------------------------------------------------------------
entity  axi_dma_register is
    generic(
        C_NUM_REGISTERS             : integer                   := 11       ;
        C_INCLUDE_SG                : integer                   := 1        ;
        C_SG_LENGTH_WIDTH           : integer range 8 to 23     := 14       ;
        C_S_AXI_LITE_DATA_WIDTH     : integer range 32 to 32    := 32       ;
        C_M_AXI_SG_ADDR_WIDTH       : integer range 32 to 64    := 32       ;
        C_MICRO_DMA                 : integer range 0 to 1      := 0        ;
        C_ENABLE_MULTI_CHANNEL             : integer range 0 to 1      := 0 
        --C_CHANNEL_IS_S2MM           : integer range 0 to 1      := 0 CR603034
    );
    port (
        m_axi_sg_aclk               : in  std_logic                         ;          --
        m_axi_sg_aresetn            : in  std_logic                         ;          --
                                                                                       --
        -- AXI Interface Control                                                       --
        axi2ip_wrce                 : in  std_logic_vector                             --
                                        (C_NUM_REGISTERS-1 downto 0)        ;          --
        axi2ip_wrdata               : in  std_logic_vector                             --
                                        (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);          --
                                                                                       --
        -- DMASR Control                                                               --
        stop_dma                    : in  std_logic                         ;          --
        halted_clr                  : in  std_logic                         ;          --
        halted_set                  : in  std_logic                         ;          --
        idle_set                    : in  std_logic                         ;          --
        idle_clr                    : in  std_logic                         ;          --
        ioc_irq_set                 : in  std_logic                         ;          --
        dly_irq_set                 : in  std_logic                         ;          --
        irqdelay_status             : in  std_logic_vector(7 downto 0)      ;          --
        irqthresh_status            : in  std_logic_vector(7 downto 0)      ;          --
        irqthresh_wren              : out std_logic                         ;          --
        irqdelay_wren               : out std_logic                         ;          --
        dlyirq_dsble                : out std_logic                         ;          -- CR605888
                                                                                       --
        -- Error Control                                                               --
        dma_interr_set              : in  std_logic                         ;          --
        dma_slverr_set              : in  std_logic                         ;          --
        dma_decerr_set              : in  std_logic                         ;          --
        ftch_interr_set             : in  std_logic                         ;          --
        ftch_slverr_set             : in  std_logic                         ;          --
        ftch_decerr_set             : in  std_logic                         ;          --
        ftch_error_addr             : in  std_logic_vector                             --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;          --
        updt_interr_set             : in  std_logic                         ;          --
        updt_slverr_set             : in  std_logic                         ;          --
        updt_decerr_set             : in  std_logic                         ;          --
        updt_error_addr             : in  std_logic_vector                             --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;          --
        error_in                    : in  std_logic                         ;          --
        error_out                   : out std_logic                         ;          --
        introut                     : out std_logic                         ;          --
        soft_reset_in               : in  std_logic                         ;          --
        soft_reset_clr              : in  std_logic                         ;          --
                                                                                       --
        -- CURDESC Update                                                              --
        update_curdesc              : in  std_logic                         ;          --
        new_curdesc                 : in  std_logic_vector                             --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;          --
        -- TAILDESC Update                                                             --
        tailpntr_updated            : out std_logic                         ;          --
                                                                                       --
        -- Channel Register Out                                                        --
        sg_ctl                      : out std_logic_vector (7 downto 0)     ;
 
        dmacr                       : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        dmasr                       : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        curdesc_lsb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        curdesc_msb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc_lsb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc_msb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        buffer_address              : out std_logic_vector                             --
                                           (C_M_AXI_SG_ADDR_WIDTH-1 downto 0);       --
        buffer_length               : out std_logic_vector                             --
                                           (C_SG_LENGTH_WIDTH-1 downto 0)   ;          --
        buffer_length_wren          : out std_logic                         ;          --
        bytes_received              : in  std_logic_vector                             --
                                           (C_SG_LENGTH_WIDTH-1 downto 0)   ;          --
        bytes_received_wren         : in  std_logic                                    --
    );                                                                                 --
end axi_dma_register;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_dma_register is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------
constant DMACR_INDEX            : integer := 0;                     -- DMACR Register index
constant DMASR_INDEX            : integer := 1;                     -- DMASR Register index
constant CURDESC_LSB_INDEX      : integer := 2;                     -- CURDESC LSB Reg index
constant CURDESC_MSB_INDEX      : integer := 3;                     -- CURDESC MSB Reg index
constant TAILDESC_LSB_INDEX     : integer := 4;                     -- TAILDESC LSB Reg index
constant TAILDESC_MSB_INDEX     : integer := 5;                     -- TAILDESC MSB Reg index
-- CR603034 moved s2mm back to offset 6
--constant SA_ADDRESS_INDEX       : integer := 6;                     -- Buffer Address Reg (SA)
--constant DA_ADDRESS_INDEX       : integer := 8;                     -- Buffer Address Reg (DA)
--
--
--constant BUFF_ADDRESS_INDEX     : integer := address_index_select   -- Buffer Address Reg (SA or DA)
--                                                    (C_CHANNEL_IS_S2MM, -- Channel Type 1=rx 0=tx
--                                                     SA_ADDRESS_INDEX,  -- Source Address Index
--                                                     DA_ADDRESS_INDEX); -- Destination Address Index
constant BUFF_ADDRESS_INDEX     : integer := 6;
constant BUFF_ADDRESS_MSB_INDEX     : integer := 7;
constant BUFF_LENGTH_INDEX      : integer := 10;                    -- Buffer Length Reg
constant SGCTL_INDEX      : integer := 11;                    -- Buffer Length Reg

constant ZERO_VALUE             : std_logic_vector(31 downto 0) := (others => '0');

constant DMA_CONFIG             : std_logic_vector(0 downto 0)
                                    := std_logic_vector(to_unsigned(C_INCLUDE_SG,1));

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
signal dmacr_i              : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal dmasr_i              : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal curdesc_lsb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 6) := (others => '0');
signal curdesc_msb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc_lsb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 6) := (others => '0');
signal taildesc_msb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal buffer_address_i     : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal buffer_address_i_64     : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal buffer_length_i      : std_logic_vector
                                (C_SG_LENGTH_WIDTH-1 downto 0)       := (others => '0');




-- DMASR Signals
signal halted               : std_logic := '0';
signal idle                 : std_logic := '0';
signal cmplt                : std_logic := '0';
signal error                : std_logic := '0';
signal dma_interr           : std_logic := '0';
signal dma_slverr           : std_logic := '0';
signal dma_decerr           : std_logic := '0';
signal sg_interr            : std_logic := '0';
signal sg_slverr            : std_logic := '0';
signal sg_decerr            : std_logic := '0';
signal ioc_irq              : std_logic := '0';
signal dly_irq              : std_logic := '0';
signal error_d1             : std_logic := '0';
signal error_re             : std_logic := '0';
signal err_irq              : std_logic := '0';

signal sg_ftch_error        : std_logic := '0';
signal sg_updt_error        : std_logic := '0';
signal error_pointer_set    : std_logic := '0';

-- interrupt coalescing support signals
signal different_delay      : std_logic := '0';
signal different_thresh     : std_logic := '0';
signal threshold_is_zero    : std_logic := '0';
-- soft reset support signals
signal soft_reset_i         : std_logic := '0';
signal run_stop_clr         : std_logic := '0';
signal sg_cache_info        : std_logic_vector (7 downto 0);
signal diff_thresh_xor      : std_logic_vector (7 downto 0);

signal sig_cur_updated : std_logic;
signal tmp11 : std_logic;

signal tailpntr_updated_d1 : std_logic;
signal tailpntr_updated_d2 : std_logic;

-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin

dmacr                   <= dmacr_i          ;
dmasr                   <= dmasr_i          ;
curdesc_lsb             <= curdesc_lsb_i (31 downto 6) & "000000"   ;
curdesc_msb             <= curdesc_msb_i    ;
taildesc_lsb            <= taildesc_lsb_i (31 downto 6) & "000000"   ;
taildesc_msb            <= taildesc_msb_i   ;

BUFF_ADDR_EQL64 : if C_M_AXI_SG_ADDR_WIDTH > 32 generate
begin

buffer_address          <= buffer_address_i_64 & buffer_address_i ;

end generate BUFF_ADDR_EQL64;


BUFF_ADDR_EQL32 : if C_M_AXI_SG_ADDR_WIDTH = 32 generate
begin

buffer_address          <= buffer_address_i ;

end generate BUFF_ADDR_EQL32;


buffer_length           <= buffer_length_i  ;

---------------------------------------------------------------------------
-- DMA Control Register
---------------------------------------------------------------------------
-- DMACR - Interrupt Delay Value
-------------------------------------------------------------------------------
DMACR_DELAY : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                dmacr_i(DMACR_IRQDELAY_MSB_BIT
                 downto DMACR_IRQDELAY_LSB_BIT) <= (others => '0');
            elsif(axi2ip_wrce(DMACR_INDEX) = '1')then
                dmacr_i(DMACR_IRQDELAY_MSB_BIT
                 downto DMACR_IRQDELAY_LSB_BIT) <= axi2ip_wrdata(DMACR_IRQDELAY_MSB_BIT
                                                          downto DMACR_IRQDELAY_LSB_BIT);
            end if;
        end if;
    end process DMACR_DELAY;

-- If written delay is different than previous value then assert write enable
different_delay <= '1' when dmacr_i(DMACR_IRQDELAY_MSB_BIT downto DMACR_IRQDELAY_LSB_BIT)
                   /= axi2ip_wrdata(DMACR_IRQDELAY_MSB_BIT downto DMACR_IRQDELAY_LSB_BIT)
              else '0';

-- delay value different, drive write of delay value to interrupt controller
NEW_DELAY_WRITE : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                irqdelay_wren <= '0';
            -- If AXI Lite write to DMACR and delay different than current
            -- setting then update delay value
            elsif(axi2ip_wrce(DMACR_INDEX) = '1' and different_delay = '1')then
                irqdelay_wren <= '1';
            else
                irqdelay_wren <= '0';
            end if;
        end if;
    end process NEW_DELAY_WRITE;

-------------------------------------------------------------------------------
-- DMACR - Interrupt Threshold Value
-------------------------------------------------------------------------------
threshold_is_zero <= '1' when axi2ip_wrdata(DMACR_IRQTHRESH_MSB_BIT
                                     downto DMACR_IRQTHRESH_LSB_BIT) = ZERO_THRESHOLD
                else '0';

DMACR_THRESH : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                dmacr_i(DMACR_IRQTHRESH_MSB_BIT
                        downto DMACR_IRQTHRESH_LSB_BIT) <= ONE_THRESHOLD;
            -- On AXI Lite write
            elsif(axi2ip_wrce(DMACR_INDEX) = '1')then

                -- If value is 0 then set threshold to 1
                if(threshold_is_zero='1')then
                    dmacr_i(DMACR_IRQTHRESH_MSB_BIT
                     downto DMACR_IRQTHRESH_LSB_BIT)    <= ONE_THRESHOLD;

                -- else set threshold to axi lite wrdata value
                else
                    dmacr_i(DMACR_IRQTHRESH_MSB_BIT
                     downto DMACR_IRQTHRESH_LSB_BIT)    <= axi2ip_wrdata(DMACR_IRQTHRESH_MSB_BIT
                                                                  downto DMACR_IRQTHRESH_LSB_BIT);
                end if;
            end if;
        end if;
    end process DMACR_THRESH;

--diff_thresh_xor <= dmacr_i(DMACR_IRQTHRESH_MSB_BIT downto DMACR_IRQTHRESH_LSB_BIT) xor 
--                   axi2ip_wrdata(DMACR_IRQTHRESH_MSB_BIT downto DMACR_IRQTHRESH_LSB_BIT);

--different_thresh <= '0' when diff_thresh_xor = "00000000" 
--                    else '1';
-- If written threshold is different than previous value then assert write enable
different_thresh <= '1' when dmacr_i(DMACR_IRQTHRESH_MSB_BIT downto DMACR_IRQTHRESH_LSB_BIT)
                    /= axi2ip_wrdata(DMACR_IRQTHRESH_MSB_BIT downto DMACR_IRQTHRESH_LSB_BIT)
              else '0';

-- new treshold written therefore drive write of threshold out
NEW_THRESH_WRITE : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                irqthresh_wren <= '0';
            -- If AXI Lite write to DMACR and threshold different than current
            -- setting then update threshold value
            elsif(axi2ip_wrce(DMACR_INDEX) = '1' and different_thresh = '1')then
                irqthresh_wren <= '1';
            else
                irqthresh_wren <= '0';
            end if;
        end if;
    end process NEW_THRESH_WRITE;

-------------------------------------------------------------------------------
-- DMACR - Remainder of DMA Control Register, Bit 3 for Key hole operation
-------------------------------------------------------------------------------
DMACR_REGISTER : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                dmacr_i(DMACR_IRQTHRESH_LSB_BIT-1
                        downto DMACR_RESERVED5_BIT)   <= (others => '0');

            elsif(axi2ip_wrce(DMACR_INDEX) = '1')then
                dmacr_i(DMACR_IRQTHRESH_LSB_BIT-1       -- bit 15
                        downto DMACR_RESERVED5_BIT)   <= ZERO_VALUE(DMACR_RESERVED15_BIT)
                                                        -- bit 14
                                                        & axi2ip_wrdata(DMACR_ERR_IRQEN_BIT)
                                                        -- bit 13
                                                        & axi2ip_wrdata(DMACR_DLY_IRQEN_BIT)
                                                        -- bit 12
                                                        & axi2ip_wrdata(DMACR_IOC_IRQEN_BIT)
                                                        -- bits 11 downto 3
                                                        & ZERO_VALUE(DMACR_RESERVED11_BIT downto DMACR_RESERVED5_BIT);

            end if;
        end if;
    end process DMACR_REGISTER;


DMACR_REGISTER1 : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0' or C_ENABLE_MULTI_CHANNEL = 1)then
                dmacr_i(DMACR_KH_BIT)  <= '0';
                dmacr_i(CYCLIC_BIT)  <= '0';
            elsif(axi2ip_wrce(DMACR_INDEX) = '1')then
                dmacr_i(DMACR_KH_BIT)  <= axi2ip_wrdata(DMACR_KH_BIT);
                dmacr_i(CYCLIC_BIT)  <= axi2ip_wrdata(CYCLIC_BIT);
            end if;
        end if;
    end process DMACR_REGISTER1;

-------------------------------------------------------------------------------
-- DMACR - Reset Bit
-------------------------------------------------------------------------------
DMACR_RESET : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(soft_reset_clr = '1')then
                dmacr_i(DMACR_RESET_BIT)  <= '0';
            -- If soft reset set in other channel then set
            -- reset bit here too
            elsif(soft_reset_in = '1')then
                dmacr_i(DMACR_RESET_BIT)  <= '1';

            -- If DMACR Write then pass axi lite write bus to DMARC reset bit
            elsif(soft_reset_i = '0' and axi2ip_wrce(DMACR_INDEX) = '1')then
                dmacr_i(DMACR_RESET_BIT)  <= axi2ip_wrdata(DMACR_RESET_BIT);

            end if;
        end if;
    end process DMACR_RESET;

soft_reset_i <= dmacr_i(DMACR_RESET_BIT);

-------------------------------------------------------------------------------
-- Tail Pointer Enable fixed at 1 for this release of axi dma
-------------------------------------------------------------------------------
dmacr_i(DMACR_TAILPEN_BIT) <= '1';

-------------------------------------------------------------------------------
-- DMACR - Run/Stop Bit
-------------------------------------------------------------------------------
run_stop_clr <= '1' when error = '1'                -- MM2S DataMover Error
                      or error_in = '1'             -- S2MM Error
                      or stop_dma = '1'             -- Stop due to error
                      or soft_reset_i = '1'         -- MM2S Soft Reset
                      or soft_reset_in  = '1'       -- S2MM Soft Reset
           else '0';


DMACR_RUNSTOP : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                dmacr_i(DMACR_RS_BIT)  <= '0';
            -- Clear on sg error (i.e. error) or other channel
            -- error (i.e. error_in) or dma error or soft reset
            elsif(run_stop_clr = '1')then
                dmacr_i(DMACR_RS_BIT)  <= '0';
            elsif(axi2ip_wrce(DMACR_INDEX) = '1')then
                dmacr_i(DMACR_RS_BIT)  <= axi2ip_wrdata(DMACR_RS_BIT);
            end if;
        end if;
    end process DMACR_RUNSTOP;

---------------------------------------------------------------------------
-- DMA Status Halted bit (BIT 0) - Set by dma controller indicating DMA
-- channel is halted.
---------------------------------------------------------------------------
DMASR_HALTED : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0' or halted_set = '1')then
                halted <= '1';
            elsif(halted_clr = '1')then
                halted <= '0';
            end if;
        end if;
    end process DMASR_HALTED;

---------------------------------------------------------------------------
-- DMA Status Idle bit (BIT 1) - Set by dma controller indicating DMA
-- channel is IDLE waiting at tail pointer.  Update of Tail Pointer
-- will cause engine to resume.  Note: Halted channels return to a
-- reset condition.
---------------------------------------------------------------------------
DMASR_IDLE : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0'
            or idle_clr = '1'
            or halted_set = '1')then
                idle   <= '0';

            elsif(idle_set = '1')then
                idle   <= '1';
            end if;
        end if;
    end process DMASR_IDLE;

---------------------------------------------------------------------------
-- DMA Status Error bit (BIT 3)
-- Note: any error will cause entire engine to halt
---------------------------------------------------------------------------
error  <= dma_interr
            or dma_slverr
            or dma_decerr
            or sg_interr
            or sg_slverr
            or sg_decerr;

-- Scatter Gather Error
--sg_ftch_error <= ftch_interr_set or ftch_slverr_set or ftch_decerr_set;

-- SG Update Errors or DMA errors assert flag on descriptor update
-- Used to latch current descriptor pointer
--sg_updt_error <= updt_interr_set or updt_slverr_set or updt_decerr_set
--              or dma_interr or dma_slverr or dma_decerr;

-- Map out to halt opposing channel
error_out   <= error;

SG_FTCH_ERROR_PROC : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                sg_ftch_error <= '0';
                sg_updt_error <= '0';
            else
                sg_ftch_error <= ftch_interr_set or ftch_slverr_set or ftch_decerr_set;
                sg_updt_error <= updt_interr_set or updt_slverr_set or updt_decerr_set
                                 or dma_interr or dma_slverr or dma_decerr;
            end if;
        end if;
    end process SG_FTCH_ERROR_PROC;


---------------------------------------------------------------------------
-- DMA Status DMA Internal Error bit (BIT 4)
---------------------------------------------------------------------------
DMASR_DMAINTERR : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                dma_interr <= '0';
            elsif(dma_interr_set = '1' )then
                dma_interr <= '1';
            end if;
        end if;
    end process DMASR_DMAINTERR;

---------------------------------------------------------------------------
-- DMA Status DMA Slave Error bit (BIT 5)
---------------------------------------------------------------------------
DMASR_DMASLVERR : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                dma_slverr <= '0';

            elsif(dma_slverr_set = '1' )then
                dma_slverr <= '1';

            end if;
        end if;
    end process DMASR_DMASLVERR;

---------------------------------------------------------------------------
-- DMA Status DMA Decode Error bit (BIT 6)
---------------------------------------------------------------------------
DMASR_DMADECERR : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                dma_decerr <= '0';

            elsif(dma_decerr_set = '1' )then
                dma_decerr <= '1';

            end if;
        end if;
    end process DMASR_DMADECERR;

---------------------------------------------------------------------------
-- DMA Status SG Internal Error bit (BIT 8)
-- (SG Mode only - trimmed at build time if simple mode)
---------------------------------------------------------------------------
DMASR_SGINTERR : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                sg_interr <= '0';

            elsif(ftch_interr_set = '1' or updt_interr_set = '1')then
                sg_interr <= '1';


            end if;
        end if;
    end process DMASR_SGINTERR;

---------------------------------------------------------------------------
-- DMA Status SG Slave Error bit (BIT 9)
-- (SG Mode only - trimmed at build time if simple mode)
---------------------------------------------------------------------------
DMASR_SGSLVERR : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                sg_slverr <= '0';

            elsif(ftch_slverr_set = '1' or updt_slverr_set = '1')then
                sg_slverr <= '1';

            end if;
        end if;
    end process DMASR_SGSLVERR;

---------------------------------------------------------------------------
-- DMA Status SG Decode Error bit (BIT 10)
-- (SG Mode only - trimmed at build time if simple mode)
---------------------------------------------------------------------------
DMASR_SGDECERR : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                sg_decerr <= '0';

            elsif(ftch_decerr_set = '1' or updt_decerr_set = '1')then
                sg_decerr <= '1';

            end if;
        end if;
    end process DMASR_SGDECERR;

---------------------------------------------------------------------------
-- DMA Status IOC Interrupt status bit (BIT 11)
---------------------------------------------------------------------------
DMASR_IOCIRQ : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                ioc_irq <= '0';

            -- CPU Writing a '1' to clear - OR'ed with setting to prevent
            -- missing a 'set' during the write.
            elsif(axi2ip_wrce(DMASR_INDEX) = '1' )then

                ioc_irq <= (ioc_irq and not(axi2ip_wrdata(DMASR_IOCIRQ_BIT)))
                             or ioc_irq_set;

            elsif(ioc_irq_set = '1')then
                ioc_irq <= '1';

            end if;
        end if;
    end process DMASR_IOCIRQ;

---------------------------------------------------------------------------
-- DMA Status Delay Interrupt status bit (BIT 12)
---------------------------------------------------------------------------
DMASR_DLYIRQ : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                dly_irq <= '0';

            -- CPU Writing a '1' to clear - OR'ed with setting to prevent
            -- missing a 'set' during the write.
            elsif(axi2ip_wrce(DMASR_INDEX) = '1' )then

                dly_irq <= (dly_irq and not(axi2ip_wrdata(DMASR_DLYIRQ_BIT)))
                             or dly_irq_set;

            elsif(dly_irq_set = '1')then
                dly_irq <= '1';

            end if;
        end if;
    end process DMASR_DLYIRQ;

-- CR605888 Disable delay timer if halted or on delay irq set
--dlyirq_dsble    <= dmasr_i(DMASR_HALTED_BIT)              -- CR606348
dlyirq_dsble    <= not dmacr_i(DMACR_RS_BIT)                -- CR606348
                    or dmasr_i(DMASR_DLYIRQ_BIT);



---------------------------------------------------------------------------
-- DMA Status Error Interrupt status bit (BIT 12)
---------------------------------------------------------------------------
-- Delay error setting for generation of error strobe
GEN_ERROR_RE : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                error_d1 <= '0';
            else
                error_d1 <= error;
            end if;
        end if;
    end process GEN_ERROR_RE;

-- Generate rising edge pulse on error
error_re   <= error and not error_d1;

DMASR_ERRIRQ : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                err_irq <= '0';

            -- CPU Writing a '1' to clear - OR'ed with setting to prevent
            -- missing a 'set' during the write.
            elsif(axi2ip_wrce(DMASR_INDEX) = '1' )then

                err_irq <= (err_irq and not(axi2ip_wrdata(DMASR_ERRIRQ_BIT)))
                             or error_re;

            elsif(error_re = '1')then
                err_irq <= '1';

            end if;
        end if;
    end process DMASR_ERRIRQ;

---------------------------------------------------------------------------
-- DMA Interrupt OUT
---------------------------------------------------------------------------
REG_INTR : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0' or soft_reset_i = '1')then
                introut <= '0';
            else
                introut <= (dly_irq and dmacr_i(DMACR_DLY_IRQEN_BIT))
                        or (ioc_irq and dmacr_i(DMACR_IOC_IRQEN_BIT))
                        or (err_irq and dmacr_i(DMACR_ERR_IRQEN_BIT));
            end if;
        end if;
    end process;

---------------------------------------------------------------------------
-- DMA Status Register
---------------------------------------------------------------------------
dmasr_i    <=  irqdelay_status         -- Bits 31 downto 24
                    & irqthresh_status -- Bits 23 downto 16
                    & '0'              -- Bit  15
                    & err_irq          -- Bit  14
                    & dly_irq          -- Bit  13
                    & ioc_irq          -- Bit  12
                    & '0'              -- Bit  11
                    & sg_decerr        -- Bit  10
                    & sg_slverr        -- Bit  9
                    & sg_interr        -- Bit  8
                    & '0'              -- Bit  7
                    & dma_decerr       -- Bit  6
                    & dma_slverr       -- Bit  5
                    & dma_interr       -- Bit  4
                    & DMA_CONFIG       -- Bit  3
                    & '0'              -- Bit  2
                    & idle             -- Bit  1
                    & halted;          -- Bit  0





-- Generate current descriptor and tail descriptor register for Scatter Gather Mode
GEN_DESC_REG_FOR_SG : if C_INCLUDE_SG = 1 generate
begin

GEN_SG_CTL_REG : if C_ENABLE_MULTI_CHANNEL = 1 generate
begin

MM2S_SGCTL : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                sg_cache_info <= "00000011"; --(others => '0');

            elsif(axi2ip_wrce(SGCTL_INDEX) = '1' ) then

                sg_cache_info <= axi2ip_wrdata(11 downto 8) & axi2ip_wrdata(3 downto 0);

            else
                sg_cache_info <= sg_cache_info;

            end if;
        end if;
    end process MM2S_SGCTL;

      sg_ctl <= sg_cache_info;
  
end generate GEN_SG_CTL_REG;

GEN_SG_NO_CTL_REG : if C_ENABLE_MULTI_CHANNEL = 0 generate
begin

                sg_ctl <= "00000011"; --(others => '0');

end generate GEN_SG_NO_CTL_REG;

    -- Signals not used for Scatter Gather Mode, only simple mode
    buffer_address_i    <= (others => '0');
    buffer_length_i     <= (others => '0');
    buffer_length_wren  <= '0';

    ---------------------------------------------------------------------------
    -- Current Descriptor LSB Register
    ---------------------------------------------------------------------------
    CURDESC_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    curdesc_lsb_i  <= (others => '0');
                    error_pointer_set   <= '0';

                -- Detected error has NOT register a desc pointer
                elsif(error_pointer_set = '0')then

                    -- Scatter Gather Fetch Error
                    if(sg_ftch_error = '1' or sg_updt_error = '1')then
                        curdesc_lsb_i       <= ftch_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 6);
                        error_pointer_set   <= '1';
                    -- Scatter Gather Update Error
            --        elsif(sg_updt_error = '1')then
            --            curdesc_lsb_i       <= updt_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
             --           error_pointer_set   <= '1';

                    -- Commanded to update descriptor value - used for indicating
                    -- current descriptor begin processed by dma controller
                    elsif(update_curdesc = '1' and dmacr_i(DMACR_RS_BIT)  = '1')then
                        curdesc_lsb_i       <= new_curdesc(C_S_AXI_LITE_DATA_WIDTH-1 downto 6);
                        error_pointer_set   <= '0';

                    -- CPU update of current descriptor pointer.  CPU
                    -- only allowed to update when engine is halted.
                    elsif(axi2ip_wrce(CURDESC_LSB_INDEX) = '1' and dmasr_i(DMASR_HALTED_BIT) = '1')then
                        curdesc_lsb_i       <= axi2ip_wrdata(CURDESC_LOWER_MSB_BIT
                                                      downto CURDESC_LOWER_LSB_BIT);
--                                              & ZERO_VALUE(CURDESC_RESERVED_BIT5
--                                                      downto CURDESC_RESERVED_BIT0);
                        error_pointer_set   <= '0';

                    end if;
                end if;
            end if;
        end process CURDESC_LSB_REGISTER;

                  

    ---------------------------------------------------------------------------
    -- Tail Descriptor LSB Register
    ---------------------------------------------------------------------------
    TAILDESC_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    taildesc_lsb_i  <= (others => '0');
                elsif(axi2ip_wrce(TAILDESC_LSB_INDEX) = '1')then
                    taildesc_lsb_i  <= axi2ip_wrdata(TAILDESC_LOWER_MSB_BIT
                                              downto TAILDESC_LOWER_LSB_BIT);
--                                       & ZERO_VALUE(TAILDESC_RESERVED_BIT5
--                                              downto TAILDESC_RESERVED_BIT0);

                end if;
            end if;
        end process TAILDESC_LSB_REGISTER;

    ---------------------------------------------------------------------------
    -- Current Descriptor MSB Register
    ---------------------------------------------------------------------------
    -- Scatter Gather Interface configured for 64-Bit SG Addresses
    GEN_SG_ADDR_EQL64 :if C_M_AXI_SG_ADDR_WIDTH = 64 generate
    begin
        CURDESC_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        curdesc_msb_i  <= (others => '0');

                    elsif(error_pointer_set = '0')then
                        -- Scatter Gather Fetch Error
                        if(sg_ftch_error = '1'  or sg_updt_error = '1')then
                            curdesc_msb_i   <= ftch_error_addr(C_M_AXI_SG_ADDR_WIDTH - 1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- Scatter Gather Update Error
              --          elsif(sg_updt_error = '1')then
              --              curdesc_msb_i   <= updt_error_addr((C_M_AXI_SG_ADDR_WIDTH
              --                                  - C_S_AXI_LITE_DATA_WIDTH)-1
              --                                  downto 0);

                        -- Commanded to update descriptor value - used for indicating
                        -- current descriptor begin processed by dma controller
                        elsif(update_curdesc = '1' and dmacr_i(DMACR_RS_BIT)  = '1')then
                            curdesc_msb_i <= new_curdesc (C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- CPU update of current descriptor pointer.  CPU
                        -- only allowed to update when engine is halted.
                        elsif(axi2ip_wrce(CURDESC_MSB_INDEX) = '1' and dmasr_i(DMASR_HALTED_BIT) = '1')then
                            curdesc_msb_i  <= axi2ip_wrdata;

                        end if;
                    end if;
                end if;
            end process CURDESC_MSB_REGISTER;

        ---------------------------------------------------------------------------
        -- Tail Descriptor MSB Register
        ---------------------------------------------------------------------------
        TAILDESC_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        taildesc_msb_i  <= (others => '0');
                    elsif(axi2ip_wrce(TAILDESC_MSB_INDEX) = '1')then
                        taildesc_msb_i  <= axi2ip_wrdata;
                    end if;
                end if;
            end process TAILDESC_MSB_REGISTER;

        end generate GEN_SG_ADDR_EQL64;

    -- Scatter Gather Interface configured for 32-Bit SG Addresses
    GEN_SG_ADDR_EQL32 : if C_M_AXI_SG_ADDR_WIDTH = 32 generate
    begin
        curdesc_msb_i  <= (others => '0');
        taildesc_msb_i <= (others => '0');
    end generate GEN_SG_ADDR_EQL32;


    -- Scatter Gather Interface configured for 32-Bit SG Addresses
    GEN_TAILUPDATE_EQL32 : if C_M_AXI_SG_ADDR_WIDTH = 32 generate
    begin
        TAILPNTR_UPDT_PROCESS : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0' or dmacr_i(DMACR_RS_BIT)='0')then
                        tailpntr_updated_d1    <= '0';
                    elsif(axi2ip_wrce(TAILDESC_LSB_INDEX) = '1')then
                        tailpntr_updated_d1    <= '1';
                    else
                        tailpntr_updated_d1    <= '0';
                    end if;
                end if;
            end process TAILPNTR_UPDT_PROCESS;

        TAILPNTR_UPDT_PROCESS_DEL : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        tailpntr_updated_d2    <= '0';
                    else
                        tailpntr_updated_d2    <= tailpntr_updated_d1;
                    end if;
                end if;
            end process TAILPNTR_UPDT_PROCESS_DEL;

          tailpntr_updated <= tailpntr_updated_d1 and (not tailpntr_updated_d2);

    end generate GEN_TAILUPDATE_EQL32;

    -- Scatter Gather Interface configured for 64-Bit SG Addresses
    GEN_TAILUPDATE_EQL64 : if C_M_AXI_SG_ADDR_WIDTH = 64 generate
    begin
        TAILPNTR_UPDT_PROCESS : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0' or dmacr_i(DMACR_RS_BIT)='0')then
                        tailpntr_updated_d1    <= '0';
                    elsif(axi2ip_wrce(TAILDESC_MSB_INDEX) = '1')then
                        tailpntr_updated_d1    <= '1';
                    else
                        tailpntr_updated_d1    <= '0';
                    end if;
                end if;
            end process TAILPNTR_UPDT_PROCESS;

        TAILPNTR_UPDT_PROCESS_DEL : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        tailpntr_updated_d2    <= '0';
                    else
                        tailpntr_updated_d2    <= tailpntr_updated_d1;
                    end if;
                end if;
            end process TAILPNTR_UPDT_PROCESS_DEL;

          tailpntr_updated <= tailpntr_updated_d1 and (not tailpntr_updated_d2);



    end generate GEN_TAILUPDATE_EQL64;

end generate GEN_DESC_REG_FOR_SG;


-- Generate Buffer Address and Length Register for Simple DMA Mode
GEN_REG_FOR_SMPL : if C_INCLUDE_SG = 0 generate
begin
    -- Signals not used for simple dma mode, only for sg mode
    curdesc_lsb_i       <= (others => '0');
    curdesc_msb_i       <= (others => '0');
    taildesc_lsb_i      <= (others => '0');
    taildesc_msb_i      <= (others => '0');
    tailpntr_updated    <= '0';
    error_pointer_set   <= '0';

    -- Buffer Address register.  Used for Source Address (SA) if MM2S
    -- and used for Destination Address (DA) if S2MM
    BUFFER_ADDR_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    buffer_address_i  <= (others => '0');
                elsif(axi2ip_wrce(BUFF_ADDRESS_INDEX) = '1')then
                    buffer_address_i  <= axi2ip_wrdata;
                end if;
            end if;
        end process BUFFER_ADDR_REGISTER;

    GEN_BUFF_ADDR_EQL64 : if C_M_AXI_SG_ADDR_WIDTH > 32 generate
    begin
    BUFFER_ADDR_REGISTER1 : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    buffer_address_i_64  <= (others => '0');
                elsif(axi2ip_wrce(BUFF_ADDRESS_MSB_INDEX) = '1')then
                    buffer_address_i_64  <= axi2ip_wrdata;
                end if;
            end if;
        end process BUFFER_ADDR_REGISTER1;


    end generate GEN_BUFF_ADDR_EQL64;

    -- Buffer Length register.  Used for number of bytes to transfer if MM2S
    -- and used for size of receive buffer is S2MM
    BUFFER_LNGTH_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    buffer_length_i  <= (others => '0');

                -- Update with actual bytes received (Only for S2MM channel)
       --         elsif(bytes_received_wren = '1')then
       --             buffer_length_i <= bytes_received;

                elsif(axi2ip_wrce(BUFF_LENGTH_INDEX) = '1')then
                    buffer_length_i  <= axi2ip_wrdata(C_SG_LENGTH_WIDTH-1 downto 0);
                end if;
            end if;
        end process BUFFER_LNGTH_REGISTER;

    -- Buffer Length Write Enable control.  Assertion of wren will
    -- begin a transfer if channel is Idle.
    BUFFER_LNGTH_WRITE : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    buffer_length_wren <= '0';
                -- Non-zero length value written
                elsif(axi2ip_wrce(BUFF_LENGTH_INDEX) = '1'
                and axi2ip_wrdata(C_SG_LENGTH_WIDTH-1 downto 0) /= ZERO_VALUE(C_SG_LENGTH_WIDTH-1 downto 0))then
                    buffer_length_wren <= '1';
                else
                    buffer_length_wren <= '0';
                end if;
            end if;
        end process BUFFER_LNGTH_WRITE;

end generate GEN_REG_FOR_SMPL;



end implementation;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:        axi_dma_register_s2mm.vhd
--
-- Description:     This entity encompasses the channel register set.
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;

-------------------------------------------------------------------------------
entity  axi_dma_register_s2mm is
    generic(
        C_NUM_REGISTERS             : integer                   := 11       ;
        C_INCLUDE_SG                : integer                   := 1        ;
        C_SG_LENGTH_WIDTH           : integer range 8 to 23     := 14       ;
        C_S_AXI_LITE_DATA_WIDTH     : integer range 32 to 32    := 32       ;
        C_M_AXI_SG_ADDR_WIDTH       : integer range 32 to 64    := 32       ;
        C_NUM_S2MM_CHANNELS         : integer range 1 to 16     := 1        ;
        C_MICRO_DMA                 : integer range 0 to 1      := 0        ;
        C_ENABLE_MULTI_CHANNEL             : integer range 0 to 1      := 0
        --C_CHANNEL_IS_S2MM           : integer range 0 to 1      := 0 CR603034
    );
    port (
        m_axi_sg_aclk               : in  std_logic                         ;          --
        m_axi_sg_aresetn            : in  std_logic                         ;          --
                                                                                       --
        -- AXI Interface Control                                                       --
        axi2ip_wrce                 : in  std_logic_vector                             --
                                        (C_NUM_REGISTERS-1 downto 0)        ;          --
        axi2ip_wrdata               : in  std_logic_vector                             --
                                        (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);          --
                                                                                       --
        -- DMASR Control                                                               --
        stop_dma                    : in  std_logic                         ;          --
        halted_clr                  : in  std_logic                         ;          --
        halted_set                  : in  std_logic                         ;          --
        idle_set                    : in  std_logic                         ;          --
        idle_clr                    : in  std_logic                         ;          --
        ioc_irq_set                 : in  std_logic                         ;          --
        dly_irq_set                 : in  std_logic                         ;          --
        irqdelay_status             : in  std_logic_vector(7 downto 0)      ;          --
        irqthresh_status            : in  std_logic_vector(7 downto 0)      ;          --
        irqthresh_wren              : out std_logic                         ;          --
        irqdelay_wren               : out std_logic                         ;          --
        dlyirq_dsble                : out std_logic                         ;          -- CR605888
                                                                                       --
        -- Error Control                                                               --
        dma_interr_set              : in  std_logic                         ;          --
        dma_slverr_set              : in  std_logic                         ;          --
        dma_decerr_set              : in  std_logic                         ;          --
        ftch_interr_set             : in  std_logic                         ;          --
        ftch_slverr_set             : in  std_logic                         ;          --
        ftch_decerr_set             : in  std_logic                         ;          --
        ftch_error_addr             : in  std_logic_vector                             --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;          --
        updt_interr_set             : in  std_logic                         ;          --
        updt_slverr_set             : in  std_logic                         ;          --
        updt_decerr_set             : in  std_logic                         ;          --
        updt_error_addr             : in  std_logic_vector                             --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;          --
        error_in                    : in  std_logic                         ;          --
        error_out                   : out std_logic                         ;          --
        introut                     : out std_logic                         ;          --
        soft_reset_in               : in  std_logic                         ;          --
        soft_reset_clr              : in  std_logic                         ;          --
                                                                                       --
        -- CURDESC Update                                                              --
        update_curdesc              : in  std_logic                         ;          --
        tdest_in                    : in  std_logic_vector (5 downto 0)     ;
        new_curdesc                 : in  std_logic_vector                             --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;          --
        -- TAILDESC Update                                                             --
        tailpntr_updated            : out std_logic                         ;          --
                                                                                       --
        -- Channel Register Out                                                        --
        sg_ctl                      : out std_logic_vector (7 downto 0)     ;

        dmacr                       : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        dmasr                       : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        curdesc_lsb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        curdesc_msb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc_lsb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc_msb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --

        curdesc1_lsb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        curdesc1_msb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc1_lsb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc1_msb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --

        curdesc2_lsb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        curdesc2_msb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc2_lsb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc2_msb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --

        curdesc3_lsb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        curdesc3_msb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc3_lsb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc3_msb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --

        curdesc4_lsb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        curdesc4_msb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc4_lsb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc4_msb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --

        curdesc5_lsb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        curdesc5_msb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc5_lsb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc5_msb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --

        curdesc6_lsb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        curdesc6_msb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc6_lsb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc6_msb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --

        curdesc7_lsb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        curdesc7_msb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc7_lsb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc7_msb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --

        curdesc8_lsb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        curdesc8_msb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc8_lsb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc8_msb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --

        curdesc9_lsb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        curdesc9_msb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc9_lsb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc9_msb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --

        curdesc10_lsb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        curdesc10_msb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc10_lsb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc10_msb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --

        curdesc11_lsb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        curdesc11_msb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc11_lsb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc11_msb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --

        curdesc12_lsb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        curdesc12_msb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc12_lsb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc12_msb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --

        curdesc13_lsb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        curdesc13_msb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc13_lsb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc13_msb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --

        curdesc14_lsb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        curdesc14_msb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc14_lsb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc14_msb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --

        curdesc15_lsb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        curdesc15_msb                 : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc15_lsb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --
        taildesc15_msb                : out std_logic_vector                             --
                                           (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);       --

        buffer_address              : out std_logic_vector                             --
                                           (C_M_AXI_SG_ADDR_WIDTH-1 downto 0);       --
        buffer_length               : out std_logic_vector                             --
                                           (C_SG_LENGTH_WIDTH-1 downto 0)   ;          --
        buffer_length_wren          : out std_logic                         ;          --
        bytes_received              : in  std_logic_vector                             --
                                           (C_SG_LENGTH_WIDTH-1 downto 0)   ;          --
        bytes_received_wren         : in  std_logic                                    --
    );                                                                                 --
end axi_dma_register_s2mm;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_dma_register_s2mm is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------
constant SGCTL_INDEX            : integer := 0;
constant DMACR_INDEX            : integer := 1;                     -- DMACR Register index
constant DMASR_INDEX            : integer := 2;                     -- DMASR Register index
constant CURDESC_LSB_INDEX      : integer := 3;                     -- CURDESC LSB Reg index
constant CURDESC_MSB_INDEX      : integer := 4;                     -- CURDESC MSB Reg index
constant TAILDESC_LSB_INDEX     : integer := 5;                     -- TAILDESC LSB Reg index
constant TAILDESC_MSB_INDEX     : integer := 6;                     -- TAILDESC MSB Reg index

constant CURDESC1_LSB_INDEX      : integer := 17;                     -- CURDESC LSB Reg index
constant CURDESC1_MSB_INDEX      : integer := 18;                     -- CURDESC MSB Reg index
constant TAILDESC1_LSB_INDEX     : integer := 19;                     -- TAILDESC LSB Reg index
constant TAILDESC1_MSB_INDEX     : integer := 20;                     -- TAILDESC MSB Reg index

constant CURDESC2_LSB_INDEX      : integer := 25;                     -- CURDESC LSB Reg index
constant CURDESC2_MSB_INDEX      : integer := 26;                     -- CURDESC MSB Reg index
constant TAILDESC2_LSB_INDEX     : integer := 27;                     -- TAILDESC LSB Reg index
constant TAILDESC2_MSB_INDEX     : integer := 28;                     -- TAILDESC MSB Reg index

constant CURDESC3_LSB_INDEX      : integer := 33;                     -- CURDESC LSB Reg index
constant CURDESC3_MSB_INDEX      : integer := 34;                     -- CURDESC MSB Reg index
constant TAILDESC3_LSB_INDEX     : integer := 35;                     -- TAILDESC LSB Reg index
constant TAILDESC3_MSB_INDEX     : integer := 36;                     -- TAILDESC MSB Reg index

constant CURDESC4_LSB_INDEX      : integer := 41;                     -- CURDESC LSB Reg index
constant CURDESC4_MSB_INDEX      : integer := 42;                     -- CURDESC MSB Reg index
constant TAILDESC4_LSB_INDEX     : integer := 43;                     -- TAILDESC LSB Reg index
constant TAILDESC4_MSB_INDEX     : integer := 44;                     -- TAILDESC MSB Reg index

constant CURDESC5_LSB_INDEX      : integer := 49;                     -- CURDESC LSB Reg index
constant CURDESC5_MSB_INDEX      : integer := 50;                     -- CURDESC MSB Reg index
constant TAILDESC5_LSB_INDEX     : integer := 51;                     -- TAILDESC LSB Reg index
constant TAILDESC5_MSB_INDEX     : integer := 52;                     -- TAILDESC MSB Reg index

constant CURDESC6_LSB_INDEX      : integer := 57;                     -- CURDESC LSB Reg index
constant CURDESC6_MSB_INDEX      : integer := 58;                     -- CURDESC MSB Reg index
constant TAILDESC6_LSB_INDEX     : integer := 59;                     -- TAILDESC LSB Reg index
constant TAILDESC6_MSB_INDEX     : integer := 60;                     -- TAILDESC MSB Reg index

constant CURDESC7_LSB_INDEX      : integer := 65;                     -- CURDESC LSB Reg index
constant CURDESC7_MSB_INDEX      : integer := 66;                     -- CURDESC MSB Reg index
constant TAILDESC7_LSB_INDEX     : integer := 67;                     -- TAILDESC LSB Reg index
constant TAILDESC7_MSB_INDEX     : integer := 68;                     -- TAILDESC MSB Reg index

constant CURDESC8_LSB_INDEX      : integer := 73;                     -- CURDESC LSB Reg index
constant CURDESC8_MSB_INDEX      : integer := 74;                     -- CURDESC MSB Reg index
constant TAILDESC8_LSB_INDEX     : integer := 75;                     -- TAILDESC LSB Reg index
constant TAILDESC8_MSB_INDEX     : integer := 76;                     -- TAILDESC MSB Reg index

constant CURDESC9_LSB_INDEX      : integer := 81;                     -- CURDESC LSB Reg index
constant CURDESC9_MSB_INDEX      : integer := 82;                     -- CURDESC MSB Reg index
constant TAILDESC9_LSB_INDEX     : integer := 83;                     -- TAILDESC LSB Reg index
constant TAILDESC9_MSB_INDEX     : integer := 84;                     -- TAILDESC MSB Reg index

constant CURDESC10_LSB_INDEX      : integer := 89;                     -- CURDESC LSB Reg index
constant CURDESC10_MSB_INDEX      : integer := 90;                     -- CURDESC MSB Reg index
constant TAILDESC10_LSB_INDEX     : integer := 91;                     -- TAILDESC LSB Reg index
constant TAILDESC10_MSB_INDEX     : integer := 92;                     -- TAILDESC MSB Reg index

constant CURDESC11_LSB_INDEX      : integer := 97;                     -- CURDESC LSB Reg index
constant CURDESC11_MSB_INDEX      : integer := 98;                     -- CURDESC MSB Reg index
constant TAILDESC11_LSB_INDEX     : integer := 99;                     -- TAILDESC LSB Reg index
constant TAILDESC11_MSB_INDEX     : integer := 100;                     -- TAILDESC MSB Reg index

constant CURDESC12_LSB_INDEX      : integer := 105;                     -- CURDESC LSB Reg index
constant CURDESC12_MSB_INDEX      : integer := 106;                     -- CURDESC MSB Reg index
constant TAILDESC12_LSB_INDEX     : integer := 107;                     -- TAILDESC LSB Reg index
constant TAILDESC12_MSB_INDEX     : integer := 108;                     -- TAILDESC MSB Reg index

constant CURDESC13_LSB_INDEX      : integer := 113;                     -- CURDESC LSB Reg index
constant CURDESC13_MSB_INDEX      : integer := 114;                     -- CURDESC MSB Reg index
constant TAILDESC13_LSB_INDEX     : integer := 115;                     -- TAILDESC LSB Reg index
constant TAILDESC13_MSB_INDEX     : integer := 116;                     -- TAILDESC MSB Reg index

constant CURDESC14_LSB_INDEX      : integer := 121;                     -- CURDESC LSB Reg index
constant CURDESC14_MSB_INDEX      : integer := 122;                     -- CURDESC MSB Reg index
constant TAILDESC14_LSB_INDEX     : integer := 123;                     -- TAILDESC LSB Reg index
constant TAILDESC14_MSB_INDEX     : integer := 124;                     -- TAILDESC MSB Reg index

constant CURDESC15_LSB_INDEX      : integer := 129;                     -- CURDESC LSB Reg index
constant CURDESC15_MSB_INDEX      : integer := 130;                     -- CURDESC MSB Reg index
constant TAILDESC15_LSB_INDEX     : integer := 131;                     -- TAILDESC LSB Reg index
constant TAILDESC15_MSB_INDEX     : integer := 132;                     -- TAILDESC MSB Reg index


-- CR603034 moved s2mm back to offset 6
--constant SA_ADDRESS_INDEX       : integer := 6;                     -- Buffer Address Reg (SA)
--constant DA_ADDRESS_INDEX       : integer := 8;                     -- Buffer Address Reg (DA)
--
--
--constant BUFF_ADDRESS_INDEX     : integer := address_index_select   -- Buffer Address Reg (SA or DA)
--                                                    (C_CHANNEL_IS_S2MM, -- Channel Type 1=rx 0=tx
--                                                     SA_ADDRESS_INDEX,  -- Source Address Index
--                                                     DA_ADDRESS_INDEX); -- Destination Address Index
constant BUFF_ADDRESS_INDEX     : integer := 7;
constant BUFF_ADDRESS_MSB_INDEX     : integer := 8;
constant BUFF_LENGTH_INDEX      : integer := 11;                    -- Buffer Length Reg

constant ZERO_VALUE             : std_logic_vector(31 downto 0) := (others => '0');

constant DMA_CONFIG             : std_logic_vector(0 downto 0)
                                    := std_logic_vector(to_unsigned(C_INCLUDE_SG,1));

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
signal dmacr_i              : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal dmasr_i              : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal curdesc_lsb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 6) := (others => '0');
signal curdesc_msb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc_lsb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 6) := (others => '0');
signal taildesc_msb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal buffer_address_i     : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal buffer_address_64_i     : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal buffer_length_i      : std_logic_vector
                                (C_SG_LENGTH_WIDTH-1 downto 0)       := (others => '0');

signal curdesc1_lsb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal curdesc1_msb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc1_lsb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc1_msb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal curdesc2_lsb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal curdesc2_msb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc2_lsb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc2_msb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal curdesc3_lsb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal curdesc3_msb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc3_lsb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc3_msb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal curdesc4_lsb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal curdesc4_msb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc4_lsb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc4_msb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal curdesc5_lsb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal curdesc5_msb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc5_lsb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc5_msb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal curdesc6_lsb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal curdesc6_msb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc6_lsb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc6_msb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal curdesc7_lsb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal curdesc7_msb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc7_lsb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc7_msb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal curdesc8_lsb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal curdesc8_msb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc8_lsb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc8_msb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal curdesc9_lsb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal curdesc9_msb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc9_lsb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc9_msb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal curdesc10_lsb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal curdesc10_msb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc10_lsb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc10_msb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal curdesc11_lsb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal curdesc11_msb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc11_lsb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc11_msb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal curdesc12_lsb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal curdesc12_msb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc12_lsb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc12_msb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal curdesc13_lsb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal curdesc13_msb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc13_lsb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc13_msb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal curdesc14_lsb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal curdesc14_msb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc14_lsb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc14_msb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal curdesc15_lsb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal curdesc15_msb_i        : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc15_lsb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal taildesc15_msb_i       : std_logic_vector
                                (C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal update_curdesc1       : std_logic := '0';
signal update_curdesc2       : std_logic := '0';
signal update_curdesc3       : std_logic := '0';
signal update_curdesc4       : std_logic := '0';
signal update_curdesc5       : std_logic := '0';
signal update_curdesc6       : std_logic := '0';
signal update_curdesc7       : std_logic := '0';
signal update_curdesc8       : std_logic := '0';
signal update_curdesc9       : std_logic := '0';
signal update_curdesc10       : std_logic := '0';
signal update_curdesc11       : std_logic := '0';
signal update_curdesc12       : std_logic := '0';
signal update_curdesc13       : std_logic := '0';
signal update_curdesc14       : std_logic := '0';
signal update_curdesc15       : std_logic := '0';


signal dest0       : std_logic := '0';
signal dest1       : std_logic := '0';
signal dest2       : std_logic := '0';
signal dest3       : std_logic := '0';
signal dest4       : std_logic := '0';
signal dest5       : std_logic := '0';
signal dest6       : std_logic := '0';
signal dest7       : std_logic := '0';
signal dest8       : std_logic := '0';
signal dest9       : std_logic := '0';
signal dest10       : std_logic := '0';
signal dest11       : std_logic := '0';
signal dest12       : std_logic := '0';
signal dest13       : std_logic := '0';
signal dest14       : std_logic := '0';
signal dest15       : std_logic := '0';

-- DMASR Signals
signal halted               : std_logic := '0';
signal idle                 : std_logic := '0';
signal cmplt                : std_logic := '0';
signal error                : std_logic := '0';
signal dma_interr           : std_logic := '0';
signal dma_slverr           : std_logic := '0';
signal dma_decerr           : std_logic := '0';
signal sg_interr            : std_logic := '0';
signal sg_slverr            : std_logic := '0';
signal sg_decerr            : std_logic := '0';
signal ioc_irq              : std_logic := '0';
signal dly_irq              : std_logic := '0';
signal error_d1             : std_logic := '0';
signal error_re             : std_logic := '0';
signal err_irq              : std_logic := '0';

signal sg_ftch_error        : std_logic := '0';
signal sg_updt_error        : std_logic := '0';
signal error_pointer_set    : std_logic := '0';
signal error_pointer_set1    : std_logic := '0';
signal error_pointer_set2    : std_logic := '0';
signal error_pointer_set3    : std_logic := '0';
signal error_pointer_set4    : std_logic := '0';
signal error_pointer_set5    : std_logic := '0';
signal error_pointer_set6    : std_logic := '0';
signal error_pointer_set7    : std_logic := '0';
signal error_pointer_set8    : std_logic := '0';
signal error_pointer_set9    : std_logic := '0';
signal error_pointer_set10    : std_logic := '0';
signal error_pointer_set11    : std_logic := '0';
signal error_pointer_set12    : std_logic := '0';
signal error_pointer_set13    : std_logic := '0';
signal error_pointer_set14    : std_logic := '0';
signal error_pointer_set15    : std_logic := '0';

-- interrupt coalescing support signals
signal different_delay      : std_logic := '0';
signal different_thresh     : std_logic := '0';
signal threshold_is_zero    : std_logic := '0';
-- soft reset support signals
signal soft_reset_i         : std_logic := '0';
signal run_stop_clr         : std_logic := '0';

signal tail_update_lsb      : std_logic := '0';
signal tail_update_msb      : std_logic := '0';

signal sg_cache_info        : std_logic_vector (7 downto 0);

signal halt_free : std_logic := '0';
signal tmp11 : std_logic := '0';
signal sig_cur_updated : std_logic := '0';

signal tailpntr_updated_d1 : std_logic;
signal tailpntr_updated_d2 : std_logic;

-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin

GEN_MULTI_CH : if C_ENABLE_MULTI_CHANNEL = 1 generate
begin
   halt_free <= '1';

end generate GEN_MULTI_CH;

GEN_NOMULTI_CH : if C_ENABLE_MULTI_CHANNEL = 0 generate
begin
   halt_free <= dmasr_i(DMASR_HALTED_BIT);

end generate GEN_NOMULTI_CH;

GEN_DESC_UPDATE_FOR_SG : if C_NUM_S2MM_CHANNELS = 1 generate
begin
update_curdesc1  <= '0';
update_curdesc2  <= '0';
update_curdesc3  <= '0';
update_curdesc4  <= '0';
update_curdesc5  <= '0';
update_curdesc6  <= '0';
update_curdesc7  <= '0';
update_curdesc8  <= '0';
update_curdesc9  <= '0';
update_curdesc10 <= '0';
update_curdesc11 <= '0';
update_curdesc12 <= '0';
update_curdesc13 <= '0';
update_curdesc14 <= '0';
update_curdesc15 <= '0';

end generate GEN_DESC_UPDATE_FOR_SG;


dest0  <= '1' when tdest_in (4 downto 0) =  "00000" else '0';
dest1  <= '1' when tdest_in (4 downto 0) =  "00001" else '0';
dest2  <= '1' when tdest_in (4 downto 0) =  "00010" else '0';
dest3  <= '1' when tdest_in (4 downto 0) =  "00011" else '0';
dest4  <= '1' when tdest_in (4 downto 0) =  "00100" else '0';
dest5  <= '1' when tdest_in (4 downto 0) =  "00101" else '0';
dest6  <= '1' when tdest_in (4 downto 0) =  "00110" else '0';
dest7  <= '1' when tdest_in (4 downto 0) =  "00111" else '0';
dest8  <= '1' when tdest_in (4 downto 0) =  "01000" else '0';
dest9  <= '1' when tdest_in (4 downto 0) =  "01001" else '0';
dest10 <= '1' when tdest_in (4 downto 0) =  "01010" else '0';
dest11 <= '1' when tdest_in (4 downto 0) =  "01011" else '0';
dest12 <= '1' when tdest_in (4 downto 0) =  "01100" else '0';
dest13 <= '1' when tdest_in (4 downto 0) =  "01101" else '0';
dest14 <= '1' when tdest_in (4 downto 0) =  "01110" else '0';
dest15 <= '1' when tdest_in (4 downto 0) =  "01111" else '0';


GEN_DESC_UPDATE_FOR_SG_CH : if C_NUM_S2MM_CHANNELS > 1 generate

update_curdesc1  <= update_curdesc when tdest_in (4 downto 0) =  "00001" else '0';
update_curdesc2  <= update_curdesc when tdest_in (4 downto 0) =  "00010" else '0';
update_curdesc3  <= update_curdesc when tdest_in (4 downto 0) =  "00011" else '0';
update_curdesc4  <= update_curdesc when tdest_in (4 downto 0) =  "00100" else '0';
update_curdesc5  <= update_curdesc when tdest_in (4 downto 0) =  "00101" else '0';
update_curdesc6  <= update_curdesc when tdest_in (4 downto 0) =  "00110" else '0';
update_curdesc7  <= update_curdesc when tdest_in (4 downto 0) =  "00111" else '0';
update_curdesc8  <= update_curdesc when tdest_in (4 downto 0) =  "01000" else '0';
update_curdesc9  <= update_curdesc when tdest_in (4 downto 0) =  "01001" else '0';
update_curdesc10 <= update_curdesc when tdest_in (4 downto 0) =  "01010" else '0';
update_curdesc11 <= update_curdesc when tdest_in (4 downto 0) =  "01011" else '0';
update_curdesc12 <= update_curdesc when tdest_in (4 downto 0) =  "01100" else '0';
update_curdesc13 <= update_curdesc when tdest_in (4 downto 0) =  "01101" else '0';
update_curdesc14 <= update_curdesc when tdest_in (4 downto 0) =  "01110" else '0';
update_curdesc15 <= update_curdesc when tdest_in (4 downto 0) =  "01111" else '0';
end generate GEN_DESC_UPDATE_FOR_SG_CH;

    GEN_DA_ADDR_EQL64 : if C_M_AXI_SG_ADDR_WIDTH > 32 generate
    begin

    buffer_address          <= buffer_address_64_i & buffer_address_i ;

    end generate GEN_DA_ADDR_EQL64;

    GEN_DA_ADDR_EQL32 : if C_M_AXI_SG_ADDR_WIDTH = 32 generate
    begin

    buffer_address          <= buffer_address_i ;

    end generate GEN_DA_ADDR_EQL32;


dmacr                   <= dmacr_i          ;
dmasr                   <= dmasr_i          ;
curdesc_lsb             <= curdesc_lsb_i (31 downto 6) & "000000"    ;
curdesc_msb             <= curdesc_msb_i    ;
taildesc_lsb            <= taildesc_lsb_i (31 downto 6) & "000000"   ;
taildesc_msb            <= taildesc_msb_i   ;
buffer_length           <= buffer_length_i  ;

curdesc1_lsb             <= curdesc1_lsb_i    ;
curdesc1_msb             <= curdesc1_msb_i    ;
taildesc1_lsb            <= taildesc1_lsb_i   ;
taildesc1_msb            <= taildesc1_msb_i   ;

curdesc2_lsb             <= curdesc2_lsb_i    ;
curdesc2_msb             <= curdesc2_msb_i    ;
taildesc2_lsb            <= taildesc2_lsb_i   ;
taildesc2_msb            <= taildesc2_msb_i   ;

curdesc3_lsb             <= curdesc3_lsb_i    ;
curdesc3_msb             <= curdesc3_msb_i    ;
taildesc3_lsb            <= taildesc3_lsb_i   ;
taildesc3_msb            <= taildesc3_msb_i   ;

curdesc4_lsb             <= curdesc4_lsb_i    ;
curdesc4_msb             <= curdesc4_msb_i    ;
taildesc4_lsb            <= taildesc4_lsb_i   ;
taildesc4_msb            <= taildesc4_msb_i   ;

curdesc5_lsb             <= curdesc5_lsb_i    ;
curdesc5_msb             <= curdesc5_msb_i    ;
taildesc5_lsb            <= taildesc5_lsb_i   ;
taildesc5_msb            <= taildesc5_msb_i   ;

curdesc6_lsb             <= curdesc6_lsb_i    ;
curdesc6_msb             <= curdesc6_msb_i    ;
taildesc6_lsb            <= taildesc6_lsb_i   ;
taildesc6_msb            <= taildesc6_msb_i   ;

curdesc7_lsb             <= curdesc7_lsb_i    ;
curdesc7_msb             <= curdesc7_msb_i    ;
taildesc7_lsb            <= taildesc7_lsb_i   ;
taildesc7_msb            <= taildesc7_msb_i   ;

curdesc8_lsb             <= curdesc8_lsb_i    ;
curdesc8_msb             <= curdesc8_msb_i    ;
taildesc8_lsb            <= taildesc8_lsb_i   ;
taildesc8_msb            <= taildesc8_msb_i   ;

curdesc9_lsb             <= curdesc9_lsb_i    ;
curdesc9_msb             <= curdesc9_msb_i    ;
taildesc9_lsb            <= taildesc9_lsb_i   ;
taildesc9_msb            <= taildesc9_msb_i   ;

curdesc10_lsb             <= curdesc10_lsb_i    ;
curdesc10_msb             <= curdesc10_msb_i    ;
taildesc10_lsb            <= taildesc10_lsb_i   ;
taildesc10_msb            <= taildesc10_msb_i   ;

curdesc11_lsb             <= curdesc11_lsb_i    ;
curdesc11_msb             <= curdesc11_msb_i    ;
taildesc11_lsb            <= taildesc11_lsb_i   ;
taildesc11_msb            <= taildesc11_msb_i   ;

curdesc12_lsb             <= curdesc12_lsb_i    ;
curdesc12_msb             <= curdesc12_msb_i    ;
taildesc12_lsb            <= taildesc12_lsb_i   ;
taildesc12_msb            <= taildesc12_msb_i   ;


curdesc13_lsb             <= curdesc13_lsb_i    ;
curdesc13_msb             <= curdesc13_msb_i    ;
taildesc13_lsb            <= taildesc13_lsb_i   ;
taildesc13_msb            <= taildesc13_msb_i   ;

curdesc14_lsb             <= curdesc14_lsb_i    ;
curdesc14_msb             <= curdesc14_msb_i    ;
taildesc14_lsb            <= taildesc14_lsb_i   ;
taildesc14_msb            <= taildesc14_msb_i   ;

curdesc15_lsb             <= curdesc15_lsb_i    ;
curdesc15_msb             <= curdesc15_msb_i    ;
taildesc15_lsb            <= taildesc15_lsb_i   ;
taildesc15_msb            <= taildesc15_msb_i   ;

---------------------------------------------------------------------------
-- DMA Control Register
---------------------------------------------------------------------------
-- DMACR - Interrupt Delay Value
-------------------------------------------------------------------------------
DMACR_DELAY : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                dmacr_i(DMACR_IRQDELAY_MSB_BIT
                 downto DMACR_IRQDELAY_LSB_BIT) <= (others => '0');
            elsif(axi2ip_wrce(DMACR_INDEX) = '1')then
                dmacr_i(DMACR_IRQDELAY_MSB_BIT
                 downto DMACR_IRQDELAY_LSB_BIT) <= axi2ip_wrdata(DMACR_IRQDELAY_MSB_BIT
                                                          downto DMACR_IRQDELAY_LSB_BIT);
            end if;
        end if;
    end process DMACR_DELAY;

-- If written delay is different than previous value then assert write enable
different_delay <= '1' when dmacr_i(DMACR_IRQDELAY_MSB_BIT downto DMACR_IRQDELAY_LSB_BIT)
                   /= axi2ip_wrdata(DMACR_IRQDELAY_MSB_BIT downto DMACR_IRQDELAY_LSB_BIT)
              else '0';

-- delay value different, drive write of delay value to interrupt controller
NEW_DELAY_WRITE : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                irqdelay_wren <= '0';
            -- If AXI Lite write to DMACR and delay different than current
            -- setting then update delay value
            elsif(axi2ip_wrce(DMACR_INDEX) = '1' and different_delay = '1')then
                irqdelay_wren <= '1';
            else
                irqdelay_wren <= '0';
            end if;
        end if;
    end process NEW_DELAY_WRITE;

-------------------------------------------------------------------------------
-- DMACR - Interrupt Threshold Value
-------------------------------------------------------------------------------
threshold_is_zero <= '1' when axi2ip_wrdata(DMACR_IRQTHRESH_MSB_BIT
                                     downto DMACR_IRQTHRESH_LSB_BIT) = ZERO_THRESHOLD
                else '0';

DMACR_THRESH : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                dmacr_i(DMACR_IRQTHRESH_MSB_BIT
                        downto DMACR_IRQTHRESH_LSB_BIT) <= ONE_THRESHOLD;
            -- On AXI Lite write
            elsif(axi2ip_wrce(DMACR_INDEX) = '1')then

                -- If value is 0 then set threshold to 1
                if(threshold_is_zero='1')then
                    dmacr_i(DMACR_IRQTHRESH_MSB_BIT
                     downto DMACR_IRQTHRESH_LSB_BIT)    <= ONE_THRESHOLD;

                -- else set threshold to axi lite wrdata value
                else
                    dmacr_i(DMACR_IRQTHRESH_MSB_BIT
                     downto DMACR_IRQTHRESH_LSB_BIT)    <= axi2ip_wrdata(DMACR_IRQTHRESH_MSB_BIT
                                                                  downto DMACR_IRQTHRESH_LSB_BIT);
                end if;
            end if;
        end if;
    end process DMACR_THRESH;

-- If written threshold is different than previous value then assert write enable
different_thresh <= '1' when dmacr_i(DMACR_IRQTHRESH_MSB_BIT downto DMACR_IRQTHRESH_LSB_BIT)
                    /= axi2ip_wrdata(DMACR_IRQTHRESH_MSB_BIT downto DMACR_IRQTHRESH_LSB_BIT)
              else '0';

-- new treshold written therefore drive write of threshold out
NEW_THRESH_WRITE : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                irqthresh_wren <= '0';
            -- If AXI Lite write to DMACR and threshold different than current
            -- setting then update threshold value
            elsif(axi2ip_wrce(DMACR_INDEX) = '1' and different_thresh = '1')then
                irqthresh_wren <= '1';
            else
                irqthresh_wren <= '0';
            end if;
        end if;
    end process NEW_THRESH_WRITE;

-------------------------------------------------------------------------------
-- DMACR - Remainder of DMA Control Register, Key Hole write bit (3)
-------------------------------------------------------------------------------
DMACR_REGISTER : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                dmacr_i(DMACR_IRQTHRESH_LSB_BIT-1
                        downto DMACR_RESERVED5_BIT)   <= (others => '0');

            elsif(axi2ip_wrce(DMACR_INDEX) = '1')then
                dmacr_i(DMACR_IRQTHRESH_LSB_BIT-1       -- bit 15
                        downto DMACR_RESERVED5_BIT)   <= ZERO_VALUE(DMACR_RESERVED15_BIT)
                                                        -- bit 14
                                                        & axi2ip_wrdata(DMACR_ERR_IRQEN_BIT)
                                                        -- bit 13
                                                        & axi2ip_wrdata(DMACR_DLY_IRQEN_BIT)
                                                        -- bit 12
                                                        & axi2ip_wrdata(DMACR_IOC_IRQEN_BIT)
                                                        -- bits 11 downto 3
                                                        & ZERO_VALUE(DMACR_RESERVED11_BIT downto DMACR_RESERVED5_BIT);

            end if;
        end if;
    end process DMACR_REGISTER;


DMACR_REGISTER1 : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0' or C_ENABLE_MULTI_CHANNEL = 1)then
                dmacr_i(DMACR_KH_BIT)  <= '0';
                dmacr_i(CYCLIC_BIT)  <= '0';

            elsif(axi2ip_wrce(DMACR_INDEX) = '1')then

                dmacr_i(DMACR_KH_BIT)  <= axi2ip_wrdata(DMACR_KH_BIT);
                dmacr_i(CYCLIC_BIT)  <= axi2ip_wrdata(CYCLIC_BIT);
            end if;
        end if;
    end process DMACR_REGISTER1;

-------------------------------------------------------------------------------
-- DMACR - Reset Bit
-------------------------------------------------------------------------------
DMACR_RESET : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(soft_reset_clr = '1')then
                dmacr_i(DMACR_RESET_BIT)  <= '0';
            -- If soft reset set in other channel then set
            -- reset bit here too
            elsif(soft_reset_in = '1')then
                dmacr_i(DMACR_RESET_BIT)  <= '1';

            -- If DMACR Write then pass axi lite write bus to DMARC reset bit
            elsif(soft_reset_i = '0' and axi2ip_wrce(DMACR_INDEX) = '1')then
                dmacr_i(DMACR_RESET_BIT)  <= axi2ip_wrdata(DMACR_RESET_BIT);

            end if;
        end if;
    end process DMACR_RESET;

soft_reset_i <= dmacr_i(DMACR_RESET_BIT);

-------------------------------------------------------------------------------
-- Tail Pointer Enable fixed at 1 for this release of axi dma
-------------------------------------------------------------------------------
dmacr_i(DMACR_TAILPEN_BIT) <= '1';

-------------------------------------------------------------------------------
-- DMACR - Run/Stop Bit
-------------------------------------------------------------------------------
run_stop_clr <= '1' when error = '1'                -- MM2S DataMover Error
                      or error_in = '1'             -- S2MM Error
                      or stop_dma = '1'             -- Stop due to error
                      or soft_reset_i = '1'         -- MM2S Soft Reset
                      or soft_reset_in  = '1'       -- S2MM Soft Reset
           else '0';


DMACR_RUNSTOP : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                dmacr_i(DMACR_RS_BIT)  <= '0';
            -- Clear on sg error (i.e. error) or other channel
            -- error (i.e. error_in) or dma error or soft reset
            elsif(run_stop_clr = '1')then
                dmacr_i(DMACR_RS_BIT)  <= '0';
            elsif(axi2ip_wrce(DMACR_INDEX) = '1')then
                dmacr_i(DMACR_RS_BIT)  <= axi2ip_wrdata(DMACR_RS_BIT);
            end if;
        end if;
    end process DMACR_RUNSTOP;

---------------------------------------------------------------------------
-- DMA Status Halted bit (BIT 0) - Set by dma controller indicating DMA
-- channel is halted.
---------------------------------------------------------------------------
DMASR_HALTED : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0' or halted_set = '1')then
                halted <= '1';
            elsif(halted_clr = '1')then
                halted <= '0';
            end if;
        end if;
    end process DMASR_HALTED;

---------------------------------------------------------------------------
-- DMA Status Idle bit (BIT 1) - Set by dma controller indicating DMA
-- channel is IDLE waiting at tail pointer.  Update of Tail Pointer
-- will cause engine to resume.  Note: Halted channels return to a
-- reset condition.
---------------------------------------------------------------------------
DMASR_IDLE : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0'
            or idle_clr = '1'
            or halted_set = '1')then
                idle   <= '0';

            elsif(idle_set = '1')then
                idle   <= '1';
            end if;
        end if;
    end process DMASR_IDLE;

---------------------------------------------------------------------------
-- DMA Status Error bit (BIT 3)
-- Note: any error will cause entire engine to halt
---------------------------------------------------------------------------
error  <= dma_interr
            or dma_slverr
            or dma_decerr
            or sg_interr
            or sg_slverr
            or sg_decerr;

-- Scatter Gather Error
--sg_ftch_error <= ftch_interr_set or ftch_slverr_set or ftch_decerr_set;

-- SG Update Errors or DMA errors assert flag on descriptor update
-- Used to latch current descriptor pointer
--sg_updt_error <= updt_interr_set or updt_slverr_set or updt_decerr_set
--              or dma_interr or dma_slverr or dma_decerr;

-- Map out to halt opposing channel
error_out   <= error;


SG_FTCH_ERROR_PROC : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                sg_ftch_error <= '0';
                sg_updt_error <= '0';
            else
                sg_ftch_error <= ftch_interr_set or ftch_slverr_set or ftch_decerr_set;
                sg_updt_error <= updt_interr_set or updt_slverr_set or updt_decerr_set
                                 or dma_interr or dma_slverr or dma_decerr;
            end if;
        end if;
    end process SG_FTCH_ERROR_PROC;

---------------------------------------------------------------------------
-- DMA Status DMA Internal Error bit (BIT 4)
---------------------------------------------------------------------------
DMASR_DMAINTERR : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                dma_interr <= '0';
            elsif(dma_interr_set = '1' )then
                dma_interr <= '1';
            end if;
        end if;
    end process DMASR_DMAINTERR;

---------------------------------------------------------------------------
-- DMA Status DMA Slave Error bit (BIT 5)
---------------------------------------------------------------------------
DMASR_DMASLVERR : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                dma_slverr <= '0';

            elsif(dma_slverr_set = '1' )then
                dma_slverr <= '1';

            end if;
        end if;
    end process DMASR_DMASLVERR;

---------------------------------------------------------------------------
-- DMA Status DMA Decode Error bit (BIT 6)
---------------------------------------------------------------------------
DMASR_DMADECERR : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                dma_decerr <= '0';

            elsif(dma_decerr_set = '1' )then
                dma_decerr <= '1';

            end if;
        end if;
    end process DMASR_DMADECERR;

---------------------------------------------------------------------------
-- DMA Status SG Internal Error bit (BIT 8)
-- (SG Mode only - trimmed at build time if simple mode)
---------------------------------------------------------------------------
DMASR_SGINTERR : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                sg_interr <= '0';

            elsif(ftch_interr_set = '1' or updt_interr_set = '1')then
                sg_interr <= '1';


            end if;
        end if;
    end process DMASR_SGINTERR;

---------------------------------------------------------------------------
-- DMA Status SG Slave Error bit (BIT 9)
-- (SG Mode only - trimmed at build time if simple mode)
---------------------------------------------------------------------------
DMASR_SGSLVERR : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                sg_slverr <= '0';

            elsif(ftch_slverr_set = '1' or updt_slverr_set = '1')then
                sg_slverr <= '1';

            end if;
        end if;
    end process DMASR_SGSLVERR;

---------------------------------------------------------------------------
-- DMA Status SG Decode Error bit (BIT 10)
-- (SG Mode only - trimmed at build time if simple mode)
---------------------------------------------------------------------------
DMASR_SGDECERR : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                sg_decerr <= '0';

            elsif(ftch_decerr_set = '1' or updt_decerr_set = '1')then
                sg_decerr <= '1';

            end if;
        end if;
    end process DMASR_SGDECERR;

---------------------------------------------------------------------------
-- DMA Status IOC Interrupt status bit (BIT 11)
---------------------------------------------------------------------------
DMASR_IOCIRQ : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                ioc_irq <= '0';

            -- CPU Writing a '1' to clear - OR'ed with setting to prevent
            -- missing a 'set' during the write.
            elsif(axi2ip_wrce(DMASR_INDEX) = '1' )then

                ioc_irq <= (ioc_irq and not(axi2ip_wrdata(DMASR_IOCIRQ_BIT)))
                             or ioc_irq_set;

            elsif(ioc_irq_set = '1')then
                ioc_irq <= '1';

            end if;
        end if;
    end process DMASR_IOCIRQ;

---------------------------------------------------------------------------
-- DMA Status Delay Interrupt status bit (BIT 12)
---------------------------------------------------------------------------
DMASR_DLYIRQ : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                dly_irq <= '0';

            -- CPU Writing a '1' to clear - OR'ed with setting to prevent
            -- missing a 'set' during the write.
            elsif(axi2ip_wrce(DMASR_INDEX) = '1' )then

                dly_irq <= (dly_irq and not(axi2ip_wrdata(DMASR_DLYIRQ_BIT)))
                             or dly_irq_set;

            elsif(dly_irq_set = '1')then
                dly_irq <= '1';

            end if;
        end if;
    end process DMASR_DLYIRQ;

-- CR605888 Disable delay timer if halted or on delay irq set
--dlyirq_dsble    <= dmasr_i(DMASR_HALTED_BIT)              -- CR606348
dlyirq_dsble    <= not dmacr_i(DMACR_RS_BIT)                -- CR606348
                    or dmasr_i(DMASR_DLYIRQ_BIT);



---------------------------------------------------------------------------
-- DMA Status Error Interrupt status bit (BIT 12)
---------------------------------------------------------------------------
-- Delay error setting for generation of error strobe
GEN_ERROR_RE : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                error_d1 <= '0';
            else
                error_d1 <= error;
            end if;
        end if;
    end process GEN_ERROR_RE;

-- Generate rising edge pulse on error
error_re   <= error and not error_d1;

DMASR_ERRIRQ : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                err_irq <= '0';

            -- CPU Writing a '1' to clear - OR'ed with setting to prevent
            -- missing a 'set' during the write.
            elsif(axi2ip_wrce(DMASR_INDEX) = '1' )then

                err_irq <= (err_irq and not(axi2ip_wrdata(DMASR_ERRIRQ_BIT)))
                             or error_re;

            elsif(error_re = '1')then
                err_irq <= '1';

            end if;
        end if;
    end process DMASR_ERRIRQ;

---------------------------------------------------------------------------
-- DMA Interrupt OUT
---------------------------------------------------------------------------
REG_INTR : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0' or soft_reset_i = '1')then
                introut <= '0';
            else
                introut <= (dly_irq and dmacr_i(DMACR_DLY_IRQEN_BIT))
                        or (ioc_irq and dmacr_i(DMACR_IOC_IRQEN_BIT))
                        or (err_irq and dmacr_i(DMACR_ERR_IRQEN_BIT));
            end if;
        end if;
    end process;

---------------------------------------------------------------------------
-- DMA Status Register
---------------------------------------------------------------------------
dmasr_i    <=  irqdelay_status         -- Bits 31 downto 24
                    & irqthresh_status -- Bits 23 downto 16
                    & '0'              -- Bit  15
                    & err_irq          -- Bit  14
                    & dly_irq          -- Bit  13
                    & ioc_irq          -- Bit  12
                    & '0'              -- Bit  11
                    & sg_decerr        -- Bit  10
                    & sg_slverr        -- Bit  9
                    & sg_interr        -- Bit  8
                    & '0'              -- Bit  7
                    & dma_decerr       -- Bit  6
                    & dma_slverr       -- Bit  5
                    & dma_interr       -- Bit  4
                    & DMA_CONFIG       -- Bit  3
                    & '0'              -- Bit  2
                    & idle             -- Bit  1
                    & halted;          -- Bit  0





-- Generate current descriptor and tail descriptor register for Scatter Gather Mode
GEN_DESC_REG_FOR_SG : if C_INCLUDE_SG = 1 generate
begin

   GEN_SG_CTL_REG : if C_ENABLE_MULTI_CHANNEL = 1 generate
   begin

   MM2S_SGCTL : process(m_axi_sg_aclk)
      begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                sg_cache_info <= "00000011"; --(others => '0');

            elsif(axi2ip_wrce(SGCTL_INDEX) = '1' ) then

                sg_cache_info <= axi2ip_wrdata(11 downto 8) & axi2ip_wrdata(3 downto 0);
            else
              sg_cache_info <= sg_cache_info;

            end if;
        end if;
      end process MM2S_SGCTL;

      sg_ctl <= sg_cache_info;

   end generate GEN_SG_CTL_REG;

   GEN_SG_NO_CTL_REG : if C_ENABLE_MULTI_CHANNEL = 0 generate
   begin

                sg_ctl <= "00000011"; --(others => '0');


   end generate GEN_SG_NO_CTL_REG;


    -- Signals not used for Scatter Gather Mode, only simple mode
    buffer_address_i    <= (others => '0');
    buffer_length_i     <= (others => '0');
    buffer_length_wren  <= '0';

    ---------------------------------------------------------------------------
    -- Current Descriptor LSB Register
    ---------------------------------------------------------------------------
    CURDESC_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    curdesc_lsb_i  <= (others => '0');
                    error_pointer_set   <= '0';

                -- Detected error has NOT register a desc pointer
                elsif(error_pointer_set = '0')then

                    -- Scatter Gather Fetch Error
                    if((sg_ftch_error = '1' or sg_updt_error = '1') and dest0 = '1')then
                        curdesc_lsb_i       <= ftch_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 6);
                        error_pointer_set   <= '1';
                    -- Scatter Gather Update Error
             --       elsif(sg_updt_error = '1' and dest0 = '1')then
             --           curdesc_lsb_i       <= updt_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
             --           error_pointer_set   <= '1';

                    -- Commanded to update descriptor value - used for indicating
                    -- current descriptor begin processed by dma controller
                    elsif(update_curdesc = '1' and dmacr_i(DMACR_RS_BIT)  = '1'  and dest0 = '1')then
                        curdesc_lsb_i       <= new_curdesc(C_S_AXI_LITE_DATA_WIDTH-1 downto 6);
                        error_pointer_set   <= '0';

                    -- CPU update of current descriptor pointer.  CPU
                    -- only allowed to update when engine is halted.
                    elsif(axi2ip_wrce(CURDESC_LSB_INDEX) = '1' and halt_free = '1')then
                        curdesc_lsb_i       <= axi2ip_wrdata(CURDESC_LOWER_MSB_BIT
                                                      downto CURDESC_LOWER_LSB_BIT);
--                                              & ZERO_VALUE(CURDESC_RESERVED_BIT5
--                                                      downto CURDESC_RESERVED_BIT0);
                        error_pointer_set   <= '0';

                    end if;
                end if;
            end if;
        end process CURDESC_LSB_REGISTER;


         


    ---------------------------------------------------------------------------
    -- Tail Descriptor LSB Register
    ---------------------------------------------------------------------------
    TAILDESC_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    taildesc_lsb_i  <= (others => '0');
                elsif(axi2ip_wrce(TAILDESC_LSB_INDEX) = '1')then
                    taildesc_lsb_i  <= axi2ip_wrdata(TAILDESC_LOWER_MSB_BIT
                                              downto TAILDESC_LOWER_LSB_BIT);
--                                       & ZERO_VALUE(TAILDESC_RESERVED_BIT5
--                                              downto TAILDESC_RESERVED_BIT0);

                end if;
            end if;
        end process TAILDESC_LSB_REGISTER;

GEN_DESC1_REG_FOR_SG : if C_NUM_S2MM_CHANNELS > 1 generate


    CURDESC1_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    curdesc1_lsb_i  <= (others => '0');
                    error_pointer_set1   <= '0';

                -- Detected error has NOT register a desc pointer
                elsif(error_pointer_set1 = '0')then

                    -- Scatter Gather Fetch Error
                    if((sg_ftch_error = '1' or sg_updt_error = '1') and dest1 = '1')then
                        curdesc1_lsb_i       <= ftch_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set1   <= '1';
                    -- Scatter Gather Update Error
--                    elsif(sg_updt_error = '1' and dest1 = '1')then
--                        curdesc1_lsb_i       <= updt_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
--                        error_pointer_set1   <= '1';

                    -- Commanded to update descriptor value - used for indicating
                    -- current descriptor begin processed by dma controller
                    elsif(update_curdesc1 = '1' and dmacr_i(DMACR_RS_BIT)  = '1'  and dest1 = '1')then
                        curdesc1_lsb_i       <= new_curdesc(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set1   <= '0';

                    -- CPU update of current descriptor pointer.  CPU
                    -- only allowed to update when engine is halted.
                    elsif(axi2ip_wrce(CURDESC1_LSB_INDEX) = '1' and halt_free = '1')then
                        curdesc1_lsb_i       <= axi2ip_wrdata(CURDESC_LOWER_MSB_BIT
                                                      downto CURDESC_LOWER_LSB_BIT)
                                              & ZERO_VALUE(CURDESC_RESERVED_BIT5
                                                      downto CURDESC_RESERVED_BIT0);
                        error_pointer_set1   <= '0';

                    end if;
                end if;
            end if;
        end process CURDESC1_LSB_REGISTER;

    ---------------------------------------------------------------------------
    -- Tail Descriptor LSB Register
    ---------------------------------------------------------------------------
    TAILDESC1_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    taildesc1_lsb_i  <= (others => '0');
                elsif(axi2ip_wrce(TAILDESC1_LSB_INDEX) = '1')then
                    taildesc1_lsb_i  <= axi2ip_wrdata(TAILDESC_LOWER_MSB_BIT
                                              downto TAILDESC_LOWER_LSB_BIT)
                                       & ZERO_VALUE(TAILDESC_RESERVED_BIT5
                                              downto TAILDESC_RESERVED_BIT0);

                end if;
            end if;
        end process TAILDESC1_LSB_REGISTER;

end generate GEN_DESC1_REG_FOR_SG;


GEN_DESC2_REG_FOR_SG : if C_NUM_S2MM_CHANNELS > 2 generate

    CURDESC2_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    curdesc2_lsb_i  <= (others => '0');
                    error_pointer_set2   <= '0';

                -- Detected error has NOT register a desc pointer
                elsif(error_pointer_set2 = '0')then

                    -- Scatter Gather Fetch Error
                    if((sg_ftch_error = '1' or sg_updt_error = '1') and dest2 = '1')then
                        curdesc2_lsb_i       <= ftch_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set2   <= '1';
                    -- Scatter Gather Update Error
--                    elsif(sg_updt_error = '1' and dest2 = '1')then
--                        curdesc2_lsb_i       <= updt_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
--                        error_pointer_set2   <= '1';

                    -- Commanded to update descriptor value - used for indicating
                    -- current descriptor begin processed by dma controller
                    elsif(update_curdesc2 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest2 = '1')then
                        curdesc2_lsb_i       <= new_curdesc(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set2   <= '0';

                    -- CPU update of current descriptor pointer.  CPU
                    -- only allowed to update when engine is halted.
                    elsif(axi2ip_wrce(CURDESC2_LSB_INDEX) = '1' and halt_free = '1')then
                        curdesc2_lsb_i       <= axi2ip_wrdata(CURDESC_LOWER_MSB_BIT
                                                      downto CURDESC_LOWER_LSB_BIT)
                                              & ZERO_VALUE(CURDESC_RESERVED_BIT5
                                                      downto CURDESC_RESERVED_BIT0);
                        error_pointer_set2   <= '0';

                    end if;
                end if;
            end if;
        end process CURDESC2_LSB_REGISTER;

    TAILDESC2_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    taildesc2_lsb_i  <= (others => '0');
                elsif(axi2ip_wrce(TAILDESC2_LSB_INDEX) = '1')then
                    taildesc2_lsb_i  <= axi2ip_wrdata(TAILDESC_LOWER_MSB_BIT
                                              downto TAILDESC_LOWER_LSB_BIT)
                                       & ZERO_VALUE(TAILDESC_RESERVED_BIT5
                                              downto TAILDESC_RESERVED_BIT0);

                end if;
            end if;
        end process TAILDESC2_LSB_REGISTER;

end generate GEN_DESC2_REG_FOR_SG;

GEN_DESC3_REG_FOR_SG : if C_NUM_S2MM_CHANNELS > 3 generate

    CURDESC3_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    curdesc3_lsb_i  <= (others => '0');
                    error_pointer_set3   <= '0';

                -- Detected error has NOT register a desc pointer
                elsif(error_pointer_set3 = '0')then

                    -- Scatter Gather Fetch Error
                    if((sg_ftch_error = '1' or sg_updt_error = '1') and dest3 = '1')then
                        curdesc3_lsb_i       <= ftch_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set3   <= '1';
                    -- Scatter Gather Update Error
--                    elsif(sg_updt_error = '1' and dest3 = '1')then
 --                       curdesc3_lsb_i       <= updt_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
 --                       error_pointer_set3   <= '1';

                    -- Commanded to update descriptor value - used for indicating
                    -- current descriptor begin processed by dma controller
                    elsif(update_curdesc3 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest3 = '1')then
                        curdesc3_lsb_i       <= new_curdesc(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set3   <= '0';

                    -- CPU update of current descriptor pointer.  CPU
                    -- only allowed to update when engine is halted.
                    elsif(axi2ip_wrce(CURDESC3_LSB_INDEX) = '1' and halt_free = '1')then
                        curdesc3_lsb_i       <= axi2ip_wrdata(CURDESC_LOWER_MSB_BIT
                                                      downto CURDESC_LOWER_LSB_BIT)
                                              & ZERO_VALUE(CURDESC_RESERVED_BIT5
                                                      downto CURDESC_RESERVED_BIT0);
                        error_pointer_set3   <= '0';

                    end if;
                end if;
            end if;
        end process CURDESC3_LSB_REGISTER;

    ---------------------------------------------------------------------------
    -- Tail Descriptor LSB Register
    ---------------------------------------------------------------------------
    TAILDESC3_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    taildesc3_lsb_i  <= (others => '0');
                elsif(axi2ip_wrce(TAILDESC3_LSB_INDEX) = '1')then
                    taildesc3_lsb_i  <= axi2ip_wrdata(TAILDESC_LOWER_MSB_BIT
                                              downto TAILDESC_LOWER_LSB_BIT)
                                       & ZERO_VALUE(TAILDESC_RESERVED_BIT5
                                              downto TAILDESC_RESERVED_BIT0);

                end if;
            end if;
        end process TAILDESC3_LSB_REGISTER;

end generate GEN_DESC3_REG_FOR_SG;

GEN_DESC4_REG_FOR_SG : if C_NUM_S2MM_CHANNELS > 4 generate

    CURDESC4_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    curdesc4_lsb_i  <= (others => '0');
                    error_pointer_set4   <= '0';

                -- Detected error has NOT register a desc pointer
                elsif(error_pointer_set4 = '0')then

                    -- Scatter Gather Fetch Error
                    if((sg_ftch_error = '1' or sg_updt_error = '1') and dest4 = '1')then
                        curdesc4_lsb_i       <= ftch_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set4   <= '1';
                    -- Scatter Gather Update Error
--                    elsif(sg_updt_error = '1' and dest4 = '1')then
--                        curdesc4_lsb_i       <= updt_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
--                        error_pointer_set4   <= '1';

                    -- Commanded to update descriptor value - used for indicating
                    -- current descriptor begin processed by dma controller
                    elsif(update_curdesc4 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest4 = '1')then
                        curdesc4_lsb_i       <= new_curdesc(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set4   <= '0';

                    -- CPU update of current descriptor pointer.  CPU
                    -- only allowed to update when engine is halted.
                    elsif(axi2ip_wrce(CURDESC4_LSB_INDEX) = '1' and halt_free = '1')then
                        curdesc4_lsb_i       <= axi2ip_wrdata(CURDESC_LOWER_MSB_BIT
                                                      downto CURDESC_LOWER_LSB_BIT)
                                              & ZERO_VALUE(CURDESC_RESERVED_BIT5
                                                      downto CURDESC_RESERVED_BIT0);
                        error_pointer_set4   <= '0';

                    end if;
                end if;
            end if;
        end process CURDESC4_LSB_REGISTER;

    ---------------------------------------------------------------------------
    -- Tail Descriptor LSB Register
    ---------------------------------------------------------------------------
    TAILDESC4_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    taildesc4_lsb_i  <= (others => '0');
                elsif(axi2ip_wrce(TAILDESC4_LSB_INDEX) = '1')then
                    taildesc4_lsb_i  <= axi2ip_wrdata(TAILDESC_LOWER_MSB_BIT
                                              downto TAILDESC_LOWER_LSB_BIT)
                                       & ZERO_VALUE(TAILDESC_RESERVED_BIT5
                                              downto TAILDESC_RESERVED_BIT0);

                end if;
            end if;
        end process TAILDESC4_LSB_REGISTER;

end generate GEN_DESC4_REG_FOR_SG;

GEN_DESC5_REG_FOR_SG : if C_NUM_S2MM_CHANNELS > 5 generate


    CURDESC5_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    curdesc5_lsb_i  <= (others => '0');
                    error_pointer_set5   <= '0';

                -- Detected error has NOT register a desc pointer
                elsif(error_pointer_set5 = '0')then

                    -- Scatter Gather Fetch Error
                    if((sg_ftch_error = '1' or sg_updt_error = '1') and dest5 = '1')then
                        curdesc5_lsb_i       <= ftch_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set5   <= '1';
                    -- Scatter Gather Update Error
--                    elsif(sg_updt_error = '1' and dest5 = '1')then
--                        curdesc5_lsb_i       <= updt_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
--                        error_pointer_set5   <= '1';

                    -- Commanded to update descriptor value - used for indicating
                    -- current descriptor begin processed by dma controller
                    elsif(update_curdesc5 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest5 = '1')then
                        curdesc5_lsb_i       <= new_curdesc(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set5   <= '0';

                    -- CPU update of current descriptor pointer.  CPU
                    -- only allowed to update when engine is halted.
                    elsif(axi2ip_wrce(CURDESC5_LSB_INDEX) = '1' and halt_free = '1')then
                        curdesc5_lsb_i       <= axi2ip_wrdata(CURDESC_LOWER_MSB_BIT
                                                      downto CURDESC_LOWER_LSB_BIT)
                                              & ZERO_VALUE(CURDESC_RESERVED_BIT5
                                                      downto CURDESC_RESERVED_BIT0);
                        error_pointer_set5   <= '0';

                    end if;
                end if;
            end if;
        end process CURDESC5_LSB_REGISTER;

    ---------------------------------------------------------------------------
    -- Tail Descriptor LSB Register
    ---------------------------------------------------------------------------
    TAILDESC5_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    taildesc5_lsb_i  <= (others => '0');
                elsif(axi2ip_wrce(TAILDESC5_LSB_INDEX) = '1')then
                    taildesc5_lsb_i  <= axi2ip_wrdata(TAILDESC_LOWER_MSB_BIT
                                              downto TAILDESC_LOWER_LSB_BIT)
                                       & ZERO_VALUE(TAILDESC_RESERVED_BIT5
                                              downto TAILDESC_RESERVED_BIT0);

                end if;
            end if;
        end process TAILDESC5_LSB_REGISTER;

end generate GEN_DESC5_REG_FOR_SG;

GEN_DESC6_REG_FOR_SG : if C_NUM_S2MM_CHANNELS > 6 generate


    CURDESC6_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    curdesc6_lsb_i  <= (others => '0');
                    error_pointer_set6   <= '0';

                -- Detected error has NOT register a desc pointer
                elsif(error_pointer_set6 = '0')then

                    -- Scatter Gather Fetch Error
                    if((sg_ftch_error = '1' or sg_updt_error = '1') and dest6 = '1')then
                        curdesc6_lsb_i       <= ftch_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set6   <= '1';
                    -- Scatter Gather Update Error
--                    elsif(sg_updt_error = '1' and dest6 = '1')then
--                        curdesc6_lsb_i       <= updt_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
--                        error_pointer_set6   <= '1';

                    -- Commanded to update descriptor value - used for indicating
                    -- current descriptor begin processed by dma controller
                    elsif(update_curdesc6 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest6 = '1')then
                        curdesc6_lsb_i       <= new_curdesc(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set6   <= '0';

                    -- CPU update of current descriptor pointer.  CPU
                    -- only allowed to update when engine is halted.
                    elsif(axi2ip_wrce(CURDESC6_LSB_INDEX) = '1' and halt_free = '1')then
                        curdesc6_lsb_i       <= axi2ip_wrdata(CURDESC_LOWER_MSB_BIT
                                                      downto CURDESC_LOWER_LSB_BIT)
                                              & ZERO_VALUE(CURDESC_RESERVED_BIT5
                                                      downto CURDESC_RESERVED_BIT0);
                        error_pointer_set6   <= '0';

                    end if;
                end if;
            end if;
        end process CURDESC6_LSB_REGISTER;

    ---------------------------------------------------------------------------
    -- Tail Descriptor LSB Register
    ---------------------------------------------------------------------------
    TAILDESC6_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    taildesc6_lsb_i  <= (others => '0');
                elsif(axi2ip_wrce(TAILDESC6_LSB_INDEX) = '1')then
                    taildesc6_lsb_i  <= axi2ip_wrdata(TAILDESC_LOWER_MSB_BIT
                                              downto TAILDESC_LOWER_LSB_BIT)
                                       & ZERO_VALUE(TAILDESC_RESERVED_BIT5
                                              downto TAILDESC_RESERVED_BIT0);

                end if;
            end if;
        end process TAILDESC6_LSB_REGISTER;

end generate GEN_DESC6_REG_FOR_SG; 

GEN_DESC7_REG_FOR_SG : if C_NUM_S2MM_CHANNELS > 7 generate

    CURDESC7_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    curdesc7_lsb_i  <= (others => '0');
                    error_pointer_set7   <= '0';

                -- Detected error has NOT register a desc pointer
                elsif(error_pointer_set7 = '0')then

                    -- Scatter Gather Fetch Error
                    if((sg_ftch_error = '1' or sg_updt_error = '1') and dest7 = '1')then
                        curdesc7_lsb_i       <= ftch_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set7   <= '1';
                    -- Scatter Gather Update Error
--                    elsif(sg_updt_error = '1' and dest7 = '1')then
--                        curdesc7_lsb_i       <= updt_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
--                        error_pointer_set7   <= '1';

                    -- Commanded to update descriptor value - used for indicating
                    -- current descriptor begin processed by dma controller
                    elsif(update_curdesc7 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest7 = '1')then
                        curdesc7_lsb_i       <= new_curdesc(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set7   <= '0';

                    -- CPU update of current descriptor pointer.  CPU
                    -- only allowed to update when engine is halted.
                    elsif(axi2ip_wrce(CURDESC7_LSB_INDEX) = '1' and halt_free = '1')then
                        curdesc7_lsb_i       <= axi2ip_wrdata(CURDESC_LOWER_MSB_BIT
                                                      downto CURDESC_LOWER_LSB_BIT)
                                              & ZERO_VALUE(CURDESC_RESERVED_BIT5
                                                      downto CURDESC_RESERVED_BIT0);
                        error_pointer_set7   <= '0';

                    end if;
                end if;
            end if;
        end process CURDESC7_LSB_REGISTER;

    ---------------------------------------------------------------------------
    -- Tail Descriptor LSB Register
    ---------------------------------------------------------------------------
    TAILDESC7_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    taildesc7_lsb_i  <= (others => '0');
                elsif(axi2ip_wrce(TAILDESC7_LSB_INDEX) = '1')then
                    taildesc7_lsb_i  <= axi2ip_wrdata(TAILDESC_LOWER_MSB_BIT
                                              downto TAILDESC_LOWER_LSB_BIT)
                                       & ZERO_VALUE(TAILDESC_RESERVED_BIT5
                                              downto TAILDESC_RESERVED_BIT0);

                end if;
            end if;
        end process TAILDESC7_LSB_REGISTER;

end generate GEN_DESC7_REG_FOR_SG;

GEN_DESC8_REG_FOR_SG : if C_NUM_S2MM_CHANNELS > 8 generate


    CURDESC8_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    curdesc8_lsb_i  <= (others => '0');
                    error_pointer_set8   <= '0';

                -- Detected error has NOT register a desc pointer
                elsif(error_pointer_set8 = '0')then

                    -- Scatter Gather Fetch Error
                    if((sg_ftch_error = '1' or sg_updt_error = '1') and dest8 = '1')then
                        curdesc8_lsb_i       <= ftch_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set8   <= '1';
                    -- Scatter Gather Update Error
--                    elsif(sg_updt_error = '1' and dest8 = '1')then
--                        curdesc8_lsb_i       <= updt_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
--                        error_pointer_set8   <= '1';

                    -- Commanded to update descriptor value - used for indicating
                    -- current descriptor begin processed by dma controller
                    elsif(update_curdesc8 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest8 = '1')then
                        curdesc8_lsb_i       <= new_curdesc(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set8   <= '0';

                    -- CPU update of current descriptor pointer.  CPU
                    -- only allowed to update when engine is halted.
                    elsif(axi2ip_wrce(CURDESC8_LSB_INDEX) = '1' and halt_free = '1')then
                        curdesc8_lsb_i       <= axi2ip_wrdata(CURDESC_LOWER_MSB_BIT
                                                      downto CURDESC_LOWER_LSB_BIT)
                                              & ZERO_VALUE(CURDESC_RESERVED_BIT5
                                                      downto CURDESC_RESERVED_BIT0);
                        error_pointer_set8   <= '0';

                    end if;
                end if;
            end if;
        end process CURDESC8_LSB_REGISTER;

    ---------------------------------------------------------------------------
    -- Tail Descriptor LSB Register
    ---------------------------------------------------------------------------
    TAILDESC8_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    taildesc8_lsb_i  <= (others => '0');
                elsif(axi2ip_wrce(TAILDESC8_LSB_INDEX) = '1')then
                    taildesc8_lsb_i  <= axi2ip_wrdata(TAILDESC_LOWER_MSB_BIT
                                              downto TAILDESC_LOWER_LSB_BIT)
                                       & ZERO_VALUE(TAILDESC_RESERVED_BIT5
                                              downto TAILDESC_RESERVED_BIT0);

                end if;
            end if;
        end process TAILDESC8_LSB_REGISTER;


end generate GEN_DESC8_REG_FOR_SG;

GEN_DESC9_REG_FOR_SG : if C_NUM_S2MM_CHANNELS > 9 generate

    CURDESC9_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    curdesc9_lsb_i  <= (others => '0');
                    error_pointer_set9   <= '0';

                -- Detected error has NOT register a desc pointer
                elsif(error_pointer_set9 = '0')then

                    -- Scatter Gather Fetch Error
                    if((sg_ftch_error = '1' or sg_updt_error = '1') and dest9 = '1')then
                        curdesc9_lsb_i       <= ftch_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set9   <= '1';
                    -- Scatter Gather Update Error
--                    elsif(sg_updt_error = '1' and dest9 = '1')then
--                        curdesc9_lsb_i       <= updt_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
--                        error_pointer_set9   <= '1';

                    -- Commanded to update descriptor value - used for indicating
                    -- current descriptor begin processed by dma controller
                    elsif(update_curdesc9 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest9 = '1')then
                        curdesc9_lsb_i       <= new_curdesc(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set9   <= '0';

                    -- CPU update of current descriptor pointer.  CPU
                    -- only allowed to update when engine is halted.
                    elsif(axi2ip_wrce(CURDESC9_LSB_INDEX) = '1' and halt_free = '1')then
                        curdesc9_lsb_i       <= axi2ip_wrdata(CURDESC_LOWER_MSB_BIT
                                                      downto CURDESC_LOWER_LSB_BIT)
                                              & ZERO_VALUE(CURDESC_RESERVED_BIT5
                                                      downto CURDESC_RESERVED_BIT0);
                        error_pointer_set9   <= '0';

                    end if;
                end if;
            end if;
        end process CURDESC9_LSB_REGISTER;

    ---------------------------------------------------------------------------
    -- Tail Descriptor LSB Register
    ---------------------------------------------------------------------------
    TAILDESC9_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    taildesc9_lsb_i  <= (others => '0');
                elsif(axi2ip_wrce(TAILDESC9_LSB_INDEX) = '1')then
                    taildesc9_lsb_i  <= axi2ip_wrdata(TAILDESC_LOWER_MSB_BIT
                                              downto TAILDESC_LOWER_LSB_BIT)
                                       & ZERO_VALUE(TAILDESC_RESERVED_BIT5
                                              downto TAILDESC_RESERVED_BIT0);

                end if;
            end if;
        end process TAILDESC9_LSB_REGISTER;

end generate GEN_DESC9_REG_FOR_SG;

GEN_DESC10_REG_FOR_SG : if C_NUM_S2MM_CHANNELS > 10 generate


    CURDESC10_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    curdesc10_lsb_i  <= (others => '0');
                    error_pointer_set10   <= '0';

                -- Detected error has NOT register a desc pointer
                elsif(error_pointer_set10 = '0')then

                    -- Scatter Gather Fetch Error
                    if((sg_ftch_error = '1' or sg_updt_error = '1') and dest10 = '1')then
                        curdesc10_lsb_i       <= ftch_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set10   <= '1';
                    -- Scatter Gather Update Error
--                    elsif(sg_updt_error = '1' and dest10 = '1')then
--                        curdesc10_lsb_i       <= updt_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
--                        error_pointer_set10   <= '1';

                    -- Commanded to update descriptor value - used for indicating
                    -- current descriptor begin processed by dma controller
                    elsif(update_curdesc10 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest10 = '1')then
                        curdesc10_lsb_i       <= new_curdesc(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set10   <= '0';

                    -- CPU update of current descriptor pointer.  CPU
                    -- only allowed to update when engine is halted.
                    elsif(axi2ip_wrce(CURDESC10_LSB_INDEX) = '1' and halt_free = '1')then
                        curdesc10_lsb_i       <= axi2ip_wrdata(CURDESC_LOWER_MSB_BIT
                                                      downto CURDESC_LOWER_LSB_BIT)
                                              & ZERO_VALUE(CURDESC_RESERVED_BIT5
                                                      downto CURDESC_RESERVED_BIT0);
                        error_pointer_set10   <= '0';

                    end if;
                end if;
            end if;
        end process CURDESC10_LSB_REGISTER;

    ---------------------------------------------------------------------------
    -- Tail Descriptor LSB Register
    ---------------------------------------------------------------------------
    TAILDESC10_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    taildesc10_lsb_i  <= (others => '0');
                elsif(axi2ip_wrce(TAILDESC10_LSB_INDEX) = '1')then
                    taildesc10_lsb_i  <= axi2ip_wrdata(TAILDESC_LOWER_MSB_BIT
                                              downto TAILDESC_LOWER_LSB_BIT)
                                       & ZERO_VALUE(TAILDESC_RESERVED_BIT5
                                              downto TAILDESC_RESERVED_BIT0);

                end if;
            end if;
        end process TAILDESC10_LSB_REGISTER;

end generate GEN_DESC10_REG_FOR_SG; 

GEN_DESC11_REG_FOR_SG : if C_NUM_S2MM_CHANNELS > 11 generate



    CURDESC11_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    curdesc11_lsb_i  <= (others => '0');
                    error_pointer_set11   <= '0';

                -- Detected error has NOT register a desc pointer
                elsif(error_pointer_set11 = '0')then

                    -- Scatter Gather Fetch Error
                    if((sg_ftch_error = '1' or sg_updt_error = '1') and dest11 = '1')then
                        curdesc11_lsb_i       <= ftch_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set11   <= '1';
                    -- Scatter Gather Update Error
--                    elsif(sg_updt_error = '1' and dest11 = '1')then
--                        curdesc11_lsb_i       <= updt_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
--                        error_pointer_set11   <= '1';

                    -- Commanded to update descriptor value - used for indicating
                    -- current descriptor begin processed by dma controller
                    elsif(update_curdesc11 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest11 = '1')then
                        curdesc11_lsb_i       <= new_curdesc(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set11   <= '0';

                    -- CPU update of current descriptor pointer.  CPU
                    -- only allowed to update when engine is halted.
                    elsif(axi2ip_wrce(CURDESC11_LSB_INDEX) = '1' and halt_free = '1')then
                        curdesc11_lsb_i       <= axi2ip_wrdata(CURDESC_LOWER_MSB_BIT
                                                      downto CURDESC_LOWER_LSB_BIT)
                                              & ZERO_VALUE(CURDESC_RESERVED_BIT5
                                                      downto CURDESC_RESERVED_BIT0);
                        error_pointer_set11   <= '0';

                    end if;
                end if;
            end if;
        end process CURDESC11_LSB_REGISTER;

    ---------------------------------------------------------------------------
    -- Tail Descriptor LSB Register
    ---------------------------------------------------------------------------
    TAILDESC11_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    taildesc11_lsb_i  <= (others => '0');
                elsif(axi2ip_wrce(TAILDESC11_LSB_INDEX) = '1')then
                    taildesc11_lsb_i  <= axi2ip_wrdata(TAILDESC_LOWER_MSB_BIT
                                              downto TAILDESC_LOWER_LSB_BIT)
                                       & ZERO_VALUE(TAILDESC_RESERVED_BIT5
                                              downto TAILDESC_RESERVED_BIT0);

                end if;
            end if;
        end process TAILDESC11_LSB_REGISTER;

end generate GEN_DESC11_REG_FOR_SG; 

GEN_DESC12_REG_FOR_SG : if C_NUM_S2MM_CHANNELS > 12 generate



    CURDESC12_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    curdesc12_lsb_i  <= (others => '0');
                    error_pointer_set12   <= '0';

                -- Detected error has NOT register a desc pointer
                elsif(error_pointer_set12 = '0')then

                    -- Scatter Gather Fetch Error
                    if((sg_ftch_error = '1' or sg_updt_error = '1') and dest12 = '1')then
                        curdesc12_lsb_i       <= ftch_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set12   <= '1';
                    -- Scatter Gather Update Error
--                    elsif(sg_updt_error = '1' and dest12 = '1')then
--                        curdesc12_lsb_i       <= updt_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
--                        error_pointer_set12   <= '1';

                    -- Commanded to update descriptor value - used for indicating
                    -- current descriptor begin processed by dma controller
                    elsif(update_curdesc12 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest12 = '1')then
                        curdesc12_lsb_i       <= new_curdesc(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set12   <= '0';

                    -- CPU update of current descriptor pointer.  CPU
                    -- only allowed to update when engine is halted.
                    elsif(axi2ip_wrce(CURDESC12_LSB_INDEX) = '1' and halt_free = '1')then
                        curdesc12_lsb_i       <= axi2ip_wrdata(CURDESC_LOWER_MSB_BIT
                                                      downto CURDESC_LOWER_LSB_BIT)
                                              & ZERO_VALUE(CURDESC_RESERVED_BIT5
                                                      downto CURDESC_RESERVED_BIT0);
                        error_pointer_set12   <= '0';

                    end if;
                end if;
            end if;
        end process CURDESC12_LSB_REGISTER;

    ---------------------------------------------------------------------------
    -- Tail Descriptor LSB Register
    ---------------------------------------------------------------------------
    TAILDESC12_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    taildesc12_lsb_i  <= (others => '0');
                elsif(axi2ip_wrce(TAILDESC12_LSB_INDEX) = '1')then
                    taildesc12_lsb_i  <= axi2ip_wrdata(TAILDESC_LOWER_MSB_BIT
                                              downto TAILDESC_LOWER_LSB_BIT)
                                       & ZERO_VALUE(TAILDESC_RESERVED_BIT5
                                              downto TAILDESC_RESERVED_BIT0);

                end if;
            end if;
        end process TAILDESC12_LSB_REGISTER;

end generate GEN_DESC12_REG_FOR_SG; 

GEN_DESC13_REG_FOR_SG : if C_NUM_S2MM_CHANNELS > 13 generate



    CURDESC13_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    curdesc13_lsb_i  <= (others => '0');
                    error_pointer_set13   <= '0';

                -- Detected error has NOT register a desc pointer
                elsif(error_pointer_set13 = '0')then

                    -- Scatter Gather Fetch Error
                    if((sg_ftch_error = '1' or sg_updt_error = '1') and dest13 = '1')then
                        curdesc13_lsb_i       <= ftch_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set13   <= '1';
                    -- Scatter Gather Update Error
--                    elsif(sg_updt_error = '1' and dest13 = '1')then
--                        curdesc13_lsb_i       <= updt_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
--                        error_pointer_set13   <= '1';

                    -- Commanded to update descriptor value - used for indicating
                    -- current descriptor begin processed by dma controller
                    elsif(update_curdesc13 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest13 = '1')then
                        curdesc13_lsb_i       <= new_curdesc(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set13   <= '0';

                    -- CPU update of current descriptor pointer.  CPU
                    -- only allowed to update when engine is halted.
                    elsif(axi2ip_wrce(CURDESC13_LSB_INDEX) = '1' and halt_free = '1')then
                        curdesc13_lsb_i       <= axi2ip_wrdata(CURDESC_LOWER_MSB_BIT
                                                      downto CURDESC_LOWER_LSB_BIT)
                                              & ZERO_VALUE(CURDESC_RESERVED_BIT5
                                                      downto CURDESC_RESERVED_BIT0);
                        error_pointer_set13   <= '0';

                    end if;
                end if;
            end if;
        end process CURDESC13_LSB_REGISTER;

    ---------------------------------------------------------------------------
    -- Tail Descriptor LSB Register
    ---------------------------------------------------------------------------
    TAILDESC13_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    taildesc13_lsb_i  <= (others => '0');
                elsif(axi2ip_wrce(TAILDESC13_LSB_INDEX) = '1')then
                    taildesc13_lsb_i  <= axi2ip_wrdata(TAILDESC_LOWER_MSB_BIT
                                              downto TAILDESC_LOWER_LSB_BIT)
                                       & ZERO_VALUE(TAILDESC_RESERVED_BIT5
                                              downto TAILDESC_RESERVED_BIT0);

                end if;
            end if;
        end process TAILDESC13_LSB_REGISTER;

end generate GEN_DESC13_REG_FOR_SG; 

GEN_DESC14_REG_FOR_SG : if C_NUM_S2MM_CHANNELS > 14 generate

    CURDESC14_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    curdesc14_lsb_i  <= (others => '0');
                    error_pointer_set14   <= '0';

                -- Detected error has NOT register a desc pointer
                elsif(error_pointer_set14 = '0')then

                    -- Scatter Gather Fetch Error
                    if((sg_ftch_error = '1' or sg_updt_error = '1') and dest14 = '1')then
                        curdesc14_lsb_i       <= ftch_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set14   <= '1';
                    -- Scatter Gather Update Error
--                    elsif(sg_updt_error = '1' and dest14 = '1')then
--                        curdesc14_lsb_i       <= updt_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
--                        error_pointer_set14   <= '1';

                    -- Commanded to update descriptor value - used for indicating
                    -- current descriptor begin processed by dma controller
                    elsif(update_curdesc14 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest14 = '1')then
                        curdesc14_lsb_i       <= new_curdesc(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set14   <= '0';

                    -- CPU update of current descriptor pointer.  CPU
                    -- only allowed to update when engine is halted.
                    elsif(axi2ip_wrce(CURDESC14_LSB_INDEX) = '1' and halt_free = '1')then
                        curdesc14_lsb_i       <= axi2ip_wrdata(CURDESC_LOWER_MSB_BIT
                                                      downto CURDESC_LOWER_LSB_BIT)
                                              & ZERO_VALUE(CURDESC_RESERVED_BIT5
                                                      downto CURDESC_RESERVED_BIT0);
                        error_pointer_set14   <= '0';

                    end if;
                end if;
            end if;
        end process CURDESC14_LSB_REGISTER;

    ---------------------------------------------------------------------------
    -- Tail Descriptor LSB Register
    ---------------------------------------------------------------------------
    TAILDESC14_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    taildesc14_lsb_i  <= (others => '0');
                elsif(axi2ip_wrce(TAILDESC14_LSB_INDEX) = '1')then
                    taildesc14_lsb_i  <= axi2ip_wrdata(TAILDESC_LOWER_MSB_BIT
                                              downto TAILDESC_LOWER_LSB_BIT)
                                       & ZERO_VALUE(TAILDESC_RESERVED_BIT5
                                              downto TAILDESC_RESERVED_BIT0);

                end if;
            end if;
        end process TAILDESC14_LSB_REGISTER;

end generate GEN_DESC14_REG_FOR_SG;

GEN_DESC15_REG_FOR_SG : if C_NUM_S2MM_CHANNELS > 15 generate


    CURDESC15_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    curdesc15_lsb_i  <= (others => '0');
                    error_pointer_set15   <= '0';

                -- Detected error has NOT register a desc pointer
                elsif(error_pointer_set15 = '0')then

                    -- Scatter Gather Fetch Error
                    if((sg_ftch_error = '1' or sg_updt_error = '1') and dest15 = '1')then
                        curdesc15_lsb_i       <= ftch_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set15   <= '1';
                    -- Scatter Gather Update Error
--                    elsif(sg_updt_error = '1' and dest15 = '1')then
--                        curdesc15_lsb_i       <= updt_error_addr(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
--                        error_pointer_set15   <= '1';

                    -- Commanded to update descriptor value - used for indicating
                    -- current descriptor begin processed by dma controller
                    elsif(update_curdesc15 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest15 = '1')then
                        curdesc15_lsb_i       <= new_curdesc(C_S_AXI_LITE_DATA_WIDTH-1 downto 0);
                        error_pointer_set15   <= '0';

                    -- CPU update of current descriptor pointer.  CPU
                    -- only allowed to update when engine is halted.
                    elsif(axi2ip_wrce(CURDESC15_LSB_INDEX) = '1' and halt_free = '1')then
                        curdesc15_lsb_i       <= axi2ip_wrdata(CURDESC_LOWER_MSB_BIT
                                                      downto CURDESC_LOWER_LSB_BIT)
                                              & ZERO_VALUE(CURDESC_RESERVED_BIT5
                                                      downto CURDESC_RESERVED_BIT0);
                        error_pointer_set15   <= '0';

                    end if;
                end if;
            end if;
        end process CURDESC15_LSB_REGISTER;

    ---------------------------------------------------------------------------
    -- Tail Descriptor LSB Register
    ---------------------------------------------------------------------------
    TAILDESC15_LSB_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    taildesc15_lsb_i  <= (others => '0');
                elsif(axi2ip_wrce(TAILDESC15_LSB_INDEX) = '1')then
                    taildesc15_lsb_i  <= axi2ip_wrdata(TAILDESC_LOWER_MSB_BIT
                                              downto TAILDESC_LOWER_LSB_BIT)
                                       & ZERO_VALUE(TAILDESC_RESERVED_BIT5
                                              downto TAILDESC_RESERVED_BIT0);

                end if;
            end if;
        end process TAILDESC15_LSB_REGISTER;

end generate GEN_DESC15_REG_FOR_SG;


    ---------------------------------------------------------------------------
    -- Current Descriptor MSB Register
    ---------------------------------------------------------------------------
    -- Scatter Gather Interface configured for 64-Bit SG Addresses
    GEN_SG_ADDR_EQL64 :if C_M_AXI_SG_ADDR_WIDTH = 64 generate
    begin
        CURDESC_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        curdesc_msb_i  <= (others => '0');

                    elsif(error_pointer_set = '0')then
                        -- Scatter Gather Fetch Error
                        if((sg_ftch_error = '1' or sg_updt_error = '1') and dest0 = '1')then
                            curdesc_msb_i   <= ftch_error_addr(C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- Scatter Gather Update Error
                        elsif(sg_updt_error = '1' and dest0 = '1')then
                            curdesc_msb_i   <= updt_error_addr(C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- Commanded to update descriptor value - used for indicating
                        -- current descriptor begin processed by dma controller
                        elsif(update_curdesc = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest0 = '1')then
                            curdesc_msb_i <= new_curdesc (C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- CPU update of current descriptor pointer.  CPU
                        -- only allowed to update when engine is halted.
                        elsif(axi2ip_wrce(CURDESC_MSB_INDEX) = '1' and halt_free = '1')then
                            curdesc_msb_i  <= axi2ip_wrdata;

                        end if;
                    end if;
                end if;
            end process CURDESC_MSB_REGISTER;

        ---------------------------------------------------------------------------
        -- Tail Descriptor MSB Register
        ---------------------------------------------------------------------------
        TAILDESC_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        taildesc_msb_i  <= (others => '0');
                    elsif(axi2ip_wrce(TAILDESC_MSB_INDEX) = '1')then
                        taildesc_msb_i  <= axi2ip_wrdata;
                    end if;
                end if;
            end process TAILDESC_MSB_REGISTER;


GEN_DESC1_MSB_FOR_SG : if C_NUM_S2MM_CHANNELS > 1 generate

        CURDESC1_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        curdesc1_msb_i  <= (others => '0');

                    elsif(error_pointer_set1 = '0')then
                        -- Scatter Gather Fetch Error
                        if((sg_ftch_error = '1' or sg_updt_error = '1') and dest1 = '1')then
                            curdesc1_msb_i   <= ftch_error_addr(C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- Scatter Gather Update Error
--                        elsif(sg_updt_error = '1' and dest1 = '1')then
--                            curdesc1_msb_i   <= updt_error_addr((C_M_AXI_SG_ADDR_WIDTH
--                                                - C_S_AXI_LITE_DATA_WIDTH)-1
--                                                downto 0);

                        -- Commanded to update descriptor value - used for indicating
                        -- current descriptor begin processed by dma controller
                        elsif(update_curdesc1 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest1 = '1')then
                            curdesc1_msb_i <= new_curdesc (C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- CPU update of current descriptor pointer.  CPU
                        -- only allowed to update when engine is halted.
                        elsif(axi2ip_wrce(CURDESC1_MSB_INDEX) = '1' and halt_free = '1')then
                            curdesc1_msb_i  <= axi2ip_wrdata;

                        end if;
                    end if;
                end if;
            end process CURDESC1_MSB_REGISTER;

        ---------------------------------------------------------------------------
        -- Tail Descriptor MSB Register
        ---------------------------------------------------------------------------
        TAILDESC1_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        taildesc1_msb_i  <= (others => '0');
                    elsif(axi2ip_wrce(TAILDESC1_MSB_INDEX) = '1')then
                        taildesc1_msb_i  <= axi2ip_wrdata;
                    end if;
                end if;
            end process TAILDESC1_MSB_REGISTER;

end generate GEN_DESC1_MSB_FOR_SG;

GEN_DESC2_MSB_FOR_SG : if C_NUM_S2MM_CHANNELS > 2 generate
        
          CURDESC2_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        curdesc2_msb_i  <= (others => '0');

                    elsif(error_pointer_set2 = '0')then
                        -- Scatter Gather Fetch Error
                        if((sg_ftch_error = '1' or sg_updt_error = '1') and dest2 = '1')then
                            curdesc2_msb_i   <= ftch_error_addr(C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- Scatter Gather Update Error
--                        elsif(sg_updt_error = '1' and dest2 = '1')then
--                            curdesc2_msb_i   <= updt_error_addr((C_M_AXI_SG_ADDR_WIDTH
--                                                - C_S_AXI_LITE_DATA_WIDTH)-1
--                                                downto 0);

                        -- Commanded to update descriptor value - used for indicating
                        -- current descriptor begin processed by dma controller
                        elsif(update_curdesc2 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest2 = '1')then
                            curdesc2_msb_i <= new_curdesc (C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- CPU update of current descriptor pointer.  CPU
                        -- only allowed to update when engine is halted.
                        elsif(axi2ip_wrce(CURDESC2_MSB_INDEX) = '1' and halt_free = '1')then
                            curdesc2_msb_i  <= axi2ip_wrdata;

                        end if;
                    end if;
                end if;
            end process CURDESC2_MSB_REGISTER;

        ---------------------------------------------------------------------------
        -- Tail Descriptor MSB Register
        ---------------------------------------------------------------------------
        TAILDESC2_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        taildesc2_msb_i  <= (others => '0');
                    elsif(axi2ip_wrce(TAILDESC2_MSB_INDEX) = '1')then
                        taildesc2_msb_i  <= axi2ip_wrdata;
                    end if;
                end if;
            end process TAILDESC2_MSB_REGISTER;

end generate GEN_DESC2_MSB_FOR_SG;

GEN_DESC3_MSB_FOR_SG : if C_NUM_S2MM_CHANNELS > 3 generate



        CURDESC3_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        curdesc3_msb_i  <= (others => '0');

                    elsif(error_pointer_set3 = '0')then
                        -- Scatter Gather Fetch Error
                        if((sg_ftch_error = '1' or sg_updt_error = '1') and dest3 = '1')then
                            curdesc3_msb_i   <= ftch_error_addr(C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- Scatter Gather Update Error
--                        elsif(sg_updt_error = '1' and dest3 = '1')then
--                            curdesc3_msb_i   <= updt_error_addr((C_M_AXI_SG_ADDR_WIDTH
--                                                - C_S_AXI_LITE_DATA_WIDTH)-1
--                                                downto 0);

                        -- Commanded to update descriptor value - used for indicating
                        -- current descriptor begin processed by dma controller
                        elsif(update_curdesc3 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest3 = '1')then
                            curdesc3_msb_i <= new_curdesc (C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- CPU update of current descriptor pointer.  CPU
                        -- only allowed to update when engine is halted.
                        elsif(axi2ip_wrce(CURDESC3_MSB_INDEX) = '1' and halt_free = '1')then
                            curdesc3_msb_i  <= axi2ip_wrdata;

                        end if;
                    end if;
                end if;
            end process CURDESC3_MSB_REGISTER;

        ---------------------------------------------------------------------------
        -- Tail Descriptor MSB Register
        ---------------------------------------------------------------------------
        TAILDESC3_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        taildesc3_msb_i  <= (others => '0');
                    elsif(axi2ip_wrce(TAILDESC3_MSB_INDEX) = '1')then
                        taildesc3_msb_i  <= axi2ip_wrdata;
                    end if;
                end if;
            end process TAILDESC3_MSB_REGISTER;

end generate GEN_DESC3_MSB_FOR_SG;

GEN_DESC4_MSB_FOR_SG : if C_NUM_S2MM_CHANNELS > 4 generate



        CURDESC4_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        curdesc4_msb_i  <= (others => '0');

                    elsif(error_pointer_set4 = '0')then
                        -- Scatter Gather Fetch Error
                        if((sg_ftch_error = '1' or sg_updt_error = '1') and dest4 = '1')then
                            curdesc4_msb_i   <= ftch_error_addr(C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- Scatter Gather Update Error
--                        elsif(sg_updt_error = '1' and dest4 = '1')then
--                            curdesc4_msb_i   <= updt_error_addr((C_M_AXI_SG_ADDR_WIDTH
--                                                - C_S_AXI_LITE_DATA_WIDTH)-1
--                                                downto 0);

                        -- Commanded to update descriptor value - used for indicating
                        -- current descriptor begin processed by dma controller
                        elsif(update_curdesc4 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest4 = '1')then
                            curdesc4_msb_i <= new_curdesc (C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- CPU update of current descriptor pointer.  CPU
                        -- only allowed to update when engine is halted.
                        elsif(axi2ip_wrce(CURDESC4_MSB_INDEX) = '1' and halt_free = '1')then
                            curdesc4_msb_i  <= axi2ip_wrdata;

                        end if;
                    end if;
                end if;
            end process CURDESC4_MSB_REGISTER;

        ---------------------------------------------------------------------------
        -- Tail Descriptor MSB Register
        ---------------------------------------------------------------------------
        TAILDESC4_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        taildesc4_msb_i  <= (others => '0');
                    elsif(axi2ip_wrce(TAILDESC4_MSB_INDEX) = '1')then
                        taildesc4_msb_i  <= axi2ip_wrdata;
                    end if;
                end if;
            end process TAILDESC4_MSB_REGISTER;
end generate GEN_DESC4_MSB_FOR_SG;

GEN_DESC5_MSB_FOR_SG : if C_NUM_S2MM_CHANNELS > 5 generate


        CURDESC5_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        curdesc5_msb_i  <= (others => '0');

                    elsif(error_pointer_set5 = '0')then
                        -- Scatter Gather Fetch Error
                        if((sg_ftch_error = '1' or sg_updt_error = '1') and dest5 = '1')then
                            curdesc5_msb_i   <= ftch_error_addr(C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- Scatter Gather Update Error
--                        elsif(sg_updt_error = '1' and dest5 = '1')then
--                            curdesc5_msb_i   <= updt_error_addr((C_M_AXI_SG_ADDR_WIDTH
--                                                - C_S_AXI_LITE_DATA_WIDTH)-1
--                                                downto 0);

                        -- Commanded to update descriptor value - used for indicating
                        -- current descriptor begin processed by dma controller
                        elsif(update_curdesc5 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest5 = '1')then
                            curdesc5_msb_i <= new_curdesc (C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- CPU update of current descriptor pointer.  CPU
                        -- only allowed to update when engine is halted.
                        elsif(axi2ip_wrce(CURDESC5_MSB_INDEX) = '1' and halt_free = '1')then
                            curdesc5_msb_i  <= axi2ip_wrdata;

                        end if;
                    end if;
                end if;
            end process CURDESC5_MSB_REGISTER;

        ---------------------------------------------------------------------------
        -- Tail Descriptor MSB Register
        ---------------------------------------------------------------------------
        TAILDESC5_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        taildesc5_msb_i  <= (others => '0');
                    elsif(axi2ip_wrce(TAILDESC5_MSB_INDEX) = '1')then
                        taildesc5_msb_i  <= axi2ip_wrdata;
                    end if;
                end if;
            end process TAILDESC5_MSB_REGISTER;

end generate GEN_DESC5_MSB_FOR_SG;

GEN_DESC6_MSB_FOR_SG : if C_NUM_S2MM_CHANNELS > 6 generate

        CURDESC6_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        curdesc6_msb_i  <= (others => '0');

                    elsif(error_pointer_set6 = '0')then
                        -- Scatter Gather Fetch Error
                        if((sg_ftch_error = '1' or sg_updt_error = '1') and dest6 = '1')then
                            curdesc6_msb_i   <= ftch_error_addr(C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- Scatter Gather Update Error
--                        elsif(sg_updt_error = '1' and dest6 = '1')then
--                            curdesc6_msb_i   <= updt_error_addr((C_M_AXI_SG_ADDR_WIDTH
--                                                - C_S_AXI_LITE_DATA_WIDTH)-1
--                                                downto 0);

                        -- Commanded to update descriptor value - used for indicating
                        -- current descriptor begin processed by dma controller
                        elsif(update_curdesc6 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest6 = '1')then
                            curdesc6_msb_i <= new_curdesc (C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- CPU update of current descriptor pointer.  CPU
                        -- only allowed to update when engine is halted.
                        elsif(axi2ip_wrce(CURDESC6_MSB_INDEX) = '1' and halt_free = '1')then
                            curdesc6_msb_i  <= axi2ip_wrdata;

                        end if;
                    end if;
                end if;
            end process CURDESC6_MSB_REGISTER;

        ---------------------------------------------------------------------------
        -- Tail Descriptor MSB Register
        ---------------------------------------------------------------------------
        TAILDESC6_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        taildesc6_msb_i  <= (others => '0');
                    elsif(axi2ip_wrce(TAILDESC6_MSB_INDEX) = '1')then
                        taildesc6_msb_i  <= axi2ip_wrdata;
                    end if;
                end if;
            end process TAILDESC6_MSB_REGISTER;
end generate GEN_DESC6_MSB_FOR_SG;

GEN_DESC7_MSB_FOR_SG : if C_NUM_S2MM_CHANNELS > 7 generate


        CURDESC7_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        curdesc7_msb_i  <= (others => '0');

                    elsif(error_pointer_set7 = '0')then
                        -- Scatter Gather Fetch Error
                        if((sg_ftch_error = '1' or sg_updt_error = '1') and dest7 = '1')then
                            curdesc7_msb_i   <= ftch_error_addr(C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- Scatter Gather Update Error
--                        elsif(sg_updt_error = '1' and dest7 = '1')then
--                            curdesc7_msb_i   <= updt_error_addr((C_M_AXI_SG_ADDR_WIDTH
--                                                - C_S_AXI_LITE_DATA_WIDTH)-1
--                                                downto 0);

                        -- Commanded to update descriptor value - used for indicating
                        -- current descriptor begin processed by dma controller
                        elsif(update_curdesc7 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest7 = '1')then
                            curdesc7_msb_i <= new_curdesc (C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- CPU update of current descriptor pointer.  CPU
                        -- only allowed to update when engine is halted.
                        elsif(axi2ip_wrce(CURDESC7_MSB_INDEX) = '1' and halt_free = '1')then
                            curdesc7_msb_i  <= axi2ip_wrdata;

                        end if;
                    end if;
                end if;
            end process CURDESC7_MSB_REGISTER;

        ---------------------------------------------------------------------------
        -- Tail Descriptor MSB Register
        ---------------------------------------------------------------------------
        TAILDESC7_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        taildesc7_msb_i  <= (others => '0');
                    elsif(axi2ip_wrce(TAILDESC7_MSB_INDEX) = '1')then
                        taildesc7_msb_i  <= axi2ip_wrdata;
                    end if;
                end if;
            end process TAILDESC7_MSB_REGISTER;

end generate GEN_DESC7_MSB_FOR_SG;

GEN_DESC8_MSB_FOR_SG : if C_NUM_S2MM_CHANNELS > 8 generate

        CURDESC8_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        curdesc8_msb_i  <= (others => '0');

                    elsif(error_pointer_set8 = '0')then
                        -- Scatter Gather Fetch Error
                        if((sg_ftch_error = '1' or sg_updt_error = '1') and dest8 = '1')then
                            curdesc8_msb_i   <= ftch_error_addr(C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- Scatter Gather Update Error
--                        elsif(sg_updt_error = '1' and dest8 = '1')then
--                            curdesc8_msb_i   <= updt_error_addr((C_M_AXI_SG_ADDR_WIDTH
--                                                - C_S_AXI_LITE_DATA_WIDTH)-1
--                                                downto 0);

                        -- Commanded to update descriptor value - used for indicating
                        -- current descriptor begin processed by dma controller
                        elsif(update_curdesc8 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest8 = '1')then
                            curdesc8_msb_i <= new_curdesc (C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- CPU update of current descriptor pointer.  CPU
                        -- only allowed to update when engine is halted.
                        elsif(axi2ip_wrce(CURDESC8_MSB_INDEX) = '1' and halt_free = '1')then
                            curdesc8_msb_i  <= axi2ip_wrdata;

                        end if;
                    end if;
                end if;
            end process CURDESC8_MSB_REGISTER;

        ---------------------------------------------------------------------------
        -- Tail Descriptor MSB Register
        ---------------------------------------------------------------------------
        TAILDESC8_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        taildesc8_msb_i  <= (others => '0');
                    elsif(axi2ip_wrce(TAILDESC8_MSB_INDEX) = '1')then
                        taildesc8_msb_i  <= axi2ip_wrdata;
                    end if;
                end if;
            end process TAILDESC8_MSB_REGISTER;


end generate  GEN_DESC8_MSB_FOR_SG;

GEN_DESC9_MSB_FOR_SG : if C_NUM_S2MM_CHANNELS > 9 generate

        CURDESC9_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        curdesc9_msb_i  <= (others => '0');

                    elsif(error_pointer_set9 = '0')then
                        -- Scatter Gather Fetch Error
                        if((sg_ftch_error = '1' or sg_updt_error = '1') and dest9 = '1')then
                            curdesc9_msb_i   <= ftch_error_addr(C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- Scatter Gather Update Error
--                        elsif(sg_updt_error = '1' and dest9 = '1')then
--                            curdesc9_msb_i   <= updt_error_addr((C_M_AXI_SG_ADDR_WIDTH
--                                                - C_S_AXI_LITE_DATA_WIDTH)-1
--                                                downto 0);

                        -- Commanded to update descriptor value - used for indicating
                        -- current descriptor begin processed by dma controller
                        elsif(update_curdesc9 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest9 = '1')then
                            curdesc9_msb_i <= new_curdesc (C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- CPU update of current descriptor pointer.  CPU
                        -- only allowed to update when engine is halted.
                        elsif(axi2ip_wrce(CURDESC9_MSB_INDEX) = '1' and halt_free = '1')then
                            curdesc9_msb_i  <= axi2ip_wrdata;

                        end if;
                    end if;
                end if;
            end process CURDESC9_MSB_REGISTER;

        ---------------------------------------------------------------------------
        -- Tail Descriptor MSB Register
        ---------------------------------------------------------------------------
        TAILDESC9_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        taildesc9_msb_i  <= (others => '0');
                    elsif(axi2ip_wrce(TAILDESC9_MSB_INDEX) = '1')then
                        taildesc9_msb_i  <= axi2ip_wrdata;
                    end if;
                end if;
            end process TAILDESC9_MSB_REGISTER;

end generate GEN_DESC9_MSB_FOR_SG;

GEN_DESC10_MSB_FOR_SG : if C_NUM_S2MM_CHANNELS > 10 generate


        CURDESC10_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        curdesc10_msb_i  <= (others => '0');

                    elsif(error_pointer_set10 = '0')then
                        -- Scatter Gather Fetch Error
                        if((sg_ftch_error = '1' or sg_updt_error = '1') and dest10 = '1')then
                            curdesc10_msb_i   <= ftch_error_addr(C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- Scatter Gather Update Error
--                        elsif(sg_updt_error = '1' and dest10 = '1')then
--                            curdesc10_msb_i   <= updt_error_addr((C_M_AXI_SG_ADDR_WIDTH
--                                                - C_S_AXI_LITE_DATA_WIDTH)-1
--                                                downto 0);

                        -- Commanded to update descriptor value - used for indicating
                        -- current descriptor begin processed by dma controller
                        elsif(update_curdesc10 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest10 = '1')then
                            curdesc10_msb_i <= new_curdesc (C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- CPU update of current descriptor pointer.  CPU
                        -- only allowed to update when engine is halted.
                        elsif(axi2ip_wrce(CURDESC10_MSB_INDEX) = '1' and halt_free = '1')then
                            curdesc10_msb_i  <= axi2ip_wrdata;

                        end if;
                    end if;
                end if;
            end process CURDESC10_MSB_REGISTER;

        ---------------------------------------------------------------------------
        -- Tail Descriptor MSB Register
        ---------------------------------------------------------------------------
        TAILDESC10_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        taildesc10_msb_i  <= (others => '0');
                    elsif(axi2ip_wrce(TAILDESC10_MSB_INDEX) = '1')then
                        taildesc10_msb_i  <= axi2ip_wrdata;
                    end if;
                end if;
            end process TAILDESC10_MSB_REGISTER;


end generate GEN_DESC10_MSB_FOR_SG;

GEN_DESC11_MSB_FOR_SG : if C_NUM_S2MM_CHANNELS > 11 generate

        CURDESC11_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        curdesc11_msb_i  <= (others => '0');

                    elsif(error_pointer_set11 = '0')then
                        -- Scatter Gather Fetch Error
                        if((sg_ftch_error = '1' or sg_updt_error = '1') and dest11 = '1')then
                            curdesc11_msb_i   <= ftch_error_addr(C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- Scatter Gather Update Error
--                        elsif(sg_updt_error = '1' and dest11 = '1')then
--                            curdesc11_msb_i   <= updt_error_addr((C_M_AXI_SG_ADDR_WIDTH
--                                                - C_S_AXI_LITE_DATA_WIDTH)-1
--                                                downto 0);

                        -- Commanded to update descriptor value - used for indicating
                        -- current descriptor begin processed by dma controller
                        elsif(update_curdesc11 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest11 = '1')then
                            curdesc11_msb_i <= new_curdesc (C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- CPU update of current descriptor pointer.  CPU
                        -- only allowed to update when engine is halted.
                        elsif(axi2ip_wrce(CURDESC11_MSB_INDEX) = '1' and halt_free = '1')then
                            curdesc11_msb_i  <= axi2ip_wrdata;

                        end if;
                    end if;
                end if;
            end process CURDESC11_MSB_REGISTER;

        ---------------------------------------------------------------------------
        -- Tail Descriptor MSB Register
        ---------------------------------------------------------------------------
        TAILDESC11_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        taildesc11_msb_i  <= (others => '0');
                    elsif(axi2ip_wrce(TAILDESC11_MSB_INDEX) = '1')then
                        taildesc11_msb_i  <= axi2ip_wrdata;
                    end if;
                end if;
            end process TAILDESC11_MSB_REGISTER;

end generate GEN_DESC11_MSB_FOR_SG;

GEN_DESC12_MSB_FOR_SG : if C_NUM_S2MM_CHANNELS > 12 generate


        CURDESC12_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        curdesc12_msb_i  <= (others => '0');

                    elsif(error_pointer_set12 = '0')then
                        -- Scatter Gather Fetch Error
                        if((sg_ftch_error = '1' or sg_updt_error = '1') and dest12 = '1')then
                            curdesc12_msb_i   <= ftch_error_addr(C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- Scatter Gather Update Error
--                        elsif(sg_updt_error = '1' and dest12 = '1')then
--                            curdesc12_msb_i   <= updt_error_addr((C_M_AXI_SG_ADDR_WIDTH
--                                                - C_S_AXI_LITE_DATA_WIDTH)-1
--                                                downto 0);

                        -- Commanded to update descriptor value - used for indicating
                        -- current descriptor begin processed by dma controller
                        elsif(update_curdesc12 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest12 = '1')then
                            curdesc12_msb_i <= new_curdesc (C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- CPU update of current descriptor pointer.  CPU
                        -- only allowed to update when engine is halted.
                        elsif(axi2ip_wrce(CURDESC12_MSB_INDEX) = '1' and halt_free = '1')then
                            curdesc12_msb_i  <= axi2ip_wrdata;

                        end if;
                    end if;
                end if;
            end process CURDESC12_MSB_REGISTER;

        ---------------------------------------------------------------------------
        -- Tail Descriptor MSB Register
        ---------------------------------------------------------------------------
        TAILDESC12_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        taildesc12_msb_i  <= (others => '0');
                    elsif(axi2ip_wrce(TAILDESC12_MSB_INDEX) = '1')then
                        taildesc12_msb_i  <= axi2ip_wrdata;
                    end if;
                end if;
            end process TAILDESC12_MSB_REGISTER;

end generate GEN_DESC12_MSB_FOR_SG;

GEN_DESC13_MSB_FOR_SG : if C_NUM_S2MM_CHANNELS > 13 generate

        CURDESC13_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        curdesc13_msb_i  <= (others => '0');

                    elsif(error_pointer_set13 = '0')then
                        -- Scatter Gather Fetch Error
                        if((sg_ftch_error = '1' or sg_updt_error = '1') and dest13 = '1')then
                            curdesc13_msb_i   <= ftch_error_addr(C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- Scatter Gather Update Error
--                        elsif(sg_updt_error = '1' and dest13 = '1')then
--                            curdesc13_msb_i   <= updt_error_addr((C_M_AXI_SG_ADDR_WIDTH
--                                                - C_S_AXI_LITE_DATA_WIDTH)-1
--                                                downto 0);

                        -- Commanded to update descriptor value - used for indicating
                        -- current descriptor begin processed by dma controller
                        elsif(update_curdesc13 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest13 = '1')then
                            curdesc13_msb_i <= new_curdesc (C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- CPU update of current descriptor pointer.  CPU
                        -- only allowed to update when engine is halted.
                        elsif(axi2ip_wrce(CURDESC13_MSB_INDEX) = '1' and halt_free = '1')then
                            curdesc13_msb_i  <= axi2ip_wrdata;

                        end if;
                    end if;
                end if;
            end process CURDESC13_MSB_REGISTER;

        ---------------------------------------------------------------------------
        -- Tail Descriptor MSB Register
        ---------------------------------------------------------------------------
        TAILDESC13_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        taildesc13_msb_i  <= (others => '0');
                    elsif(axi2ip_wrce(TAILDESC13_MSB_INDEX) = '1')then
                        taildesc13_msb_i  <= axi2ip_wrdata;
                    end if;
                end if;
            end process TAILDESC13_MSB_REGISTER;

end generate GEN_DESC13_MSB_FOR_SG;


GEN_DESC14_MSB_FOR_SG : if C_NUM_S2MM_CHANNELS > 14 generate

        CURDESC14_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        curdesc14_msb_i  <= (others => '0');

                    elsif(error_pointer_set14 = '0')then
                        -- Scatter Gather Fetch Error
                        if((sg_ftch_error = '1' or sg_updt_error = '1') and dest14 = '1')then
                            curdesc14_msb_i   <= ftch_error_addr(C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- Scatter Gather Update Error
--                        elsif(sg_updt_error = '1' and dest14 = '1')then
--                            curdesc14_msb_i   <= updt_error_addr((C_M_AXI_SG_ADDR_WIDTH
--                                                - C_S_AXI_LITE_DATA_WIDTH)-1
--                                                downto 0);

                        -- Commanded to update descriptor value - used for indicating
                        -- current descriptor begin processed by dma controller
                        elsif(update_curdesc14 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest14 = '1')then
                            curdesc14_msb_i <= new_curdesc (C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- CPU update of current descriptor pointer.  CPU
                        -- only allowed to update when engine is halted.
                        elsif(axi2ip_wrce(CURDESC14_MSB_INDEX) = '1' and halt_free = '1')then
                            curdesc14_msb_i  <= axi2ip_wrdata;

                        end if;
                    end if;
                end if;
            end process CURDESC14_MSB_REGISTER;

        ---------------------------------------------------------------------------
        -- Tail Descriptor MSB Register
        ---------------------------------------------------------------------------
        TAILDESC14_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        taildesc14_msb_i  <= (others => '0');
                    elsif(axi2ip_wrce(TAILDESC14_MSB_INDEX) = '1')then
                        taildesc14_msb_i  <= axi2ip_wrdata;
                    end if;
                end if;
            end process TAILDESC14_MSB_REGISTER;

end generate GEN_DESC14_MSB_FOR_SG;


GEN_DESC15_MSB_FOR_SG : if C_NUM_S2MM_CHANNELS > 15 generate

        CURDESC15_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        curdesc15_msb_i  <= (others => '0');

                    elsif(error_pointer_set15 = '0')then
                        -- Scatter Gather Fetch Error
                        if((sg_ftch_error = '1' or sg_updt_error = '1') and dest15 = '1')then
                            curdesc15_msb_i   <= ftch_error_addr(C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- Scatter Gather Update Error
--                        elsif(sg_updt_error = '1' and dest15 = '1')then
--                            curdesc15_msb_i   <= updt_error_addr((C_M_AXI_SG_ADDR_WIDTH
--                                                - C_S_AXI_LITE_DATA_WIDTH)-1
--                                                downto 0);

                        -- Commanded to update descriptor value - used for indicating
                        -- current descriptor begin processed by dma controller
                        elsif(update_curdesc15 = '1' and dmacr_i(DMACR_RS_BIT)  = '1' and dest15 = '1')then
                            curdesc15_msb_i <= new_curdesc (C_M_AXI_SG_ADDR_WIDTH-1 downto C_S_AXI_LITE_DATA_WIDTH);

                        -- CPU update of current descriptor pointer.  CPU
                        -- only allowed to update when engine is halted.
                        elsif(axi2ip_wrce(CURDESC15_MSB_INDEX) = '1' and halt_free = '1')then
                            curdesc15_msb_i  <= axi2ip_wrdata;

                        end if;
                    end if;
                end if;
            end process CURDESC15_MSB_REGISTER;

        ---------------------------------------------------------------------------
        -- Tail Descriptor MSB Register
        ---------------------------------------------------------------------------
        TAILDESC15_MSB_REGISTER : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        taildesc15_msb_i  <= (others => '0');
                    elsif(axi2ip_wrce(TAILDESC15_MSB_INDEX) = '1')then
                        taildesc15_msb_i  <= axi2ip_wrdata;
                    end if;
                end if;
            end process TAILDESC15_MSB_REGISTER;

end generate GEN_DESC15_MSB_FOR_SG;

        end generate GEN_SG_ADDR_EQL64;

    -- Scatter Gather Interface configured for 32-Bit SG Addresses
    GEN_SG_ADDR_EQL32 : if C_M_AXI_SG_ADDR_WIDTH = 32 generate
    begin
        curdesc_msb_i  <= (others => '0');
        taildesc_msb_i <= (others => '0');
    -- Extending this to the extra registers
        curdesc1_msb_i  <= (others => '0');
        taildesc1_msb_i <= (others => '0');
        curdesc2_msb_i  <= (others => '0');
        taildesc2_msb_i <= (others => '0');
        curdesc3_msb_i  <= (others => '0');
        taildesc3_msb_i <= (others => '0');
        curdesc4_msb_i  <= (others => '0');
        taildesc4_msb_i <= (others => '0');
        curdesc5_msb_i  <= (others => '0');
        taildesc5_msb_i <= (others => '0');
        curdesc6_msb_i  <= (others => '0');
        taildesc6_msb_i <= (others => '0');
        curdesc7_msb_i  <= (others => '0');
        taildesc7_msb_i <= (others => '0');
        curdesc8_msb_i  <= (others => '0');
        taildesc8_msb_i <= (others => '0');
        curdesc9_msb_i  <= (others => '0');
        taildesc9_msb_i <= (others => '0');
        curdesc10_msb_i  <= (others => '0');
        taildesc10_msb_i <= (others => '0');
        curdesc11_msb_i  <= (others => '0');
        taildesc11_msb_i <= (others => '0');
        curdesc12_msb_i  <= (others => '0');
        taildesc12_msb_i <= (others => '0');
        curdesc13_msb_i  <= (others => '0');
        taildesc13_msb_i <= (others => '0');
        curdesc14_msb_i  <= (others => '0');
        taildesc14_msb_i <= (others => '0');
        curdesc15_msb_i  <= (others => '0');
        taildesc15_msb_i <= (others => '0');

    end generate GEN_SG_ADDR_EQL32;


    -- Scatter Gather Interface configured for 32-Bit SG Addresses
    GEN_TAILUPDATE_EQL32 : if C_M_AXI_SG_ADDR_WIDTH = 32 generate
    begin

 -- Added dest so that BD can be dynamically updated

GENERATE_MULTI_CH : if C_ENABLE_MULTI_CHANNEL = 1 generate
        tail_update_lsb <= (axi2ip_wrce(TAILDESC_LSB_INDEX) and dest0) or 
                           (axi2ip_wrce(TAILDESC1_LSB_INDEX) and dest1) or 
                           (axi2ip_wrce(TAILDESC2_LSB_INDEX) and dest2) or 
                           (axi2ip_wrce(TAILDESC3_LSB_INDEX) and dest3) or 
                           (axi2ip_wrce(TAILDESC4_LSB_INDEX) and dest4) or 
                           (axi2ip_wrce(TAILDESC5_LSB_INDEX) and dest5) or 
                           (axi2ip_wrce(TAILDESC6_LSB_INDEX) and dest6) or 
                           (axi2ip_wrce(TAILDESC7_LSB_INDEX) and dest7) or 
                           (axi2ip_wrce(TAILDESC8_LSB_INDEX) and dest8) or 
                           (axi2ip_wrce(TAILDESC9_LSB_INDEX) and dest9) or 
                           (axi2ip_wrce(TAILDESC10_LSB_INDEX) and dest10) or 
                           (axi2ip_wrce(TAILDESC11_LSB_INDEX) and dest11) or 
                           (axi2ip_wrce(TAILDESC12_LSB_INDEX) and dest12) or 
                           (axi2ip_wrce(TAILDESC13_LSB_INDEX) and dest13) or 
                           (axi2ip_wrce(TAILDESC14_LSB_INDEX) and dest14) or 
                           (axi2ip_wrce(TAILDESC15_LSB_INDEX) and dest15); 
end generate GENERATE_MULTI_CH;

GENERATE_NO_MULTI_CH : if C_ENABLE_MULTI_CHANNEL = 0 generate
        tail_update_lsb <= (axi2ip_wrce(TAILDESC_LSB_INDEX) and dest0);

end generate GENERATE_NO_MULTI_CH;
        
        TAILPNTR_UPDT_PROCESS : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0' or dmacr_i(DMACR_RS_BIT)='0')then
                        tailpntr_updated_d1    <= '0';
                    elsif (tail_update_lsb = '1' and tdest_in(5) = '0')then  
                        tailpntr_updated_d1    <= '1';
                    else
                        tailpntr_updated_d1    <= '0';
                    end if;
                end if;
            end process TAILPNTR_UPDT_PROCESS;

        TAILPNTR_UPDT_PROCESS_DEL : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        tailpntr_updated_d2    <= '0';
                    else
                        tailpntr_updated_d2    <= tailpntr_updated_d1;
                    end if;
                end if;
            end process TAILPNTR_UPDT_PROCESS_DEL;
   
           tailpntr_updated <= tailpntr_updated_d1 and (not tailpntr_updated_d2);

    end generate GEN_TAILUPDATE_EQL32;

    -- Scatter Gather Interface configured for 64-Bit SG Addresses
    GEN_TAILUPDATE_EQL64 : if C_M_AXI_SG_ADDR_WIDTH = 64 generate
    begin

 -- Added dest so that BD can be dynamically updated
GENERATE_NO_MULTI_CH1 : if C_ENABLE_MULTI_CHANNEL = 1 generate
        tail_update_msb <= (axi2ip_wrce(TAILDESC_MSB_INDEX) and dest0) or 
                           (axi2ip_wrce(TAILDESC1_MSB_INDEX) and dest1) or 
                           (axi2ip_wrce(TAILDESC2_MSB_INDEX) and dest2) or 
                           (axi2ip_wrce(TAILDESC3_MSB_INDEX) and dest3) or 
                           (axi2ip_wrce(TAILDESC4_MSB_INDEX) and dest4) or 
                           (axi2ip_wrce(TAILDESC5_MSB_INDEX) and dest5) or 
                           (axi2ip_wrce(TAILDESC6_MSB_INDEX) and dest6) or 
                           (axi2ip_wrce(TAILDESC7_MSB_INDEX) and dest7) or 
                           (axi2ip_wrce(TAILDESC8_MSB_INDEX) and dest8) or 
                           (axi2ip_wrce(TAILDESC9_MSB_INDEX) and dest9) or 
                           (axi2ip_wrce(TAILDESC10_MSB_INDEX) and dest10) or 
                           (axi2ip_wrce(TAILDESC11_MSB_INDEX) and dest11) or 
                           (axi2ip_wrce(TAILDESC12_MSB_INDEX) and dest12) or 
                           (axi2ip_wrce(TAILDESC13_MSB_INDEX) and dest13) or 
                           (axi2ip_wrce(TAILDESC14_MSB_INDEX) and dest14) or 
                           (axi2ip_wrce(TAILDESC15_MSB_INDEX) and dest15); 
end generate GENERATE_NO_MULTI_CH1;


GENERATE_NO_MULTI_CH2 : if C_ENABLE_MULTI_CHANNEL = 0 generate

        tail_update_msb <= (axi2ip_wrce(TAILDESC_MSB_INDEX) and dest0);

end generate GENERATE_NO_MULTI_CH2;

        TAILPNTR_UPDT_PROCESS : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0' or dmacr_i(DMACR_RS_BIT)='0')then
                        tailpntr_updated_d1    <= '0';
                    elsif (tail_update_msb = '1'  and tdest_in(5) = '0')then
                        tailpntr_updated_d1    <= '1';
                    else
                        tailpntr_updated_d1    <= '0';
                    end if;
                end if;
            end process TAILPNTR_UPDT_PROCESS;


        TAILPNTR_UPDT_PROCESS_DEL : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        tailpntr_updated_d2    <= '0';
                    else
                        tailpntr_updated_d2    <= tailpntr_updated_d1;
                    end if;
                end if;
            end process TAILPNTR_UPDT_PROCESS_DEL;
   
           tailpntr_updated <= tailpntr_updated_d1 and (not tailpntr_updated_d2);

    end generate GEN_TAILUPDATE_EQL64;

end generate GEN_DESC_REG_FOR_SG;


-- Generate Buffer Address and Length Register for Simple DMA Mode
GEN_REG_FOR_SMPL : if C_INCLUDE_SG = 0 generate
begin
    -- Signals not used for simple dma mode, only for sg mode
    curdesc_lsb_i       <= (others => '0');
    curdesc_msb_i       <= (others => '0');
    taildesc_lsb_i      <= (others => '0');
    taildesc_msb_i      <= (others => '0');
-- Extending this to new registers
    curdesc1_msb_i  <= (others => '0');
    taildesc1_msb_i <= (others => '0');
    curdesc2_msb_i  <= (others => '0');
    taildesc2_msb_i <= (others => '0');
    curdesc3_msb_i  <= (others => '0');
    taildesc3_msb_i <= (others => '0');
    curdesc4_msb_i  <= (others => '0');
    taildesc4_msb_i <= (others => '0');
    curdesc5_msb_i  <= (others => '0');
    taildesc5_msb_i <= (others => '0');
    curdesc6_msb_i  <= (others => '0');
    taildesc6_msb_i <= (others => '0');
    curdesc7_msb_i  <= (others => '0');
    taildesc7_msb_i <= (others => '0');
    curdesc8_msb_i  <= (others => '0');
    taildesc8_msb_i <= (others => '0');
    curdesc9_msb_i  <= (others => '0');
    taildesc9_msb_i <= (others => '0');
    curdesc10_msb_i  <= (others => '0');
    taildesc10_msb_i <= (others => '0');
    curdesc11_msb_i  <= (others => '0');
    taildesc11_msb_i <= (others => '0');
    curdesc12_msb_i  <= (others => '0');
    taildesc12_msb_i <= (others => '0');
    curdesc13_msb_i  <= (others => '0');
    taildesc13_msb_i <= (others => '0');
    curdesc14_msb_i  <= (others => '0');
    taildesc14_msb_i <= (others => '0');
    curdesc15_msb_i  <= (others => '0');
    taildesc15_msb_i <= (others => '0');

    curdesc1_lsb_i  <= (others => '0');
    taildesc1_lsb_i <= (others => '0');
    curdesc2_lsb_i  <= (others => '0');
    taildesc2_lsb_i <= (others => '0');
    curdesc3_lsb_i  <= (others => '0');
    taildesc3_lsb_i <= (others => '0');
    curdesc4_lsb_i  <= (others => '0');
    taildesc4_lsb_i <= (others => '0');
    curdesc5_lsb_i  <= (others => '0');
    taildesc5_lsb_i <= (others => '0');
    curdesc6_lsb_i  <= (others => '0');
    taildesc6_lsb_i <= (others => '0');
    curdesc7_lsb_i  <= (others => '0');
    taildesc7_lsb_i <= (others => '0');
    curdesc8_lsb_i  <= (others => '0');
    taildesc8_lsb_i <= (others => '0');
    curdesc9_lsb_i  <= (others => '0');
    taildesc9_lsb_i <= (others => '0');
    curdesc10_lsb_i  <= (others => '0');
    taildesc10_lsb_i <= (others => '0');
    curdesc11_lsb_i  <= (others => '0');
    taildesc11_lsb_i <= (others => '0');
    curdesc12_lsb_i  <= (others => '0');
    taildesc12_lsb_i <= (others => '0');
    curdesc13_lsb_i  <= (others => '0');
    taildesc13_lsb_i <= (others => '0');
    curdesc14_lsb_i  <= (others => '0');
    taildesc14_lsb_i <= (others => '0');
    curdesc15_lsb_i  <= (others => '0');
    taildesc15_lsb_i <= (others => '0');

    tailpntr_updated    <= '0';
    error_pointer_set   <= '0';

    -- Buffer Address register.  Used for Source Address (SA) if MM2S
    -- and used for Destination Address (DA) if S2MM
    BUFFER_ADDR_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    buffer_address_i  <= (others => '0');
                elsif(axi2ip_wrce(BUFF_ADDRESS_INDEX) = '1')then
                    buffer_address_i  <= axi2ip_wrdata;
                end if;
            end if;
        end process BUFFER_ADDR_REGISTER;

    GEN_BUF_ADDR_EQL64 : if C_M_AXI_SG_ADDR_WIDTH > 32 generate
    begin

    BUFFER_ADDR_REGISTER1 : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    buffer_address_64_i  <= (others => '0');
                elsif(axi2ip_wrce(BUFF_ADDRESS_MSB_INDEX) = '1')then
                    buffer_address_64_i  <= axi2ip_wrdata;
                end if;
            end if;
        end process BUFFER_ADDR_REGISTER1;

    end generate GEN_BUF_ADDR_EQL64;


    -- Buffer Length register.  Used for number of bytes to transfer if MM2S
    -- and used for size of receive buffer is S2MM
    BUFFER_LNGTH_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    buffer_length_i  <= (others => '0');

                -- Update with actual bytes received (Only for S2MM channel)
                elsif(bytes_received_wren = '1' and C_MICRO_DMA = 0)then
                    buffer_length_i <= bytes_received;

                elsif(axi2ip_wrce(BUFF_LENGTH_INDEX) = '1')then
                    buffer_length_i  <= axi2ip_wrdata(C_SG_LENGTH_WIDTH-1 downto 0);
                end if;
            end if;
        end process BUFFER_LNGTH_REGISTER;

    -- Buffer Length Write Enable control.  Assertion of wren will
    -- begin a transfer if channel is Idle.
    BUFFER_LNGTH_WRITE : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    buffer_length_wren <= '0';
                -- Non-zero length value written
                elsif(axi2ip_wrce(BUFF_LENGTH_INDEX) = '1'
                and axi2ip_wrdata(C_SG_LENGTH_WIDTH-1 downto 0) /= ZERO_VALUE(C_SG_LENGTH_WIDTH-1 downto 0))then
                    buffer_length_wren <= '1';
                else
                    buffer_length_wren <= '0';
                end if;
            end if;
        end process BUFFER_LNGTH_WRITE;

end generate GEN_REG_FOR_SMPL;



end implementation;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:          axi_dma_reg_module.vhd
-- Description: This entity is AXI DMA Register Module Top Level
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library lib_cdc_v1_0_2;
library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;

-------------------------------------------------------------------------------
entity  axi_dma_reg_module is
    generic(
        C_INCLUDE_MM2S              : integer range 0 to 1      := 1        ;
        C_INCLUDE_S2MM              : integer range 0 to 1      := 1        ;
        C_INCLUDE_SG                : integer range 0 to 1      := 1        ;
        C_SG_LENGTH_WIDTH           : integer range 8 to 23     := 14       ;
        C_AXI_LITE_IS_ASYNC         : integer range 0 to 1      := 0        ;
        C_S_AXI_LITE_ADDR_WIDTH     : integer range 2 to 32    := 32       ;
        C_S_AXI_LITE_DATA_WIDTH     : integer range 32 to 32    := 32       ;
        C_M_AXI_SG_ADDR_WIDTH       : integer range 32 to 64    := 32       ;
        C_M_AXI_MM2S_ADDR_WIDTH     : integer range 32 to 64    := 32       ;
        C_M_AXI_S2MM_ADDR_WIDTH     : integer range 32 to 64    := 32       ;
        C_NUM_S2MM_CHANNELS         : integer range 1 to 16     := 1        ;
        C_MICRO_DMA                 : integer range 0 to 1      := 0        ;
        C_ENABLE_MULTI_CHANNEL             : integer range 0 to 1      := 0

    );
    port (
        -----------------------------------------------------------------------
        -- AXI Lite Control Interface
        -----------------------------------------------------------------------
        m_axi_sg_aclk               : in  std_logic                         ;                   --
        m_axi_sg_aresetn            : in  std_logic                         ;                   --
        m_axi_sg_hrdresetn          : in  std_logic                         ;                   --
                                                                                                --
        s_axi_lite_aclk             : in  std_logic                         ;                   --
        axi_lite_reset_n            : in  std_logic                         ;                   --
                                                                                                --
        -- AXI Lite Write Address Channel                                                       --
        s_axi_lite_awvalid          : in  std_logic                         ;                   --
        s_axi_lite_awready          : out std_logic                         ;                   --
        s_axi_lite_awaddr           : in  std_logic_vector                                      --
                                        (C_S_AXI_LITE_ADDR_WIDTH-1 downto 0);                   --
                                                                                                --
        -- AXI Lite Write Data Channel                                                          --
        s_axi_lite_wvalid           : in  std_logic                         ;                   --
        s_axi_lite_wready           : out std_logic                         ;                   --
        s_axi_lite_wdata            : in  std_logic_vector                                      --
                                        (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);                   --
                                                                                                --
        -- AXI Lite Write Response Channel                                                      --
        s_axi_lite_bresp            : out std_logic_vector(1 downto 0)      ;                   --
        s_axi_lite_bvalid           : out std_logic                         ;                   --
        s_axi_lite_bready           : in  std_logic                         ;                   --
                                                                                                --
        -- AXI Lite Read Address Channel                                                        --
        s_axi_lite_arvalid          : in  std_logic                         ;                   --
        s_axi_lite_arready          : out std_logic                         ;                   --
        s_axi_lite_araddr           : in  std_logic_vector                                      --
                                        (C_S_AXI_LITE_ADDR_WIDTH-1 downto 0);                   --
        s_axi_lite_rvalid           : out std_logic                         ;                   --
        s_axi_lite_rready           : in  std_logic                         ;                   --
        s_axi_lite_rdata            : out std_logic_vector                                      --
                                        (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);                   --
        s_axi_lite_rresp            : out std_logic_vector(1 downto 0)      ;                   --
                                                                                                --
                                                                                                --
        -- MM2S Signals                                                                         --
        mm2s_stop                   : in  std_logic                         ;                   --
        mm2s_halted_clr             : in  std_logic                         ;                   --
        mm2s_halted_set             : in  std_logic                         ;                   --
        mm2s_idle_set               : in  std_logic                         ;                   --
        mm2s_idle_clr               : in  std_logic                         ;                   --
        mm2s_dma_interr_set         : in  std_logic                         ;                   --
        mm2s_dma_slverr_set         : in  std_logic                         ;                   --
        mm2s_dma_decerr_set         : in  std_logic                         ;                   --
        mm2s_ioc_irq_set            : in  std_logic                         ;                   --
        mm2s_dly_irq_set            : in  std_logic                         ;                   --
        mm2s_irqdelay_status        : in  std_logic_vector(7 downto 0)      ;                   --
        mm2s_irqthresh_status       : in  std_logic_vector(7 downto 0)      ;                   --
        mm2s_ftch_interr_set        : in  std_logic                         ;                   --
        mm2s_ftch_slverr_set        : in  std_logic                         ;                   --
        mm2s_ftch_decerr_set        : in  std_logic                         ;                   --
        mm2s_updt_interr_set        : in  std_logic                         ;                   --
        mm2s_updt_slverr_set        : in  std_logic                         ;                   --
        mm2s_updt_decerr_set        : in  std_logic                         ;                   --
        mm2s_new_curdesc_wren       : in  std_logic                         ;                   --
        mm2s_new_curdesc            : in  std_logic_vector                                      --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                   --
        mm2s_dlyirq_dsble           : out std_logic                         ; -- CR605888       --
        mm2s_irqthresh_rstdsbl      : out std_logic                         ; -- CR572013       --
        mm2s_irqthresh_wren         : out std_logic                         ;                   --
        mm2s_irqdelay_wren          : out std_logic                         ;                   --
        mm2s_tailpntr_updated       : out std_logic                         ;                   --
        mm2s_dmacr                  : out std_logic_vector                                      --
                                        (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);                   --
        mm2s_dmasr                  : out std_logic_vector                                      --
                                        (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);                   --
        mm2s_curdesc                : out std_logic_vector                                      --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                   --
        mm2s_taildesc               : out std_logic_vector                                      --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                   --
        mm2s_sa                     : out std_logic_vector                                      --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0);                   --
        mm2s_length                 : out std_logic_vector                                      --
                                        (C_SG_LENGTH_WIDTH-1 downto 0)      ;                   --
        mm2s_length_wren            : out std_logic                         ;                   --
                                                                                                --
        -- S2MM Signals                                                                         --
        tdest_in                    : in std_logic_vector (6 downto 0)      ;
        same_tdest_in               : in std_logic;
        sg_ctl                      : out std_logic_vector (7 downto 0)     ;

        s2mm_sof                    : in  std_logic                         ;
        s2mm_eof                    : in  std_logic                         ;
        s2mm_stop                   : in  std_logic                         ;                   --
        s2mm_halted_clr             : in  std_logic                         ;                   --
        s2mm_halted_set             : in  std_logic                         ;                   --
        s2mm_idle_set               : in  std_logic                         ;                   --
        s2mm_idle_clr               : in  std_logic                         ;                   --
        s2mm_dma_interr_set         : in  std_logic                         ;                   --
        s2mm_dma_slverr_set         : in  std_logic                         ;                   --
        s2mm_dma_decerr_set         : in  std_logic                         ;                   --
        s2mm_ioc_irq_set            : in  std_logic                         ;                   --
        s2mm_dly_irq_set            : in  std_logic                         ;                   --
        s2mm_irqdelay_status        : in  std_logic_vector(7 downto 0)      ;                   --
        s2mm_irqthresh_status       : in  std_logic_vector(7 downto 0)      ;                   --
        s2mm_ftch_interr_set        : in  std_logic                         ;                   --
        s2mm_ftch_slverr_set        : in  std_logic                         ;                   --
        s2mm_ftch_decerr_set        : in  std_logic                         ;                   --
        s2mm_updt_interr_set        : in  std_logic                         ;                   --
        s2mm_updt_slverr_set        : in  std_logic                         ;                   --
        s2mm_updt_decerr_set        : in  std_logic                         ;                   --
        s2mm_new_curdesc_wren       : in  std_logic                         ;                   --
        s2mm_new_curdesc            : in  std_logic_vector                                      --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                   --
        s2mm_tvalid                 : in std_logic;
        s2mm_dlyirq_dsble           : out std_logic                         ; -- CR605888       --
        s2mm_irqthresh_rstdsbl      : out std_logic                         ; -- CR572013       --
        s2mm_irqthresh_wren         : out std_logic                         ;                   --
        s2mm_irqdelay_wren          : out std_logic                         ;                   --
        s2mm_tailpntr_updated       : out std_logic                         ;                   --
        s2mm_tailpntr_updated_pkt       : out std_logic                         ;                   --
        s2mm_tvalid_latch           : out std_logic                         ;
        s2mm_tvalid_latch_del           : out std_logic                         ;
        strm_valid                  : in std_logic                          ;
        s2mm_dmacr                  : out std_logic_vector                                      --
                                        (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);                   --
        s2mm_dmasr                  : out std_logic_vector                                      --
                                        (C_S_AXI_LITE_DATA_WIDTH-1 downto 0);                   --
        s2mm_curdesc                : out std_logic_vector                                      --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                   --
        s2mm_taildesc               : out std_logic_vector                                      --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                   --
        s2mm_da                     : out std_logic_vector                                      --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0);                   --
        s2mm_length                 : out std_logic_vector                                      --
                                        (C_SG_LENGTH_WIDTH-1 downto 0)      ;                   --
        s2mm_length_wren            : out std_logic                         ;                   --
        s2mm_bytes_rcvd             : in  std_logic_vector                                      --
                                        (C_SG_LENGTH_WIDTH-1 downto 0)      ;                   --
        s2mm_bytes_rcvd_wren        : in  std_logic                         ;                   --
                                                                                                --
        soft_reset                  : out std_logic                         ;                   --
        soft_reset_clr              : in  std_logic                         ;                   --
                                                                                                --
        -- Fetch/Update error addresses                                                         --
        ftch_error_addr             : in  std_logic_vector                                      --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                   --
        updt_error_addr             : in  std_logic_vector                                      --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                   --
        mm2s_introut                : out std_logic                         ;                   --
        s2mm_introut                : out std_logic                         ;                    --
        bd_eq                       : in std_logic


    );
end axi_dma_reg_module;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_dma_reg_module is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  ATTRIBUTE async_reg                      : STRING;


-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------
constant LENGTH_PAD_WIDTH   : integer := C_S_AXI_LITE_DATA_WIDTH - C_SG_LENGTH_WIDTH;
constant LENGTH_PAD         : std_logic_vector(LENGTH_PAD_WIDTH-1 downto 0) := (others => '0');

constant ZERO_BYTES         : std_logic_vector(C_SG_LENGTH_WIDTH-1 downto 0) := (others => '0');
constant NUM_REG_PER_S2MM_INT : integer := NUM_REG_PER_CHANNEL + ((NUM_REG_PER_S2MM+1)*C_ENABLE_MULTI_CHANNEL);

-- Specifies to axi_dma_register which block belongs to S2MM channel
-- so simple dma s2mm_da register offset can be correctly assigned
-- CR603034
--constant NOT_S2MM_CHANNEL   : integer := 0;
--constant IS_S2MM_CHANNEL    : integer := 1;

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
signal axi2ip_wrce          : std_logic_vector(23+(121*C_ENABLE_MULTI_CHANNEL) - 1 downto 0)      := (others => '0');
signal axi2ip_wrdata        : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal axi2ip_rdce          : std_logic_vector(23+(121*C_ENABLE_MULTI_CHANNEL) - 1 downto 0)      := (others => '0');
signal axi2ip_rdaddr        : std_logic_vector(C_S_AXI_LITE_ADDR_WIDTH-1 downto 0) := (others => '0');
signal ip2axi_rddata        : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal mm2s_dmacr_i         : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal mm2s_dmasr_i         : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal mm2s_curdesc_lsb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal mm2s_curdesc_msb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal mm2s_taildesc_lsb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal mm2s_taildesc_msb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal mm2s_sa_i            : std_logic_vector(C_M_AXI_SG_ADDR_WIDTH-1 downto 0) := (others => '0');
signal mm2s_length_i        : std_logic_vector(C_SG_LENGTH_WIDTH-1 downto 0) := (others => '0');
signal mm2s_error_in        : std_logic := '0';
signal mm2s_error_out       : std_logic := '0';

signal s2mm_curdesc_int     : std_logic_vector                                      --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                   --
signal s2mm_taildesc_int    : std_logic_vector                                      --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                   --

signal s2mm_curdesc_int2     : std_logic_vector                                      --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                   --
signal s2mm_taildesc_int2    : std_logic_vector                                      --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                   --
signal s2mm_taildesc_int3    : std_logic_vector                                      --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                   --
signal s2mm_dmacr_i         : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_dmasr_i         : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_curdesc_lsb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_curdesc_msb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc_lsb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc_msb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal s2mm_curdesc1_lsb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_curdesc1_msb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc1_lsb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc1_msb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal s2mm_curdesc2_lsb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_curdesc2_msb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc2_lsb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc2_msb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal s2mm_curdesc3_lsb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_curdesc3_msb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc3_lsb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc3_msb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal s2mm_curdesc4_lsb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_curdesc4_msb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc4_lsb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc4_msb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal s2mm_curdesc5_lsb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_curdesc5_msb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc5_lsb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc5_msb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal s2mm_curdesc6_lsb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_curdesc6_msb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc6_lsb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc6_msb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal s2mm_curdesc7_lsb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_curdesc7_msb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc7_lsb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc7_msb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal s2mm_curdesc8_lsb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_curdesc8_msb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc8_lsb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc8_msb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal s2mm_curdesc9_lsb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_curdesc9_msb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc9_lsb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc9_msb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal s2mm_curdesc10_lsb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_curdesc10_msb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc10_lsb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc10_msb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal s2mm_curdesc11_lsb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_curdesc11_msb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc11_lsb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc11_msb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal s2mm_curdesc12_lsb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_curdesc12_msb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc12_lsb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc12_msb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal s2mm_curdesc13_lsb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_curdesc13_msb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc13_lsb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc13_msb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal s2mm_curdesc14_lsb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_curdesc14_msb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc14_lsb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc14_msb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal s2mm_curdesc15_lsb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_curdesc15_msb_i   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc15_lsb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc15_msb_i  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal s2mm_curdesc_lsb_muxed   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_curdesc_msb_muxed   : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc_lsb_muxed  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');
signal s2mm_taildesc_msb_muxed  : std_logic_vector(C_S_AXI_LITE_DATA_WIDTH-1 downto 0) := (others => '0');

signal s2mm_da_i            : std_logic_vector(C_M_AXI_SG_ADDR_WIDTH-1 downto 0) := (others => '0');
signal s2mm_length_i        : std_logic_vector(C_SG_LENGTH_WIDTH-1 downto 0) := (others => '0');
signal s2mm_error_in        : std_logic := '0';
signal s2mm_error_out       : std_logic := '0';

signal read_addr            : std_logic_vector(9 downto 0) := (others => '0');

signal mm2s_introut_i_cdc_from       : std_logic := '0';
signal mm2s_introut_d1_cdc_tig      : std_logic := '0';
signal mm2s_introut_to      : std_logic := '0';
signal s2mm_introut_i_cdc_from       : std_logic := '0';
signal s2mm_introut_d1_cdc_tig      : std_logic := '0';
signal s2mm_introut_to      : std_logic := '0';

signal mm2s_sgctl           : std_logic_vector (7 downto 0);
signal s2mm_sgctl           : std_logic_vector (7 downto 0);

signal or_sgctl            : std_logic_vector (7 downto 0);

signal open_window, wren          : std_logic;
signal s2mm_tailpntr_updated_int  : std_logic;
signal s2mm_tailpntr_updated_int1  : std_logic;
signal s2mm_tailpntr_updated_int2  : std_logic;
signal s2mm_tailpntr_updated_int3  : std_logic;


signal tvalid_int : std_logic;
signal tvalid_int1 : std_logic;
signal tvalid_int2 : std_logic;
signal new_tdest : std_logic;
signal tvalid_latch : std_logic;

signal tdest_changed : std_logic;
signal tdest_fix  : std_logic_vector (4 downto 0);

signal same_tdest_int1 : std_logic;
signal same_tdest_int2 : std_logic;
signal same_tdest_int3 : std_logic;
signal same_tdest_arrived : std_logic;

signal s2mm_msb_sa : std_logic_vector (31 downto 0);
signal mm2s_msb_sa : std_logic_vector (31 downto 0);


signal strm_valid_int2, strm_valid_int_cdc_to : std_logic;
signal strm_valid_int3 : std_logic;

signal s2mm_tailpntr_updated_int_pkt : std_logic;
signal s2mm_tailpntr_updated_int2_pkt : std_logic;

  --ATTRIBUTE async_reg OF mm2s_introut_d1_cdc_tig  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF s2mm_introut_d1_cdc_tig  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF mm2s_introut_to  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF s2mm_introut_to  : SIGNAL IS "true";
-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin

or_sgctl        <= mm2s_sgctl or s2mm_sgctl;
sg_ctl          <= mm2s_sgctl or s2mm_sgctl;
mm2s_dmacr      <= mm2s_dmacr_i;        -- MM2S DMA Control Register
mm2s_dmasr      <= mm2s_dmasr_i;        -- MM2S DMA Status Register
mm2s_sa         <= mm2s_sa_i;           -- MM2S Source Address (Simple Only)
mm2s_length     <= mm2s_length_i;       -- MM2S Length (Simple Only)

s2mm_dmacr      <= s2mm_dmacr_i;        -- S2MM DMA Control Register
s2mm_dmasr      <= s2mm_dmasr_i;        -- S2MM DMA Status Register
s2mm_da         <= s2mm_da_i;           -- S2MM Destination Address (Simple Only)
s2mm_length     <= s2mm_length_i;       -- S2MM Length (Simple Only)

-- Soft reset set in mm2s DMACR or s2MM DMACR
soft_reset      <= mm2s_dmacr_i(DMACR_RESET_BIT)
                or s2mm_dmacr_i(DMACR_RESET_BIT);

-- CR572013 - added to match legacy SDMA operation
mm2s_irqthresh_rstdsbl <= not mm2s_dmacr_i(DMACR_DLY_IRQEN_BIT);
s2mm_irqthresh_rstdsbl <= not s2mm_dmacr_i(DMACR_DLY_IRQEN_BIT);




--GEN_S2MM_TDEST : if (C_NUM_S2MM_CHANNELS > 1) generate
GEN_S2MM_TDEST : if (C_ENABLE_MULTI_CHANNEL = 1 and C_INCLUDE_S2MM = 1) generate
begin
   PROC_WREN : process (m_axi_sg_aclk)
          begin
               if (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
                   if (m_axi_sg_aresetn = '0') then
                     s2mm_taildesc_int3 <= (others => '0');  
                     s2mm_tailpntr_updated_int <= '0';
                     s2mm_tailpntr_updated_int2 <= '0';
                     s2mm_tailpntr_updated <= '0';
                     s2mm_tailpntr_updated_int_pkt <= '0';
                     s2mm_tailpntr_updated_int2_pkt <= '0';
                     s2mm_tailpntr_updated_pkt <= '0';
                   else -- (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
                 --    s2mm_tailpntr_updated_int <= new_tdest or same_tdest_arrived;
                 --    s2mm_tailpntr_updated_int2 <= s2mm_tailpntr_updated_int;
                 --    s2mm_tailpntr_updated <= s2mm_tailpntr_updated_int2;
 
                 -- Commenting this code as it is causing SG to start early
                     s2mm_tailpntr_updated_int <= (s2mm_tailpntr_updated_int1 and strm_valid_int3);
                     s2mm_tailpntr_updated_int2 <= s2mm_tailpntr_updated_int;
                     s2mm_tailpntr_updated <= s2mm_tailpntr_updated_int2;

                     s2mm_tailpntr_updated_int_pkt <= new_tdest or (same_tdest_arrived and (not bd_eq));
                     s2mm_tailpntr_updated_int2_pkt <= s2mm_tailpntr_updated_int_pkt;
                     s2mm_tailpntr_updated_pkt <= s2mm_tailpntr_updated_int2_pkt;

                   end if;
                end if;
          end process PROC_WREN; 

          -- this is always '1' as MCH needs to have all desc reg programmed before hand
 
--s2mm_tailpntr_updated_int3_i <= s2mm_tailpntr_updated_int2_i and (not s2mm_tailpntr_updated_int_i); -- and tvalid_latch; 

           tdest_fix <= "11111";


    new_tdest <= tvalid_int1 xor tvalid_int2;  

   process (m_axi_sg_aclk)
   begin
        if (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
           if (m_axi_sg_aresetn = '0') then
              tvalid_int <= '0';
              tvalid_int1 <= '0';
              tvalid_int2 <= '0';
              tvalid_latch <= '0';
              strm_valid_int_cdc_to <= '0';
              strm_valid_int2 <= '0';
              strm_valid_int3 <= '0';
           else --if (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
              tvalid_int <= tdest_in (6); --s2mm_tvalid;
              tvalid_int1 <= tvalid_int;
              tvalid_int2 <= tvalid_int1;
              strm_valid_int_cdc_to <= strm_valid;
              strm_valid_int2 <= strm_valid_int_cdc_to;
              strm_valid_int3 <= strm_valid_int2;
              s2mm_tvalid_latch_del <= tvalid_latch;
              if (new_tdest = '1') then
                tvalid_latch <= '0';
              else
                tvalid_latch <= '1';
              end if;
           end if;
        end if;
    end process;

-- will trigger tailptrupdtd and it will then get SG out of pause
    same_tdest_arrived <= same_tdest_int2 xor same_tdest_int3;  

   process (m_axi_sg_aclk)
   begin
        if (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
           if (m_axi_sg_aresetn = '0') then
             same_tdest_int1 <= '0';
             same_tdest_int2 <= '0';
             same_tdest_int3 <= '0';
           else --if (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
             same_tdest_int1 <= same_tdest_in;
             same_tdest_int2 <= same_tdest_int1;
             same_tdest_int3 <= same_tdest_int2;
           end if;
        end if;
    end process;

--   process (m_axi_sg_aclk)
--   begin
--        if (m_axi_sg_aresetn = '0') then
--           tvalid_int <= '0';
--           tvalid_int1 <= '0';
--           tvalid_latch <= '0';
--           tdest_in_int <= (others => '0');
--        elsif (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
--           tvalid_int <= s2mm_tvalid;
--           tvalid_int1 <= tvalid_int;
--           tdest_in_int <= tdest_in;
--         --  if (tvalid_int1 = '1' and (tdest_in_int /= tdest_in)) then
--           if (tvalid_int1 = '1' and tdest_in_int = "00000" and (tdest_in_int = tdest_in)) then
--              tvalid_latch <= '1';
--           elsif (tvalid_int1 = '1' and (tdest_in_int /= tdest_in)) then
--              tvalid_latch <= '0';
--           elsif (tvalid_int1 = '1' and (tdest_in_int = tdest_in)) then
--              tvalid_latch <= '1';
--           end if;
--        end if;
--    end process;

   s2mm_tvalid_latch <= tvalid_latch;

   PROC_TDEST_IN : process (m_axi_sg_aclk)
          begin
               if (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
                  if (m_axi_sg_aresetn = '0') then
                     s2mm_curdesc_int2 <= (others => '0');
                     s2mm_taildesc_int2 <= (others => '0');  
                  else --if (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
                     s2mm_curdesc_int2    <= s2mm_curdesc_int;
                     s2mm_taildesc_int2   <= s2mm_taildesc_int; 
                  end if;
               end if;
          end process PROC_TDEST_IN; 
                     
                 s2mm_curdesc <= s2mm_curdesc_int2;
                 s2mm_taildesc <= s2mm_taildesc_int2;  

end generate GEN_S2MM_TDEST;

GEN_S2MM_NO_TDEST : if (C_ENABLE_MULTI_CHANNEL = 0) generate
--GEN_S2MM_NO_TDEST : if (C_NUM_S2MM_CHANNELS = 1 and C_ENABLE_MULTI_CHANNEL = 0) generate
begin
                   s2mm_tailpntr_updated <= s2mm_tailpntr_updated_int1; 
                   s2mm_curdesc <= s2mm_curdesc_int;
                   s2mm_taildesc <= s2mm_taildesc_int; 

                   s2mm_tvalid_latch <= '1'; 
                   s2mm_tvalid_latch_del <= '1'; 

end generate GEN_S2MM_NO_TDEST;


-- For 32 bit address map only lsb registers out
GEN_DESC_ADDR_EQL32 : if C_M_AXI_SG_ADDR_WIDTH = 32 generate
begin
    mm2s_curdesc    <= mm2s_curdesc_lsb_i;
    mm2s_taildesc   <= mm2s_taildesc_lsb_i;

    s2mm_curdesc_int    <= s2mm_curdesc_lsb_muxed; 
    s2mm_taildesc_int   <= s2mm_taildesc_lsb_muxed;
end generate GEN_DESC_ADDR_EQL32;

-- For 64 bit address map lsb and msb registers out
GEN_DESC_ADDR_EQL64 : if C_M_AXI_SG_ADDR_WIDTH = 64 generate
begin
    mm2s_curdesc    <= mm2s_curdesc_msb_i & mm2s_curdesc_lsb_i;
    mm2s_taildesc   <= mm2s_taildesc_msb_i & mm2s_taildesc_lsb_i;

    s2mm_curdesc_int    <= s2mm_curdesc_msb_muxed & s2mm_curdesc_lsb_muxed;
    s2mm_taildesc_int   <= s2mm_taildesc_msb_muxed & s2mm_taildesc_lsb_muxed;
end generate GEN_DESC_ADDR_EQL64;

-------------------------------------------------------------------------------
-- Generate AXI Lite Inteface
-------------------------------------------------------------------------------
GEN_AXI_LITE_IF : if C_INCLUDE_MM2S = 1 or C_INCLUDE_S2MM = 1 generate
begin
    AXI_LITE_IF_I : entity axi_dma_v7_1_13.axi_dma_lite_if
        generic map(
            C_NUM_CE                    => 23+(121*C_ENABLE_MULTI_CHANNEL)   ,
            C_AXI_LITE_IS_ASYNC         => C_AXI_LITE_IS_ASYNC      ,
            C_S_AXI_LITE_ADDR_WIDTH     => C_S_AXI_LITE_ADDR_WIDTH  ,
            C_S_AXI_LITE_DATA_WIDTH     => C_S_AXI_LITE_DATA_WIDTH
        )
        port map(
            ip2axi_aclk                 => m_axi_sg_aclk            ,
            ip2axi_aresetn              => m_axi_sg_hrdresetn       ,

            s_axi_lite_aclk             => s_axi_lite_aclk          ,
            s_axi_lite_aresetn          => axi_lite_reset_n         ,

            -- AXI Lite Write Address Channel
            s_axi_lite_awvalid          => s_axi_lite_awvalid       ,
            s_axi_lite_awready          => s_axi_lite_awready       ,
            s_axi_lite_awaddr           => s_axi_lite_awaddr        ,

            -- AXI Lite Write Data Channel
            s_axi_lite_wvalid           => s_axi_lite_wvalid        ,
            s_axi_lite_wready           => s_axi_lite_wready        ,
            s_axi_lite_wdata            => s_axi_lite_wdata         ,

            -- AXI Lite Write Response Channel
            s_axi_lite_bresp            => s_axi_lite_bresp         ,
            s_axi_lite_bvalid           => s_axi_lite_bvalid        ,
            s_axi_lite_bready           => s_axi_lite_bready        ,

            -- AXI Lite Read Address Channel
            s_axi_lite_arvalid          => s_axi_lite_arvalid       ,
            s_axi_lite_arready          => s_axi_lite_arready       ,
            s_axi_lite_araddr           => s_axi_lite_araddr        ,
            s_axi_lite_rvalid           => s_axi_lite_rvalid        ,
            s_axi_lite_rready           => s_axi_lite_rready        ,
            s_axi_lite_rdata            => s_axi_lite_rdata         ,
            s_axi_lite_rresp            => s_axi_lite_rresp         ,

            -- User IP Interface
            axi2ip_wrce                 => axi2ip_wrce              ,
            axi2ip_wrdata               => axi2ip_wrdata            ,

            axi2ip_rdce                 => open                     ,
            axi2ip_rdaddr               => axi2ip_rdaddr            ,
            ip2axi_rddata               => ip2axi_rddata

        );
end generate GEN_AXI_LITE_IF;

-------------------------------------------------------------------------------
-- No channels therefore do not generate an AXI Lite interface
-------------------------------------------------------------------------------
GEN_NO_AXI_LITE_IF : if C_INCLUDE_MM2S = 0 and C_INCLUDE_S2MM = 0 generate
begin
    s_axi_lite_awready          <= '0';
    s_axi_lite_wready           <= '0';
    s_axi_lite_bresp            <= (others => '0');
    s_axi_lite_bvalid           <= '0';
    s_axi_lite_arready          <= '0';
    s_axi_lite_rvalid           <= '0';
    s_axi_lite_rdata            <= (others => '0');
    s_axi_lite_rresp            <= (others => '0');

end generate GEN_NO_AXI_LITE_IF;

-------------------------------------------------------------------------------
-- Generate MM2S Registers if included
-------------------------------------------------------------------------------
GEN_MM2S_REGISTERS : if C_INCLUDE_MM2S = 1 generate
begin
    I_MM2S_DMA_REGISTER : entity axi_dma_v7_1_13.axi_dma_register
    generic map (
        C_NUM_REGISTERS             => NUM_REG_PER_CHANNEL      ,
        C_INCLUDE_SG                => C_INCLUDE_SG             ,
        C_SG_LENGTH_WIDTH           => C_SG_LENGTH_WIDTH        ,
        C_S_AXI_LITE_DATA_WIDTH     => C_S_AXI_LITE_DATA_WIDTH  ,
        C_M_AXI_SG_ADDR_WIDTH       => C_M_AXI_SG_ADDR_WIDTH    ,
        C_MICRO_DMA                 => C_MICRO_DMA              ,
        C_ENABLE_MULTI_CHANNEL      => C_ENABLE_MULTI_CHANNEL             
  --      C_NUM_S2MM_CHANNELS         => 1 --C_S2MM_NUM_CHANNELS
        --C_CHANNEL_IS_S2MM           => NOT_S2MM_CHANNEL  CR603034
    )
    port map(
        -- Secondary Clock / Reset
        m_axi_sg_aclk               => m_axi_sg_aclk            ,
        m_axi_sg_aresetn            => m_axi_sg_aresetn         ,

        -- CPU Write Control (via AXI Lite)
        axi2ip_wrdata               => axi2ip_wrdata            ,
        axi2ip_wrce                 => axi2ip_wrce              
                                        (RESERVED_2C_INDEX        
                                        downto MM2S_DMACR_INDEX),
                                         --(MM2S_LENGTH_INDEX

        -- DMASR Register bit control/status
        stop_dma                    => mm2s_stop                ,
        halted_clr                  => mm2s_halted_clr          ,
        halted_set                  => mm2s_halted_set          ,
        idle_set                    => mm2s_idle_set            ,
        idle_clr                    => mm2s_idle_clr            ,
        ioc_irq_set                 => mm2s_ioc_irq_set         ,
        dly_irq_set                 => mm2s_dly_irq_set         ,
        irqdelay_status             => mm2s_irqdelay_status     ,
        irqthresh_status            => mm2s_irqthresh_status    ,

        -- SG Error Control
        ftch_interr_set             => mm2s_ftch_interr_set     ,
        ftch_slverr_set             => mm2s_ftch_slverr_set     ,
        ftch_decerr_set             => mm2s_ftch_decerr_set     ,
        ftch_error_addr             => ftch_error_addr          ,
        updt_interr_set             => mm2s_updt_interr_set     ,
        updt_slverr_set             => mm2s_updt_slverr_set     ,
        updt_decerr_set             => mm2s_updt_decerr_set     ,
        updt_error_addr             => updt_error_addr          ,
        dma_interr_set              => mm2s_dma_interr_set      ,
        dma_slverr_set              => mm2s_dma_slverr_set      ,
        dma_decerr_set              => mm2s_dma_decerr_set      ,
        irqthresh_wren              => mm2s_irqthresh_wren      ,
        irqdelay_wren               => mm2s_irqdelay_wren       ,
        dlyirq_dsble                => mm2s_dlyirq_dsble        , -- CR605888
        error_in                    => s2mm_error_out           ,
        error_out                   => mm2s_error_out           ,
        introut                     => mm2s_introut_i_cdc_from           ,
        soft_reset_in               => s2mm_dmacr_i(DMACR_RESET_BIT),
        soft_reset_clr              => soft_reset_clr           ,


        -- CURDESC Update
        update_curdesc              => mm2s_new_curdesc_wren    ,
        new_curdesc                 => mm2s_new_curdesc         ,

        -- TAILDESC Update
        tailpntr_updated            => mm2s_tailpntr_updated    ,

        -- Channel Registers
        sg_ctl                      => mm2s_sgctl               ,
        dmacr                       => mm2s_dmacr_i             ,
        dmasr                       => mm2s_dmasr_i             ,
        curdesc_lsb                 => mm2s_curdesc_lsb_i       ,
        curdesc_msb                 => mm2s_curdesc_msb_i       ,
        taildesc_lsb                => mm2s_taildesc_lsb_i      ,
        taildesc_msb                => mm2s_taildesc_msb_i      ,

--        curdesc1_lsb                 => open       ,
--        curdesc1_msb                 => open       ,
--        taildesc1_lsb                => open      ,
--        taildesc1_msb                => open      ,

--        curdesc2_lsb                 => open       ,
--        curdesc2_msb                 => open       ,
--        taildesc2_lsb                => open      ,
--        taildesc2_msb                => open      ,
--
--        curdesc3_lsb                 => open       ,
--        curdesc3_msb                 => open       ,
--        taildesc3_lsb                => open      ,
--        taildesc3_msb                => open      ,
--
--        curdesc4_lsb                 => open       ,
--        curdesc4_msb                 => open       ,
--        taildesc4_lsb                => open      ,
--        taildesc4_msb                => open      ,
--
--        curdesc5_lsb                 => open       ,
--        curdesc5_msb                 => open       ,
--        taildesc5_lsb                => open      ,
--        taildesc5_msb                => open      ,
--
--        curdesc6_lsb                 => open       ,
--        curdesc6_msb                 => open       ,
--        taildesc6_lsb                => open      ,
--        taildesc6_msb                => open      ,
--
--        curdesc7_lsb                 => open       ,
--        curdesc7_msb                 => open       ,
--        taildesc7_lsb                => open      ,
--        taildesc7_msb                => open      ,
--
--        curdesc8_lsb                 => open       ,
--        curdesc8_msb                 => open       ,
--        taildesc8_lsb                => open      ,
--        taildesc8_msb                => open      ,
--
--        curdesc9_lsb                 => open       ,
--        curdesc9_msb                 => open       ,
--        taildesc9_lsb                => open      ,
--        taildesc9_msb                => open      ,
--
--        curdesc10_lsb                 => open       ,
--        curdesc10_msb                 => open       ,
--        taildesc10_lsb                => open      ,
--        taildesc10_msb                => open      ,
--
--        curdesc11_lsb                 => open       ,
--        curdesc11_msb                 => open       ,
--        taildesc11_lsb                => open      ,
--        taildesc11_msb                => open      ,
--
--        curdesc12_lsb                 => open       ,
--        curdesc12_msb                 => open       ,
--        taildesc12_lsb                => open      ,
--        taildesc12_msb                => open      ,
--
--        curdesc13_lsb                 => open       ,
--        curdesc13_msb                 => open       ,
--        taildesc13_lsb                => open      ,
--        taildesc13_msb                => open      ,
--
--        curdesc14_lsb                 => open       ,
--        curdesc14_msb                 => open       ,
--        taildesc14_lsb                => open      ,
--        taildesc14_msb                => open      ,
--
--
--        curdesc15_lsb                 => open       ,
--        curdesc15_msb                 => open       ,
--        taildesc15_lsb                => open      ,
--        taildesc15_msb                => open      ,
--     
--        tdest_in                    => "00000" ,

        buffer_address              => mm2s_sa_i                ,
        buffer_length               => mm2s_length_i            ,
        buffer_length_wren          => mm2s_length_wren         ,
        bytes_received              => ZERO_BYTES               ,   -- Not used on transmit
        bytes_received_wren         => '0'                          -- Not used on transmit

    );

    -- If async clocks then cross interrupt out to AXI Lite clock domain
    GEN_INTROUT_ASYNC : if C_AXI_LITE_IS_ASYNC = 1 generate
    begin

PROC_REG_INTR2LITE : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => mm2s_introut_i_cdc_from,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => s_axi_lite_aclk,
        scndry_resetn              => '0',
        scndry_out                 => mm2s_introut_to,
        scndry_vect_out            => open
    );


--        PROC_REG_INTR2LITE : process(s_axi_lite_aclk)
--            begin
--                if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
--                 --   if(axi_lite_reset_n = '0')then
--                 --       mm2s_introut_d1_cdc_tig <= '0';
--                 --       mm2s_introut_to    <= '0';
--                 --   else
--                        mm2s_introut_d1_cdc_tig <= mm2s_introut_i_cdc_from;
--                        mm2s_introut_to    <= mm2s_introut_d1_cdc_tig;
--                 --   end if;
--                end if;
--            end process PROC_REG_INTR2LITE;

             mm2s_introut <= mm2s_introut_to;
 
    end generate GEN_INTROUT_ASYNC;

    -- If sync then simply pass out
    GEN_INTROUT_SYNC : if C_AXI_LITE_IS_ASYNC = 0 generate
    begin
        mm2s_introut    <= mm2s_introut_i_cdc_from;
    end generate GEN_INTROUT_SYNC;

end generate GEN_MM2S_REGISTERS;

-------------------------------------------------------------------------------
-- Tie MM2S Register outputs to zero if excluded
-------------------------------------------------------------------------------
GEN_NO_MM2S_REGISTERS : if C_INCLUDE_MM2S = 0 generate
begin
    mm2s_dmacr_i            <= (others => '0');
    mm2s_dmasr_i            <= (others => '0');
    mm2s_curdesc_lsb_i      <= (others => '0');
    mm2s_curdesc_msb_i      <= (others => '0');
    mm2s_taildesc_lsb_i     <= (others => '0');
    mm2s_taildesc_msb_i     <= (others => '0');
    mm2s_tailpntr_updated   <= '0';
    mm2s_sa_i               <= (others => '0');
    mm2s_length_i           <= (others => '0');
    mm2s_length_wren        <= '0';

    mm2s_irqthresh_wren         <= '0';
    mm2s_irqdelay_wren          <= '0';
    mm2s_tailpntr_updated       <= '0';
    mm2s_introut                <= '0';
    mm2s_sgctl                  <= (others => '0');
    mm2s_dlyirq_dsble       <= '0';
end generate GEN_NO_MM2S_REGISTERS;



-------------------------------------------------------------------------------
-- Generate S2MM Registers if included
-------------------------------------------------------------------------------
GEN_S2MM_REGISTERS : if C_INCLUDE_S2MM = 1 generate
begin
    I_S2MM_DMA_REGISTER : entity axi_dma_v7_1_13.axi_dma_register_s2mm
    generic map (
        C_NUM_REGISTERS             => NUM_REG_PER_S2MM_INT, --NUM_REG_TOTAL, --NUM_REG_PER_CHANNEL      ,
        C_INCLUDE_SG                => C_INCLUDE_SG             ,
        C_SG_LENGTH_WIDTH           => C_SG_LENGTH_WIDTH        ,
        C_S_AXI_LITE_DATA_WIDTH     => C_S_AXI_LITE_DATA_WIDTH  ,
        C_M_AXI_SG_ADDR_WIDTH       => C_M_AXI_SG_ADDR_WIDTH    ,
        C_NUM_S2MM_CHANNELS         => C_NUM_S2MM_CHANNELS      ,
        C_MICRO_DMA                 => C_MICRO_DMA              ,
        C_ENABLE_MULTI_CHANNEL      => C_ENABLE_MULTI_CHANNEL             
        --C_CHANNEL_IS_S2MM           => IS_S2MM_CHANNEL    CR603034
    )
    port map(
        -- Secondary Clock / Reset
        m_axi_sg_aclk               => m_axi_sg_aclk            ,
        m_axi_sg_aresetn            => m_axi_sg_aresetn         ,

        -- CPU Write Control (via AXI Lite)
        axi2ip_wrdata               => axi2ip_wrdata            ,
        axi2ip_wrce                 => axi2ip_wrce              
                                        ((23+(121*C_ENABLE_MULTI_CHANNEL)-1)
                                         downto RESERVED_2C_INDEX) ,  
--                                        downto S2MM_DMACR_INDEX),
--S2MM_LENGTH_INDEX
        -- DMASR Register bit control/status
        stop_dma                    => s2mm_stop                ,
        halted_clr                  => s2mm_halted_clr          ,
        halted_set                  => s2mm_halted_set          ,
        idle_set                    => s2mm_idle_set            ,
        idle_clr                    => s2mm_idle_clr            ,
        ioc_irq_set                 => s2mm_ioc_irq_set         ,
        dly_irq_set                 => s2mm_dly_irq_set         ,
        irqdelay_status             => s2mm_irqdelay_status     ,
        irqthresh_status            => s2mm_irqthresh_status    ,

        -- SG Error Control
        dma_interr_set              => s2mm_dma_interr_set      ,
        dma_slverr_set              => s2mm_dma_slverr_set      ,
        dma_decerr_set              => s2mm_dma_decerr_set      ,
        ftch_interr_set             => s2mm_ftch_interr_set     ,
        ftch_slverr_set             => s2mm_ftch_slverr_set     ,
        ftch_decerr_set             => s2mm_ftch_decerr_set     ,
        ftch_error_addr             => ftch_error_addr          ,
        updt_interr_set             => s2mm_updt_interr_set     ,
        updt_slverr_set             => s2mm_updt_slverr_set     ,
        updt_decerr_set             => s2mm_updt_decerr_set     ,
        updt_error_addr             => updt_error_addr          ,
        irqthresh_wren              => s2mm_irqthresh_wren      ,
        irqdelay_wren               => s2mm_irqdelay_wren       ,
        dlyirq_dsble                => s2mm_dlyirq_dsble        , -- CR605888
        error_in                    => mm2s_error_out           ,
        error_out                   => s2mm_error_out           ,
        introut                     => s2mm_introut_i_cdc_from           ,
        soft_reset_in               => mm2s_dmacr_i(DMACR_RESET_BIT),
        soft_reset_clr              => soft_reset_clr           ,

        -- CURDESC Update
        update_curdesc              => s2mm_new_curdesc_wren    ,
        new_curdesc                 => s2mm_new_curdesc         ,

        -- TAILDESC Update
        tailpntr_updated            => s2mm_tailpntr_updated_int1     ,

        -- Channel Registers
        sg_ctl                      => s2mm_sgctl               ,
        dmacr                       => s2mm_dmacr_i             ,
        dmasr                       => s2mm_dmasr_i             ,
        curdesc_lsb                 => s2mm_curdesc_lsb_i       ,
        curdesc_msb                 => s2mm_curdesc_msb_i       ,
        taildesc_lsb                => s2mm_taildesc_lsb_i      ,
        taildesc_msb                => s2mm_taildesc_msb_i      ,

        curdesc1_lsb                 => s2mm_curdesc1_lsb_i       ,
        curdesc1_msb                 => s2mm_curdesc1_msb_i       ,
        taildesc1_lsb                => s2mm_taildesc1_lsb_i      ,
        taildesc1_msb                => s2mm_taildesc1_msb_i      ,

        curdesc2_lsb                 => s2mm_curdesc2_lsb_i       ,
        curdesc2_msb                 => s2mm_curdesc2_msb_i       ,
        taildesc2_lsb                => s2mm_taildesc2_lsb_i      ,
        taildesc2_msb                => s2mm_taildesc2_msb_i      ,

        curdesc3_lsb                 => s2mm_curdesc3_lsb_i       ,
        curdesc3_msb                 => s2mm_curdesc3_msb_i       ,
        taildesc3_lsb                => s2mm_taildesc3_lsb_i      ,
        taildesc3_msb                => s2mm_taildesc3_msb_i      ,

        curdesc4_lsb                 => s2mm_curdesc4_lsb_i       ,
        curdesc4_msb                 => s2mm_curdesc4_msb_i       ,
        taildesc4_lsb                => s2mm_taildesc4_lsb_i      ,
        taildesc4_msb                => s2mm_taildesc4_msb_i      ,

        curdesc5_lsb                 => s2mm_curdesc5_lsb_i       ,
        curdesc5_msb                 => s2mm_curdesc5_msb_i       ,
        taildesc5_lsb                => s2mm_taildesc5_lsb_i      ,
        taildesc5_msb                => s2mm_taildesc5_msb_i      ,

        curdesc6_lsb                 => s2mm_curdesc6_lsb_i       ,
        curdesc6_msb                 => s2mm_curdesc6_msb_i       ,
        taildesc6_lsb                => s2mm_taildesc6_lsb_i      ,
        taildesc6_msb                => s2mm_taildesc6_msb_i      ,

        curdesc7_lsb                 => s2mm_curdesc7_lsb_i       ,
        curdesc7_msb                 => s2mm_curdesc7_msb_i       ,
        taildesc7_lsb                => s2mm_taildesc7_lsb_i      ,
        taildesc7_msb                => s2mm_taildesc7_msb_i      ,

        curdesc8_lsb                 => s2mm_curdesc8_lsb_i       ,
        curdesc8_msb                 => s2mm_curdesc8_msb_i       ,
        taildesc8_lsb                => s2mm_taildesc8_lsb_i      ,
        taildesc8_msb                => s2mm_taildesc8_msb_i      ,

        curdesc9_lsb                 => s2mm_curdesc9_lsb_i       ,
        curdesc9_msb                 => s2mm_curdesc9_msb_i       ,
        taildesc9_lsb                => s2mm_taildesc9_lsb_i      ,
        taildesc9_msb                => s2mm_taildesc9_msb_i      ,

        curdesc10_lsb                => s2mm_curdesc10_lsb_i       ,
        curdesc10_msb                => s2mm_curdesc10_msb_i       ,
        taildesc10_lsb               => s2mm_taildesc10_lsb_i      ,
        taildesc10_msb               => s2mm_taildesc10_msb_i      ,

        curdesc11_lsb                => s2mm_curdesc11_lsb_i       ,
        curdesc11_msb                => s2mm_curdesc11_msb_i       ,
        taildesc11_lsb               => s2mm_taildesc11_lsb_i      ,
        taildesc11_msb               => s2mm_taildesc11_msb_i      ,

        curdesc12_lsb                => s2mm_curdesc12_lsb_i       ,
        curdesc12_msb                => s2mm_curdesc12_msb_i       ,
        taildesc12_lsb               => s2mm_taildesc12_lsb_i      ,
        taildesc12_msb               => s2mm_taildesc12_msb_i      ,

        curdesc13_lsb                => s2mm_curdesc13_lsb_i       ,
        curdesc13_msb                => s2mm_curdesc13_msb_i       ,
        taildesc13_lsb               => s2mm_taildesc13_lsb_i      ,
        taildesc13_msb               => s2mm_taildesc13_msb_i      ,

        curdesc14_lsb                => s2mm_curdesc14_lsb_i       ,
        curdesc14_msb                => s2mm_curdesc14_msb_i       ,
        taildesc14_lsb               => s2mm_taildesc14_lsb_i      ,
        taildesc14_msb               => s2mm_taildesc14_msb_i      ,

        curdesc15_lsb                => s2mm_curdesc15_lsb_i       ,
        curdesc15_msb                => s2mm_curdesc15_msb_i       ,
        taildesc15_lsb               => s2mm_taildesc15_lsb_i      ,
        taildesc15_msb               => s2mm_taildesc15_msb_i      ,

        tdest_in                    => tdest_in (5 downto 0)                ,

        buffer_address              => s2mm_da_i                ,
        buffer_length               => s2mm_length_i            ,
        buffer_length_wren          => s2mm_length_wren         ,
        bytes_received              => s2mm_bytes_rcvd          ,
        bytes_received_wren         => s2mm_bytes_rcvd_wren
    );

    GEN_DESC_MUX_SINGLE_CH : if C_NUM_S2MM_CHANNELS = 1 generate
    begin
        
                     s2mm_curdesc_lsb_muxed <= s2mm_curdesc_lsb_i;
                     s2mm_curdesc_msb_muxed <= s2mm_curdesc_msb_i;
                     s2mm_taildesc_lsb_muxed <= s2mm_taildesc_lsb_i;
                     s2mm_taildesc_msb_muxed <= s2mm_taildesc_msb_i;  
    end generate GEN_DESC_MUX_SINGLE_CH;
   
    GEN_DESC_MUX : if C_NUM_S2MM_CHANNELS > 1 generate
    begin
        
      PROC_DESC_SEL : process (tdest_in, s2mm_curdesc_lsb_i,s2mm_curdesc_msb_i, s2mm_taildesc_lsb_i, s2mm_taildesc_msb_i,
                               s2mm_curdesc1_lsb_i,s2mm_curdesc1_msb_i, s2mm_taildesc1_lsb_i, s2mm_taildesc1_msb_i,
                               s2mm_curdesc2_lsb_i,s2mm_curdesc2_msb_i, s2mm_taildesc2_lsb_i, s2mm_taildesc2_msb_i,
                               s2mm_curdesc3_lsb_i,s2mm_curdesc3_msb_i, s2mm_taildesc3_lsb_i, s2mm_taildesc3_msb_i,
                               s2mm_curdesc4_lsb_i,s2mm_curdesc4_msb_i, s2mm_taildesc4_lsb_i, s2mm_taildesc4_msb_i,
                               s2mm_curdesc5_lsb_i,s2mm_curdesc5_msb_i, s2mm_taildesc5_lsb_i, s2mm_taildesc5_msb_i,
                               s2mm_curdesc6_lsb_i,s2mm_curdesc6_msb_i, s2mm_taildesc6_lsb_i, s2mm_taildesc6_msb_i,
                               s2mm_curdesc7_lsb_i,s2mm_curdesc7_msb_i, s2mm_taildesc7_lsb_i, s2mm_taildesc7_msb_i,
                               s2mm_curdesc8_lsb_i,s2mm_curdesc8_msb_i, s2mm_taildesc8_lsb_i, s2mm_taildesc8_msb_i,
                               s2mm_curdesc9_lsb_i,s2mm_curdesc9_msb_i, s2mm_taildesc9_lsb_i, s2mm_taildesc9_msb_i,
                               s2mm_curdesc10_lsb_i,s2mm_curdesc10_msb_i, s2mm_taildesc10_lsb_i, s2mm_taildesc10_msb_i,
                               s2mm_curdesc11_lsb_i,s2mm_curdesc11_msb_i, s2mm_taildesc11_lsb_i, s2mm_taildesc11_msb_i,
                               s2mm_curdesc12_lsb_i,s2mm_curdesc12_msb_i, s2mm_taildesc12_lsb_i, s2mm_taildesc12_msb_i,
                               s2mm_curdesc13_lsb_i,s2mm_curdesc13_msb_i, s2mm_taildesc13_lsb_i, s2mm_taildesc13_msb_i,
                               s2mm_curdesc14_lsb_i,s2mm_curdesc14_msb_i, s2mm_taildesc14_lsb_i, s2mm_taildesc14_msb_i,
                               s2mm_curdesc15_lsb_i,s2mm_curdesc15_msb_i, s2mm_taildesc15_lsb_i, s2mm_taildesc15_msb_i
                               )
         begin
              case tdest_in (3 downto 0) is
                 when "0000" =>
                     s2mm_curdesc_lsb_muxed <= s2mm_curdesc_lsb_i;
                     s2mm_curdesc_msb_muxed <= s2mm_curdesc_msb_i;
                     s2mm_taildesc_lsb_muxed <= s2mm_taildesc_lsb_i;
                     s2mm_taildesc_msb_muxed <= s2mm_taildesc_msb_i;  
                 when "0001" =>
                     s2mm_curdesc_lsb_muxed <= s2mm_curdesc1_lsb_i;
                     s2mm_curdesc_msb_muxed <= s2mm_curdesc1_msb_i;
                     s2mm_taildesc_lsb_muxed <= s2mm_taildesc1_lsb_i;
                     s2mm_taildesc_msb_muxed <= s2mm_taildesc1_msb_i;  
                 when "0010" =>
                     s2mm_curdesc_lsb_muxed <= s2mm_curdesc2_lsb_i;
                     s2mm_curdesc_msb_muxed <= s2mm_curdesc2_msb_i;
                     s2mm_taildesc_lsb_muxed <= s2mm_taildesc2_lsb_i;
                     s2mm_taildesc_msb_muxed <= s2mm_taildesc2_msb_i;  
                 when "0011" =>
                     s2mm_curdesc_lsb_muxed <= s2mm_curdesc3_lsb_i;
                     s2mm_curdesc_msb_muxed <= s2mm_curdesc3_msb_i;
                     s2mm_taildesc_lsb_muxed <= s2mm_taildesc3_lsb_i;
                     s2mm_taildesc_msb_muxed <= s2mm_taildesc3_msb_i;  
                 when "0100" =>
                     s2mm_curdesc_lsb_muxed <= s2mm_curdesc4_lsb_i;
                     s2mm_curdesc_msb_muxed <= s2mm_curdesc4_msb_i;
                     s2mm_taildesc_lsb_muxed <= s2mm_taildesc4_lsb_i;
                     s2mm_taildesc_msb_muxed <= s2mm_taildesc4_msb_i;  
                 when "0101" =>
                     s2mm_curdesc_lsb_muxed <= s2mm_curdesc5_lsb_i;
                     s2mm_curdesc_msb_muxed <= s2mm_curdesc5_msb_i;
                     s2mm_taildesc_lsb_muxed <= s2mm_taildesc5_lsb_i;
                     s2mm_taildesc_msb_muxed <= s2mm_taildesc5_msb_i;  
                 when "0110" =>
                     s2mm_curdesc_lsb_muxed <= s2mm_curdesc6_lsb_i;
                     s2mm_curdesc_msb_muxed <= s2mm_curdesc6_msb_i;
                     s2mm_taildesc_lsb_muxed <= s2mm_taildesc6_lsb_i;
                     s2mm_taildesc_msb_muxed <= s2mm_taildesc6_msb_i;  
                 when "0111" =>
                     s2mm_curdesc_lsb_muxed <= s2mm_curdesc7_lsb_i;
                     s2mm_curdesc_msb_muxed <= s2mm_curdesc7_msb_i;
                     s2mm_taildesc_lsb_muxed <= s2mm_taildesc7_lsb_i;
                     s2mm_taildesc_msb_muxed <= s2mm_taildesc7_msb_i;  
                 when "1000" =>
                     s2mm_curdesc_lsb_muxed <= s2mm_curdesc8_lsb_i;
                     s2mm_curdesc_msb_muxed <= s2mm_curdesc8_msb_i;
                     s2mm_taildesc_lsb_muxed <= s2mm_taildesc8_lsb_i;
                     s2mm_taildesc_msb_muxed <= s2mm_taildesc8_msb_i;  
                 when "1001" =>
                     s2mm_curdesc_lsb_muxed <= s2mm_curdesc9_lsb_i;
                     s2mm_curdesc_msb_muxed <= s2mm_curdesc9_msb_i;
                     s2mm_taildesc_lsb_muxed <= s2mm_taildesc9_lsb_i;
                     s2mm_taildesc_msb_muxed <= s2mm_taildesc9_msb_i;  
                 when "1010" =>
                     s2mm_curdesc_lsb_muxed <= s2mm_curdesc10_lsb_i;
                     s2mm_curdesc_msb_muxed <= s2mm_curdesc10_msb_i;
                     s2mm_taildesc_lsb_muxed <= s2mm_taildesc10_lsb_i;
                     s2mm_taildesc_msb_muxed <= s2mm_taildesc10_msb_i;  
                 when "1011" =>
                     s2mm_curdesc_lsb_muxed <= s2mm_curdesc11_lsb_i;
                     s2mm_curdesc_msb_muxed <= s2mm_curdesc11_msb_i;
                     s2mm_taildesc_lsb_muxed <= s2mm_taildesc11_lsb_i;
                     s2mm_taildesc_msb_muxed <= s2mm_taildesc11_msb_i;  
                 when "1100" =>
                     s2mm_curdesc_lsb_muxed <= s2mm_curdesc12_lsb_i;
                     s2mm_curdesc_msb_muxed <= s2mm_curdesc12_msb_i;
                     s2mm_taildesc_lsb_muxed <= s2mm_taildesc12_lsb_i;
                     s2mm_taildesc_msb_muxed <= s2mm_taildesc12_msb_i;  
                 when "1101" =>
                     s2mm_curdesc_lsb_muxed <= s2mm_curdesc13_lsb_i;
                     s2mm_curdesc_msb_muxed <= s2mm_curdesc13_msb_i;
                     s2mm_taildesc_lsb_muxed <= s2mm_taildesc13_lsb_i;
                     s2mm_taildesc_msb_muxed <= s2mm_taildesc13_msb_i;  
                 when "1110" =>
                     s2mm_curdesc_lsb_muxed <= s2mm_curdesc14_lsb_i;
                     s2mm_curdesc_msb_muxed <= s2mm_curdesc14_msb_i;
                     s2mm_taildesc_lsb_muxed <= s2mm_taildesc14_lsb_i;
                     s2mm_taildesc_msb_muxed <= s2mm_taildesc14_msb_i;  
                 when "1111" =>
                     s2mm_curdesc_lsb_muxed <= s2mm_curdesc15_lsb_i;
                     s2mm_curdesc_msb_muxed <= s2mm_curdesc15_msb_i;
                     s2mm_taildesc_lsb_muxed <= s2mm_taildesc15_lsb_i;
                     s2mm_taildesc_msb_muxed <= s2mm_taildesc15_msb_i;  
                 when others =>
                     s2mm_curdesc_lsb_muxed <= (others => '0');
                     s2mm_curdesc_msb_muxed <= (others => '0');
                     s2mm_taildesc_lsb_muxed <= (others => '0');
                     s2mm_taildesc_msb_muxed <= (others => '0');
               end case;
         end process PROC_DESC_SEL;
    end generate GEN_DESC_MUX;

    -- If async clocks then cross interrupt out to AXI Lite clock domain
    GEN_INTROUT_ASYNC : if C_AXI_LITE_IS_ASYNC = 1 generate
    begin
        -- Cross interrupt out to AXI Lite clock domain

PROC_REG_INTR2LITE : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => s2mm_introut_i_cdc_from,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => s_axi_lite_aclk,
        scndry_resetn              => '0',
        scndry_out                 => s2mm_introut_to,
        scndry_vect_out            => open
    );

--        PROC_REG_INTR2LITE : process(s_axi_lite_aclk)
--            begin
--                if(s_axi_lite_aclk'EVENT and s_axi_lite_aclk = '1')then
--                    if(axi_lite_reset_n = '0')then
--                        s2mm_introut_d1_cdc_tig <= '0';
--                        s2mm_introut_to    <= '0';
--                    else
--                        s2mm_introut_d1_cdc_tig <= s2mm_introut_i_cdc_from;
--                        s2mm_introut_to    <= s2mm_introut_d1_cdc_tig;
--                    end if;
--                end if;
--            end process PROC_REG_INTR2LITE;
                   
              s2mm_introut <= s2mm_introut_to;

    end generate GEN_INTROUT_ASYNC;

    -- If sync then simply pass out
    GEN_INTROUT_SYNC : if C_AXI_LITE_IS_ASYNC = 0 generate
    begin
        s2mm_introut    <= s2mm_introut_i_cdc_from;
    end generate GEN_INTROUT_SYNC;




end generate GEN_S2MM_REGISTERS;

-------------------------------------------------------------------------------
-- Tie S2MM Register outputs to zero if excluded
-------------------------------------------------------------------------------
GEN_NO_S2MM_REGISTERS : if C_INCLUDE_S2MM = 0 generate
begin
    s2mm_dmacr_i                <= (others => '0');
    s2mm_dmasr_i                <= (others => '0');
    s2mm_curdesc_lsb_i          <= (others => '0');
    s2mm_curdesc_msb_i          <= (others => '0');
    s2mm_taildesc_lsb_i         <= (others => '0');
    s2mm_taildesc_msb_i         <= (others => '0');
    s2mm_da_i                   <= (others => '0');
    s2mm_length_i               <= (others => '0');
    s2mm_length_wren            <= '0';

    s2mm_tailpntr_updated       <= '0';
    s2mm_introut                <= '0';
    s2mm_irqthresh_wren         <= '0';
    s2mm_irqdelay_wren          <= '0';
    s2mm_tailpntr_updated       <= '0';
    s2mm_dlyirq_dsble           <= '0';
    s2mm_tailpntr_updated_int1  <= '0';
    s2mm_sgctl                  <= (others => '0'); 
end generate GEN_NO_S2MM_REGISTERS;


-------------------------------------------------------------------------------
-- AXI LITE READ MUX
-------------------------------------------------------------------------------
read_addr <= axi2ip_rdaddr(9 downto 0);

-- Generate read mux for Scatter Gather Mode
GEN_READ_MUX_FOR_SG : if C_INCLUDE_SG = 1 generate
begin

    AXI_LITE_READ_MUX : process(read_addr            ,
                                mm2s_dmacr_i         ,
                                mm2s_dmasr_i         ,
                                mm2s_curdesc_lsb_i   ,
                                mm2s_curdesc_msb_i   ,
                                mm2s_taildesc_lsb_i  ,
                                mm2s_taildesc_msb_i  ,
                                s2mm_dmacr_i         ,
                                s2mm_dmasr_i         ,
                                s2mm_curdesc_lsb_i   ,
                                s2mm_curdesc_msb_i   ,
                                s2mm_taildesc_lsb_i  ,
                                s2mm_taildesc_msb_i  ,
                                s2mm_curdesc1_lsb_i   ,
                                s2mm_curdesc1_msb_i   ,
                                s2mm_taildesc1_lsb_i  ,
                                s2mm_taildesc1_msb_i  ,
                                s2mm_curdesc2_lsb_i   ,
                                s2mm_curdesc2_msb_i   ,
                                s2mm_taildesc2_lsb_i  ,
                                s2mm_taildesc2_msb_i  ,
                                s2mm_curdesc3_lsb_i   ,
                                s2mm_curdesc3_msb_i   ,
                                s2mm_taildesc3_lsb_i  ,
                                s2mm_taildesc3_msb_i  ,
                                s2mm_curdesc4_lsb_i   ,
                                s2mm_curdesc4_msb_i   ,
                                s2mm_taildesc4_lsb_i  ,
                                s2mm_taildesc4_msb_i  ,
                                s2mm_curdesc5_lsb_i   ,
                                s2mm_curdesc5_msb_i   ,
                                s2mm_taildesc5_lsb_i  ,
                                s2mm_taildesc5_msb_i  ,
                                s2mm_curdesc6_lsb_i   ,
                                s2mm_curdesc6_msb_i   ,
                                s2mm_taildesc6_lsb_i  ,
                                s2mm_taildesc6_msb_i  ,
                                s2mm_curdesc7_lsb_i   ,
                                s2mm_curdesc7_msb_i   ,
                                s2mm_taildesc7_lsb_i  ,
                                s2mm_taildesc7_msb_i  ,
                                s2mm_curdesc8_lsb_i   ,
                                s2mm_curdesc8_msb_i   ,
                                s2mm_taildesc8_lsb_i  ,
                                s2mm_taildesc8_msb_i  ,
                                s2mm_curdesc9_lsb_i   ,
                                s2mm_curdesc9_msb_i   ,
                                s2mm_taildesc9_lsb_i  ,
                                s2mm_taildesc9_msb_i  ,
                                s2mm_curdesc10_lsb_i   ,
                                s2mm_curdesc10_msb_i   ,
                                s2mm_taildesc10_lsb_i  ,
                                s2mm_taildesc10_msb_i  ,
                                s2mm_curdesc11_lsb_i   ,
                                s2mm_curdesc11_msb_i   ,
                                s2mm_taildesc11_lsb_i  ,
                                s2mm_taildesc11_msb_i  ,
                                s2mm_curdesc12_lsb_i   ,
                                s2mm_curdesc12_msb_i   ,
                                s2mm_taildesc12_lsb_i  ,
                                s2mm_taildesc12_msb_i  ,
                                s2mm_curdesc13_lsb_i   ,
                                s2mm_curdesc13_msb_i   ,
                                s2mm_taildesc13_lsb_i  ,
                                s2mm_taildesc13_msb_i  ,
                                s2mm_curdesc14_lsb_i   ,
                                s2mm_curdesc14_msb_i   ,
                                s2mm_taildesc14_lsb_i  ,
                                s2mm_taildesc14_msb_i  ,
                                s2mm_curdesc15_lsb_i   ,
                                s2mm_curdesc15_msb_i   ,
                                s2mm_taildesc15_lsb_i  ,
                                s2mm_taildesc15_msb_i  ,
                                or_sgctl
                                )
        begin
            case read_addr is
                when MM2S_DMACR_OFFSET        =>
                    ip2axi_rddata <= mm2s_dmacr_i;
                when MM2S_DMASR_OFFSET        =>
                    ip2axi_rddata <= mm2s_dmasr_i;
                when MM2S_CURDESC_LSB_OFFSET  =>
                    ip2axi_rddata <= mm2s_curdesc_lsb_i;
                when MM2S_CURDESC_MSB_OFFSET  =>
                    ip2axi_rddata <= mm2s_curdesc_msb_i;
                when MM2S_TAILDESC_LSB_OFFSET =>
                    ip2axi_rddata <= mm2s_taildesc_lsb_i;
                when MM2S_TAILDESC_MSB_OFFSET =>
                    ip2axi_rddata <= mm2s_taildesc_msb_i;
                when SGCTL_OFFSET =>
                    ip2axi_rddata <= x"00000" & or_sgctl (7 downto 4) & "0000" & or_sgctl (3 downto 0);
                when S2MM_DMACR_OFFSET        =>
                    ip2axi_rddata <= s2mm_dmacr_i;
                when S2MM_DMASR_OFFSET       =>
                    ip2axi_rddata <= s2mm_dmasr_i;
                when S2MM_CURDESC_LSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc_lsb_i;
                when S2MM_CURDESC_MSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc_msb_i;
                when S2MM_TAILDESC_LSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc_lsb_i;
                when S2MM_TAILDESC_MSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc_msb_i;
                when S2MM_CURDESC1_LSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc1_lsb_i;
                when S2MM_CURDESC1_MSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc1_msb_i;
                when S2MM_TAILDESC1_LSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc1_lsb_i;
                when S2MM_TAILDESC1_MSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc1_msb_i;
                when S2MM_CURDESC2_LSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc2_lsb_i;
                when S2MM_CURDESC2_MSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc2_msb_i;
                when S2MM_TAILDESC2_LSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc2_lsb_i;
                when S2MM_TAILDESC2_MSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc2_msb_i;
                when S2MM_CURDESC3_LSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc3_lsb_i;
                when S2MM_CURDESC3_MSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc3_msb_i;
                when S2MM_TAILDESC3_LSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc3_lsb_i;
                when S2MM_TAILDESC3_MSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc3_msb_i;
                when S2MM_CURDESC4_LSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc4_lsb_i;
                when S2MM_CURDESC4_MSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc4_msb_i;
                when S2MM_TAILDESC4_LSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc4_lsb_i;
                when S2MM_TAILDESC4_MSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc4_msb_i;
                when S2MM_CURDESC5_LSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc5_lsb_i;
                when S2MM_CURDESC5_MSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc5_msb_i;
                when S2MM_TAILDESC5_LSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc5_lsb_i;
                when S2MM_TAILDESC5_MSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc5_msb_i;
                when S2MM_CURDESC6_LSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc6_lsb_i;
                when S2MM_CURDESC6_MSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc6_msb_i;
                when S2MM_TAILDESC6_LSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc6_lsb_i;
                when S2MM_TAILDESC6_MSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc6_msb_i;
                when S2MM_CURDESC7_LSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc7_lsb_i;
                when S2MM_CURDESC7_MSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc7_msb_i;
                when S2MM_TAILDESC7_LSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc7_lsb_i;
                when S2MM_TAILDESC7_MSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc7_msb_i;
                when S2MM_CURDESC8_LSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc8_lsb_i;
                when S2MM_CURDESC8_MSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc8_msb_i;
                when S2MM_TAILDESC8_LSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc8_lsb_i;
                when S2MM_TAILDESC8_MSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc8_msb_i;
                when S2MM_CURDESC9_LSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc9_lsb_i;
                when S2MM_CURDESC9_MSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc9_msb_i;
                when S2MM_TAILDESC9_LSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc9_lsb_i;
                when S2MM_TAILDESC9_MSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc9_msb_i;
                when S2MM_CURDESC10_LSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc10_lsb_i;
                when S2MM_CURDESC10_MSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc10_msb_i;
                when S2MM_TAILDESC10_LSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc10_lsb_i;
                when S2MM_TAILDESC10_MSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc10_msb_i;
                when S2MM_CURDESC11_LSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc11_lsb_i;
                when S2MM_CURDESC11_MSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc11_msb_i;
                when S2MM_TAILDESC11_LSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc11_lsb_i;
                when S2MM_TAILDESC11_MSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc11_msb_i;
                when S2MM_CURDESC12_LSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc12_lsb_i;
                when S2MM_CURDESC12_MSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc12_msb_i;
                when S2MM_TAILDESC12_LSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc12_lsb_i;
                when S2MM_TAILDESC12_MSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc12_msb_i;
                when S2MM_CURDESC13_LSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc13_lsb_i;
                when S2MM_CURDESC13_MSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc13_msb_i;
                when S2MM_TAILDESC13_LSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc13_lsb_i;
                when S2MM_TAILDESC13_MSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc13_msb_i;
                when S2MM_CURDESC14_LSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc14_lsb_i;
                when S2MM_CURDESC14_MSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc14_msb_i;
                when S2MM_TAILDESC14_LSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc14_lsb_i;
                when S2MM_TAILDESC14_MSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc14_msb_i;
                when S2MM_CURDESC15_LSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc15_lsb_i;
                when S2MM_CURDESC15_MSB_OFFSET  =>
                    ip2axi_rddata <= s2mm_curdesc15_msb_i;
                when S2MM_TAILDESC15_LSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc15_lsb_i;
                when S2MM_TAILDESC15_MSB_OFFSET =>
                    ip2axi_rddata <= s2mm_taildesc15_msb_i;

                -- coverage off
                when others =>
                    ip2axi_rddata <= (others => '0');

                -- coverage on
            end case;
        end process AXI_LITE_READ_MUX;
end generate GEN_READ_MUX_FOR_SG;

-- Generate read mux for Simple DMA Mode
GEN_READ_MUX_FOR_SMPL_DMA : if C_INCLUDE_SG = 0 generate
begin

ADDR32_MSB : if C_M_AXI_SG_ADDR_WIDTH = 32 generate
begin
mm2s_msb_sa <= (others => '0');
s2mm_msb_sa <= (others => '0');
end generate ADDR32_MSB;


ADDR64_MSB : if C_M_AXI_SG_ADDR_WIDTH > 32 generate
begin
mm2s_msb_sa <= mm2s_sa_i (63 downto 32);
s2mm_msb_sa <= s2mm_da_i (63 downto 32);
end generate ADDR64_MSB;


    AXI_LITE_READ_MUX : process(read_addr            ,
                                mm2s_dmacr_i         ,
                                mm2s_dmasr_i         ,
                                mm2s_sa_i (31 downto 0)            ,
                                mm2s_length_i        ,
                                s2mm_dmacr_i         ,
                                s2mm_dmasr_i         ,
                                s2mm_da_i (31 downto 0)            ,
                                s2mm_length_i        ,
                                mm2s_msb_sa          ,
                                s2mm_msb_sa
                                )
        begin
            case read_addr is
                when MM2S_DMACR_OFFSET        =>
                    ip2axi_rddata <= mm2s_dmacr_i;
                when MM2S_DMASR_OFFSET        =>
                    ip2axi_rddata <= mm2s_dmasr_i;
                when MM2S_SA_OFFSET  =>
                    ip2axi_rddata <= mm2s_sa_i (31 downto 0);
                when MM2S_SA2_OFFSET  =>
                    ip2axi_rddata <= mm2s_msb_sa; --mm2s_sa_i (63 downto 32);
                when MM2S_LENGTH_OFFSET  =>
                    ip2axi_rddata <= LENGTH_PAD & mm2s_length_i;
                when S2MM_DMACR_OFFSET        =>
                    ip2axi_rddata <= s2mm_dmacr_i;
                when S2MM_DMASR_OFFSET       =>
                    ip2axi_rddata <= s2mm_dmasr_i;
                when S2MM_DA_OFFSET  =>
                    ip2axi_rddata <= s2mm_da_i (31 downto 0);
                when S2MM_DA2_OFFSET  =>
                    ip2axi_rddata <= s2mm_msb_sa; --s2mm_da_i (63 downto 32);
                when S2MM_LENGTH_OFFSET  =>
                    ip2axi_rddata <= LENGTH_PAD & s2mm_length_i;
                when others =>
                    ip2axi_rddata <= (others => '0');
            end case;
        end process AXI_LITE_READ_MUX;
end generate GEN_READ_MUX_FOR_SMPL_DMA;



end implementation;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:        axi_dma_skid_buf.vhd
--
-- Description:
--  Implements the AXi Skid Buffer in the Option 2 (Registerd outputs) mode.
--
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

-------------------------------------------------------------------------------

entity axi_dma_skid_buf is
  generic (
    C_WDATA_WIDTH : INTEGER range 8 to 256 := 32
       --  Width of the Stream Data bus (in bits)

    );
  port (
  -- System Ports
     ACLK         : In  std_logic ;                                         --
     ARST         : In  std_logic ;                                         --
                                                                            --
   -- Shutdown control (assert for 1 clk pulse)                             --
     skid_stop    : In std_logic  ;                                         --
   -- Slave Side (Stream Data Input)                                        --
     S_VALID      : In  std_logic ;                                         --
     S_READY      : Out std_logic ;                                         --
     S_Data       : In  std_logic_vector(C_WDATA_WIDTH-1 downto 0);         --
     S_STRB       : In  std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0);     --
     S_Last       : In  std_logic ;                                         --
                                                                            --
   -- Master Side (Stream Data Output                                       --
     M_VALID      : Out std_logic ;                                         --
     M_READY      : In  std_logic ;                                         --
     M_Data       : Out std_logic_vector(C_WDATA_WIDTH-1 downto 0);         --
     M_STRB       : Out std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0);     --
     M_Last       : Out std_logic                                           --
    );

end entity axi_dma_skid_buf;


architecture implementation of axi_dma_skid_buf is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";



-- Signals decalrations -------------------------

  Signal sig_reset_reg         : std_logic := '0';
  signal sig_spcl_s_ready_set  : std_logic := '0';

  signal sig_data_skid_reg     : std_logic_vector(C_WDATA_WIDTH-1 downto 0) := (others => '0');
  signal sig_strb_skid_reg     : std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0) := (others => '0');
  signal sig_last_skid_reg     : std_logic := '0';
  signal sig_skid_reg_en       : std_logic := '0';

  signal sig_data_skid_mux_out : std_logic_vector(C_WDATA_WIDTH-1 downto 0) := (others => '0');
  signal sig_strb_skid_mux_out : std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0) := (others => '0');
  signal sig_last_skid_mux_out : std_logic := '0';
  signal sig_skid_mux_sel      : std_logic := '0';

  signal sig_data_reg_out      : std_logic_vector(C_WDATA_WIDTH-1 downto 0) := (others => '0');
  signal sig_strb_reg_out      : std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0) := (others => '0');
  signal sig_last_reg_out      : std_logic := '0';
  signal sig_data_reg_out_en   : std_logic := '0';

  signal sig_m_valid_out       : std_logic := '0';
  signal sig_m_valid_dup       : std_logic := '0';
  signal sig_m_valid_comb      : std_logic := '0';

  signal sig_s_ready_out       : std_logic := '0';
  signal sig_s_ready_dup       : std_logic := '0';
  signal sig_s_ready_comb      : std_logic := '0';

  signal sig_stop_request      : std_logic := '0';
  signal sig_stopped           : std_logic := '0';
  signal sig_sready_stop       : std_logic := '0';
  signal sig_sready_stop_reg   : std_logic := '0';
  signal sig_s_last_xfered     : std_logic := '0';

  signal sig_m_last_xfered     : std_logic := '0';
  signal sig_mvalid_stop_reg   : std_logic := '0';
  signal sig_mvalid_stop       : std_logic := '0';

  signal sig_slast_with_stop   : std_logic := '0';
  signal sig_sstrb_stop_mask   : std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0) := (others => '0');
  signal sig_sstrb_with_stop   : std_logic_vector((C_WDATA_WIDTH/8)-1 downto 0) := (others => '0');


-- Register duplication attribute assignments to control fanout
-- on handshake output signals

  Attribute KEEP : string; -- declaration
  Attribute EQUIVALENT_REGISTER_REMOVAL : string; -- declaration

  Attribute KEEP of sig_m_valid_out : signal is "TRUE"; -- definition
  Attribute KEEP of sig_m_valid_dup : signal is "TRUE"; -- definition
  Attribute KEEP of sig_s_ready_out : signal is "TRUE"; -- definition
  Attribute KEEP of sig_s_ready_dup : signal is "TRUE"; -- definition

  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_m_valid_out : signal is "no";
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_m_valid_dup : signal is "no";
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_s_ready_out : signal is "no";
  Attribute EQUIVALENT_REGISTER_REMOVAL of sig_s_ready_dup : signal is "no";




begin --(architecture implementation)

   M_VALID <= sig_m_valid_out;
   S_READY <= sig_s_ready_out;

   M_STRB  <= sig_strb_reg_out;
   M_Last  <= sig_last_reg_out;
   M_Data  <= sig_data_reg_out;

  -- Special shutdown logic version od Slast.
  -- A halt request forces a tlast through the skig buffer
  sig_slast_with_stop <= s_last or sig_stop_request;
  sig_sstrb_with_stop <= s_strb or sig_sstrb_stop_mask;
  -- Assign the special S_READY FLOP set signal
  sig_spcl_s_ready_set <= sig_reset_reg;


  -- Generate the ouput register load enable control
   sig_data_reg_out_en <= M_READY or not(sig_m_valid_dup);

  -- Generate the skid input register load enable control
   sig_skid_reg_en     <= sig_s_ready_dup;

  -- Generate the skid mux select control
   sig_skid_mux_sel    <= not(sig_s_ready_dup);


 -- Skid Mux
   sig_data_skid_mux_out <=  sig_data_skid_reg
     When (sig_skid_mux_sel = '1')
     Else  S_Data;

   sig_strb_skid_mux_out <=  sig_strb_skid_reg
     When (sig_skid_mux_sel = '1')
     Else  sig_sstrb_with_stop;

   sig_last_skid_mux_out <=  sig_last_skid_reg
     When (sig_skid_mux_sel = '1')
     Else  sig_slast_with_stop;


   -- m_valid combinational logic
   sig_m_valid_comb <= S_VALID or
                      (sig_m_valid_dup and
                      (not(sig_s_ready_dup) or
                       not(M_READY)));



   -- s_ready combinational logic
   sig_s_ready_comb <= M_READY or
                      (sig_s_ready_dup and
                      (not(sig_m_valid_dup) or
                       not(S_VALID)));



   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: REG_THE_RST
   --
   -- Process Description:
   -- Register input reset
   --
   -------------------------------------------------------------
   REG_THE_RST : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then

            sig_reset_reg <= ARST;

        end if;
      end process REG_THE_RST;




   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: S_READY_FLOP
   --
   -- Process Description:
   -- Registers S_READY handshake signals per Skid Buffer
   -- Option 2 scheme
   --
   -------------------------------------------------------------
   S_READY_FLOP : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST            = '1' or
               sig_sready_stop = '1') then  -- Special stop condition

             sig_s_ready_out  <= '0';
             sig_s_ready_dup  <= '0';

           Elsif (sig_spcl_s_ready_set = '1') Then

             sig_s_ready_out  <= '1';
             sig_s_ready_dup  <= '1';

           else

             sig_s_ready_out  <= sig_s_ready_comb;
             sig_s_ready_dup  <= sig_s_ready_comb;

           end if;
        end if;
      end process S_READY_FLOP;






   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: M_VALID_FLOP
   --
   -- Process Description:
   -- Registers M_VALID handshake signals per Skid Buffer
   -- Option 2 scheme
   --
   -------------------------------------------------------------
   M_VALID_FLOP : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST                 = '1' or
               sig_spcl_s_ready_set = '1' or    -- Fix from AXI DMA
               sig_mvalid_stop      = '1') then -- Special stop condition
             sig_m_valid_out  <= '0';
             sig_m_valid_dup  <= '0';

           else

             sig_m_valid_out  <= sig_m_valid_comb;
             sig_m_valid_dup  <= sig_m_valid_comb;

           end if;
        end if;
      end process M_VALID_FLOP;






   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: SKID_REG
   --
   -- Process Description:
   -- This process implements the output registers for the
   -- Skid Buffer Data signals
   --
   -------------------------------------------------------------
   SKID_REG : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST = '1') then

             sig_data_skid_reg <= (others => '0');
             sig_strb_skid_reg <= (others => '0');
             sig_last_skid_reg <= '0';

           elsif (sig_skid_reg_en = '1') then

             sig_data_skid_reg <= S_Data;
             sig_strb_skid_reg <= sig_sstrb_with_stop;
             sig_last_skid_reg <= sig_slast_with_stop;

           else
             null;  -- hold current state
           end if;
        end if;
      end process SKID_REG;





   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: OUTPUT_REG
   --
   -- Process Description:
   -- This process implements the output registers for the
   -- Skid Buffer Data signals
   --
   -------------------------------------------------------------
   OUTPUT_REG : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST                = '1' or
               sig_mvalid_stop_reg = '1') then

             --sig_data_reg_out <= (others => '0');         -- CR585409
             sig_strb_reg_out <= (others => '0');
             sig_last_reg_out <= '0';

           elsif (sig_data_reg_out_en = '1') then

             --sig_data_reg_out <= sig_data_skid_mux_out;   -- CR585409
             sig_strb_reg_out <= sig_strb_skid_mux_out;
             sig_last_reg_out <= sig_last_skid_mux_out;

           else
             null;  -- hold current state
           end if;
        end if;
      end process OUTPUT_REG;

   -- CR585409 - To lower reset fanout and improve FPGA fmax timing
   -- resets have been removed from AXI Stream data buses
   DATA_OUTPUT_REG : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (sig_data_reg_out_en = '1') then
             sig_data_reg_out <= sig_data_skid_mux_out;
           else
             null;  -- hold current state
           end if;
        end if;
      end process DATA_OUTPUT_REG;



   -------- Special Stop Logic --------------------------------------


   sig_s_last_xfered  <=  sig_s_ready_dup and
                          s_valid         and
                          sig_slast_with_stop;


   sig_sready_stop    <=  (sig_s_last_xfered and
                          sig_stop_request) or
                          sig_sready_stop_reg;






   sig_m_last_xfered  <=  sig_m_valid_dup and
                          m_ready         and
                          sig_last_reg_out;


   sig_mvalid_stop    <=  (sig_m_last_xfered and
                          sig_stop_request)  or
                          sig_mvalid_stop_reg;




   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: IMP_STOP_REQ_FLOP
   --
   -- Process Description:
   -- This process implements the Stop request flop. It is a
   -- sample and hold register that can only be cleared by reset.
   --
   -------------------------------------------------------------
   IMP_STOP_REQ_FLOP : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST = '1') then

             sig_stop_request <= '0';
             sig_sstrb_stop_mask <= (others => '0');

           elsif (skid_stop = '1') then

             sig_stop_request <= '1';
             sig_sstrb_stop_mask <= (others => '1');

           else
             null;  -- hold current state
           end if;
        end if;
      end process IMP_STOP_REQ_FLOP;









   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: IMP_CLR_SREADY_FLOP
   --
   -- Process Description:
   -- This process implements the flag to clear the s_ready
   -- flop at a stop condition.
   --
   -------------------------------------------------------------
   IMP_CLR_SREADY_FLOP : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST = '1') then

             sig_sready_stop_reg <= '0';

           elsif (sig_s_last_xfered = '1' and
                  sig_stop_request  = '1') then

             sig_sready_stop_reg <= '1';

           else
             null;  -- hold current state
           end if;
        end if;
      end process IMP_CLR_SREADY_FLOP;





   -------------------------------------------------------------
   -- Synchronous Process with Sync Reset
   --
   -- Label: IMP_CLR_MREADY_FLOP
   --
   -- Process Description:
   -- This process implements the flag to clear the m_ready
   -- flop at a stop condition.
   --
   -------------------------------------------------------------
   IMP_CLR_MVALID_FLOP : process (ACLK)
      begin
        if (ACLK'event and ACLK = '1') then
           if (ARST = '1') then

             sig_mvalid_stop_reg <= '0';

           elsif (sig_m_last_xfered = '1' and
                  sig_stop_request  = '1') then

             sig_mvalid_stop_reg <= '1';

           else
             null;  -- hold current state
           end if;
        end if;
      end process IMP_CLR_MVALID_FLOP;



end implementation;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-- *************************************************************************
--
-------------------------------------------------------------------------------
-- Filename:        axi_dma_afifo_autord.vhd
-- Version:         initial
-- Description:
--    This file contains the logic to generate a CoreGen call to create a
-- asynchronous FIFO as part of the synthesis process of XST. This eliminates
-- the need for multiple fixed netlists for various sizes and widths of FIFOs.
--
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

library lib_cdc_v1_0_2;
library lib_fifo_v1_0_8;
use lib_fifo_v1_0_8.async_fifo_fg;

library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;


-----------------------------------------------------------------------------
-- Entity section
-----------------------------------------------------------------------------

entity axi_dma_afifo_autord is
  generic (
     C_DWIDTH        : integer := 32;
     C_DEPTH         : integer := 16;
     C_CNT_WIDTH     : Integer := 5;
     C_USE_BLKMEM    : Integer := 0 ;
     C_USE_AUTORD    : Integer := 1;
     C_PRMRY_IS_ACLK_ASYNC : integer := 1;
     C_FAMILY        : String  := "virtex7"
    );
  port (
    -- Inputs
     AFIFO_Ainit                : In  std_logic;                                 --
     AFIFO_Wr_clk               : In  std_logic;                                 --
     AFIFO_Wr_en                : In  std_logic;                                 --
     AFIFO_Din                  : In  std_logic_vector(C_DWIDTH-1 downto 0);     --
     AFIFO_Rd_clk               : In  std_logic;                                 --
     AFIFO_Rd_en                : In  std_logic;                                 --
     AFIFO_Clr_Rd_Data_Valid    : In  std_logic;                                 --
                                                                                 --
    -- Outputs                                                                   --
     AFIFO_DValid               : Out std_logic;                                 --
     AFIFO_Dout                 : Out std_logic_vector(C_DWIDTH-1 downto 0);     --
     AFIFO_Full                 : Out std_logic;                                 --
     AFIFO_Empty                : Out std_logic;                                 --
     AFIFO_Almost_full          : Out std_logic;                                 --
     AFIFO_Almost_empty         : Out std_logic;                                 --
     AFIFO_Wr_count             : Out std_logic_vector(C_CNT_WIDTH-1 downto 0);  --
     AFIFO_Rd_count             : Out std_logic_vector(C_CNT_WIDTH-1 downto 0);  --
     AFIFO_Corr_Rd_count        : Out std_logic_vector(C_CNT_WIDTH downto 0);    --
     AFIFO_Corr_Rd_count_minus1 : Out std_logic_vector(C_CNT_WIDTH downto 0);    --
     AFIFO_Rd_ack               : Out std_logic                                  --
    );
end entity axi_dma_afifo_autord;


-----------------------------------------------------------------------------
-- Architecture section
-----------------------------------------------------------------------------

architecture imp of axi_dma_afifo_autord is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of imp : architecture is "yes";

-- Constant declarations
  ATTRIBUTE async_reg                      : STRING;


-- Signal declarations
   signal write_data_lil_end       : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');
   signal read_data_lil_end        : std_logic_vector(C_DWIDTH-1 downto 0) := (others => '0');

   signal wr_count_lil_end         : std_logic_vector(C_CNT_WIDTH-1 downto 0) := (others => '0');
   signal rd_count_lil_end         : std_logic_vector(C_CNT_WIDTH-1 downto 0) := (others => '0');
   signal rd_count_int             : integer range 0 to C_DEPTH+1 := 0;
   signal rd_count_int_corr        : integer range 0 to C_DEPTH+1 := 0;
   signal rd_count_int_corr_minus1 : integer range 0 to C_DEPTH+1 := 0;


   Signal corrected_empty          : std_logic := '0';
   Signal corrected_almost_empty   : std_logic := '0';
   Signal sig_afifo_empty          : std_logic := '0';
   Signal sig_afifo_almost_empty   : std_logic := '0';


 -- backend fifo read ack sample and hold
   Signal sig_rddata_valid         : std_logic := '0';
   Signal hold_ff_q                : std_logic := '0';
   Signal ored_ack_ff_reset        : std_logic := '0';
   Signal autoread                 : std_logic := '0';
   Signal sig_wrfifo_rdack         : std_logic := '0';
   Signal fifo_read_enable         : std_logic := '0';

   Signal first_write              : std_logic := '0';
   Signal first_read_cdc_tig               : std_logic := '0';
   Signal first_read1              : std_logic := '0';
   Signal first_read2              : std_logic := '0';
   signal AFIFO_Ainit_d1_cdc_tig   : std_logic;
   signal AFIFO_Ainit_d2   : std_logic;

  --ATTRIBUTE async_reg OF AFIFO_Ainit_d1_cdc_tig  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF AFIFO_Ainit_d2 : SIGNAL IS "true";

  --ATTRIBUTE async_reg OF first_read_cdc_tig  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF first_read1 : SIGNAL IS "true";

-- Component declarations



-----------------------------------------------------------------------------
-- Begin architecture
-----------------------------------------------------------------------------
begin

 -- Bit ordering translations

    write_data_lil_end   <=  AFIFO_Din;  -- translate from Big Endian to little
                                         -- endian.
    AFIFO_Rd_ack         <= sig_wrfifo_rdack;

    AFIFO_Dout           <= read_data_lil_end;  -- translate from Little Endian to
                                                -- Big endian.

    AFIFO_Almost_empty   <= corrected_almost_empty;
GEN_EMPTY : if (C_USE_AUTORD = 1) generate
begin
    AFIFO_Empty          <= corrected_empty;
end generate GEN_EMPTY;
GEN_EMPTY1 : if (C_USE_AUTORD = 0) generate
begin
    AFIFO_Empty          <= sig_afifo_empty;
end generate GEN_EMPTY1;

    AFIFO_Wr_count       <= wr_count_lil_end;

    AFIFO_Rd_count       <= rd_count_lil_end;

    AFIFO_Corr_Rd_count  <= CONV_STD_LOGIC_VECTOR(rd_count_int_corr,
                                                  C_CNT_WIDTH+1);

    AFIFO_Corr_Rd_count_minus1 <= CONV_STD_LOGIC_VECTOR(rd_count_int_corr_minus1,
                                                        C_CNT_WIDTH+1);

    AFIFO_DValid         <= sig_rddata_valid; -- Output data valid indicator


    fifo_read_enable     <= AFIFO_Rd_en or autoread;



   -------------------------------------------------------------------------------
   -- Instantiate the CoreGen FIFO
   --
   -- NOTE:
   -- This instance refers to a wrapper file that interm will use the
   -- CoreGen FIFO Generator Async FIFO utility.
   --
   -------------------------------------------------------------------------------
    I_ASYNC_FIFOGEN_FIFO : entity lib_fifo_v1_0_8.async_fifo_fg
       generic map (
--          C_ALLOW_2N_DEPTH      =>  1,
          C_ALLOW_2N_DEPTH      =>  0,
          C_FAMILY              =>  C_FAMILY,
          C_DATA_WIDTH          =>  C_DWIDTH,
          C_ENABLE_RLOCS        =>  0,
          C_FIFO_DEPTH          =>  C_DEPTH,
          C_HAS_ALMOST_EMPTY    =>  1,
          C_HAS_ALMOST_FULL     =>  1,
          C_HAS_RD_ACK          =>  1,
          C_HAS_RD_COUNT        =>  1,
          C_EN_SAFETY_CKT       =>  1,
          C_HAS_RD_ERR          =>  0,
          C_HAS_WR_ACK          =>  0,
          C_HAS_WR_COUNT        =>  1,
          C_HAS_WR_ERR          =>  0,
          C_RD_ACK_LOW          =>  0,
          C_RD_COUNT_WIDTH      =>  C_CNT_WIDTH,
          C_RD_ERR_LOW          =>  0,
          C_USE_BLOCKMEM        =>  C_USE_BLKMEM,
          C_WR_ACK_LOW          =>  0,
          C_WR_COUNT_WIDTH      =>  C_CNT_WIDTH,
          C_WR_ERR_LOW          =>  0,
          C_SYNCHRONIZER_STAGE  =>  C_FIFO_MTBF,
          C_USE_EMBEDDED_REG    =>  0 -- 0 ;
    --      C_PRELOAD_REGS        =>  0, -- 0 ;
    --      C_PRELOAD_LATENCY     =>  1  -- 1 ;
         )
      port Map (
         Din                 =>  write_data_lil_end,
         Wr_en               =>  AFIFO_Wr_en,
         Wr_clk              =>  AFIFO_Wr_clk,
         Rd_en               =>  fifo_read_enable,
         Rd_clk              =>  AFIFO_Rd_clk,
         Ainit               =>  AFIFO_Ainit,
         Dout                =>  read_data_lil_end,
         Full                =>  AFIFO_Full,
         Empty               =>  sig_afifo_empty,
         Almost_full         =>  AFIFO_Almost_full,
         Almost_empty        =>  sig_afifo_almost_empty,
         Wr_count            =>  wr_count_lil_end,
         Rd_count            =>  rd_count_lil_end,
         Rd_ack              =>  sig_wrfifo_rdack,
         Rd_err              =>  open,              -- Not used by axi_dma
         Wr_ack              =>  open,              -- Not used by axi_dma
         Wr_err              =>  open               -- Not used by axi_dma
        );


   ----------------------------------------------------------------------------
   -- Read Ack assert & hold logic (needed because:
   --     1) The Async FIFO has to be read once to get valid
   --        data to the read data port (data is discarded).
   --     2) The Read ack from the fifo is only asserted for 1 clock.
   --     3) A signal is needed that indicates valid data is at the read
   --        port of the FIFO and has not yet been read. This signal needs
   --        to be held until the next read operation occurs or a clear
   --        signal is received.


    ored_ack_ff_reset  <=  fifo_read_enable or
                           AFIFO_Ainit_d2 or
                           AFIFO_Clr_Rd_Data_Valid;

    sig_rddata_valid   <=  hold_ff_q or
                           sig_wrfifo_rdack;




    -------------------------------------------------------------
    -- Synchronous Process with Sync Reset
    --
    -- Label: IMP_ACK_HOLD_FLOP
    --
    -- Process Description:
    --  Flop for registering the hold flag
    --
    -------------------------------------------------------------
ASYNC_CDC_SYNC : if C_PRMRY_IS_ACLK_ASYNC = 1 generate

IMP_SYNC_FLOP : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => AFIFO_Ainit,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => AFIFO_Rd_clk,
        scndry_resetn              => '0',
        scndry_out                 => AFIFO_Ainit_d2,
        scndry_vect_out            => open
    );
end generate ASYNC_CDC_SYNC;

SYNC_CDC_SYNC : if C_PRMRY_IS_ACLK_ASYNC = 0 generate

        AFIFO_Ainit_d2 <= AFIFO_Ainit;

end generate SYNC_CDC_SYNC;

--    IMP_SYNC_FLOP : process (AFIFO_Rd_clk)
--       begin
--         if (AFIFO_Rd_clk'event and AFIFO_Rd_clk = '1') then
--             AFIFO_Ainit_d1_cdc_tig <= AFIFO_Ainit;
--             AFIFO_Ainit_d2 <= AFIFO_Ainit_d1_cdc_tig;
--         end if;
--       end process IMP_SYNC_FLOP;

    IMP_ACK_HOLD_FLOP : process (AFIFO_Rd_clk)
       begin
         if (AFIFO_Rd_clk'event and AFIFO_Rd_clk = '1') then
           if (ored_ack_ff_reset = '1') then
             hold_ff_q  <= '0';
           else
             hold_ff_q  <= sig_rddata_valid;
           end if;
         end if;
       end process IMP_ACK_HOLD_FLOP;



   --  I_ACK_HOLD_FF : FDRE
   --    port map(
   --      Q  =>  hold_ff_q,
   --      C  =>  AFIFO_Rd_clk,
   --      CE =>  '1',
   --      D  =>  sig_rddata_valid,
   --      R  =>  ored_ack_ff_reset
   --    );



  -- generate auto-read enable. This keeps fresh data at the output
  -- of the FIFO whenever it is available.

GEN_AUTORD1 : if C_USE_AUTORD = 1 generate
    autoread <= '1'                     -- create a read strobe when the
      when (sig_rddata_valid = '0' and  -- output data is NOT valid
            sig_afifo_empty = '0')      -- and the FIFO is not empty
      Else '0';
end generate GEN_AUTORD1;


GEN_AUTORD2 : if C_USE_AUTORD = 0 generate
    process (AFIFO_Wr_clk)
    begin
       if (AFIFO_Wr_clk'event and AFIFO_Wr_clk = '1') then
          if (AFIFO_Ainit = '0') then
              first_write <= '0';
          elsif (AFIFO_Wr_en = '1') then
                 first_write <= '1';
          end if;
       end if; 
    end process;



IMP_SYNC_FLOP1 : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => first_write,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => AFIFO_Rd_clk,
        scndry_resetn              => '0',
        scndry_out                 => first_read1,
        scndry_vect_out            => open
    );

    process (AFIFO_Rd_clk)
    begin
           if (AFIFO_Rd_clk'event and AFIFO_Rd_clk = '1') then
              if (AFIFO_Ainit_d2 = '0') then 
                first_read2 <= '0';
              elsif (sig_afifo_empty = '0') then
                 first_read2 <= first_read1;
              end if;
           end if; 
    end process;
    autoread <= first_read1 xor first_read2;             

end generate GEN_AUTORD2;


    rd_count_int <=  CONV_INTEGER(rd_count_lil_end);


    -------------------------------------------------------------
    -- Combinational Process
    --
    -- Label: CORRECT_RD_CNT
    --
    -- Process Description:
    --  This process corrects the FIFO Read Count output for the
    -- auto read function.
    --
    -------------------------------------------------------------
    CORRECT_RD_CNT : process (sig_rddata_valid,
                              sig_afifo_empty,
                              sig_afifo_almost_empty,
                              rd_count_int)
       begin

          if (sig_rddata_valid = '0') then

             rd_count_int_corr        <= 0;
             rd_count_int_corr_minus1 <= 0;
             corrected_empty          <= '1';
             corrected_almost_empty   <= '0';

          elsif (sig_afifo_empty = '1') then         -- rddata valid and fifo empty

             rd_count_int_corr        <= 1;
             rd_count_int_corr_minus1 <= 0;
             corrected_empty          <= '0';
             corrected_almost_empty   <= '1';

          Elsif (sig_afifo_almost_empty = '1') Then  -- rddata valid and fifo almost empty

             rd_count_int_corr        <= 2;
             rd_count_int_corr_minus1 <= 1;
             corrected_empty          <= '0';
             corrected_almost_empty   <= '0';

          else                                   -- rddata valid and modify rd count from FIFO

             rd_count_int_corr        <= rd_count_int+1;
             rd_count_int_corr_minus1 <= rd_count_int;
             corrected_empty          <= '0';
             corrected_almost_empty   <= '0';

          end if;

       end process CORRECT_RD_CNT;



end imp;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;
library lib_fifo_v1_0_8;
use lib_fifo_v1_0_8.async_fifo_fg;

entity axi_dma_s2mm is
     generic (
             C_FAMILY : string := "virtex7"
             );
     port (
           clk_in      : in std_logic;
           sg_clk      : in std_logic;
           resetn      : in std_logic;
           reset_sg     : in std_logic;
           s2mm_tvalid : in std_logic;
           s2mm_tlast  : in std_logic;
           s2mm_tdest  : in std_logic_vector (4 downto 0);
           s2mm_tuser  : in std_logic_vector (3 downto 0);
           s2mm_tid  : in std_logic_vector (4 downto 0);
           s2mm_tready : in std_logic;
           desc_available : in std_logic;
       --    s2mm_eof       : in std_logic;
           s2mm_eof_det       : in std_logic_vector (1 downto 0);
           ch2_update_active : in std_logic;

           tdest_out       : out std_logic_vector (6 downto 0);  -- to select desc
           same_tdest      : out std_logic;  -- to select desc
-- to DM
           s2mm_desc_info  : out std_logic_vector (13 downto 0);
--           updt_cmpt       : out std_logic;
           s2mm_tvalid_out : out std_logic;
           s2mm_tlast_out  : out std_logic;
           s2mm_tready_out : out std_logic;
           s2mm_tdest_out  : out std_logic_vector (4 downto 0);
           strm_valid : out std_logic
          );
end entity axi_dma_s2mm;

architecture implementation of axi_dma_s2mm is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


signal first_data : std_logic;
signal first_stream : std_logic;
signal first_stream_del : std_logic;

signal last_received : std_logic;
signal first_received : std_logic;
signal first_received1 : std_logic;
signal open_window : std_logic;

signal tdest_out_int : std_logic_vector (6 downto 0);
signal fifo_wr : std_logic;

signal last_update_over_int : std_logic;
signal last_update_over_int1 : std_logic;
signal last_update_over : std_logic;

signal ch_updt_over_int : std_logic;

signal ch_updt_over_int_cdc_from : std_logic;
signal ch_updt_over_int_cdc_to : std_logic;
signal ch_updt_over_int_cdc_to1 : std_logic;
signal ch_updt_over_int_cdc_to2 : std_logic;
  -- Prevent x-propagation on clock-domain crossing register
  ATTRIBUTE async_reg                      : STRING;
  --ATTRIBUTE async_reg OF ch_updt_over_int_cdc_to  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF ch_updt_over_int_cdc_to1  : SIGNAL IS "true";


signal fifo_rd : std_logic;
signal first_read : std_logic;
signal first_rd_en : std_logic;
signal fifo_rd_int : std_logic;
signal first_read_int : std_logic;

signal fifo_empty : std_logic;
signal fifo_full : std_logic;
signal s2mm_desc_info_int : std_logic_vector (13 downto 0);

signal updt_cmpt : std_logic;
signal tdest_capture : std_logic_vector (4 downto 0);
signal noread : std_logic;

signal same_tdest_b2b : std_logic;
signal fifo_reset : std_logic;

begin


     process (sg_clk)
       begin
            if (sg_clk'event and sg_clk = '1') then
              if (reset_sg = '0') then
                ch_updt_over_int_cdc_from <= '0';
              else --if (sg_clk'event and sg_clk = '1') then
                ch_updt_over_int_cdc_from <= ch2_update_active;
              end if;
            end if;
     end process;

     process (clk_in)
       begin
            if (clk_in'event and clk_in = '1') then
              if (resetn = '0') then
               ch_updt_over_int_cdc_to <= '0';
               ch_updt_over_int_cdc_to1 <= '0';
               ch_updt_over_int_cdc_to2 <= '0';
              else --if (clk_in'event and clk_in = '1') then
               ch_updt_over_int_cdc_to <= ch_updt_over_int_cdc_from;
               ch_updt_over_int_cdc_to1 <= ch_updt_over_int_cdc_to;
               ch_updt_over_int_cdc_to2 <= ch_updt_over_int_cdc_to1;
              end if;
            end if;
     end process;

         updt_cmpt <= (not ch_updt_over_int_cdc_to1) and ch_updt_over_int_cdc_to2;

  --   process (sg_clk)
  --     begin
  --          if (resetn = '0') then
  --             ch_updt_over_int <= '0';
  --          elsif (sg_clk'event and sg_clk = '1') then
  --             ch_updt_over_int <= ch2_update_active;
  --          end if;
  --   end process;
  
     --    updt_cmpt <= (not ch2_update_active) and ch_updt_over_int;

     process (sg_clk)
       begin
          if (sg_clk'event and sg_clk = '1') then
            if (reset_sg = '0') then
               last_update_over_int <= '0';
               last_update_over_int1 <= '0';
                   noread <= '0'; 
         --   else --if (sg_clk'event and sg_clk = '1') then
                   last_update_over_int1 <= last_update_over_int; 
             elsif (s2mm_eof_det(1) = '1' and noread = '0') then
                   last_update_over_int <= '1';
                   noread <= '1'; 
             elsif (s2mm_eof_det(0) = '1') then
                   noread <= '0';
                   last_update_over_int <= '0'; 
             elsif (fifo_empty = '0') then -- (updt_cmpt = '1') then
                   last_update_over_int <= '0'; 
             else
                   last_update_over_int <= last_update_over_int; 
             end if; 
            end if;
      --    end if;
     end process;
 
        last_update_over <= (not last_update_over_int) and last_update_over_int1;

     process (sg_clk)
       begin
          if (sg_clk'event and sg_clk = '1') then
            if (reset_sg = '0') then
               fifo_rd_int <= '0';
               first_read <= '0';
       --     else --if (sg_clk'event and sg_clk = '1') then
            elsif (last_update_over_int = '1' and fifo_rd_int = '0') then
                   fifo_rd_int <= '1';
            else
                   fifo_rd_int <= '0';
            end if;
          end if; 
     end process;

     process (sg_clk)
       begin
          if (sg_clk'event and sg_clk = '1') then
            if (reset_sg = '0') then
               first_read_int <= '0';
            else --if (sg_clk'event and sg_clk = '1') then
                   first_read_int <= first_read;
            end if;
          end if;
     end process;

         first_rd_en <= first_read and (not first_read_int);
         fifo_rd <= last_update_over_int; --(fifo_rd_int or first_rd_en);


--     process (clk_in)
--       begin
--            if (resetn = '0') then
--                first_data <= '0';
--                first_stream_del <= '0';
--           elsif (clk_in'event and clk_in = '1') then
--               if (s2mm_tvalid = '1' and first_data = '0' and s2mm_tready = '1') then   -- no tlast
--                  first_data <= '1';         -- just after the system comes out of reset
--               end if;
--               first_stream_del <= first_stream;
--           end if;
--    end process;

            first_stream <= (s2mm_tvalid and (not first_data));  -- pulse when first stream comes after reset

     process (clk_in)
       begin
          if (clk_in'event and clk_in = '1') then
            if (resetn = '0') then
                first_received1 <= '0';
                first_stream_del <= '0';
            else --if (clk_in'event and clk_in = '1') then
                first_received1 <= first_received; --'0';
                first_stream_del <= first_stream;
            end if;
          end if;
     end process;

     process (clk_in)
       begin
          if (clk_in'event and clk_in = '1') then
            if (resetn = '0') then
                last_received <= '0';
                first_received <= '0';
                tdest_capture <= (others => '0'); 
                first_data <= '0';
                strm_valid <= '0';
        --    else --if (clk_in'event and clk_in = '1') then
            elsif (s2mm_tvalid = '1' and first_data = '0' and s2mm_tready = '1') then   -- first stream afetr reset
                   s2mm_desc_info_int <= s2mm_tuser & s2mm_tid & s2mm_tdest;
                   tdest_capture <= s2mm_tdest;  -- latching tdest on first beat
                   first_data <= '1';         -- just after the system comes out of reset
                   strm_valid <= '1';
            elsif (s2mm_tlast = '1' and s2mm_tvalid = '1' and s2mm_tready = '1') then  -- catch for last beat 
                   last_received <= '1';         
                   first_received <= '0';
                   s2mm_desc_info_int <= s2mm_desc_info_int;
                   strm_valid <= '0';
            elsif (last_received = '1' and s2mm_tvalid = '1' and s2mm_tready = '1') then -- catch for following first beat
                   last_received <= '0';
                   first_received <= '1';
                   tdest_capture <= s2mm_tdest;  -- latching tdest on first beat
                   s2mm_desc_info_int <= s2mm_tuser & s2mm_tid & s2mm_tdest;
                   strm_valid <= '1';
            else
                   s2mm_desc_info_int <= s2mm_desc_info_int;
                   last_received <= last_received;
                   if (updt_cmpt = '1') then
                      first_received <= '0';
                   else
                      first_received <= first_received;  -- hold the first received until update comes for previous tlast
                   end if; 
            end if;
          end if;
     end process;

           fifo_wr <= first_stream_del or (first_received and not (first_received1)); -- writing the tdest,tuser,tid into FIFO





     process (clk_in)
       begin
          if (clk_in'event and clk_in = '1') then
            if (resetn = '0') then
                   tdest_out_int <= "0100000";
                   same_tdest_b2b <= '0';
        --    else --if (clk_in'event and clk_in = '1') then
            elsif (first_received = '1' or first_stream = '1') then
                   if (first_stream = '1') then    -- when first stream is received, capture the tdest
                      tdest_out_int (6) <= not tdest_out_int (6);  -- signifies a new stream has come
                      tdest_out_int (5 downto 0) <=  '0' & s2mm_tdest;
                      same_tdest_b2b <= '0';
                 --  elsif (updt_cmpt = '1' or (first_received = '1' and first_received1 = '0')) then    -- when subsequent streams are received, pass the latched value of tdest
                 --  elsif (first_received = '1' and first_received1 = '0') then    -- when subsequent streams are received, pass the latched value of tdest
                 -- Following change made to allow b2b same channel pkt
                   elsif ((first_received = '1' and first_received1 = '0') and (tdest_out_int (4 downto 0) /= tdest_capture)) then    -- when subsequent streams are received, pass the latched value of tdest
                      tdest_out_int (6) <= not tdest_out_int (6);
                      tdest_out_int (5 downto 0) <=  '0' & tdest_capture; --s2mm_tdest;
                   elsif (first_received = '1' and first_received1 = '0') then
                      same_tdest_b2b <= not (same_tdest_b2b);
                   end if;
             else
                   tdest_out_int <= tdest_out_int;
             end if;
          end if;
     end process;

           tdest_out <= tdest_out_int;
           same_tdest <= same_tdest_b2b;

     process (clk_in)
       begin
          if (clk_in'event and clk_in = '1') then
            if (resetn = '0') then
                open_window <= '0';
          --  else --if (clk_in'event and clk_in = '1') then
            elsif (desc_available = '1') then
                   open_window <= '1';
            elsif (s2mm_tlast = '1') then
                   open_window <= '0';
            else
                   open_window <= open_window;
            end if;
          end if;
     end process;


     process (clk_in)
       begin
          if (clk_in'event and clk_in = '1') then
            if (resetn = '0') then
                s2mm_tvalid_out <= '0';
                s2mm_tready_out <= '0';
                s2mm_tlast_out  <= '0';
                s2mm_tdest_out      <= "00000";
          --  else --if (clk_in'event and clk_in = '1') then
            elsif (open_window = '1') then
                s2mm_tvalid_out <= s2mm_tvalid;
                s2mm_tready_out <= s2mm_tready;
                s2mm_tlast_out  <= s2mm_tlast;
                s2mm_tdest_out  <= s2mm_tdest;
            else
                s2mm_tready_out <= '0';
                s2mm_tvalid_out <= '0';
                s2mm_tlast_out  <= '0';
                s2mm_tdest_out      <= "00000";
            end if;
          end if;
     end process;

    fifo_reset <= not (resetn);

--    s2mm_desc_info_int <= s2mm_tuser & s2mm_tid & s2mm_tdest;
    -- Following FIFO is used to store the Tuser, Tid and xCache info
    I_ASYNC_FIFOGEN_FIFO : entity lib_fifo_v1_0_8.async_fifo_fg
       generic map (
--          C_ALLOW_2N_DEPTH      =>  1,
          C_ALLOW_2N_DEPTH      =>  0,
          C_FAMILY              =>  C_FAMILY,
          C_DATA_WIDTH          =>  14,
          C_ENABLE_RLOCS        =>  0,
          C_FIFO_DEPTH          =>  31,
          C_HAS_ALMOST_EMPTY    =>  1,
          C_EN_SAFETY_CKT       =>  1,
          C_HAS_ALMOST_FULL     =>  1,
          C_HAS_RD_ACK          =>  1,
          C_HAS_RD_COUNT        =>  1,
          C_HAS_RD_ERR          =>  0,
          C_HAS_WR_ACK          =>  0,
          C_HAS_WR_COUNT        =>  1,
          C_HAS_WR_ERR          =>  0,
          C_RD_ACK_LOW          =>  0,
          C_RD_COUNT_WIDTH      =>  5,
          C_RD_ERR_LOW          =>  0,
          C_USE_BLOCKMEM        =>  0,
          C_WR_ACK_LOW          =>  0,
          C_WR_COUNT_WIDTH      =>  5,
          C_WR_ERR_LOW          =>  0,
          C_SYNCHRONIZER_STAGE  =>  C_FIFO_MTBF,
          C_USE_EMBEDDED_REG    =>  0 -- 0 ;
    --      C_PRELOAD_REGS        =>  0, -- 0 ;
    --      C_PRELOAD_LATENCY     =>  1  -- 1 ;
         )
      port Map (
         Din                 =>  s2mm_desc_info_int,
         Wr_en               =>  fifo_wr,
         Wr_clk              =>  clk_in,
         Rd_en               =>  fifo_rd,
         Rd_clk              =>  sg_clk,
         Ainit               =>  fifo_reset,
         Dout                =>  s2mm_desc_info,
         Full                =>  fifo_Full,
         Empty               =>  fifo_empty,
         Almost_full         =>  open,
         Almost_empty        =>  open,
         Wr_count            =>  open,
         Rd_count            =>  open,
         Rd_ack              =>  open,
         Rd_err              =>  open,              -- Not used by axi_dma
         Wr_ack              =>  open,              -- Not used by axi_dma
         Wr_err              =>  open               -- Not used by axi_dma
        );



end implementation;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:    axi_dma_sofeof_gen.vhd
-- Description: This entity manages
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library lib_cdc_v1_0_2;
library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;

-------------------------------------------------------------------------------
entity  axi_dma_sofeof_gen is
    generic (
        C_PRMRY_IS_ACLK_ASYNC           : integer range 0 to 1         := 0
            -- Primary MM2S/S2MM sync/async mode
            -- 0 = synchronous mode     - all clocks are synchronous
            -- 1 = asynchronous mode    - Primary data path channels (MM2S and S2MM)
            --                            run asynchronous to AXI Lite, DMA Control,
            --                            and SG.
    );
    port (
        -----------------------------------------------------------------------
        -- AXI Scatter Gather Interface
        -----------------------------------------------------------------------
        axi_prmry_aclk              : in  std_logic                         ;           --
        p_reset_n                   : in  std_logic                         ;           --
                                                                                        --
        m_axi_sg_aclk               : in  std_logic                         ;           --
        m_axi_sg_aresetn            : in  std_logic                         ;           --
                                                                                        --
        axis_tready                 : in  std_logic                         ;           --
        axis_tvalid                 : in  std_logic                         ;           --
        axis_tlast                  : in  std_logic                         ;           --
                                                                                        --
        packet_sof                  : out std_logic                         ;           --
        packet_eof                  : out std_logic                                     --
                                                                                        --

    );

end axi_dma_sofeof_gen;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_dma_sofeof_gen is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


  ATTRIBUTE async_reg                      : STRING;

-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------

-- No Constants Declared

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
signal p_ready          : std_logic := '0';
signal p_valid          : std_logic := '0';
signal p_valid_d1       : std_logic := '0';
signal p_valid_re       : std_logic := '0';
signal p_last           : std_logic := '0';
signal p_last_d1        : std_logic := '0';
signal p_last_re        : std_logic := '0';


signal s_ready          : std_logic := '0';
signal s_valid          : std_logic := '0';
signal s_valid_d1       : std_logic := '0';
signal s_valid_re       : std_logic := '0';
signal s_last           : std_logic := '0';
signal s_last_d1        : std_logic := '0';
signal s_last_re        : std_logic := '0';



signal s_sof_d1_cdc_tig         : std_logic := '0';
signal s_sof_d2         : std_logic := '0';
  --ATTRIBUTE async_reg OF s_sof_d1_cdc_tig  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF s_sof_d2  : SIGNAL IS "true";

signal s_sof_d3         : std_logic := '0';
signal s_sof_re         : std_logic := '0';

signal s_sof            : std_logic := '0';
signal p_sof            : std_logic := '0';

signal s_eof_d1_cdc_tig         : std_logic := '0';
signal s_eof_d2         : std_logic := '0';

  --ATTRIBUTE async_reg OF s_eof_d1_cdc_tig  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF s_eof_d2 : SIGNAL IS "true";
signal s_eof_d3         : std_logic := '0';
signal s_eof_re         : std_logic := '0';

signal p_eof            : std_logic := '0';
signal p_eof_d1_cdc_tig         : std_logic := '0';
signal p_eof_d2         : std_logic := '0';
  --ATTRIBUTE async_reg OF p_eof_d1_cdc_tig  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF p_eof_d2 : SIGNAL IS "true";

signal p_eof_d3         : std_logic := '0';
signal p_eof_clr        : std_logic := '0';

signal s_sof_generated  : std_logic := '0';
signal sof_generated_fe : std_logic := '0';
signal s_eof_re_latch   : std_logic := '0';

-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin

-- pass internal version out
packet_sof <= s_sof_re;
packet_eof <= s_eof_re;


-- Generate for when primary clock is asynchronous
GEN_FOR_ASYNC : if C_PRMRY_IS_ACLK_ASYNC = 1 generate
begin

    ---------------------------------------------------------------------------
    -- Generate Packet SOF
    ---------------------------------------------------------------------------

    -- Register stream control in to isolate wrt clock
    -- for timing closure
    REG_STRM_IN : process(axi_prmry_aclk)
        begin
            if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
                if(p_reset_n = '0')then
                    p_valid <= '0';
                    p_last  <= '0';
                    p_ready <= '0';
                else
                    p_valid <= axis_tvalid;
                    p_last  <= axis_tlast ;
                    p_ready <= axis_tready;
                end if;
            end if;
        end process REG_STRM_IN;


    -- Generate rising edge pulse on valid to use for
    -- smaple and hold register
    REG_FOR_RE : process(axi_prmry_aclk)
        begin
            if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
                if(p_reset_n = '0')then
                    p_valid_d1  <= '0';
                    p_last_d1   <= '0';
                    p_last_re   <= '0';
                else
                    p_valid_d1  <= p_valid and p_ready;
                    p_last_d1   <= p_last and p_valid and p_ready;

                    -- register to aligne with setting of p_sof
                    p_last_re   <= p_ready and p_valid and p_last and not p_last_d1;
                end if;
            end if;
        end process REG_FOR_RE;

    p_valid_re  <= p_ready and p_valid and not p_valid_d1;


    -- Sample and hold valid re to create sof
    SOF_SMPL_N_HOLD : process(axi_prmry_aclk)
        begin
            if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
                -- clear at end of packet
                if(p_reset_n = '0' or p_eof_clr = '1')then
                    p_sof <= '0';

                -- assert at beginning of packet hold to allow
                -- clock crossing to slower secondary clk
                elsif(p_valid_re = '1')then
                    p_sof <= '1';

                end if;
            end if;
        end process SOF_SMPL_N_HOLD;

    -- Register p_sof into secondary clock domain to
    -- generate packet_sof and also to clear sample and held p_sof
SOF_REG2SCNDRY : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => p_sof,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => m_axi_sg_aclk,
        scndry_resetn              => '0',
        scndry_out                 => s_sof_d2,
        scndry_vect_out            => open
    );


    SOF_REG2SCNDRY1 : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
--                    s_sof_d1_cdc_tig <= '0';
--                    s_sof_d2 <= '0';
                    s_sof_d3 <= '0';
                else
--                    s_sof_d1_cdc_tig <= p_sof;
--                    s_sof_d2 <= s_sof_d1_cdc_tig;
                    s_sof_d3 <= s_sof_d2;
                end if;
            end if;
        end process SOF_REG2SCNDRY1;

    s_sof_re <= s_sof_d2 and not s_sof_d3;

    ---------------------------------------------------------------------------
    -- Generate Packet EOF
    ---------------------------------------------------------------------------
    -- Sample and hold valid re to create sof
    EOF_SMPL_N_HOLD : process(axi_prmry_aclk)
        begin
            if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
                if(p_reset_n = '0' or p_eof_clr = '1')then
                    p_eof <= '0';

                -- if p_last but p_sof not set then it means between pkt
                -- gap was too small to catch new sof.  therefor do not
                -- generate eof
                elsif(p_last_re = '1' and p_sof = '0')then
                    p_eof <= '0';

                elsif(p_last_re = '1')then
                    p_eof <= '1';
                end if;
            end if;
        end process EOF_SMPL_N_HOLD;

    -- Register p_sof into secondary clock domain to
    -- generate packet_sof and also to clear sample and held p_sof
    -- CDC register has to be a pure flop

EOF_REG2SCNDRY : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => p_eof,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => m_axi_sg_aclk,
        scndry_resetn              => '0',
        scndry_out                 => s_eof_d2,
        scndry_vect_out            => open
    );

    EOF_REG2SCNDRY1 : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
--                    s_eof_d1_cdc_tig <= '0';
--                    s_eof_d2 <= '0';
                    s_eof_d3 <= '0';                      -- CR605883
                else
--                    s_eof_d1_cdc_tig <= p_eof;
--                    s_eof_d2 <= s_eof_d1_cdc_tig;
                    s_eof_d3 <= s_eof_d2;                 -- CR605883
                end if;
            end if;
        end process EOF_REG2SCNDRY1;

                    s_eof_re <= s_eof_d2 and not s_eof_d3;

    EOF_latch : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    s_eof_re_latch <= '0';
                elsif (s_eof_re = '1') then 
                    s_eof_re_latch <= not s_eof_re_latch;
                end if;
            end if;
        end process EOF_latch;


    -- Register s_sof_re back into primary clock domain to use
    -- as clear of p_sof.

EOF_REG2PRMRY : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => s_eof_re_latch,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => axi_prmry_aclk,
        scndry_resetn              => '0',
        scndry_out                 => p_eof_d2,
        scndry_vect_out            => open
    );


    EOF_REG2PRMRY1 : process(axi_prmry_aclk)
        begin
            if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
                if(p_reset_n = '0')then
               --     p_eof_d1_cdc_tig <= '0';
               --     p_eof_d2 <= '0';
                    p_eof_d3 <= '0';
                else
                --    p_eof_d1_cdc_tig <= s_eof_re_latch;
                --    p_eof_d2 <= p_eof_d1_cdc_tig;
                    p_eof_d3 <= p_eof_d2;

                end if;
            end if;
        end process EOF_REG2PRMRY1;


--    p_eof_clr <= p_eof_d2 and not p_eof_d3;-- CR565366
    -- drive eof clear for minimum of 2 scndry clocks
    -- to guarentee secondary capture.  this allows
    -- new valid assertions to not be missed in
    -- creating next sof.
    p_eof_clr <= p_eof_d2 xor p_eof_d3;



end generate GEN_FOR_ASYNC;

-- Generate for when primary clock is synchronous
GEN_FOR_SYNC : if C_PRMRY_IS_ACLK_ASYNC = 0 generate
begin

    ---------------------------------------------------------------------------
    -- Generate Packet EOF and SOF
    ---------------------------------------------------------------------------

    -- Register stream control in to isolate wrt clock
    -- for timing closure
    REG_STRM_IN : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    s_valid <= '0';
                    s_last  <= '0';
                    s_ready <= '0';
                else
                    s_valid <= axis_tvalid;
                    s_last  <= axis_tlast ;
                    s_ready <= axis_tready;
                end if;
            end if;
        end process REG_STRM_IN;

    -- Generate rising edge pulse on valid to use for
    -- smaple and hold register
    REG_FOR_RE : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    s_valid_d1  <= '0';
                    s_last_d1   <= '0';
                else
                    s_valid_d1  <= s_valid and s_ready;
                    s_last_d1   <= s_last and s_valid and s_ready;
                end if;
            end if;
        end process REG_FOR_RE;

-- CR565366 investigating delay interurpt issue discovered
-- this coding issue.
--    s_valid_re  <= s_ready and s_valid and not s_last_d1;
    s_valid_re  <= s_ready and s_valid and not s_valid_d1;

    s_last_re   <= s_ready and s_valid and s_last and not s_last_d1;

    -- Sample and hold valid re to create sof
    SOF_SMPL_N_HOLD : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(p_reset_n = '0' or s_eof_re = '1')then
                    s_sof_generated <= '0';
                -- new
                elsif((s_valid_re = '1')
                   or (sof_generated_fe = '1' and s_ready = '1' and s_valid = '1'))then
                    s_sof_generated <= '1';
                end if;
            end if;
        end process SOF_SMPL_N_HOLD;


    -- Register p_sof into secondary clock domain to
    -- generate packet_sof and also to clear sample and held p_sof
    SOF_REG2SCNDRY : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    s_sof_d1_cdc_tig <= '0';
                else
                    s_sof_d1_cdc_tig <= s_sof_generated;
                end if;
            end if;
        end process SOF_REG2SCNDRY;

    -- generate falling edge pulse on end of packet for use if
    -- need to generate an immediate sof.
    sof_generated_fe <= not s_sof_generated and s_sof_d1_cdc_tig;

    -- generate SOF on rising edge of valid if not already in a packet OR...
    s_sof_re <= '1' when (s_valid_re = '1' and s_sof_generated = '0')
                      or (sof_generated_fe = '1'        -- If end of previous packet
                               and s_ready = '1'        -- and ready asserted
                               and s_valid = '1')       -- and valid asserted
           else '0';

    -- generate eof on rising edge of valid last assertion OR...
    s_eof_re <= '1' when (s_last_re = '1')
                      or (sof_generated_fe = '1'        -- If end of previous packet
                               and s_ready = '1'        -- and ready asserted
                               and s_valid = '1'        -- and valid asserted
                               and s_last = '1')        -- and last asserted
           else '0';

end generate GEN_FOR_SYNC;



end implementation;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:          axi_dma_smple_sm.vhd
-- Description: This entity contains the DMA Controller State Machine for
--              Simple DMA mode.
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;

library lib_pkg_v1_0_2;
use lib_pkg_v1_0_2.lib_pkg.clog2;


-------------------------------------------------------------------------------
entity  axi_dma_smple_sm is
    generic (
        C_M_AXI_ADDR_WIDTH          : integer range 32 to 64    := 32;
            -- Master AXI Memory Map Address Width for MM2S Read Port

        C_SG_LENGTH_WIDTH           : integer range 8 to 23     := 14;
            -- Width of Buffer Length, Transferred Bytes, and BTT fields

        C_MICRO_DMA                 : integer range 0 to 1      := 0
    );
    port (
        m_axi_sg_aclk               : in  std_logic                         ;                      --
        m_axi_sg_aresetn            : in  std_logic                         ;                      --
                                                                                                   --
        -- Channel 1 Control and Status                                                            --
        run_stop                    : in  std_logic                         ;                      --
        keyhole                     : in  std_logic                         ;
        stop                        : in  std_logic                         ;                      --
        cmnd_idle                   : out std_logic                         ;                      --
        sts_idle                    : out std_logic                         ;                      --
                                                                                                   --
        -- DataMover Status                                                                        --
        sts_received                : in  std_logic                         ;                      --
        sts_received_clr            : out std_logic                         ;                      --
                                                                                                   --
        -- DataMover Command                                                                       --
        cmnd_wr                     : out std_logic                         ;                      --
        cmnd_data                   : out std_logic_vector                                         --
                                           ((C_M_AXI_ADDR_WIDTH-32+2*32+CMD_BASE_WIDTH+46)-1 downto 0);       --
        cmnd_pending                : in std_logic                          ;                      --
                                                                                                   --
        -- Trasnfer Qualifiers                                                                     --
        xfer_length_wren            : in  std_logic                         ;                      --
        xfer_address                : in  std_logic_vector                                         --
                                        (C_M_AXI_ADDR_WIDTH-1 downto 0)     ;                      --
        xfer_length                 : in  std_logic_vector                                         --
                                        (C_SG_LENGTH_WIDTH - 1 downto 0)                           --
    );

end axi_dma_smple_sm;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_dma_smple_sm is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------
-- DataMover Command Destination Stream Offset
constant CMD_DSA            : std_logic_vector(5 downto 0)  := (others => '0');
-- DataMover Cmnd Reserved Bits
constant CMD_RSVD           : std_logic_vector(
                                DATAMOVER_CMD_RSVMSB_BOFST + C_M_AXI_ADDR_WIDTH downto
                                DATAMOVER_CMD_RSVLSB_BOFST + C_M_AXI_ADDR_WIDTH)
                                := (others => '0');


-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
type SMPL_STATE_TYPE        is (
                                IDLE,
                                EXECUTE_XFER,
                                WAIT_STATUS
                                );

signal smpl_cs                  : SMPL_STATE_TYPE;
signal smpl_ns                  : SMPL_STATE_TYPE;



-- State Machine Signals
signal write_cmnd_cmb           : std_logic := '0';
signal cmnd_wr_i                : std_logic := '0';
signal sts_received_clr_cmb     : std_logic := '0';

signal cmnds_queued             : std_logic := '0';
signal cmd_dumb                 : std_logic_vector (31 downto 0) := (others => '0');
signal zeros                    : std_logic_vector (45 downto 0) := (others => '0');

signal burst_type               : std_logic;
-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin
-- Pass command write control out
cmnd_wr    <= cmnd_wr_i;

burst_type <= '1' and (not keyhole);
-- 0 means fixed burst
-- 1 means increment burst

-------------------------------------------------------------------------------
-- MM2S Transfer State Machine
-------------------------------------------------------------------------------
MM2S_MACHINE : process(smpl_cs,
                       run_stop,
                       xfer_length_wren,
                       sts_received,
                       cmnd_pending,
                       cmnds_queued,
                       stop
                       )
    begin

        -- Default signal assignment
        write_cmnd_cmb          <= '0';
        sts_received_clr_cmb    <= '0';
        cmnd_idle               <= '0';
        smpl_ns                 <= smpl_cs;

        case smpl_cs is

            -------------------------------------------------------------------
            when IDLE =>
                -- Running, no errors, and new length written,then execute
                -- transfer
                if( run_stop = '1' and xfer_length_wren = '1' and stop = '0'
                and cmnds_queued = '0') then
                    smpl_ns <= EXECUTE_XFER;
                else
                    cmnd_idle <= '1';
                end if;


            -------------------------------------------------------------------
            when EXECUTE_XFER =>
                -- error detected
                if(stop = '1')then
                    smpl_ns     <= IDLE;
                -- Write another command if there is not one already pending
                elsif(cmnd_pending = '0')then
                    write_cmnd_cmb  <= '1';
                    smpl_ns         <= WAIT_STATUS;
                else
                    smpl_ns         <= EXECUTE_XFER;
                end if;

            -------------------------------------------------------------------
            when WAIT_STATUS =>
                -- wait until desc update complete or error occurs
                if(sts_received = '1' or stop = '1')then
                    sts_received_clr_cmb <= '1';
                    smpl_ns              <= IDLE;
                else
                    smpl_ns <= WAIT_STATUS;
                end if;

            -------------------------------------------------------------------
--          coverage off
            when others =>
                smpl_ns <= IDLE;
--          coverage on

        end case;
    end process MM2S_MACHINE;

-------------------------------------------------------------------------------
-- register state machine states
-------------------------------------------------------------------------------
REGISTER_STATE : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                smpl_cs     <= IDLE;
            else
                smpl_cs     <= smpl_ns;
            end if;
        end if;
    end process REGISTER_STATE;

-- Register state machine signals
REGISTER_STATE_SIGS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn ='0')then
                sts_received_clr    <= '0';
            else
                sts_received_clr    <= sts_received_clr_cmb;
            end if;
        end if;
    end process REGISTER_STATE_SIGS;

-------------------------------------------------------------------------------
-- Build DataMover command
-------------------------------------------------------------------------------
-- If Bytes To Transfer (BTT) width less than 23, need to add pad
GEN_CMD_BTT_LESS_23 : if C_SG_LENGTH_WIDTH < 23 generate
constant PAD_VALUE : std_logic_vector(22 - C_SG_LENGTH_WIDTH downto 0)
                        := (others => '0');
begin
    -- When command by sm, drive command to mm2s_cmdsts_if
    GEN_DATAMOVER_CMND : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    cmnd_wr_i       <= '0';
                    cmnd_data       <= (others => '0');

                -- SM issued a command write
                elsif(write_cmnd_cmb = '1')then
                    cmnd_wr_i       <= '1';
                    cmnd_data       <=  zeros
                                        & cmd_dumb 
                                        & CMD_RSVD
                                        -- Command Tag
                                        & '0'               -- Tag Not Used in Simple Mode
                                        & '0'               -- Tag Not Used in Simple Mode
                                        & '0'               -- Tag Not Used in Simple Mode
                                        & '0'               -- Tag Not Used in Simple Mode
                                        -- Command
                                        & xfer_address      -- Command Address
                                        & '1'               -- Command SOF
                                        & '1'               -- Command EOF
                                        & CMD_DSA           -- Stream Offset
                                        & burst_type  -- Key Hole Operation'1'               -- Not Used
                                        & PAD_VALUE
                                        & xfer_length;

                else
                    cmnd_wr_i       <= '0';

                end if;
            end if;
        end process GEN_DATAMOVER_CMND;

end generate GEN_CMD_BTT_LESS_23;

-- If Bytes To Transfer (BTT) width equal 23, no required pad
GEN_CMD_BTT_EQL_23 : if C_SG_LENGTH_WIDTH = 23 generate
begin
    -- When command by sm, drive command to mm2s_cmdsts_if
    GEN_DATAMOVER_CMND : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    cmnd_wr_i       <= '0';
                    cmnd_data       <= (others => '0');

                -- SM issued a command write
                elsif(write_cmnd_cmb = '1')then
                    cmnd_wr_i       <= '1';
                    cmnd_data       <=  zeros 
                                        & cmd_dumb
                                        & CMD_RSVD
                                        -- Command Tag
                                        & '0'                   -- Tag Not Used in Simple Mode
                                        & '0'                   -- Tag Not Used in Simple Mode
                                        & '0'                   -- Tag Not Used in Simple Mode
                                        & '0'                   -- Tag Not Used in Simple Mode
                                        -- Command
                                        & xfer_address          -- Command Address
                                        & '1'                   -- Command SOF
                                        & '1'                   -- Command EOF
                                        & CMD_DSA               -- Stream Offset
                                        & burst_type  -- key Hole Operation '1'                   -- Not Used
                                        & xfer_length;

                else
                    cmnd_wr_i       <= '0';

                end if;
            end if;
        end process GEN_DATAMOVER_CMND;

end generate GEN_CMD_BTT_EQL_23;


-------------------------------------------------------------------------------
-- Flag indicating command being processed by Datamover
-------------------------------------------------------------------------------

-- count number of queued commands to keep track of what datamover is still
-- working on
CMD2STS_COUNTER : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0' or stop = '1')then
                cmnds_queued <= '0';
            elsif(cmnd_wr_i = '1')then
                cmnds_queued <= '1';
            elsif(sts_received = '1')then
                cmnds_queued <= '0';
            end if;
        end if;
    end process CMD2STS_COUNTER;

-- Indicate status is idle when no cmnd/sts queued
sts_idle <= '1' when  cmnds_queued = '0'
       else '0';

end implementation;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:          axi_dma_mm2s_sg_if.vhd
-- Description: This entity is the MM2S Scatter Gather Interface for Descriptor
--              Fetches and Updates.
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;


library lib_cdc_v1_0_2;
library lib_srl_fifo_v1_0_2;
use lib_srl_fifo_v1_0_2.srl_fifo_f;

-------------------------------------------------------------------------------
entity  axi_dma_mm2s_sg_if is
    generic (
        C_PRMRY_IS_ACLK_ASYNC        : integer range 0 to 1          := 0       ;
            -- Primary MM2S/S2MM sync/async mode
            -- 0 = synchronous mode     - all clocks are synchronous
            -- 1 = asynchronous mode    - Any one of the 4 clock inputs is not
            --                            synchronous to the other

        -----------------------------------------------------------------------
        -- Scatter Gather Parameters
        -----------------------------------------------------------------------
        C_SG_INCLUDE_STSCNTRL_STRM      : integer range 0 to 1          := 1    ;
            -- Include or Exclude AXI Status and AXI Control Streams
            -- 0 = Exclude Status and Control Streams
            -- 1 = Include Status and Control Streams

        C_SG_INCLUDE_DESC_QUEUE         : integer range 0 to 1          := 0    ;
            -- Include or Exclude Scatter Gather Descriptor Queuing
            -- 0 = Exclude SG Descriptor Queuing
            -- 1 = Include SG Descriptor Queuing

        C_M_AXIS_SG_TDATA_WIDTH          : integer range 32 to 32        := 32  ;
            -- AXI Master Stream in for descriptor fetch


        C_S_AXIS_UPDPTR_TDATA_WIDTH      : integer range 32 to 32        := 32   ;
            -- 32 Update Status Bits

        C_S_AXIS_UPDSTS_TDATA_WIDTH      : integer range 33 to 33        := 33   ;
            -- 1 IOC bit + 32 Update Status Bits

        C_M_AXI_SG_ADDR_WIDTH           : integer range 32 to 64        := 32   ;
            -- Master AXI Memory Map Data Width for Scatter Gather R/W Port

        C_M_AXI_MM2S_ADDR_WIDTH         : integer range 32 to 64        := 32   ;
            -- Master AXI Memory Map Address Width for MM2S Read Port

        C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH  : integer range 32 to 32        := 32  ;
            -- Master AXI Control Stream Data Width
        C_ENABLE_MULTI_CHANNEL                  : integer range 0 to 1          := 0 ;
        C_MICRO_DMA                     : integer range 0 to 1 := 0;
        C_FAMILY                        : string                        := "virtex5"
            -- Target FPGA Device Family

    );
    port (

        m_axi_sg_aclk               : in  std_logic                         ;           --
        m_axi_sg_aresetn            : in  std_logic                         ;           --
                                                                                        --
        -- SG MM2S Descriptor Fetch AXI Stream In                                       --
        m_axis_mm2s_ftch_tdata      : in  std_logic_vector                              --
                                        (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0);           --
        m_axis_mm2s_ftch_tvalid     : in  std_logic                         ;           --
        m_axis_mm2s_ftch_tready     : out std_logic                         ;           --
        m_axis_mm2s_ftch_tlast      : in  std_logic                         ;           --

        m_axis_mm2s_ftch_tdata_new      : in  std_logic_vector                              --
                                        (96+31*0+(0+2)*(C_M_AXI_SG_ADDR_WIDTH-32) downto 0);           --
        m_axis_mm2s_ftch_tdata_mcdma_new      : in  std_logic_vector                              --
                                        (63 downto 0);           --
        m_axis_mm2s_ftch_tvalid_new     : in  std_logic                         ;           --
        m_axis_ftch1_desc_available     : in std_logic;
                                                                                        --
                                                                                        --
        -- SG MM2S Descriptor Update AXI Stream Out                                     --
        s_axis_mm2s_updtptr_tdata   : out std_logic_vector                              --
                                     (C_M_AXI_SG_ADDR_WIDTH-1 downto 0);          --
        s_axis_mm2s_updtptr_tvalid  : out std_logic                         ;           --
        s_axis_mm2s_updtptr_tready  : in  std_logic                         ;           --
        s_axis_mm2s_updtptr_tlast   : out std_logic                         ;           --
                                                                                        --
        s_axis_mm2s_updtsts_tdata   : out std_logic_vector                              --
                                     (C_S_AXIS_UPDSTS_TDATA_WIDTH-1 downto 0);          --
        s_axis_mm2s_updtsts_tvalid  : out std_logic                         ;           --
        s_axis_mm2s_updtsts_tready  : in  std_logic                         ;           --
        s_axis_mm2s_updtsts_tlast   : out std_logic                         ;           --

                                                                                        --
                                                                                        --
        -- MM2S Descriptor Fetch Request (from mm2s_sm)                                 --
        desc_available              : out std_logic                         ;           --
        desc_fetch_req              : in  std_logic                         ;           --
        desc_fetch_done             : out std_logic                         ;           --
        updt_pending                : out std_logic                         ;
        packet_in_progress          : out std_logic                         ;           --
                                                                                        --
        -- MM2S Descriptor Update Request (from mm2s_sm)                                --
        desc_update_done            : out std_logic                         ;           --
                                                                                        --
        mm2s_sts_received_clr       : out std_logic                         ;           --
        mm2s_sts_received           : in  std_logic                         ;           --
        mm2s_ftch_stale_desc        : in  std_logic                         ;           --
        mm2s_done                   : in  std_logic                         ;           --
        mm2s_interr                 : in  std_logic                         ;           --
        mm2s_slverr                 : in  std_logic                         ;           --
        mm2s_decerr                 : in  std_logic                         ;           --
        mm2s_tag                    : in  std_logic_vector(3 downto 0)      ;           --
        mm2s_halt                   : in  std_logic                         ;           --
                                                                                        --
        -- Control Stream Output                                                        --
        cntrlstrm_fifo_wren         : out std_logic                         ;           --
        cntrlstrm_fifo_din          : out std_logic_vector                              --
                                        (C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH downto 0);     --
        cntrlstrm_fifo_full         : in  std_logic                         ;           --
                                                                                        --
                                                                                        --
        -- MM2S Descriptor Field Output                                                 --
        mm2s_new_curdesc            : out std_logic_vector                              --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;           --
        mm2s_new_curdesc_wren       : out std_logic                         ;           --
                                                                                        --
        mm2s_desc_baddress          : out std_logic_vector                              --
                                        (C_M_AXI_MM2S_ADDR_WIDTH-1 downto 0);           --
        mm2s_desc_blength           : out std_logic_vector                              --
                                        (BUFFER_LENGTH_WIDTH-1 downto 0)    ;           --
        mm2s_desc_blength_v         : out std_logic_vector                              --
                                        (BUFFER_LENGTH_WIDTH-1 downto 0)    ;           --
        mm2s_desc_blength_s         : out std_logic_vector                              --
                                        (BUFFER_LENGTH_WIDTH-1 downto 0)    ;           --
        mm2s_desc_eof               : out std_logic                         ;           --
        mm2s_desc_sof               : out std_logic                         ;           --
        mm2s_desc_cmplt             : out std_logic                         ;           --
        mm2s_desc_info              : out std_logic_vector                              --
                                        (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0) ;          --
        mm2s_desc_app0              : out std_logic_vector                              --
                                        (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0) ;          --
        mm2s_desc_app1              : out std_logic_vector                              --
                                        (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0) ;          --
        mm2s_desc_app2              : out std_logic_vector                              --
                                        (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0) ;          --
        mm2s_desc_app3              : out std_logic_vector                              --
                                        (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0) ;          --
        mm2s_desc_app4              : out std_logic_vector                              --
                                        (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0)            --
    );

end axi_dma_mm2s_sg_if;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_dma_mm2s_sg_if is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  ATTRIBUTE async_reg                      : STRING;

-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------
-- Status reserved bits
constant RESERVED_STS           : std_logic_vector(4 downto 0) := (others => '0');

-- Used to determine when Control word is coming, in order to check SOF bit.
-- This then indicates that the app fields need to be directed towards the
-- control stream fifo.
-- Word Five Count
-- Incrementing these counts by 2 as i am now sending two extra fields from BD
--constant SEVEN_COUNT             : std_logic_vector(3 downto 0) := "1011"; --"0111";
constant SEVEN_COUNT             : std_logic_vector(3 downto 0) := "0001";
-- Word Six Count
--constant EIGHT_COUNT              : std_logic_vector(3 downto 0) := "0101"; --"1000";
constant EIGHT_COUNT              : std_logic_vector(3 downto 0) := "0010";
-- Word Seven Count
--constant NINE_COUNT            : std_logic_vector(3 downto 0) := "1010"; --"1001";
constant NINE_COUNT            : std_logic_vector(3 downto 0) := "0011";

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
signal ftch_shftenbl            : std_logic := '0';
signal ftch_tready              : std_logic := '0';
signal desc_fetch_done_i        : std_logic := '0';

signal desc_reg12               : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg11               : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg10               : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg9                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg8                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg7                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg6                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg5                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg4                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg3                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg2                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg1                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg0                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_dummy                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_dummy1                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');

signal mm2s_desc_curdesc_lsb    : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal mm2s_desc_curdesc_msb    : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal mm2s_desc_baddr_lsb      : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal mm2s_desc_baddr_msb      : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal mm2s_desc_blength_i      : std_logic_vector(BUFFER_LENGTH_WIDTH - 1 downto 0)    := (others => '0');
signal mm2s_desc_blength_v_i      : std_logic_vector(BUFFER_LENGTH_WIDTH - 1 downto 0)    := (others => '0');
signal mm2s_desc_blength_s_i      : std_logic_vector(BUFFER_LENGTH_WIDTH - 1 downto 0)    := (others => '0');

-- Fetch control signals for driving out control app stream
signal analyze_control          : std_logic := '0';
signal redirect_app             : std_logic := '0';
signal redirect_app_d1          : std_logic := '0';
signal redirect_app_re          : std_logic := '0';
signal redirect_app_hold        : std_logic := '0';
signal mask_fifo_write          : std_logic := '0';

-- Current descriptor control and fetch throttle control
signal mm2s_new_curdesc_wren_i  : std_logic := '0';
signal mm2s_pending_update      : std_logic := '0';
signal mm2s_pending_ptr_updt    : std_logic := '0';

-- Descriptor Update Signals
signal mm2s_complete            : std_logic := '0';
signal mm2s_xferd_bytes         : std_logic_vector(BUFFER_LENGTH_WIDTH-1 downto 0)      := (others => '0');
signal mm2s_xferd_bytes_int         : std_logic_vector(BUFFER_LENGTH_WIDTH-1 downto 0)      := (others => '0');

-- Update Descriptor Pointer Holding Registers
signal updt_desc_reg0           : std_logic_vector(C_M_AXI_SG_ADDR_WIDTH-1 downto 0) := (others => '0');
signal updt_desc_64_reg0           : std_logic_vector(C_S_AXIS_UPDPTR_TDATA_WIDTH-1 downto 0) := (others => '0');
signal updt_desc_reg1           : std_logic_vector(C_S_AXIS_UPDPTR_TDATA_WIDTH downto 0) := (others => '0');
-- Update Descriptor Status Holding Register
signal updt_desc_reg2           : std_logic_vector(C_S_AXIS_UPDSTS_TDATA_WIDTH downto 0) := (others => '0');

-- Pointer shift control
signal updt_shftenbl            : std_logic := '0';

-- Update pointer stream
signal updtptr_tvalid           : std_logic := '0';
signal updtptr_tlast            : std_logic := '0';
signal updtptr_tdata            : std_logic_vector(C_M_AXI_SG_ADDR_WIDTH-1 downto 0) := (others => '0');

-- Update status stream
signal updtsts_tvalid           : std_logic := '0';
signal updtsts_tlast            : std_logic := '0';
signal updtsts_tdata            : std_logic_vector(C_S_AXIS_UPDSTS_TDATA_WIDTH-1 downto 0) := (others => '0');

-- Status control
signal sts_received             : std_logic := '0';
signal sts_received_d1          : std_logic := '0';
signal sts_received_re          : std_logic := '0';

-- Queued Update signals
signal updt_data_clr            : std_logic := '0';
signal updt_sts_clr             : std_logic := '0';
signal updt_data                : std_logic := '0';
signal updt_sts                 : std_logic := '0';

signal packet_start             : std_logic := '0';
signal packet_end               : std_logic := '0';

signal mm2s_halt_d1_cdc_tig             : std_logic := '0';
signal mm2s_halt_cdc_d2             : std_logic := '0';
signal mm2s_halt_d2             : std_logic := '0';
  --ATTRIBUTE async_reg OF mm2s_halt_d1_cdc_tig  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF mm2s_halt_cdc_d2  : SIGNAL IS "true";


signal temp                     : std_logic := '0';
signal m_axis_mm2s_ftch_tlast_new : std_logic := '1';
-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin
-- Drive buffer length out
mm2s_desc_blength <= mm2s_desc_blength_i;
mm2s_desc_blength_v <= mm2s_desc_blength_v_i;
mm2s_desc_blength_s <= mm2s_desc_blength_s_i;


-- Drive fetch request done on tlast

desc_fetch_done_i   <= m_axis_mm2s_ftch_tlast_new
                 and m_axis_mm2s_ftch_tvalid_new;

-- pass out of module
desc_fetch_done <= desc_fetch_done_i;


-- Shift in data from SG engine if tvalid and fetch request
ftch_shftenbl     <= m_axis_mm2s_ftch_tvalid_new
                        and ftch_tready
                        and desc_fetch_req
                        and not mm2s_pending_update;

-- Passed curdes write out to register module
mm2s_new_curdesc_wren   <= desc_fetch_done_i; --mm2s_new_curdesc_wren_i;

-- tvalid asserted means descriptor availble
desc_available          <= m_axis_ftch1_desc_available; --m_axis_mm2s_ftch_tvalid_new;


--***************************************************************************--
--** Register DataMover Halt to secondary if needed
--***************************************************************************--
GEN_FOR_ASYNC : if C_PRMRY_IS_ACLK_ASYNC = 1 generate
begin
    -- Double register to secondary clock domain.  This is sufficient
    -- because halt will remain asserted until halt_cmplt detected in
    -- reset module in secondary clock domain.

REG_TO_SECONDARY : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => mm2s_halt,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => m_axi_sg_aclk,
        scndry_resetn              => '0',
        scndry_out                 => mm2s_halt_cdc_d2,
        scndry_vect_out            => open
    );


--    REG_TO_SECONDARY : process(m_axi_sg_aclk)
--        begin
--            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
--             --   if(m_axi_sg_aresetn = '0')then
--             --       mm2s_halt_d1_cdc_tig <= '0';
--             --       mm2s_halt_d2 <= '0';
--             --   else
--                    mm2s_halt_d1_cdc_tig <= mm2s_halt;
--                    mm2s_halt_cdc_d2 <= mm2s_halt_d1_cdc_tig;
--             --   end if;
--            end if;
--        end process REG_TO_SECONDARY;

                    mm2s_halt_d2 <= mm2s_halt_cdc_d2;

end generate GEN_FOR_ASYNC;

GEN_FOR_SYNC : if C_PRMRY_IS_ACLK_ASYNC = 0 generate
begin
    -- No clock crossing required therefore simple pass through
    mm2s_halt_d2 <= mm2s_halt;

end generate GEN_FOR_SYNC;




--***************************************************************************--
--**                        Descriptor Fetch Logic                         **--
--***************************************************************************--

packet_start <= '1' when mm2s_new_curdesc_wren_i ='1'
                     and desc_reg6(DESC_SOF_BIT) = '1'
           else '0';

packet_end <= '1' when mm2s_new_curdesc_wren_i ='1'
                   and desc_reg6(DESC_EOF_BIT) = '1'
           else '0';

REG_PACKET_PROGRESS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0' or packet_end = '1')then
                packet_in_progress <= '0';
            elsif(packet_start = '1')then
                packet_in_progress <= '1';
            end if;
        end if;
    end process REG_PACKET_PROGRESS;


-- Status/Control stream enabled therefore APP fields are included
GEN_FTCHIF_WITH_APP : if (C_SG_INCLUDE_STSCNTRL_STRM = 1 and C_ENABLE_MULTI_CHANNEL = 0) generate
-- Control Stream Ethernet TAG
constant ETHERNET_CNTRL_TAG     : std_logic_vector
                                    (C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH - 1 downto 0)
                                    := X"A000_0000";
begin

                            desc_reg7(30 downto 0)     <= (others => '0');

                            desc_reg7 (DESC_STS_CMPLTD_BIT)     <= m_axis_mm2s_ftch_tdata_new (64); -- downto 64);
                            desc_reg6     <= m_axis_mm2s_ftch_tdata_new (63 downto 32);
                            desc_reg2     <= m_axis_mm2s_ftch_tdata_new (31 downto 0);
                            desc_reg0     <= m_axis_mm2s_ftch_tdata_new (96 downto 65);


ADDR_64BIT : if C_M_AXI_SG_ADDR_WIDTH > 32 generate
begin

    mm2s_desc_baddr_msb   <= m_axis_mm2s_ftch_tdata_new (128 downto 97);
    mm2s_desc_curdesc_msb     <= m_axis_mm2s_ftch_tdata_new (160 downto 129);

end generate ADDR_64BIT;

ADDR_32BIT : if C_M_AXI_SG_ADDR_WIDTH = 32 generate
begin

    mm2s_desc_curdesc_msb   <= (others => '0');
    mm2s_desc_baddr_msb     <= (others => '0');

end generate ADDR_32BIT;


    mm2s_desc_curdesc_lsb   <= desc_reg0;
    mm2s_desc_baddr_lsb     <= desc_reg2;
    -- desc 5 are reserved and thus don't care
    -- CR 583779, need to pass on tuser and cache information
    mm2s_desc_info          <= (others => '0'); --desc_reg4; -- this coincides with desc_fetch_done
    mm2s_desc_blength_i     <= desc_reg6(DESC_BLENGTH_MSB_BIT downto DESC_BLENGTH_LSB_BIT);
    mm2s_desc_blength_v_i     <= (others => '0');
    mm2s_desc_blength_s_i     <= (others => '0');
    mm2s_desc_eof           <= desc_reg6(DESC_EOF_BIT);
    mm2s_desc_sof           <= desc_reg6(DESC_SOF_BIT);
    mm2s_desc_cmplt         <= desc_reg7(DESC_STS_CMPLTD_BIT);
    mm2s_desc_app0          <= desc_reg8;
    mm2s_desc_app1          <= desc_reg9;
    mm2s_desc_app2          <= desc_reg10;
    mm2s_desc_app3          <= desc_reg11;
    mm2s_desc_app4          <= desc_reg12;


    -- Drive ready if descriptor fetch request is being made
    -- If not redirecting app fields then drive ready based on sm request
    -- If redirecting app fields then drive ready based on room in cntrl strm fifo

    ftch_tready     <= desc_fetch_req               -- desc fetch request
                   and not mm2s_pending_update; -- no pntr updates pending



    m_axis_mm2s_ftch_tready <= ftch_tready;




                    redirect_app <= '0';

                cntrlstrm_fifo_din  <= (others => '0');
                cntrlstrm_fifo_wren <= '0';

end generate GEN_FTCHIF_WITH_APP;


-- Status/Control stream diabled therefore APP fields are NOT included
GEN_FTCHIF_WITHOUT_APP : if C_SG_INCLUDE_STSCNTRL_STRM = 0 generate



GEN_NO_MCDMA : if C_ENABLE_MULTI_CHANNEL = 0 generate

                            desc_reg7(30 downto 0)     <= (others => '0');

                            desc_reg7(DESC_STS_CMPLTD_BIT)     <= m_axis_mm2s_ftch_tdata_new (64); --95 downto 64);
                            desc_reg6     <= m_axis_mm2s_ftch_tdata_new (63 downto 32);
                            desc_reg2     <= m_axis_mm2s_ftch_tdata_new (31 downto 0);
                            desc_reg0     <= m_axis_mm2s_ftch_tdata_new (96 downto 65); --127 downto 96);


ADDR1_64BIT : if C_M_AXI_SG_ADDR_WIDTH > 32 generate
begin

    mm2s_desc_baddr_msb   <= m_axis_mm2s_ftch_tdata_new (128 downto 97);
    mm2s_desc_curdesc_msb     <= m_axis_mm2s_ftch_tdata_new (160 downto 129);

end generate ADDR1_64BIT;

ADDR1_32BIT : if C_M_AXI_SG_ADDR_WIDTH = 32 generate
begin

    mm2s_desc_curdesc_msb   <= (others => '0');
    mm2s_desc_baddr_msb     <= (others => '0');

end generate ADDR1_32BIT;


    mm2s_desc_curdesc_lsb   <= desc_reg0;
    mm2s_desc_baddr_lsb     <= desc_reg2;
    -- desc 4 and desc 5 are reserved and thus don't care
    -- CR 583779, need to send the user and xchache info
    mm2s_desc_info          <= (others => '0'); --desc_reg4;
    mm2s_desc_blength_i     <= desc_reg6(DESC_BLENGTH_MSB_BIT downto DESC_BLENGTH_LSB_BIT);
    mm2s_desc_blength_v_i     <= (others => '0');
    mm2s_desc_blength_s_i     <= (others => '0');
    mm2s_desc_eof           <= desc_reg6(DESC_EOF_BIT);
    mm2s_desc_sof           <= desc_reg6(DESC_SOF_BIT);
    mm2s_desc_cmplt         <= desc_reg7(DESC_STS_CMPLTD_BIT);
    mm2s_desc_app0          <= (others => '0');
    mm2s_desc_app1          <= (others => '0');
    mm2s_desc_app2          <= (others => '0');
    mm2s_desc_app3          <= (others => '0');
    mm2s_desc_app4          <= (others => '0');
end generate GEN_NO_MCDMA;


GEN_MCDMA : if C_ENABLE_MULTI_CHANNEL = 1 generate



                            desc_reg7(30 downto 0)     <= (others => '0');

                            desc_reg7 (DESC_STS_CMPLTD_BIT)     <= m_axis_mm2s_ftch_tdata_new (64); --95 downto 64);
                            desc_reg6     <= m_axis_mm2s_ftch_tdata_new (63 downto 32);
                            desc_reg2     <= m_axis_mm2s_ftch_tdata_new (31 downto 0);
                            desc_reg0     <= m_axis_mm2s_ftch_tdata_new (96 downto 65); --127 downto 96);

                            desc_reg4     <= m_axis_mm2s_ftch_tdata_mcdma_new (31 downto 0); --63 downto 32);
                            desc_reg5     <= m_axis_mm2s_ftch_tdata_mcdma_new (63 downto 32);

ADDR2_64BIT : if C_M_AXI_SG_ADDR_WIDTH > 32 generate
begin

    mm2s_desc_curdesc_msb   <= m_axis_mm2s_ftch_tdata_new (128 downto 97);
    mm2s_desc_baddr_msb     <= m_axis_mm2s_ftch_tdata_new (160 downto 129);

end generate ADDR2_64BIT;

ADDR2_32BIT : if C_M_AXI_SG_ADDR_WIDTH = 32 generate
begin

    mm2s_desc_curdesc_msb   <= (others => '0');
    mm2s_desc_baddr_msb     <= (others => '0');

end generate ADDR2_32BIT;


    mm2s_desc_curdesc_lsb   <= desc_reg0;
    mm2s_desc_baddr_lsb     <= desc_reg2;
-- As per new MCDMA descriptor
    mm2s_desc_info          <= desc_reg4; -- (31 downto 24) & desc_reg7 (23 downto 0);
    mm2s_desc_blength_s_i   <= "0000000" & desc_reg5(15 downto 0); 
    mm2s_desc_blength_v_i   <= "0000000000" & desc_reg5(31 downto 19);
    mm2s_desc_blength_i     <= "0000000" & desc_reg6(15 downto 0); 
    mm2s_desc_eof           <= desc_reg6(DESC_EOF_BIT);
    mm2s_desc_sof           <= desc_reg6(DESC_SOF_BIT);
    mm2s_desc_cmplt         <= '0' ; --desc_reg7(DESC_STS_CMPLTD_BIT); -- we are not considering the completed bit
    mm2s_desc_app0          <= (others => '0');
    mm2s_desc_app1          <= (others => '0');
    mm2s_desc_app2          <= (others => '0');
    mm2s_desc_app3          <= (others => '0');
    mm2s_desc_app4          <= (others => '0');
end generate GEN_MCDMA;

    -- Drive ready if descriptor fetch request is being made
    ftch_tready     <= desc_fetch_req               -- desc fetch request
                   and not mm2s_pending_update; -- no pntr updates pending


    m_axis_mm2s_ftch_tready <= ftch_tready;

    cntrlstrm_fifo_wren     <= '0';
    cntrlstrm_fifo_din      <= (others => '0');


end generate GEN_FTCHIF_WITHOUT_APP;

-------------------------------------------------------------------------------
-- BUFFER ADDRESS
-------------------------------------------------------------------------------
-- If 64 bit addressing then concatinate msb to lsb
GEN_NEW_64BIT_BUFADDR : if C_M_AXI_MM2S_ADDR_WIDTH > 32 generate
    mm2s_desc_baddress <= mm2s_desc_baddr_msb & mm2s_desc_baddr_lsb;
end generate GEN_NEW_64BIT_BUFADDR;

-- If 32 bit addressing then simply pass lsb out
GEN_NEW_32BIT_BUFADDR : if C_M_AXI_MM2S_ADDR_WIDTH = 32 generate
    mm2s_desc_baddress <= mm2s_desc_baddr_lsb;
end generate GEN_NEW_32BIT_BUFADDR;

-------------------------------------------------------------------------------
-- NEW CURRENT DESCRIPTOR
-------------------------------------------------------------------------------
-- If 64 bit addressing then concatinate msb to lsb
GEN_NEW_64BIT_CURDESC : if C_M_AXI_SG_ADDR_WIDTH > 32 generate
    mm2s_new_curdesc <= mm2s_desc_curdesc_msb & mm2s_desc_curdesc_lsb;
end generate GEN_NEW_64BIT_CURDESC;

-- If 32 bit addressing then simply pass lsb out
GEN_NEW_32BIT_CURDESC : if C_M_AXI_SG_ADDR_WIDTH = 32 generate
    mm2s_new_curdesc <= mm2s_desc_curdesc_lsb;
end generate GEN_NEW_32BIT_CURDESC;

                mm2s_new_curdesc_wren_i <= desc_fetch_done_i;

--***************************************************************************--
--**                       Descriptor Update Logic                         **--
--***************************************************************************--

--*****************************************************************************
--** Pointer Update Logic
--*****************************************************************************

    -----------------------------------------------------------------------
    -- Capture LSB cur descriptor on write for use on descriptor update.
    -- This will be the address the descriptor is updated to
    -----------------------------------------------------------------------
    UPDT_DESC_WRD0: process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    updt_desc_reg0 (31 downto 0) <= (others => '0');
                elsif(mm2s_new_curdesc_wren_i = '1')then
                    updt_desc_reg0 (31 downto 0) <= mm2s_desc_curdesc_lsb;
                end if;
            end if;
        end process UPDT_DESC_WRD0;

UPDT_ADDR_64BIT : if C_M_AXI_MM2S_ADDR_WIDTH > 32 generate
begin
    UPDT_DESC_WRD0_1: process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    updt_desc_reg0 (C_M_AXI_SG_ADDR_WIDTH-1 downto 32) <= (others => '0');
                elsif(mm2s_new_curdesc_wren_i = '1')then
                    updt_desc_reg0 (C_M_AXI_SG_ADDR_WIDTH-1 downto 32) <= mm2s_desc_curdesc_msb;
                end if;
            end if;
        end process UPDT_DESC_WRD0_1;


end generate UPDT_ADDR_64BIT;

    -----------------------------------------------------------------------
    -- Capture MSB cur descriptor on write for use on descriptor update.
    -- This will be the address the descriptor is updated to
    -----------------------------------------------------------------------
    UPDT_DESC_WRD1: process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    updt_desc_reg1 <= (others => '0');
                elsif(mm2s_new_curdesc_wren_i = '1')then
                    updt_desc_reg1 <= DESC_LAST
                                      & mm2s_desc_curdesc_msb;

                -- Shift data out on shift enable
                elsif(updt_shftenbl = '1')then
                    updt_desc_reg1 <= (others => '0');

                end if;
            end if;
        end process UPDT_DESC_WRD1;


    -- Shift in data from SG engine if tvalid, tready, and not on last word
    updt_shftenbl <=  updt_data and updtptr_tvalid and s_axis_mm2s_updtptr_tready;


    -- Update data done when updating data and tlast received and target
    -- (i.e. SG Engine) is ready
    updt_data_clr <= '1' when updtptr_tvalid = '1' and updtptr_tlast = '1'
                          and s_axis_mm2s_updtptr_tready = '1'
                else '0';


    -- When desc data ready for update set and hold flag until
    -- data can be updated to queue.  Note it may
    -- be held off due to update of status
    UPDT_DATA_PROCESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0' or updt_data_clr = '1')then
                    updt_data   <= '0';
                -- clear flag when data update complete
 --               elsif(updt_data_clr = '1')then
 --                   updt_data <= '0';
 --               -- set flag when desc fetched as indicated
 --               -- by curdesc wren
                elsif(mm2s_new_curdesc_wren_i = '1')then
                    updt_data <= '1';
                end if;
            end if;
        end process UPDT_DATA_PROCESS;

    updtptr_tvalid  <= updt_data;
    updtptr_tlast   <= DESC_LAST; --updt_desc_reg0(C_S_AXIS_UPDPTR_TDATA_WIDTH);
    updtptr_tdata   <= updt_desc_reg0(C_M_AXI_SG_ADDR_WIDTH-1 downto 0);


--*****************************************************************************
--** Status Update Logic
--*****************************************************************************

    mm2s_complete <= '1'; -- Fixed at '1'

    ---------------------------------------------------------------------------
    -- Descriptor queuing turned on in sg engine therefore need to instantiate
    -- fifo to hold fetch buffer lengths.  Also need to throttle fetches
    -- if pointer has not been updated yet or length fifo is full
    ---------------------------------------------------------------------------
    GEN_UPDT_FOR_QUEUE : if C_SG_INCLUDE_DESC_QUEUE = 1 generate
    signal xb_fifo_reset   : std_logic; -- xfer'ed bytes fifo reset
    signal xb_fifo_full    : std_logic; -- xfer'ed bytes fifo full
    begin
        -----------------------------------------------------------------------
        -- Need to flag a pending pointer update to prevent subsequent fetch of
        -- descriptor from stepping on the stored pointer, and buffer length
        -----------------------------------------------------------------------
        REG_PENDING_UPDT : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0' or updt_data_clr = '1')then
                        mm2s_pending_ptr_updt <= '0';
                    elsif (desc_fetch_done_i = '1') then --(mm2s_new_curdesc_wren_i = '1')then
                        mm2s_pending_ptr_updt <= '1';
                    end if;
                end if;
            end process REG_PENDING_UPDT;

        -- Pointer pending update or xferred bytes fifo full
        mm2s_pending_update <= mm2s_pending_ptr_updt or xb_fifo_full;
        updt_pending <= mm2s_pending_update;
        -----------------------------------------------------------------------
        -- On MM2S transferred bytes equals buffer length.  Capture length
        -- on curdesc write.
        -----------------------------------------------------------------------

GEN_MICRO_DMA : if C_MICRO_DMA = 1 generate

mm2s_xferd_bytes <= (others => '0');
xb_fifo_full <= '0';

end generate GEN_MICRO_DMA;

GEN_NO_MICRO_DMA : if C_MICRO_DMA = 0 generate

        XFERRED_BYTE_FIFO : entity lib_srl_fifo_v1_0_2.srl_fifo_f
          generic map(
            C_DWIDTH        => BUFFER_LENGTH_WIDTH          ,
            C_DEPTH         => 16                           ,
            C_FAMILY        => C_FAMILY
            )
          port map(
            Clk             => m_axi_sg_aclk              ,
            Reset           => xb_fifo_reset                ,
            FIFO_Write      => desc_fetch_done_i, --mm2s_new_curdesc_wren_i      ,
            Data_In         => mm2s_desc_blength_i          ,
            FIFO_Read       => sts_received_re              ,
            Data_Out        => mm2s_xferd_bytes             ,
            FIFO_Empty      => open                         ,
            FIFO_Full       => xb_fifo_full                 ,
            Addr            => open
            );

end generate GEN_NO_MICRO_DMA;


        xb_fifo_reset      <= not m_axi_sg_aresetn;

        -- clear status received flag in cmdsts_if to
        -- allow more status to be received from datamover
        mm2s_sts_received_clr <= updt_sts_clr;

        -- Generate a rising edge off status received in order to
        -- flag status update
        REG_STATUS : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        sts_received_d1 <= '0';
                    else
                        sts_received_d1 <= mm2s_sts_received;
                    end if;
                end if;
            end process REG_STATUS;

        -- CR566306 - status invalid during halt
        --sts_received_re <= mm2s_sts_received and not sts_received_d1;
        sts_received_re <= mm2s_sts_received and not sts_received_d1 and not mm2s_halt_d2;

    end generate GEN_UPDT_FOR_QUEUE;

    ---------------------------------------------------------------------------
    -- If no queue in sg engine then do not need to instantiate a
    -- fifo to hold buffer lengths.   Also do not need to hold off
    -- fetch based on if status has been updated or not because
    -- descriptors are only processed one at a time
    ---------------------------------------------------------------------------
    GEN_UPDT_FOR_NO_QUEUE : if C_SG_INCLUDE_DESC_QUEUE = 0 generate
    begin

        mm2s_sts_received_clr   <= '1'; -- Not needed for the No Queue configuration

        mm2s_pending_update     <= '0'; -- Not needed for the No Queue configuration

        -----------------------------------------------------------------------
        -- On MM2S transferred bytes equals buffer length.  Capture length
        -- on curdesc write.
        -----------------------------------------------------------------------
        REG_XFERRED_BYTES : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        mm2s_xferd_bytes <= (others => '0');
                    elsif(mm2s_new_curdesc_wren_i = '1')then
                        mm2s_xferd_bytes <= mm2s_desc_blength_i;
                    end if;
                end if;
            end process REG_XFERRED_BYTES;

        -- Status received based on a DONE or an ERROR from DataMover
        sts_received <= mm2s_done or mm2s_interr or mm2s_decerr or mm2s_slverr;

        -- Generate a rising edge off status received in order to
        -- flag status update
        REG_STATUS : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        sts_received_d1 <= '0';
                    else
                        sts_received_d1 <= sts_received;
                    end if;
                end if;
            end process REG_STATUS;

        -- CR566306 - status invalid during halt
        --sts_received_re <= mm2s_sts_received and not sts_received_d1;
        sts_received_re <= sts_received and not sts_received_d1 and not mm2s_halt_d2;

    end generate GEN_UPDT_FOR_NO_QUEUE;



    -----------------------------------------------------------------------
    -- Receive Status SG Update Logic
    -----------------------------------------------------------------------
    -- clear flag when updating status and see a tlast and target
    -- (i.e. sg engine) is ready
    updt_sts_clr <= '1' when updt_sts = '1'
                         and updtsts_tlast = '1'
                         and updtsts_tvalid = '1'
                         and s_axis_mm2s_updtsts_tready = '1'
                else '0';

    -- When status received set and hold flag until
    -- status can be updated to queue.  Note it may
    -- be held off due to update of data
    UPDT_STS_PROCESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0' or updt_sts_clr = '1')then
                    updt_sts                <= '0';
                -- clear flag when status update done
                -- or datamover halted
   --             elsif(updt_sts_clr = '1')then
   --                 updt_sts                <= '0';
   --             -- set flag when status received
                elsif(sts_received_re = '1')then
                    updt_sts                <= '1';
                end if;
            end if;
        end process UPDT_STS_PROCESS;

    -----------------------------------------------------------------------
    -- Catpure Status.  Status is built from status word from DataMover
    -- and from transferred bytes value.
    -----------------------------------------------------------------------
    UPDT_DESC_WRD2 : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    updt_desc_reg2  <= (others => '0');

                elsif(sts_received_re = '1')then
                    updt_desc_reg2  <= DESC_LAST
                                     & mm2s_tag(DATAMOVER_STS_TAGLSB_BIT)  -- Desc_IOC
                                     & mm2s_complete
                                     & mm2s_decerr
                                     & mm2s_slverr
                                     & mm2s_interr
                                     & RESERVED_STS
                                     & mm2s_xferd_bytes;
                end if;
            end if;
        end process UPDT_DESC_WRD2;


    updtsts_tdata  <= updt_desc_reg2(C_S_AXIS_UPDSTS_TDATA_WIDTH-1 downto 0);
    -- MSB asserts last on last word of update stream
    updtsts_tlast  <= updt_desc_reg2(C_S_AXIS_UPDSTS_TDATA_WIDTH);
    -- Drive tvalid
    updtsts_tvalid <= updt_sts;


    -- Drive update done to mm2s sm for the no queue case to indicate
    -- readyd to fetch next descriptor
    UPDT_DONE_PROCESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then

                if(m_axi_sg_aresetn = '0')then
                    desc_update_done <= '0';
                else
                    desc_update_done <= updt_sts_clr;
                end if;
            end if;
       end process UPDT_DONE_PROCESS;


    -- Update Pointer Stream
    s_axis_mm2s_updtptr_tvalid <= updtptr_tvalid;
    s_axis_mm2s_updtptr_tlast  <= updtptr_tlast and updtptr_tvalid;
    s_axis_mm2s_updtptr_tdata  <= updtptr_tdata ;

    -- Update Status Stream
    s_axis_mm2s_updtsts_tvalid <= updtsts_tvalid;
    s_axis_mm2s_updtsts_tlast  <= updtsts_tlast and updtsts_tvalid;
    s_axis_mm2s_updtsts_tdata  <= updtsts_tdata ;

    -----------------------------------------------------------------------


end implementation;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:          axi_dma_mm2s_sm.vhd
-- Description: This entity contains the MM2S DMA Controller State Machine
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;

library lib_pkg_v1_0_2;
use lib_pkg_v1_0_2.lib_pkg.clog2;


-------------------------------------------------------------------------------
entity  axi_dma_mm2s_sm is
    generic (
        C_M_AXI_MM2S_ADDR_WIDTH     : integer range 32 to 64    := 32;
            -- Master AXI Memory Map Address Width for MM2S Read Port

        C_SG_LENGTH_WIDTH           : integer range 8 to 23     := 14;
            -- Width of Buffer Length, Transferred Bytes, and BTT fields

        C_SG_INCLUDE_DESC_QUEUE     : integer range 0 to 1      := 0;
            -- Include or Exclude Scatter Gather Descriptor Queuing
            -- 0 = Exclude SG Descriptor Queuing
            -- 1 = Include SG Descriptor Queuing

        C_PRMY_CMDFIFO_DEPTH        : integer range 1 to 16     := 1;
            -- Depth of DataMover command FIFO

        C_ENABLE_MULTI_CHANNEL             : integer range 0 to 1      := 0
    );
    port (
        m_axi_sg_aclk               : in  std_logic                         ;                      --
        m_axi_sg_aresetn            : in  std_logic                         ;                      --
                                                                                                   --
        -- Channel 1 Control and Status                                                            --
        mm2s_run_stop               : in  std_logic                         ;                      --
        mm2s_keyhole                : in  std_logic                         ;
        mm2s_ftch_idle              : in  std_logic                         ;                      --
        mm2s_stop                   : in  std_logic                         ;                      --
        mm2s_cmnd_idle              : out std_logic                         ;                      --
        mm2s_sts_idle               : out std_logic                         ;                      --
        mm2s_desc_flush             : out std_logic                         ;                      --
                                                                                                   --
        -- MM2S Descriptor Fetch Request (from mm2s_sm)                                            --
        desc_available              : in  std_logic                         ;                      --
        desc_fetch_req              : out std_logic                         ;                      --
        desc_fetch_done             : in  std_logic                         ;                      --
        desc_update_done            : in  std_logic                         ;                      --
        updt_pending                : in  std_logic                         ;
        packet_in_progress          : in  std_logic                         ;                      --
                                                                                                   --
        -- DataMover Command                                                                       --
        mm2s_cmnd_wr                : out std_logic                         ;                      --
        mm2s_cmnd_data              : out std_logic_vector                                         --
                                        ((C_M_AXI_MM2S_ADDR_WIDTH-32+64+CMD_BASE_WIDTH+46)-1 downto 0);     --
        mm2s_cmnd_pending           : in std_logic                          ;                      --
                                                                                                   --
        -- Descriptor Fields                                                                       --
        mm2s_cache_info             : in std_logic_vector
                                        (32-1 downto 0);                      --
        mm2s_desc_baddress          : in  std_logic_vector                                         --
                                        (C_M_AXI_MM2S_ADDR_WIDTH-1 downto 0);                      --
        mm2s_desc_blength           : in  std_logic_vector                                         --
                                        (BUFFER_LENGTH_WIDTH-1 downto 0)    ;                      --
        mm2s_desc_blength_v           : in  std_logic_vector                                         --
                                        (BUFFER_LENGTH_WIDTH-1 downto 0)    ;                      --
        mm2s_desc_blength_s           : in  std_logic_vector                                         --
                                        (BUFFER_LENGTH_WIDTH-1 downto 0)    ;                      --
        mm2s_desc_eof               : in  std_logic                         ;                      --
        mm2s_desc_sof               : in  std_logic                                                --

    );

end axi_dma_mm2s_sm;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_dma_mm2s_sm is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------
-- DataMover Commmand TAG
constant MM2S_CMD_TAG       : std_logic_vector(2 downto 0)  := (others => '0');
-- DataMover Command Destination Stream Offset
constant MM2S_CMD_DSA       : std_logic_vector(5 downto 0)  := (others => '0');
-- DataMover Cmnd Reserved Bits
constant MM2S_CMD_RSVD      : std_logic_vector(
                                DATAMOVER_CMD_RSVMSB_BOFST + C_M_AXI_MM2S_ADDR_WIDTH downto
                                DATAMOVER_CMD_RSVLSB_BOFST + C_M_AXI_MM2S_ADDR_WIDTH)
                                := (others => '0');

-- Queued commands counter width
constant COUNTER_WIDTH      : integer := clog2(C_PRMY_CMDFIFO_DEPTH+1);
-- Queued commands zero count
constant ZERO_COUNT         : std_logic_vector(COUNTER_WIDTH - 1 downto 0)
                                := (others => '0');



-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
type SG_MM2S_STATE_TYPE      is (
                                IDLE,
                                FETCH_DESCRIPTOR,
                              --  EXECUTE_XFER,
                                WAIT_STATUS
                                );

signal mm2s_cs                  : SG_MM2S_STATE_TYPE;
signal mm2s_ns                  : SG_MM2S_STATE_TYPE;



-- State Machine Signals
signal desc_fetch_req_cmb       : std_logic := '0';
signal write_cmnd_cmb           : std_logic := '0';
signal mm2s_cmnd_wr_i           : std_logic := '0';

signal cmnds_queued             : std_logic_vector(COUNTER_WIDTH - 1 downto 0) := (others => '0');
signal cmnds_queued_shift             : std_logic_vector(C_PRMY_CMDFIFO_DEPTH - 1 downto 0) := (others => '0');
signal count_incr               : std_logic := '0';
signal count_decr               : std_logic := '0';
signal mm2s_desc_flush_i        : std_logic := '0';

signal queue_more               : std_logic := '0';

signal burst_type               : std_logic;

-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin

mm2s_cmnd_wr    <= mm2s_cmnd_wr_i;
mm2s_desc_flush <= mm2s_desc_flush_i;

-- Flush any fetch descriptors if stopped due to errors or soft reset
-- or if not in middle of packet and run/stop clears
mm2s_desc_flush_i <= '1' when (mm2s_stop = '1')

                          or  (packet_in_progress = '0'
                               and mm2s_run_stop  = '0')
              else '0';

burst_type <= '1' and (not mm2s_keyhole);
-- A 0 on mm2s_kyhole means increment type burst
-- 1 means fixed burst

-------------------------------------------------------------------------------
-- MM2S Transfer State Machine
-------------------------------------------------------------------------------
MM2S_MACHINE : process(mm2s_cs,
                       mm2s_run_stop,
                       packet_in_progress,
                       desc_available,
                       updt_pending,
                --       desc_fetch_done,
                       desc_update_done,
                       mm2s_cmnd_pending,
                       mm2s_stop,
                       mm2s_desc_flush_i
                   --    queue_more
                       )
    begin

        -- Default signal assignment
        desc_fetch_req_cmb      <= '0';
        write_cmnd_cmb          <= '0';
        mm2s_cmnd_idle          <= '0';
        mm2s_ns                 <= mm2s_cs;

        case mm2s_cs is

            -------------------------------------------------------------------
            when IDLE =>
                -- Running or Stopped but in middle of xfer and Descriptor
                -- data available, No errors logged, and Room to queue more
                -- commands, then fetch descriptor
            --    if (updt_pending = '1') then
            --       mm2s_ns <= IDLE;
                if( (mm2s_run_stop = '1' or packet_in_progress = '1')
--                and desc_available = '1' and mm2s_stop = '0' and queue_more = '1' and updt_pending = '0') then
                and desc_available = '1' and mm2s_stop = '0' and updt_pending = '0') then
                  if (C_SG_INCLUDE_DESC_QUEUE = 0) then
            -- coverage off
                    mm2s_ns <= WAIT_STATUS;
                    write_cmnd_cmb  <= '1';
            -- coverage on
                  else
                    mm2s_ns <= FETCH_DESCRIPTOR;
                    desc_fetch_req_cmb  <= '1';
                  end if;
                else
                    mm2s_cmnd_idle <= '1';
                    write_cmnd_cmb  <= '0';
                end if;

            -------------------------------------------------------------------
            when FETCH_DESCRIPTOR =>
                -- error detected or run/stop cleared
                if(mm2s_desc_flush_i = '1' or mm2s_stop = '1')then
                    mm2s_ns     <= IDLE;
                -- descriptor fetch complete
          --      elsif(desc_fetch_done = '1')then
          --          desc_fetch_req_cmb  <= '0';
          --          mm2s_ns             <= EXECUTE_XFER;
                elsif(mm2s_cmnd_pending = '0')then
                    desc_fetch_req_cmb  <= '0';
                    if (updt_pending = '0') then
                        if(C_SG_INCLUDE_DESC_QUEUE = 1)then
                           mm2s_ns         <= IDLE;
            -- coverage off
                           write_cmnd_cmb  <= '1';
            -- coverage on
                        else
                           mm2s_ns         <= WAIT_STATUS;
                        end if;
                    end if;
                else
                        mm2s_ns            <= FETCH_DESCRIPTOR;
                    desc_fetch_req_cmb  <= '0';
                end if;

            -------------------------------------------------------------------
--            when EXECUTE_XFER =>
--                -- error detected
--                if(mm2s_stop = '1')then
--                    mm2s_ns     <= IDLE;
--                -- Write another command if there is not one already pending
--                elsif(mm2s_cmnd_pending = '0')then
--                    if (updt_pending = '0') then
--                        write_cmnd_cmb  <= '1';
--                    end if;
--                    if(C_SG_INCLUDE_DESC_QUEUE = 1)then
--                        mm2s_ns         <= IDLE;
--                    else
--                        mm2s_ns         <= WAIT_STATUS;
--                    end if;
--                else
--                    mm2s_ns             <= EXECUTE_XFER;
--                end if;
--
            -------------------------------------------------------------------
            -- coverage off
            when WAIT_STATUS =>
                -- wait until desc update complete or error occurs
                if(desc_update_done = '1' or mm2s_stop = '1')then
                    mm2s_ns <= IDLE;
                else
                    mm2s_ns <= WAIT_STATUS;
                end if;
            -- coverage on

            -------------------------------------------------------------------
            -- coverage off
            when others =>
                mm2s_ns <= IDLE;
            -- coverage on
        end case;
    end process MM2S_MACHINE;

-------------------------------------------------------------------------------
-- register state machine states
-------------------------------------------------------------------------------
REGISTER_STATE : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                mm2s_cs     <= IDLE;
            else
                mm2s_cs     <= mm2s_ns;
            end if;
        end if;
    end process REGISTER_STATE;

-------------------------------------------------------------------------------
-- register state machine signals
-------------------------------------------------------------------------------
--SM_SIG_REGISTER : process(m_axi_sg_aclk)
--    begin
--        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
--            if(m_axi_sg_aresetn = '0')then
--                desc_fetch_req      <= '0'      ;
--            else
--                if (C_SG_INCLUDE_DESC_QUEUE = 0) then
--                     desc_fetch_req      <= '1'; --desc_fetch_req_cmb   ;
--                else
--                     desc_fetch_req      <= desc_fetch_req_cmb   ;
--                end if; 
--            end if;
--        end if;
--    end process SM_SIG_REGISTER;

           desc_fetch_req <= '1' when (C_SG_INCLUDE_DESC_QUEUE = 0) else
                             desc_fetch_req_cmb ;



-------------------------------------------------------------------------------
-- Build DataMover command
-------------------------------------------------------------------------------
-- If Bytes To Transfer (BTT) width less than 23, need to add pad
GEN_CMD_BTT_LESS_23 : if C_SG_LENGTH_WIDTH < 23 generate
constant PAD_VALUE : std_logic_vector(22 - C_SG_LENGTH_WIDTH downto 0)
                        := (others => '0');
begin
    -- When command by sm, drive command to mm2s_cmdsts_if
    GEN_DATAMOVER_CMND : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    mm2s_cmnd_wr_i  <= '0';
        --            mm2s_cmnd_data  <= (others => '0');

                -- Fetch SM issued a command write
                --
                -- Note: change to mode where EOF generates IOC interrupt as
                -- opposed to a IOC bit in the descriptor negated need for an
                -- EOF and IOC tag.  Given time, these two bits could be combined
                -- into 1.  Associated logic in SG engine would also need to be
                -- modified as well as in mm2s_sg_if.
                elsif(write_cmnd_cmb = '1')then
                    mm2s_cmnd_wr_i  <= '1';
        --            mm2s_cmnd_data  <=  mm2s_cache_info 
        --                                & mm2s_desc_blength_v
        --                                & mm2s_desc_blength_s   
        --                                & MM2S_CMD_RSVD
        --                                -- Command Tag
        --                                & '0'
        --                                & '0'
        --                                & mm2s_desc_eof  -- Cat. EOF to CMD Tag
        --                                & mm2s_desc_eof  -- Cat. IOC to CMD Tag
        --                                -- Command
        --                                & mm2s_desc_baddress
        --                                & mm2s_desc_sof
        --                                & mm2s_desc_eof
        --                                & MM2S_CMD_DSA
        --                                & burst_type          -- key Hole operation'1'       -- mm2s_desc_type IR#545697
        --                                & PAD_VALUE
        --                                & mm2s_desc_blength(C_SG_LENGTH_WIDTH-1 downto 0);

                else
                    mm2s_cmnd_wr_i  <= '0';

                end if;
            end if;
        end process GEN_DATAMOVER_CMND;


                    mm2s_cmnd_data  <=  mm2s_cache_info 
                                        & mm2s_desc_blength_v
                                        & mm2s_desc_blength_s   
                                        & MM2S_CMD_RSVD
                                        -- Command Tag
                                        & '0'
                                        & '0'
                                        & mm2s_desc_eof  -- Cat. EOF to CMD Tag
                                        & mm2s_desc_eof  -- Cat. IOC to CMD Tag
                                        -- Command
                                        & mm2s_desc_baddress
                                        & mm2s_desc_sof
                                        & mm2s_desc_eof
                                        & MM2S_CMD_DSA
                                        & burst_type          -- key Hole operation'1'       -- mm2s_desc_type IR#545697
                                        & PAD_VALUE
                                        & mm2s_desc_blength(C_SG_LENGTH_WIDTH-1 downto 0);

end generate GEN_CMD_BTT_LESS_23;

-- If Bytes To Transfer (BTT) width equal 23, no required pad
GEN_CMD_BTT_EQL_23 : if C_SG_LENGTH_WIDTH = 23 generate
begin
    -- When command by sm, drive command to mm2s_cmdsts_if
    GEN_DATAMOVER_CMND : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    mm2s_cmnd_wr_i  <= '0';
          --          mm2s_cmnd_data  <= (others => '0');

                -- Fetch SM issued a command write
                --
                -- Note: change to mode where EOF generates IOC interrupt as
                -- opposed to a IOC bit in the descriptor negated need for an
                -- EOF and IOC tag.  Given time, these two bits could be combined
                -- into 1.  Associated logic in SG engine would also need to be
                -- modified as well as in mm2s_sg_if.
                elsif(write_cmnd_cmb = '1')then
                    mm2s_cmnd_wr_i  <= '1';
         --           mm2s_cmnd_data  <=  mm2s_cache_info
         --                               & mm2s_desc_blength_v
         --                               & mm2s_desc_blength_s   
         --                               & MM2S_CMD_RSVD
         --                               -- Command Tag
         --                               & '0'
         --                               & '0'
         --                               & mm2s_desc_eof  -- Cat. EOF to CMD Tag
         --                               & mm2s_desc_eof  -- Cat. IOC to CMD Tag (ioc changed to EOF)
         --                               -- Command
         --                               & mm2s_desc_baddress
         --                               & mm2s_desc_sof
         --                               & mm2s_desc_eof
         --                               & MM2S_CMD_DSA
         --                               & burst_type    -- key Hole Operation'1'               -- mm2s_desc_type IR#545697
         --                               & mm2s_desc_blength;

                else
                    mm2s_cmnd_wr_i  <= '0';

                end if;
            end if;
        end process GEN_DATAMOVER_CMND;

                    mm2s_cmnd_data  <=  mm2s_cache_info
                                        & mm2s_desc_blength_v
                                        & mm2s_desc_blength_s   
                                        & MM2S_CMD_RSVD
                                        -- Command Tag
                                        & '0'
                                        & '0'
                                        & mm2s_desc_eof  -- Cat. EOF to CMD Tag
                                        & mm2s_desc_eof  -- Cat. IOC to CMD Tag (ioc changed to EOF)
                                        -- Command
                                        & mm2s_desc_baddress
                                        & mm2s_desc_sof
                                        & mm2s_desc_eof
                                        & MM2S_CMD_DSA
                                        & burst_type    -- key Hole Operation'1'               -- mm2s_desc_type IR#545697
                                        & mm2s_desc_blength;

end generate GEN_CMD_BTT_EQL_23;


-------------------------------------------------------------------------------
-- Counter for keepting track of pending commands/status in primary datamover
-- Use this to determine if primary datamover for mm2s is Idle.
-------------------------------------------------------------------------------
-- increment with each command written
count_incr  <= '1' when mm2s_cmnd_wr_i = '1' and desc_update_done = '0'
          else '0';

-- decrement with each status received
count_decr  <= '1' when mm2s_cmnd_wr_i = '0' and desc_update_done = '1'
          else '0';

-- count number of queued commands to keep track of what datamover is still
-- working on
--CMD2STS_COUNTER : process(m_axi_sg_aclk)
--    begin
--        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
--            if(m_axi_sg_aresetn = '0' or mm2s_stop = '1')then
--                cmnds_queued <= (others => '0');
--            elsif(count_incr = '1')then
--                cmnds_queued <= std_logic_vector(unsigned(cmnds_queued(COUNTER_WIDTH - 1 downto 0)) + 1);
--            elsif(count_decr = '1')then
--                cmnds_queued <= std_logic_vector(unsigned(cmnds_queued(COUNTER_WIDTH - 1 downto 0)) - 1);
--            end if;
--        end if;
--    end process CMD2STS_COUNTER;

QUEUE_COUNT : if C_SG_INCLUDE_DESC_QUEUE = 1 generate
begin

CMD2STS_COUNTER1 : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0' or mm2s_stop = '1')then
                cmnds_queued_shift <= (others => '0');
            elsif(count_incr = '1')then
                cmnds_queued_shift <= cmnds_queued_shift (2 downto 0) & '1';
            elsif(count_decr = '1')then
                cmnds_queued_shift <= '0' & cmnds_queued_shift (3 downto 1);
            end if;
        end if;
    end process CMD2STS_COUNTER1;

end generate QUEUE_COUNT;


NOQUEUE_COUNT : if C_SG_INCLUDE_DESC_QUEUE = 0 generate
begin

            -- coverage off
CMD2STS_COUNTER1 : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0' or mm2s_stop = '1')then
                cmnds_queued_shift(0) <= '0';
            elsif(count_incr = '1')then
                cmnds_queued_shift (0) <= '1';
            elsif(count_decr = '1')then
                cmnds_queued_shift (0) <= '0';
            end if;
        end if;
    end process CMD2STS_COUNTER1;

end generate NOQUEUE_COUNT;
            -- coverage on

-- Indicate status is idle when no cmnd/sts queued
--mm2s_sts_idle <= '1' when  cmnds_queued_shift = "0000"
--            else '0';

mm2s_sts_idle <= not  cmnds_queued_shift (0);

-------------------------------------------------------------------------------
-- Queue only the amount of commands that can be queued on descriptor update
-- else lock up can occur. Note datamover command fifo depth is set to number
-- of descriptors to queue.
-------------------------------------------------------------------------------
--QUEUE_MORE_PROCESS : process(m_axi_sg_aclk)
--    begin
--        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
--            if(m_axi_sg_aresetn = '0')then
--                queue_more <= '0';
--            elsif(cmnds_queued < std_logic_vector(to_unsigned(C_PRMY_CMDFIFO_DEPTH,COUNTER_WIDTH)))then
--                queue_more <= '1';
--            else
--                queue_more <= '0';
--            end if;
--        end if;
--    end process QUEUE_MORE_PROCESS;

QUEUE_MORE_PROCESS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                queue_more <= '0';
--            elsif(cmnds_queued_shift(3) /= '1') then -- < std_logic_vector(to_unsigned(C_PRMY_CMDFIFO_DEPTH,COUNTER_WIDTH)))then
--                queue_more <= '1';
            else
                queue_more <= not (cmnds_queued_shift(C_PRMY_CMDFIFO_DEPTH-1));
            end if;
        end if;
    end process QUEUE_MORE_PROCESS;

end implementation;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:    axi_dma_mm2s_cmdsts_if.vhd
-- Description: This entity is the descriptor fetch command and status inteface
--              for the Scatter Gather Engine AXI DataMover.
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;

-------------------------------------------------------------------------------
entity  axi_dma_mm2s_cmdsts_if is
    generic (
        C_M_AXI_MM2S_ADDR_WIDTH       : integer range 32 to 64        := 32;
        C_ENABLE_QUEUE                : integer range 0 to 1          := 1;
        C_ENABLE_MULTI_CHANNEL               : integer range 0 to 1          := 0
            -- Master AXI Memory Map Address Width for Scatter Gather R/W Port

    );
    port (
        -----------------------------------------------------------------------
        -- AXI Scatter Gather Interface
        -----------------------------------------------------------------------
        m_axi_sg_aclk               : in  std_logic                         ;                   --
        m_axi_sg_aresetn            : in  std_logic                         ;                   --
                                                                                                --
        -- Command write interface from mm2s sm                                                 --
        mm2s_cmnd_wr                : in  std_logic                         ;                   --
        mm2s_cmnd_data              : in  std_logic_vector                                      --
                                        ((C_M_AXI_MM2S_ADDR_WIDTH-32+2*32+CMD_BASE_WIDTH+46)-1 downto 0);  --
        mm2s_cmnd_pending           : out std_logic                         ;                   --
        mm2s_sts_received_clr       : in  std_logic                         ;                   --
        mm2s_sts_received           : out std_logic                         ;                   --
        mm2s_tailpntr_enble         : in  std_logic                         ;                   --
        mm2s_desc_cmplt             : in  std_logic                         ;                   --
                                                                                                --
        -- User Command Interface Ports (AXI Stream)                                            --
        s_axis_mm2s_cmd_tvalid      : out std_logic                         ;                   --
        s_axis_mm2s_cmd_tready      : in  std_logic                         ;                   --
        s_axis_mm2s_cmd_tdata       : out std_logic_vector                                      --
                                        ((C_M_AXI_MM2S_ADDR_WIDTH-32+2*32+CMD_BASE_WIDTH+46)-1 downto 0);  --
                                                                                                --
        -- User Status Interface Ports (AXI Stream)                                             --
        m_axis_mm2s_sts_tvalid      : in  std_logic                         ;                   --
        m_axis_mm2s_sts_tready      : out std_logic                         ;                   --
        m_axis_mm2s_sts_tdata       : in  std_logic_vector(7 downto 0)      ;                   --
        m_axis_mm2s_sts_tkeep       : in  std_logic_vector(0 downto 0)      ;                   --
                                                                                                --
        -- Scatter Gather Fetch Status                                                          --
        mm2s_err                    : in  std_logic                         ;                   --
        mm2s_done                   : out std_logic                         ;                   --
        mm2s_error                  : out std_logic                         ;                   --
        mm2s_interr                 : out std_logic                         ;                   --
        mm2s_slverr                 : out std_logic                         ;                   --
        mm2s_decerr                 : out std_logic                         ;                   --
        mm2s_tag                    : out std_logic_vector(3 downto 0)                          --

    );

end axi_dma_mm2s_cmdsts_if;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_dma_mm2s_cmdsts_if is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------

-- No Constants Declared

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
signal sts_tready       : std_logic := '0';
signal sts_received_i   : std_logic := '0';
signal stale_desc       : std_logic := '0';
signal log_status       : std_logic := '0';

signal mm2s_slverr_i    : std_logic := '0';
signal mm2s_decerr_i    : std_logic := '0';
signal mm2s_interr_i    : std_logic := '0';
signal mm2s_error_or    : std_logic := '0';

-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin

mm2s_slverr <= mm2s_slverr_i;
mm2s_decerr <= mm2s_decerr_i;
mm2s_interr <= mm2s_interr_i;

-- Stale descriptor if complete bit already set and in tail pointer mode.
stale_desc <= '1' when mm2s_desc_cmplt = '1' and mm2s_tailpntr_enble = '1'
         else '0';


-------------------------------------------------------------------------------
-- DataMover Command Interface
-------------------------------------------------------------------------------


-------------------------------------------------------------------------------
-- When command by fetch sm, drive descriptor fetch command to data mover.
-- Hold until data mover indicates ready.
-------------------------------------------------------------------------------
GEN_NO_HOLD_DATA : if C_ENABLE_QUEUE = 1 generate
begin
GEN_DATAMOVER_CMND : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                s_axis_mm2s_cmd_tvalid  <= '0';
          --      s_axis_mm2s_cmd_tdata   <= (others => '0');
                mm2s_cmnd_pending       <= '0';
            -- New command write and not flagged as stale descriptor
            elsif(mm2s_cmnd_wr = '1' and stale_desc = '0')then
                s_axis_mm2s_cmd_tvalid  <= '1';
          --      s_axis_mm2s_cmd_tdata   <= mm2s_cmnd_data;
                mm2s_cmnd_pending       <= '1';
            -- Clear flags when command excepted by datamover
            elsif(s_axis_mm2s_cmd_tready = '1')then
                s_axis_mm2s_cmd_tvalid  <= '0';
          --      s_axis_mm2s_cmd_tdata   <= (others => '0');
                mm2s_cmnd_pending       <= '0';

            end if;
        end if;
    end process GEN_DATAMOVER_CMND;

                s_axis_mm2s_cmd_tdata   <= mm2s_cmnd_data;

end generate GEN_NO_HOLD_DATA;

GEN_HOLD_DATA : if C_ENABLE_QUEUE = 0 generate
begin

GEN_DATAMOVER_CMND : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                s_axis_mm2s_cmd_tvalid  <= '0';
                s_axis_mm2s_cmd_tdata   <= (others => '0');
                mm2s_cmnd_pending       <= '0';
            -- New command write and not flagged as stale descriptor
            elsif(mm2s_cmnd_wr = '1' and stale_desc = '0')then
                s_axis_mm2s_cmd_tvalid  <= '1';
                s_axis_mm2s_cmd_tdata   <= mm2s_cmnd_data;
                mm2s_cmnd_pending       <= '1';
            -- Clear flags when command excepted by datamover
            elsif(s_axis_mm2s_cmd_tready = '1')then
                s_axis_mm2s_cmd_tvalid  <= '0';
                s_axis_mm2s_cmd_tdata   <= (others => '0');
                mm2s_cmnd_pending       <= '0';

            end if;
        end if;
    end process GEN_DATAMOVER_CMND;
             --   s_axis_mm2s_cmd_tdata   <= mm2s_cmnd_data;
end generate GEN_HOLD_DATA;
-------------------------------------------------------------------------------
-- DataMover Status Interface
-------------------------------------------------------------------------------
-- Drive ready low during reset to indicate not ready
REG_STS_READY : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                sts_tready <= '0';

            -- De-assert tready on acceptance of status to prevent
            -- over writing current status
            elsif(sts_tready = '1' and m_axis_mm2s_sts_tvalid = '1')then
                sts_tready <= '0';

            -- If not status received assert ready to datamover
            elsif(sts_received_i = '0') then
                sts_tready <= '1';
            end if;
        end if;
    end process REG_STS_READY;

-- Pass to DataMover
m_axis_mm2s_sts_tready <= sts_tready;

-------------------------------------------------------------------------------
-- Log status bits out of data mover.
-------------------------------------------------------------------------------
log_status <= '1' when m_axis_mm2s_sts_tvalid = '1' and sts_received_i = '0'
         else '0';

DATAMOVER_STS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                mm2s_done       <= '0';
                mm2s_slverr_i   <= '0';
                mm2s_decerr_i   <= '0';
                mm2s_interr_i   <= '0';
                mm2s_tag        <= (others => '0');
            -- Status valid, therefore capture status
            elsif(log_status = '1')then
                mm2s_done       <= m_axis_mm2s_sts_tdata(DATAMOVER_STS_CMDDONE_BIT);
                mm2s_slverr_i   <= m_axis_mm2s_sts_tdata(DATAMOVER_STS_SLVERR_BIT);
                mm2s_decerr_i   <= m_axis_mm2s_sts_tdata(DATAMOVER_STS_DECERR_BIT);
                mm2s_interr_i   <= m_axis_mm2s_sts_tdata(DATAMOVER_STS_INTERR_BIT);
                mm2s_tag        <= m_axis_mm2s_sts_tdata(DATAMOVER_STS_TAGMSB_BIT downto DATAMOVER_STS_TAGLSB_BIT);
            -- Only assert when valid
            else
                mm2s_done       <= '0';
                mm2s_slverr_i   <= '0';
                mm2s_decerr_i   <= '0';
                mm2s_interr_i   <= '0';
                mm2s_tag        <= (others => '0');
            end if;
        end if;
    end process DATAMOVER_STS;

-- Flag when status is received.  Used to hold status until sg if
-- can use status.  This only has meaning when SG Engine Queues are turned
-- on
STS_RCVD_FLAG : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            -- Clear flag on reset or sg_if status clear
            if(m_axi_sg_aresetn = '0' or mm2s_sts_received_clr = '1')then
                sts_received_i  <= '0';
            -- Status valid, therefore capture status
            elsif(m_axis_mm2s_sts_tvalid = '1' and sts_received_i = '0')then
                sts_received_i  <= '1';
            end if;
        end if;
    end process STS_RCVD_FLAG;

mm2s_sts_received    <= sts_received_i;


-------------------------------------------------------------------------------
-- Register global error from data mover.
-------------------------------------------------------------------------------
mm2s_error_or <= mm2s_slverr_i or mm2s_decerr_i or mm2s_interr_i;

-- Log errors into a global error output
MM2S_ERROR_PROCESS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                mm2s_error <= '0';
            -- If Datamover issues error on the transfer or if a stale descriptor is
            -- detected when in tailpointer mode then issue an error
            elsif((mm2s_error_or = '1')
               or (stale_desc = '1' and mm2s_cmnd_wr='1'))then
                mm2s_error <= '1';
            end if;
        end if;
    end process MM2S_ERROR_PROCESS;



end implementation;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:    axi_dma_mm2s_sts_mngr.vhd
-- Description: This entity mangages 'halt' and 'idle' status for the MM2S
--              channel
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library lib_cdc_v1_0_2;
library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;

-------------------------------------------------------------------------------
entity  axi_dma_mm2s_sts_mngr is
    generic (
        C_PRMRY_IS_ACLK_ASYNC        : integer range 0 to 1          := 0
            -- Primary MM2S/S2MM sync/async mode
            -- 0 = synchronous mode     - all clocks are synchronous
            -- 1 = asynchronous mode    - Any one of the 4 clock inputs is not
            --                            synchronous to the other
    );
    port (
        -- system signals
        m_axi_sg_aclk               : in  std_logic                         ;           --
        m_axi_sg_aresetn            : in  std_logic                         ;           --
                                                                                        --
        -- dma control and sg engine status signals                                     --
        mm2s_run_stop               : in  std_logic                         ;           --
                                                                                        --
        mm2s_ftch_idle              : in  std_logic                         ;           --
        mm2s_updt_idle              : in  std_logic                         ;           --
        mm2s_cmnd_idle              : in  std_logic                         ;           --
        mm2s_sts_idle               : in  std_logic                         ;           --
                                                                                        --
        -- stop and halt control/status                                                 --
        mm2s_stop                   : in  std_logic                         ;           --
        mm2s_halt_cmplt             : in  std_logic                         ;           --
                                                                                        --
        -- system state and control                                                     --
        mm2s_all_idle               : out std_logic                         ;           --
        mm2s_halted_clr             : out std_logic                         ;           --
        mm2s_halted_set             : out std_logic                         ;           --
        mm2s_idle_set               : out std_logic                         ;           --
        mm2s_idle_clr               : out std_logic                                     --

    );

end axi_dma_mm2s_sts_mngr;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_dma_mm2s_sts_mngr is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  ATTRIBUTE async_reg                      : STRING;
-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------

-- No Constants Declared

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
signal all_is_idle          : std_logic := '0';
signal all_is_idle_d1       : std_logic := '0';
signal all_is_idle_re       : std_logic := '0';
signal all_is_idle_fe       : std_logic := '0';
signal mm2s_datamover_idle  : std_logic := '0';

signal mm2s_halt_cmpt_d1_cdc_tig    : std_logic := '0';
signal mm2s_halt_cmpt_cdc_d2    : std_logic := '0';
  --ATTRIBUTE async_reg OF mm2s_halt_cmpt_d1_cdc_tig  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF mm2s_halt_cmpt_cdc_d2  : SIGNAL IS "true";

signal mm2s_halt_cmpt_d2    : std_logic := '0';

-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin

-- Everything is idle when everything is idle
all_is_idle <=  mm2s_ftch_idle
            and mm2s_updt_idle
            and mm2s_cmnd_idle
            and mm2s_sts_idle;

-- Pass out for soft reset use
mm2s_all_idle <= all_is_idle;



-------------------------------------------------------------------------------
-- For data mover halting look at halt complete to determine when halt
-- is done and datamover has completly halted.  If datamover not being
-- halted then can ignore flag thus simply flag as idle.
-------------------------------------------------------------------------------
GEN_FOR_ASYNC : if C_PRMRY_IS_ACLK_ASYNC = 1 generate
begin
    -- Double register to secondary clock domain.  This is sufficient
    -- because halt_cmplt will remain asserted until detected in
    -- reset module in secondary clock domain.
AWVLD_CDC_TO : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => mm2s_halt_cmplt,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => m_axi_sg_aclk,
        scndry_resetn              => '0',
        scndry_out                 => mm2s_halt_cmpt_cdc_d2,
        scndry_vect_out            => open
    );


--    REG_TO_SECONDARY : process(m_axi_sg_aclk)
--        begin
--            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
--            --    if(m_axi_sg_aresetn = '0')then
--            --        mm2s_halt_cmpt_d1_cdc_tig <= '0';
--            --        mm2s_halt_cmpt_d2 <= '0';
--            --    else
--                    mm2s_halt_cmpt_d1_cdc_tig <= mm2s_halt_cmplt;
--                    mm2s_halt_cmpt_cdc_d2 <= mm2s_halt_cmpt_d1_cdc_tig;
--            --    end if;
--            end if;
--        end process REG_TO_SECONDARY;

                    mm2s_halt_cmpt_d2 <= mm2s_halt_cmpt_cdc_d2;

end generate GEN_FOR_ASYNC;

GEN_FOR_SYNC : if C_PRMRY_IS_ACLK_ASYNC = 0 generate
begin
    -- No clock crossing required therefore simple pass through
    mm2s_halt_cmpt_d2 <= mm2s_halt_cmplt;

end generate GEN_FOR_SYNC;




mm2s_datamover_idle  <= '1' when (mm2s_stop = '1' and mm2s_halt_cmpt_d2 = '1')
                              or (mm2s_stop = '0')
                   else '0';

-------------------------------------------------------------------------------
-- Set halt bit if run/stop cleared and all processes are idle
-------------------------------------------------------------------------------
HALT_PROCESS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                mm2s_halted_set <= '0';

            -- DMACR.Run/Stop is cleared, all processes are idle, datamover halt cmplted
            elsif(mm2s_run_stop = '0' and all_is_idle = '1' and mm2s_datamover_idle = '1')then
                mm2s_halted_set <= '1';
            else
                mm2s_halted_set <= '0';
            end if;
        end if;
    end process HALT_PROCESS;

-------------------------------------------------------------------------------
-- Clear halt bit if run/stop is set and SG engine begins to fetch descriptors
-------------------------------------------------------------------------------
NOT_HALTED_PROCESS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                mm2s_halted_clr <= '0';
            elsif(mm2s_run_stop = '1')then
                mm2s_halted_clr <= '1';
            else
                mm2s_halted_clr <= '0';
            end if;
        end if;
    end process NOT_HALTED_PROCESS;

-------------------------------------------------------------------------------
-- Register ALL is Idle to create rising and falling edges on idle flag
-------------------------------------------------------------------------------
IDLE_REG_PROCESS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                all_is_idle_d1 <= '0';
            else
                all_is_idle_d1 <= all_is_idle;
            end if;
        end if;
    end process IDLE_REG_PROCESS;

all_is_idle_re  <= all_is_idle and not all_is_idle_d1;
all_is_idle_fe  <= not all_is_idle and all_is_idle_d1;

-- Set or Clear IDLE bit in DMASR
mm2s_idle_set <= all_is_idle_re and mm2s_run_stop;
mm2s_idle_clr <= all_is_idle_fe;


end implementation;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:          axi_dma_mm2s_cntrl_strm.vhd
-- Description: This entity is MM2S control stream logic
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;

library lib_pkg_v1_0_2;
use lib_pkg_v1_0_2.lib_pkg.clog2;
use lib_pkg_v1_0_2.lib_pkg.max2;

library lib_fifo_v1_0_8;

-------------------------------------------------------------------------------
entity  axi_dma_mm2s_cntrl_strm is
    generic(
        C_PRMRY_IS_ACLK_ASYNC           : integer range 0 to 1      := 0;
            -- Primary MM2S/S2MM sync/async mode
            -- 0 = synchronous mode     - all clocks are synchronous
            -- 1 = asynchronous mode    - Primary data path channels (MM2S and S2MM)
            --                            run asynchronous to AXI Lite, DMA Control,
            --                            and SG.

        C_PRMY_CMDFIFO_DEPTH        : integer range 1 to 16         := 1;
            -- Depth of DataMover command FIFO

        C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH : integer range 32 to 32    := 32;
            -- Master AXI Control Stream Data Width

        C_FAMILY                        : string                    := "virtex7"
            -- Target FPGA Device Family
    );
    port (
        -- Secondary clock / reset
        m_axi_sg_aclk               : in  std_logic                         ;           --
        m_axi_sg_aresetn            : in  std_logic                         ;           --
                                                                                        --
        -- Primary clock / reset                                                        --
        axi_prmry_aclk              : in  std_logic                         ;           --
        p_reset_n                   : in  std_logic                         ;           --
                                                                                        --
        -- MM2S Error                                                                   --
        mm2s_stop                   : in  std_logic                         ;           --
                                                                                        --
        -- Control Stream FIFO write signals (from axi_dma_mm2s_sg_if)                  --
        cntrlstrm_fifo_wren         : in  std_logic                         ;           --
        cntrlstrm_fifo_din          : in  std_logic_vector                              --
                                        (C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH downto 0);     --
        cntrlstrm_fifo_full         : out std_logic                         ;           --
                                                                                        --
                                                                                        --
        -- Memory Map to Stream Control Stream Interface                                --
        m_axis_mm2s_cntrl_tdata     : out std_logic_vector                              --
                                        (C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH-1 downto 0);   --
        m_axis_mm2s_cntrl_tkeep     : out std_logic_vector                              --
                                        ((C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH/8)-1 downto 0);--
        m_axis_mm2s_cntrl_tvalid    : out std_logic                         ;           --
        m_axis_mm2s_cntrl_tready    : in  std_logic                         ;           --
        m_axis_mm2s_cntrl_tlast     : out std_logic                                     --



    );
end axi_dma_mm2s_cntrl_strm;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_dma_mm2s_cntrl_strm is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------

-- Number of words deep fifo needs to be
-- Only 5 app fields, but set to 8 so depth is a power of 2
constant CNTRL_FIFO_DEPTH       : integer := max2(16,8 * C_PRMY_CMDFIFO_DEPTH);


-- Width of fifo rd and wr counts - only used for proper fifo operation
constant CNTRL_FIFO_CNT_WIDTH   : integer   := clog2(CNTRL_FIFO_DEPTH+1);

constant USE_LOGIC_FIFOS        : integer   := 0; -- Use Logic FIFOs
constant USE_BRAM_FIFOS         : integer   := 1; -- Use BRAM FIFOs

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
-- FIFO signals
signal cntrl_fifo_rden  : std_logic := '0';
signal cntrl_fifo_empty : std_logic := '0';
signal cntrl_fifo_dout  : std_logic_vector
                            (C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH downto 0) := (others => '0');
signal cntrl_fifo_dvalid: std_logic := '0';

signal cntrl_tdata      : std_logic_vector
                            (C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH-1 downto 0) := (others => '0');
signal cntrl_tkeep      : std_logic_vector
                            ((C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH/8)-1 downto 0) := (others => '0');
signal cntrl_tvalid     : std_logic := '0';
signal cntrl_tready     : std_logic := '0';
signal cntrl_tlast      : std_logic := '0';
signal sinit            : std_logic := '0';

signal m_valid          : std_logic := '0';
signal m_ready          : std_logic := '0';
signal m_data           : std_logic_vector(C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH-1 downto 0) := (others => '0');
signal m_strb           : std_logic_vector((C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH/8)-1 downto 0) := (others => '0');
signal m_last           : std_logic := '0';

signal skid_rst         : std_logic := '0';
-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin
-- All bytes always valid
cntrl_tkeep <= (others => '1');


-- Primary Clock is synchronous to Secondary Clock therfore
-- instantiate a sync fifo.
GEN_SYNC_FIFO : if C_PRMRY_IS_ACLK_ASYNC = 0 generate
signal mm2s_stop_d1     : std_logic := '0';
signal mm2s_stop_re     : std_logic := '0';
signal xfer_in_progress : std_logic := '0';
begin
    -- reset on hard reset or mm2s stop
    sinit   <= not m_axi_sg_aresetn or mm2s_stop;

    -- Generate Synchronous FIFO
    I_CNTRL_FIFO : entity lib_fifo_v1_0_8.sync_fifo_fg
    generic map (
        C_FAMILY                =>  C_FAMILY                ,
        C_MEMORY_TYPE           =>  USE_LOGIC_FIFOS,
        C_WRITE_DATA_WIDTH      =>  C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH + 1,
        C_WRITE_DEPTH           =>  CNTRL_FIFO_DEPTH       ,
        C_READ_DATA_WIDTH       =>  C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH + 1,
        C_READ_DEPTH            =>  CNTRL_FIFO_DEPTH       ,
        C_PORTS_DIFFER          =>  0,
        C_HAS_DCOUNT            =>  1, --req for proper fifo operation
        C_DCOUNT_WIDTH          =>  CNTRL_FIFO_CNT_WIDTH,
        C_HAS_ALMOST_FULL       =>  0,
        C_HAS_RD_ACK            =>  0,
        C_HAS_RD_ERR            =>  0,
        C_HAS_WR_ACK            =>  0,
        C_HAS_WR_ERR            =>  0,
        C_RD_ACK_LOW            =>  0,
        C_RD_ERR_LOW            =>  0,
        C_WR_ACK_LOW            =>  0,
        C_WR_ERR_LOW            =>  0,
        C_PRELOAD_REGS          =>  1,-- 1 = first word fall through
        C_PRELOAD_LATENCY       =>  0 -- 0 = first word fall through
 --       C_USE_EMBEDDED_REG      =>  1 -- 0 ;
    )
    port map (

        Clk             =>  m_axi_sg_aclk       ,
        Sinit           =>  sinit               ,
        Din             =>  cntrlstrm_fifo_din  ,
        Wr_en           =>  cntrlstrm_fifo_wren ,
        Rd_en           =>  cntrl_fifo_rden     ,
        Dout            =>  cntrl_fifo_dout     ,
        Full            =>  cntrlstrm_fifo_full ,
        Empty           =>  cntrl_fifo_empty    ,
        Almost_full     =>  open                ,
        Data_count      =>  open                ,
        Rd_ack          =>  open                ,
        Rd_err          =>  open                ,
        Wr_ack          =>  open                ,
        Wr_err          =>  open

    );

    -----------------------------------------------------------------------
    -- Control Stream OUT Side
    -----------------------------------------------------------------------
    -- Read if fifo is not empty and target is ready
    cntrl_fifo_rden  <= not cntrl_fifo_empty
                        and cntrl_tready;

    -- Drive valid if fifo is not empty or in the middle
    -- of transfer and stop issued.
    cntrl_tvalid  <= not cntrl_fifo_empty
                    or (xfer_in_progress and mm2s_stop_re);

    -- Pass data out to control channel with MSB driving tlast
    cntrl_tlast   <= (cntrl_tvalid and cntrl_fifo_dout(C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH))
                    or (xfer_in_progress and mm2s_stop_re);

    cntrl_tdata   <= cntrl_fifo_dout(C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH-1 downto 0);

    -- Register stop to create re pulse for cleaning shutting down
    -- stream out during soft reset.
    REG_STOP : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    mm2s_stop_d1    <= '0';
                else
                    mm2s_stop_d1    <= mm2s_stop;
                end if;
            end if;
        end process REG_STOP;

    mm2s_stop_re <= mm2s_stop and not mm2s_stop_d1;

    -------------------------------------------------------------
    -- Flag transfer in progress. If xfer in progress then
    -- a fake tlast and tvalid need to be asserted during soft
    -- reset else no need of tlast.
    -------------------------------------------------------------
    TRANSFER_IN_PROGRESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(cntrl_tlast = '1' and cntrl_tvalid = '1' and cntrl_tready = '1')then
                    xfer_in_progress <= '0';
                elsif(xfer_in_progress = '0' and cntrl_tvalid = '1')then
                    xfer_in_progress <= '1';
                end if;
            end if;
        end process TRANSFER_IN_PROGRESS;

    skid_rst   <= not m_axi_sg_aresetn;

    ---------------------------------------------------------------------------
    -- Buffer AXI Signals
    ---------------------------------------------------------------------------
    CNTRL_SKID_BUF_I : entity axi_dma_v7_1_13.axi_dma_skid_buf
        generic map(
            C_WDATA_WIDTH           => C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH
        )
        port map(
            -- System Ports
            ACLK            => m_axi_sg_aclk                          ,
            ARST            => skid_rst                                 ,

            skid_stop       => mm2s_stop_re                             ,

            -- Slave Side (Stream Data Input)
            S_VALID         => cntrl_tvalid                             ,
            S_READY         => cntrl_tready                             ,
            S_Data          => cntrl_tdata                              ,
            S_STRB          => cntrl_tkeep                              ,
            S_Last          => cntrl_tlast                              ,

            -- Master Side (Stream Data Output
            M_VALID         => m_axis_mm2s_cntrl_tvalid                 ,
            M_READY         => m_axis_mm2s_cntrl_tready                 ,
            M_Data          => m_axis_mm2s_cntrl_tdata                  ,
            M_STRB          => m_axis_mm2s_cntrl_tkeep                  ,
            M_Last          => m_axis_mm2s_cntrl_tlast
        );


end generate GEN_SYNC_FIFO;

-- Primary Clock is asynchronous to Secondary Clock therfore
-- instantiate an async fifo.
GEN_ASYNC_FIFO : if C_PRMRY_IS_ACLK_ASYNC = 1 generate
signal mm2s_stop_reg        : std_logic := '0'; -- CR605883
signal p_mm2s_stop_d1       : std_logic := '0';
signal p_mm2s_stop_d2       : std_logic := '0';
signal p_mm2s_stop_d3       : std_logic := '0';
signal p_mm2s_stop_re       : std_logic := '0';
signal xfer_in_progress     : std_logic := '0';
begin

    -- reset on hard reset, soft reset, or mm2s error
    sinit   <= not p_reset_n or p_mm2s_stop_d2;

    -- Generate Asynchronous FIFO
    I_CNTRL_STRM_FIFO : entity axi_dma_v7_1_13.axi_dma_afifo_autord
      generic map(
         C_DWIDTH        => C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH + 1  ,
-- Temp work around for issue in async fifo model
--         C_DEPTH         => CNTRL_FIFO_DEPTH                    ,
--         C_CNT_WIDTH     => CNTRL_FIFO_CNT_WIDTH                ,
         C_DEPTH         => 31                                  ,
         C_CNT_WIDTH     => 5                                   ,
         C_USE_BLKMEM    => USE_LOGIC_FIFOS                     ,
         C_FAMILY        => C_FAMILY
        )
      port map(
        -- Inputs
         AFIFO_Ainit                => sinit                    ,
         AFIFO_Wr_clk               => m_axi_sg_aclk            ,
         AFIFO_Wr_en                => cntrlstrm_fifo_wren      ,
         AFIFO_Din                  => cntrlstrm_fifo_din       ,
         AFIFO_Rd_clk               => axi_prmry_aclk           ,
         AFIFO_Rd_en                => cntrl_fifo_rden          ,
         AFIFO_Clr_Rd_Data_Valid    => '0'                      ,

        -- Outputs
         AFIFO_DValid               => cntrl_fifo_dvalid        ,
         AFIFO_Dout                 => cntrl_fifo_dout          ,
         AFIFO_Full                 => cntrlstrm_fifo_full      ,
         AFIFO_Empty                => cntrl_fifo_empty         ,
         AFIFO_Almost_full          => open                     ,
         AFIFO_Almost_empty         => open                     ,
         AFIFO_Wr_count             => open                     ,
         AFIFO_Rd_count             => open                     ,
         AFIFO_Corr_Rd_count        => open                     ,
         AFIFO_Corr_Rd_count_minus1 => open                     ,
         AFIFO_Rd_ack               => open
        );


    -----------------------------------------------------------------------
    -- Control Stream OUT Side
    -----------------------------------------------------------------------
    -- Read if fifo is not empty and target is ready
    cntrl_fifo_rden <= not cntrl_fifo_empty        -- fifo has data
                       and cntrl_tready;           -- target ready


    -- Drive valid if fifo is not empty or in the middle
    -- of transfer and stop issued.
    cntrl_tvalid  <= cntrl_fifo_dvalid
                    or (xfer_in_progress and p_mm2s_stop_re);

    -- Pass data out to control channel with MSB driving tlast
    cntrl_tlast   <= cntrl_tvalid and cntrl_fifo_dout(C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH);

    cntrl_tdata   <= cntrl_fifo_dout(C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH-1 downto 0);

    -- CR605883
    -- Register stop to provide pure FF output for synchronizer
    REG_STOP : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    mm2s_stop_reg <= '0';
                else
                    mm2s_stop_reg <= mm2s_stop;
                end if;
            end if;
        end process REG_STOP;


    -- Double/triple register mm2s error into primary clock domain
    -- Triple register to give two versions with min double reg for use
    -- in rising edge detection.
    REG_ERR2PRMRY : process(axi_prmry_aclk)
        begin
            if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
                if(p_reset_n = '0')then
                    p_mm2s_stop_d1 <= '0';
                    p_mm2s_stop_d2 <= '0';
                    p_mm2s_stop_d3 <= '0';
                else
                    --p_mm2s_stop_d1 <= mm2s_stop;
                    p_mm2s_stop_d1 <= mm2s_stop_reg;
                    p_mm2s_stop_d2 <= p_mm2s_stop_d1;
                    p_mm2s_stop_d3 <= p_mm2s_stop_d2;
                end if;
            end if;
        end process REG_ERR2PRMRY;

    -- Rising edge pulse for use in shutting down stream output
    p_mm2s_stop_re <= p_mm2s_stop_d2 and not p_mm2s_stop_d3;

    -------------------------------------------------------------
    -- Flag transfer in progress. If xfer in progress then
    -- a fake tlast needs to be asserted during soft reset.
    -- else no need of tlast.
    -------------------------------------------------------------
    TRANSFER_IN_PROGRESS : process(axi_prmry_aclk)
        begin
            if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
                if(cntrl_tlast = '1' and cntrl_tvalid = '1' and cntrl_tready = '1')then
                    xfer_in_progress <= '0';
                elsif(xfer_in_progress = '0' and cntrl_tvalid = '1')then
                    xfer_in_progress <= '1';
                end if;
            end if;
        end process TRANSFER_IN_PROGRESS;


    skid_rst   <= not p_reset_n;

    ---------------------------------------------------------------------------
    -- Buffer AXI Signals
    ---------------------------------------------------------------------------
    CNTRL_SKID_BUF_I : entity axi_dma_v7_1_13.axi_dma_skid_buf
        generic map(
            C_WDATA_WIDTH           => C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH
        )
        port map(
            -- System Ports
            ACLK            => axi_prmry_aclk                           ,
            ARST            => skid_rst                                 ,

            skid_stop       => p_mm2s_stop_re                           ,

            -- Slave Side (Stream Data Input)
            S_VALID         => cntrl_tvalid                             ,
            S_READY         => cntrl_tready                             ,
            S_Data          => cntrl_tdata                              ,
            S_STRB          => cntrl_tkeep                              ,
            S_Last          => cntrl_tlast                              ,

            -- Master Side (Stream Data Output
            M_VALID         => m_axis_mm2s_cntrl_tvalid                 ,
            M_READY         => m_axis_mm2s_cntrl_tready                 ,
            M_Data          => m_axis_mm2s_cntrl_tdata                  ,
            M_STRB          => m_axis_mm2s_cntrl_tkeep                  ,
            M_Last          => m_axis_mm2s_cntrl_tlast
        );


end generate GEN_ASYNC_FIFO;


end implementation;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:          axi_dma_mm2s_mngr.vhd
-- Description: This entity is the top level entity for the AXI DMA MM2S
--              manager.
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;

-------------------------------------------------------------------------------
entity  axi_dma_mm2s_mngr is
    generic(

        C_PRMRY_IS_ACLK_ASYNC       : integer range 0 to 1         := 0;
            -- Primary MM2S/S2MM sync/async mode
            -- 0 = synchronous mode     - all clocks are synchronous
            -- 1 = asynchronous mode    - Primary data path channels (MM2S and S2MM)
            --                            run asynchronous to AXI Lite, DMA Control,
            --                            and SG.

        C_PRMY_CMDFIFO_DEPTH        : integer range 1 to 16         := 1;
            -- Depth of DataMover command FIFO

        -----------------------------------------------------------------------
        -- Scatter Gather Parameters
        -----------------------------------------------------------------------
        C_INCLUDE_SG                : integer range 0 to 1          := 1;
            -- Include or Exclude the Scatter Gather Engine
            -- 0 = Exclude SG Engine - Enables Simple DMA Mode
            -- 1 = Include SG Engine - Enables Scatter Gather Mode

        C_SG_INCLUDE_STSCNTRL_STRM      : integer range 0 to 1      := 1;
            -- Include or Exclude AXI Status and AXI Control Streams
            -- 0 = Exclude Status and Control Streams
            -- 1 = Include Status and Control Streams

        C_SG_INCLUDE_DESC_QUEUE     : integer range 0 to 1          := 0;
            -- Include or Exclude Scatter Gather Descriptor Queuing
            -- 0 = Exclude SG Descriptor Queuing
            -- 1 = Include SG Descriptor Queuing

        C_SG_LENGTH_WIDTH               : integer range 8 to 23     := 14;
            -- Descriptor Buffer Length, Transferred Bytes, and Status Stream
            -- Rx Length Width.  Indicates the least significant valid bits of
            -- descriptor buffer length, transferred bytes, or Rx Length value
            -- in the status word coincident with tlast.

        C_M_AXI_SG_ADDR_WIDTH           : integer range 32 to 64    := 32;
            -- Master AXI Memory Map Address Width for Scatter Gather R/W Port

        C_M_AXIS_SG_TDATA_WIDTH         : integer range 32 to 32    := 32;
            -- AXI Master Stream in for descriptor fetch

        C_S_AXIS_UPDPTR_TDATA_WIDTH : integer range 32 to 32     := 32;
            -- 32 Update Status Bits

        C_S_AXIS_UPDSTS_TDATA_WIDTH : integer range 33 to 33     := 33;
            -- 1 IOC bit + 32 Update Status Bits

        C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH : integer range 32 to 32    := 32;
            -- Master AXI Control Stream Data Width

        -----------------------------------------------------------------------
        -- Memory Map to Stream (MM2S) Parameters
        -----------------------------------------------------------------------
        C_INCLUDE_MM2S                  : integer range 0 to 1      := 1;
            -- Include or exclude MM2S primary data path
            -- 0 = Exclude MM2S primary data path
            -- 1 = Include MM2S primary data path

        C_M_AXI_MM2S_ADDR_WIDTH         : integer range 32 to 64    := 32;
            -- Master AXI Memory Map Address Width for MM2S Read Port
 
        C_ENABLE_MULTI_CHANNEL                 : integer range 0 to 1 := 0;
        C_MICRO_DMA                            : integer range 0 to 1 := 0;

        C_FAMILY                        : string            := "virtex7"
            -- Target FPGA Device Family
    );
    port (

        -- Secondary Clock and Reset
        m_axi_sg_aclk               : in  std_logic                         ;                 --
        m_axi_sg_aresetn            : in  std_logic                         ;                 --
                                                                                              --
        -- Primary Clock and Reset                                                            --
        axi_prmry_aclk              : in  std_logic                         ;                 --
        p_reset_n                   : in  std_logic                         ;                 --
                                                                                              --
        soft_reset                  : in  std_logic                         ;                 --
                                                                                              --
        -- MM2S Control and Status                                                            --
        mm2s_run_stop               : in  std_logic                         ;                 --
        mm2s_keyhole                : in  std_logic                         ;
        mm2s_halted                 : in  std_logic                         ;                 --
        mm2s_ftch_idle              : in  std_logic                         ;                 --
        mm2s_updt_idle              : in  std_logic                         ;                 --
        mm2s_ftch_err_early         : in  std_logic                         ;                 --
        mm2s_ftch_stale_desc        : in  std_logic                         ;                 --
        mm2s_tailpntr_enble         : in  std_logic                         ;                 --
        mm2s_halt                   : in  std_logic                         ;                 --
        mm2s_halt_cmplt             : in  std_logic                         ;                 --
        mm2s_halted_clr             : out std_logic                         ;                 --
        mm2s_halted_set             : out std_logic                         ;                 --
        mm2s_idle_set               : out std_logic                         ;                 --
        mm2s_idle_clr               : out std_logic                         ;                 --
        mm2s_new_curdesc            : out std_logic_vector                                    --
                                            (C_M_AXI_SG_ADDR_WIDTH-1 downto 0);               --
        mm2s_new_curdesc_wren       : out std_logic                         ;                 --
        mm2s_stop                   : out std_logic                         ;                 --
        mm2s_desc_flush             : out std_logic                         ;                 --
        cntrl_strm_stop             : out std_logic                         ;
        mm2s_all_idle               : out std_logic                         ;                 --
                                                                                              --
        mm2s_error                  : out std_logic                         ;                 --
        s2mm_error                  : in  std_logic                         ;                 --

        -- Simple DMA Mode Signals
        mm2s_sa                     : in  std_logic_vector                                    --
                                        (C_M_AXI_MM2S_ADDR_WIDTH-1 downto 0);                 --
        mm2s_length_wren            : in  std_logic                         ;                 --
        mm2s_length                 : in  std_logic_vector                                    --
                                        (C_SG_LENGTH_WIDTH-1 downto 0)      ;                 --
        mm2s_smple_done             : out std_logic                         ;                 --
        mm2s_interr_set             : out std_logic                         ;                 --
        mm2s_slverr_set             : out std_logic                         ;                 --
        mm2s_decerr_set             : out std_logic                         ;                 --
        
        m_axis_mm2s_aclk            : in std_logic;
        mm2s_strm_tlast             : in std_logic;
        mm2s_strm_tready            : in std_logic;
        mm2s_axis_info              : out std_logic_vector
                                        (13 downto 0);
                                                                                              --
        -- SG MM2S Descriptor Fetch AXI Stream In                                             --
        m_axis_mm2s_ftch_tdata      : in  std_logic_vector                                    --
                                        (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0);                 --
        m_axis_mm2s_ftch_tvalid     : in  std_logic                         ;                 --
        m_axis_mm2s_ftch_tready     : out std_logic                         ;                 --
        m_axis_mm2s_ftch_tlast      : in  std_logic                         ;                 --

        m_axis_mm2s_ftch_tdata_new      : in  std_logic_vector                                    --
                                        (96+31*0+(0+2)*(C_M_AXI_SG_ADDR_WIDTH-32) downto 0);                 --
        m_axis_mm2s_ftch_tdata_mcdma_new      : in  std_logic_vector                                    --
                                        (63 downto 0);                 --
        m_axis_mm2s_ftch_tvalid_new     : in  std_logic                         ;                 --
        m_axis_ftch1_desc_available  : in std_logic;
                                                                                              --
        -- SG MM2S Descriptor Update AXI Stream Out                                           --
        s_axis_mm2s_updtptr_tdata   : out std_logic_vector                                    --
                                     (C_M_AXI_SG_ADDR_WIDTH-1 downto 0);                --
        s_axis_mm2s_updtptr_tvalid  : out std_logic                         ;                 --
        s_axis_mm2s_updtptr_tready  : in  std_logic                         ;                 --
        s_axis_mm2s_updtptr_tlast   : out std_logic                         ;                 --
                                                                                              --
        s_axis_mm2s_updtsts_tdata   : out std_logic_vector                                    --
                                     (C_S_AXIS_UPDSTS_TDATA_WIDTH-1 downto 0);                --
        s_axis_mm2s_updtsts_tvalid  : out std_logic                         ;                 --
        s_axis_mm2s_updtsts_tready  : in  std_logic                         ;                 --
        s_axis_mm2s_updtsts_tlast   : out std_logic                         ;                 --
                                                                                              --
        -- User Command Interface Ports (AXI Stream)                                          --
        s_axis_mm2s_cmd_tvalid      : out std_logic                         ;                 --
        s_axis_mm2s_cmd_tready      : in  std_logic                         ;                 --
        s_axis_mm2s_cmd_tdata       : out std_logic_vector                                    --
                                        ((C_M_AXI_MM2S_ADDR_WIDTH-32+2*32+CMD_BASE_WIDTH+46)-1 downto 0);--
                                                                                              --
        -- User Status Interface Ports (AXI Stream)                                           --
        m_axis_mm2s_sts_tvalid      : in  std_logic                         ;                 --
        m_axis_mm2s_sts_tready      : out std_logic                         ;                 --
        m_axis_mm2s_sts_tdata       : in  std_logic_vector(7 downto 0)      ;                 --
        m_axis_mm2s_sts_tkeep       : in  std_logic_vector(0 downto 0)      ;                 --
        mm2s_err                    : in  std_logic                         ;                 --
                                                                                              --
        ftch_error                  : in  std_logic                         ;                 --
        updt_error                  : in  std_logic                         ;                 --
                                                                                              --
        -- Memory Map to Stream Control Stream Interface                                      --
        m_axis_mm2s_cntrl_tdata     : out std_logic_vector                                    --
                                        (C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH-1 downto 0);         --
        m_axis_mm2s_cntrl_tkeep     : out std_logic_vector                                    --
                                        ((C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH/8)-1 downto 0);     --
        m_axis_mm2s_cntrl_tvalid    : out std_logic                         ;                 --
        m_axis_mm2s_cntrl_tready    : in  std_logic                         ;                 --
        m_axis_mm2s_cntrl_tlast     : out std_logic                                           --

    );

end axi_dma_mm2s_mngr;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_dma_mm2s_mngr is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";



-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------

-- No Constants Declared

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
-- Primary DataMover Command signals
signal mm2s_cmnd_wr                 : std_logic := '0';
signal mm2s_cmnd_data               : std_logic_vector
                                        ((C_M_AXI_MM2S_ADDR_WIDTH-32+2*32+CMD_BASE_WIDTH+46)-1 downto 0) := (others => '0');
signal mm2s_cmnd_pending            : std_logic := '0';
-- Primary DataMover Status signals
signal mm2s_done                    : std_logic := '0';
signal mm2s_stop_i                  : std_logic := '0';
signal mm2s_interr                  : std_logic := '0';
signal mm2s_slverr                  : std_logic := '0';
signal mm2s_decerr                  : std_logic := '0';
signal mm2s_tag                     : std_logic_vector(3 downto 0) := (others => '0');
signal dma_mm2s_error               : std_logic := '0';
signal soft_reset_d1                : std_logic := '0';
signal soft_reset_d2                : std_logic := '0';
signal soft_reset_re                : std_logic := '0';
signal mm2s_error_i                 : std_logic := '0';
--signal cntrl_strm_stop              : std_logic := '0';
signal mm2s_halted_set_i            : std_logic := '0';

signal mm2s_sts_received_clr        : std_logic := '0';
signal mm2s_sts_received            : std_logic := '0';

signal mm2s_cmnd_idle               : std_logic := '0';
signal mm2s_sts_idle                : std_logic := '0';

-- Scatter Gather Interface signals
signal desc_fetch_req               : std_logic := '0';
signal desc_fetch_done              : std_logic := '0';
signal desc_fetch_done_del              : std_logic := '0';
signal desc_update_req              : std_logic := '0';
signal desc_update_done             : std_logic := '0';
signal desc_available               : std_logic := '0';
signal packet_in_progress           : std_logic := '0';

signal mm2s_desc_baddress           : std_logic_vector(C_M_AXI_MM2S_ADDR_WIDTH-1 downto 0)  := (others => '0');
signal mm2s_desc_blength            : std_logic_vector(BUFFER_LENGTH_WIDTH-1 downto 0)    := (others => '0');
signal mm2s_desc_blength_v            : std_logic_vector(BUFFER_LENGTH_WIDTH-1 downto 0)    := (others => '0');
signal mm2s_desc_blength_s            : std_logic_vector(BUFFER_LENGTH_WIDTH-1 downto 0)    := (others => '0');
signal mm2s_desc_eof                : std_logic := '0';
signal mm2s_desc_sof                : std_logic := '0';
signal mm2s_desc_cmplt              : std_logic := '0';
signal mm2s_desc_info               : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH-1 downto 0)    := (others => '0');
signal mm2s_desc_app0               : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH-1 downto 0)    := (others => '0');
signal mm2s_desc_app1               : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH-1 downto 0)    := (others => '0');
signal mm2s_desc_app2               : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH-1 downto 0)    := (others => '0');
signal mm2s_desc_app3               : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH-1 downto 0)    := (others => '0');
signal mm2s_desc_app4               : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH-1 downto 0)    := (others => '0');
signal mm2s_desc_info_int               : std_logic_vector(13 downto 0)    := (others => '0');
signal mm2s_strm_tlast_int          : std_logic;
signal rd_en_hold, rd_en_hold_int   : std_logic;

-- Control Stream Fifo write signals
signal cntrlstrm_fifo_wren          : std_logic := '0';
signal cntrlstrm_fifo_full          : std_logic := '0';
signal cntrlstrm_fifo_din           : std_logic_vector(C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH downto 0) := (others => '0');

signal info_fifo_full               : std_logic;
signal info_fifo_empty              : std_logic;

signal updt_pending                 : std_logic := '0';
signal mm2s_cmnd_wr_1               : std_logic := '0'; 
signal fifo_rst : std_logic;


signal fifo_empty : std_logic;
signal fifo_empty_first  : std_logic;
signal fifo_empty_first1  : std_logic;
signal first_read_pulse : std_logic;
signal fifo_read : std_logic;

-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin


-------------------------------------------------------------------------------
-- Include MM2S State Machine and support logic
-------------------------------------------------------------------------------
GEN_MM2S_DMA_CONTROL : if C_INCLUDE_MM2S = 1 generate
begin

    -- Pass out to register module
    mm2s_halted_set <= mm2s_halted_set_i;


    -------------------------------------------------------------------------------
    -- Graceful shut down logic
    -------------------------------------------------------------------------------
    -- Error from DataMover (DMAIntErr, DMADecErr, or DMASlvErr) or SG Update error
    -- or SG Fetch error, or Stale Descriptor Error
    mm2s_error_i <= dma_mm2s_error              -- Primary data mover reports error
                    or updt_error               -- SG Update engine reports error
                    or ftch_error               -- SG Fetch engine reports error
                    or mm2s_ftch_err_early      -- SG Fetch engine reports early error on mm2s
                    or mm2s_ftch_stale_desc;    -- SG Fetch stale descriptor error

    -- pass out to shut down s2mm
    mm2s_error <= mm2s_error_i;

    -- Clear run/stop and stop state machines due to errors or soft reset
    -- Error based on datamover error report or sg update error or sg fetch error
    -- SG update error and fetch error included because need to shut down, no way
    -- to update descriptors on sg update error and on fetch error descriptor
    -- data is corrupt therefor do not want to issue the xfer command to primary datamover
--CR#566306 status for both mm2s and s2mm datamover are masked during shutdown therefore
-- need to stop all processes regardless of the source of the error.
--    mm2s_stop_i    <= mm2s_error                -- Error
--                   or soft_reset;               -- Soft Reset issued
    mm2s_stop_i    <= mm2s_error_i              -- Error on MM2S
                   or s2mm_error                -- Error on S2MM
                   or soft_reset;               -- Soft Reset issued

    -- Reg stop out
    REG_STOP_OUT : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    mm2s_stop <= '0';
                else
                    mm2s_stop      <= mm2s_stop_i;
                end if;
            end if;
        end process REG_STOP_OUT;

    -- Generate DMA Controller For Scatter Gather Mode
    GEN_SCATTER_GATHER_MODE : if C_INCLUDE_SG = 1 generate
    begin
        -- Not Used in SG Mode (Errors are imbedded in updated descriptor and
        -- generate error after descriptor update is complete)
        mm2s_interr_set  <=  '0';
        mm2s_slverr_set  <=  '0';
        mm2s_decerr_set  <=  '0';
        mm2s_smple_done  <=  '0';

mm2s_cmnd_wr_1 <= m_axis_mm2s_ftch_tvalid_new;

        ---------------------------------------------------------------------------
        -- MM2S Primary DMA Controller State Machine
        ---------------------------------------------------------------------------
        I_MM2S_SM : entity  axi_dma_v7_1_13.axi_dma_mm2s_sm
            generic map(
                C_M_AXI_MM2S_ADDR_WIDTH     => C_M_AXI_MM2S_ADDR_WIDTH          ,
                C_SG_LENGTH_WIDTH           => C_SG_LENGTH_WIDTH                ,
                C_SG_INCLUDE_DESC_QUEUE     => C_SG_INCLUDE_DESC_QUEUE          ,
                C_PRMY_CMDFIFO_DEPTH        => C_PRMY_CMDFIFO_DEPTH             ,
                C_ENABLE_MULTI_CHANNEL             => C_ENABLE_MULTI_CHANNEL
            )
            port map(
                m_axi_sg_aclk               => m_axi_sg_aclk                    ,
                m_axi_sg_aresetn            => m_axi_sg_aresetn                 ,

                -- Channel 1 Control and Status
                mm2s_run_stop               => mm2s_run_stop                    ,
                mm2s_keyhole                => mm2s_keyhole                     ,
                mm2s_ftch_idle              => mm2s_ftch_idle                   ,
                mm2s_cmnd_idle              => mm2s_cmnd_idle                   ,
                mm2s_sts_idle               => mm2s_sts_idle                    ,
                mm2s_stop                   => mm2s_stop_i                      ,
                mm2s_desc_flush             => mm2s_desc_flush                  ,

                -- MM2S Descriptor Fetch Request (from mm2s_sm)
                desc_available              => desc_available                   ,
                desc_fetch_req              => desc_fetch_req                   ,
                desc_fetch_done             => desc_fetch_done                  ,
                desc_update_done            => desc_update_done                 ,
                updt_pending                => updt_pending                     ,
                packet_in_progress          => packet_in_progress               ,

                -- DataMover Command
                mm2s_cmnd_wr                => open, --mm2s_cmnd_wr_1                     ,
                mm2s_cmnd_data              => mm2s_cmnd_data                   ,
                mm2s_cmnd_pending           => mm2s_cmnd_pending                ,

                -- Descriptor Fields
                mm2s_cache_info             => mm2s_desc_info                   ,
                mm2s_desc_baddress          => mm2s_desc_baddress               ,
                mm2s_desc_blength           => mm2s_desc_blength                ,
                mm2s_desc_blength_v           => mm2s_desc_blength_v                ,
                mm2s_desc_blength_s           => mm2s_desc_blength_s                ,
                mm2s_desc_eof               => mm2s_desc_eof                    ,
                mm2s_desc_sof               => mm2s_desc_sof
            );

        ---------------------------------------------------------------------------
        -- MM2S Scatter Gather State Machine
        ---------------------------------------------------------------------------
        I_MM2S_SG_IF : entity  axi_dma_v7_1_13.axi_dma_mm2s_sg_if
            generic map(

                -------------------------------------------------------------------
                -- Scatter Gather Parameters
                -------------------------------------------------------------------
                C_PRMRY_IS_ACLK_ASYNC       => C_PRMRY_IS_ACLK_ASYNC            ,
                C_SG_INCLUDE_DESC_QUEUE         => C_SG_INCLUDE_DESC_QUEUE      ,
                C_SG_INCLUDE_STSCNTRL_STRM      => C_SG_INCLUDE_STSCNTRL_STRM   ,
                C_M_AXIS_SG_TDATA_WIDTH         => C_M_AXIS_SG_TDATA_WIDTH      ,
                C_S_AXIS_UPDPTR_TDATA_WIDTH     => C_S_AXIS_UPDPTR_TDATA_WIDTH  ,
                C_S_AXIS_UPDSTS_TDATA_WIDTH     => C_S_AXIS_UPDSTS_TDATA_WIDTH  ,
                C_M_AXI_SG_ADDR_WIDTH           => C_M_AXI_SG_ADDR_WIDTH        ,
                C_M_AXI_MM2S_ADDR_WIDTH         => C_M_AXI_MM2S_ADDR_WIDTH      ,
                C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH => C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH,
                C_ENABLE_MULTI_CHANNEL                 => C_ENABLE_MULTI_CHANNEL              , 
                C_MICRO_DMA                 => C_MICRO_DMA,
                C_FAMILY                        => C_FAMILY
            )
            port map(

                m_axi_sg_aclk                   => m_axi_sg_aclk                ,
                m_axi_sg_aresetn                => m_axi_sg_aresetn             ,

                -- SG MM2S Descriptor Fetch AXI Stream In
                m_axis_mm2s_ftch_tdata          => m_axis_mm2s_ftch_tdata       ,
                m_axis_mm2s_ftch_tvalid         => m_axis_mm2s_ftch_tvalid      ,
                m_axis_mm2s_ftch_tready         => m_axis_mm2s_ftch_tready      ,
                m_axis_mm2s_ftch_tlast          => m_axis_mm2s_ftch_tlast       ,

                m_axis_mm2s_ftch_tdata_new          => m_axis_mm2s_ftch_tdata_new       ,
                m_axis_mm2s_ftch_tdata_mcdma_new          => m_axis_mm2s_ftch_tdata_mcdma_new       ,
                m_axis_mm2s_ftch_tvalid_new         => m_axis_mm2s_ftch_tvalid_new      ,
                m_axis_ftch1_desc_available         => m_axis_ftch1_desc_available      ,

                -- SG MM2S Descriptor Update AXI Stream Out
                s_axis_mm2s_updtptr_tdata       => s_axis_mm2s_updtptr_tdata    ,
                s_axis_mm2s_updtptr_tvalid      => s_axis_mm2s_updtptr_tvalid   ,
                s_axis_mm2s_updtptr_tready      => s_axis_mm2s_updtptr_tready   ,
                s_axis_mm2s_updtptr_tlast       => s_axis_mm2s_updtptr_tlast    ,

                s_axis_mm2s_updtsts_tdata       => s_axis_mm2s_updtsts_tdata    ,
                s_axis_mm2s_updtsts_tvalid      => s_axis_mm2s_updtsts_tvalid   ,
                s_axis_mm2s_updtsts_tready      => s_axis_mm2s_updtsts_tready   ,
                s_axis_mm2s_updtsts_tlast       => s_axis_mm2s_updtsts_tlast    ,


                -- MM2S Descriptor Fetch Request (from mm2s_sm)
                desc_available                  => desc_available               ,
                desc_fetch_req                  => desc_fetch_req               ,
                desc_fetch_done                 => desc_fetch_done              ,
                updt_pending                    => updt_pending                 ,
                packet_in_progress              => packet_in_progress           ,

                -- MM2S Descriptor Update Request
                desc_update_done                => desc_update_done             ,

                mm2s_ftch_stale_desc            => mm2s_ftch_stale_desc         ,
                mm2s_sts_received_clr           => mm2s_sts_received_clr        ,
                mm2s_sts_received               => mm2s_sts_received            ,
                mm2s_desc_cmplt                 => mm2s_desc_cmplt              ,
                mm2s_done                       => mm2s_done                    ,
                mm2s_interr                     => mm2s_interr                  ,
                mm2s_slverr                     => mm2s_slverr                  ,
                mm2s_decerr                     => mm2s_decerr                  ,
                mm2s_tag                        => mm2s_tag                     ,
                mm2s_halt                       => mm2s_halt                    , -- CR566306

                -- Control Stream Output
                cntrlstrm_fifo_wren             => cntrlstrm_fifo_wren          ,
                cntrlstrm_fifo_full             => cntrlstrm_fifo_full          ,
                cntrlstrm_fifo_din              => cntrlstrm_fifo_din           ,

                -- MM2S Descriptor Field Output
                mm2s_new_curdesc                => mm2s_new_curdesc             ,
                mm2s_new_curdesc_wren           => mm2s_new_curdesc_wren        ,
                mm2s_desc_baddress              => mm2s_desc_baddress           ,
                mm2s_desc_blength               => mm2s_desc_blength            ,
                mm2s_desc_blength_v               => mm2s_desc_blength_v            ,
                mm2s_desc_blength_s               => mm2s_desc_blength_s            ,
                mm2s_desc_info                  => mm2s_desc_info               ,
                mm2s_desc_eof                   => mm2s_desc_eof                ,
                mm2s_desc_sof                   => mm2s_desc_sof                ,
                mm2s_desc_app0                  => mm2s_desc_app0               ,
                mm2s_desc_app1                  => mm2s_desc_app1               ,
                mm2s_desc_app2                  => mm2s_desc_app2               ,
                mm2s_desc_app3                  => mm2s_desc_app3               ,
                mm2s_desc_app4                  => mm2s_desc_app4
            );

        cntrlstrm_fifo_full         <= '0';

    end generate GEN_SCATTER_GATHER_MODE;


    -- Generate DMA Controller for Simple DMA Mode
    GEN_SIMPLE_DMA_MODE : if C_INCLUDE_SG = 0 generate
    begin

        -- Scatter Gather signals not used in Simple DMA Mode
        m_axis_mm2s_ftch_tready     <= '0';
        s_axis_mm2s_updtptr_tdata   <= (others => '0');
        s_axis_mm2s_updtptr_tvalid  <= '0';
        s_axis_mm2s_updtptr_tlast   <= '0';
        s_axis_mm2s_updtsts_tdata   <= (others => '0');
        s_axis_mm2s_updtsts_tvalid  <= '0';
        s_axis_mm2s_updtsts_tlast   <= '0';
        desc_available              <= '0';
        desc_fetch_done             <= '0';
        packet_in_progress          <= '0';
        desc_update_done            <= '0';
        cntrlstrm_fifo_wren         <= '0';
        cntrlstrm_fifo_din          <= (others => '0');
        mm2s_new_curdesc            <= (others => '0');
        mm2s_new_curdesc_wren       <= '0';
        mm2s_desc_baddress          <= (others => '0');
        mm2s_desc_blength           <= (others => '0');
        mm2s_desc_blength_v           <= (others => '0');
        mm2s_desc_blength_s           <= (others => '0');
        mm2s_desc_eof               <= '0';
        mm2s_desc_sof               <= '0';
        mm2s_desc_cmplt             <= '0';
        mm2s_desc_app0              <= (others => '0');
        mm2s_desc_app1              <= (others => '0');
        mm2s_desc_app2              <= (others => '0');
        mm2s_desc_app3              <= (others => '0');
        mm2s_desc_app4              <= (others => '0');
        desc_fetch_req              <= '0';

        -- Simple DMA State Machine
        I_MM2S_SMPL_SM : entity axi_dma_v7_1_13.axi_dma_smple_sm
            generic map(
                C_M_AXI_ADDR_WIDTH          => C_M_AXI_MM2S_ADDR_WIDTH  ,
                C_SG_LENGTH_WIDTH           => C_SG_LENGTH_WIDTH,
                C_MICRO_DMA                 => C_MICRO_DMA
            )
            port map(
                m_axi_sg_aclk               => m_axi_sg_aclk            ,
                m_axi_sg_aresetn            => m_axi_sg_aresetn         ,

                -- Channel 1 Control and Status
                run_stop                    => mm2s_run_stop            ,
                keyhole                     => mm2s_keyhole             ,
                stop                        => mm2s_stop_i              ,
                cmnd_idle                   => mm2s_cmnd_idle           ,
                sts_idle                    => mm2s_sts_idle            ,

                -- DataMover Status
                sts_received                => mm2s_sts_received        ,
                sts_received_clr            => mm2s_sts_received_clr    ,

                -- DataMover Command
                cmnd_wr                     => mm2s_cmnd_wr_1             ,
                cmnd_data                   => mm2s_cmnd_data           ,
                cmnd_pending                => mm2s_cmnd_pending        ,

                -- Trasnfer Qualifiers
                xfer_length_wren            => mm2s_length_wren         ,
                xfer_address                => mm2s_sa                  ,
                xfer_length                 => mm2s_length
            );


        -- Pass Done/Error Status out to DMASR
        mm2s_interr_set                 <= mm2s_interr;
        mm2s_slverr_set                 <= mm2s_slverr;
        mm2s_decerr_set                 <= mm2s_decerr;

        -- S2MM Simple DMA Transfer Done - used to assert IOC bit in DMASR.
                                      -- Receive clear when not shutting down
        mm2s_smple_done                 <= mm2s_sts_received_clr when mm2s_stop_i = '0'
                                      -- Else halt set prior to halted being set
                                      else mm2s_halted_set_i when mm2s_halted = '0'
                                      else '0';



    end generate GEN_SIMPLE_DMA_MODE;

    -------------------------------------------------------------------------------
    -- MM2S Primary DataMover command status interface
    -------------------------------------------------------------------------------
    I_MM2S_CMDSTS : entity  axi_dma_v7_1_13.axi_dma_mm2s_cmdsts_if
        generic map(
            C_M_AXI_MM2S_ADDR_WIDTH         => C_M_AXI_MM2S_ADDR_WIDTH,
            C_ENABLE_MULTI_CHANNEL                 => C_ENABLE_MULTI_CHANNEL,
            C_ENABLE_QUEUE                  => C_SG_INCLUDE_DESC_QUEUE 
        )
        port map(
            m_axi_sg_aclk                   => m_axi_sg_aclk                ,
            m_axi_sg_aresetn                => m_axi_sg_aresetn             ,

            -- Fetch command write interface from mm2s sm
            mm2s_cmnd_wr                    => mm2s_cmnd_wr_1                 ,
            mm2s_cmnd_data                  => mm2s_cmnd_data               ,
            mm2s_cmnd_pending               => mm2s_cmnd_pending            ,

            mm2s_sts_received_clr           => mm2s_sts_received_clr        ,
            mm2s_sts_received               => mm2s_sts_received            ,
            mm2s_tailpntr_enble             => mm2s_tailpntr_enble          ,
            mm2s_desc_cmplt                 => mm2s_desc_cmplt              ,

            -- User Command Interface Ports (AXI Stream)
            s_axis_mm2s_cmd_tvalid          => s_axis_mm2s_cmd_tvalid       ,
            s_axis_mm2s_cmd_tready          => s_axis_mm2s_cmd_tready       ,
            s_axis_mm2s_cmd_tdata           => s_axis_mm2s_cmd_tdata        ,

            -- User Status Interface Ports (AXI Stream)
            m_axis_mm2s_sts_tvalid          => m_axis_mm2s_sts_tvalid       ,
            m_axis_mm2s_sts_tready          => m_axis_mm2s_sts_tready       ,
            m_axis_mm2s_sts_tdata           => m_axis_mm2s_sts_tdata        ,
            m_axis_mm2s_sts_tkeep           => m_axis_mm2s_sts_tkeep        ,

            -- MM2S Primary DataMover Status
            mm2s_err                        => mm2s_err                     ,
            mm2s_done                       => mm2s_done                    ,
            mm2s_error                      => dma_mm2s_error               ,
            mm2s_interr                     => mm2s_interr                  ,
            mm2s_slverr                     => mm2s_slverr                  ,
            mm2s_decerr                     => mm2s_decerr                  ,
            mm2s_tag                        => mm2s_tag
        );

    ---------------------------------------------------------------------------
    -- Halt / Idle Status Manager
    ---------------------------------------------------------------------------
    I_MM2S_STS_MNGR : entity  axi_dma_v7_1_13.axi_dma_mm2s_sts_mngr
        generic map(
            C_PRMRY_IS_ACLK_ASYNC       => C_PRMRY_IS_ACLK_ASYNC
        )
        port map(
            m_axi_sg_aclk               => m_axi_sg_aclk                    ,
            m_axi_sg_aresetn            => m_axi_sg_aresetn                 ,

            -- dma control and sg engine status signals
            mm2s_run_stop               => mm2s_run_stop                    ,
            mm2s_ftch_idle              => mm2s_ftch_idle                   ,
            mm2s_updt_idle              => mm2s_updt_idle                   ,
            mm2s_cmnd_idle              => mm2s_cmnd_idle                   ,
            mm2s_sts_idle               => mm2s_sts_idle                    ,

            -- stop and halt control/status
            mm2s_stop                   => mm2s_stop_i                      ,
            mm2s_halt_cmplt             => mm2s_halt_cmplt                  ,

            -- system state and control
            mm2s_all_idle               => mm2s_all_idle                    ,
            mm2s_halted_clr             => mm2s_halted_clr                  ,
            mm2s_halted_set             => mm2s_halted_set_i                ,
            mm2s_idle_set               => mm2s_idle_set                    ,
            mm2s_idle_clr               => mm2s_idle_clr
        );


    -- MM2S Control Stream Included
    GEN_CNTRL_STREAM : if C_SG_INCLUDE_STSCNTRL_STRM = 1 and C_INCLUDE_SG = 1 generate
    begin

        -- Register soft reset to create rising edge pulse to use for shut down.
        -- soft_reset from DMACR does not clear until after all reset processes
        -- are done.  This causes stop to assert too long causing issue with
        -- status stream skid buffer.
        REG_SFT_RST : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        soft_reset_d1 <= '0';
                        soft_reset_d2 <= '0';
                    else
                        soft_reset_d1 <= soft_reset;
                        soft_reset_d2 <= soft_reset_d1;
                    end if;
                end if;
            end process REG_SFT_RST;

        -- Rising edge soft reset pulse
        soft_reset_re <= soft_reset_d1 and not soft_reset_d2;

        -- Control Stream module stop requires rising edge of soft reset to
        -- shut down due to DMACR.SoftReset does not deassert on internal hard reset
        -- It clears after therefore do not want to issue another stop to cntrl strm
        -- skid buffer.
        cntrl_strm_stop <= mm2s_error_i             -- Error
                        or soft_reset_re;           -- Soft Reset issued

        -- Control stream interface
--        I_MM2S_CNTRL_STREAM : entity axi_dma_v7_1_13.axi_dma_mm2s_cntrl_strm
--            generic map(
--                C_PRMRY_IS_ACLK_ASYNC           => C_PRMRY_IS_ACLK_ASYNC            ,
--                C_PRMY_CMDFIFO_DEPTH            => C_PRMY_CMDFIFO_DEPTH             ,
--                C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH => C_M_AXIS_MM2S_CNTRL_TDATA_WIDTH  ,
--                C_FAMILY                        => C_FAMILY
--            )
--            port map(
--                -- Secondary clock / reset
--                m_axi_sg_aclk               => m_axi_sg_aclk                ,
--                m_axi_sg_aresetn            => m_axi_sg_aresetn             ,
--
--                -- Primary clock / reset
--                axi_prmry_aclk              => axi_prmry_aclk               ,
--                p_reset_n                   => p_reset_n                    ,
--
--                -- MM2S Error
--                mm2s_stop                   => cntrl_strm_stop              ,
--
--                -- Control Stream input
----                cntrlstrm_fifo_wren         => cntrlstrm_fifo_wren          ,
--                cntrlstrm_fifo_full         => cntrlstrm_fifo_full          ,
--                cntrlstrm_fifo_din          => cntrlstrm_fifo_din           ,
--
--                -- Memory Map to Stream Control Stream Interface
--                m_axis_mm2s_cntrl_tdata     => m_axis_mm2s_cntrl_tdata      ,
--                m_axis_mm2s_cntrl_tkeep     => m_axis_mm2s_cntrl_tkeep      ,
--                m_axis_mm2s_cntrl_tvalid    => m_axis_mm2s_cntrl_tvalid     ,
--                m_axis_mm2s_cntrl_tready    => m_axis_mm2s_cntrl_tready     ,
--                m_axis_mm2s_cntrl_tlast     => m_axis_mm2s_cntrl_tlast
--
--            );
    end generate GEN_CNTRL_STREAM;


    -- MM2S Control Stream Excluded
    GEN_NO_CNTRL_STREAM : if C_SG_INCLUDE_STSCNTRL_STRM = 0 or C_INCLUDE_SG = 0 generate
    begin
        soft_reset_d1               <= '0';
        soft_reset_d2               <= '0';
        soft_reset_re               <= '0';
        cntrl_strm_stop             <= '0';

    end generate GEN_NO_CNTRL_STREAM;

        m_axis_mm2s_cntrl_tdata     <= (others => '0');
        m_axis_mm2s_cntrl_tkeep     <= (others => '0');
        m_axis_mm2s_cntrl_tvalid    <= '0';
        m_axis_mm2s_cntrl_tlast     <= '0';

end generate GEN_MM2S_DMA_CONTROL;


-------------------------------------------------------------------------------
-- Exclude MM2S State Machine and support logic
-------------------------------------------------------------------------------
GEN_NO_MM2S_DMA_CONTROL : if C_INCLUDE_MM2S = 0 generate
begin
        m_axis_mm2s_ftch_tready     <= '0';
        s_axis_mm2s_updtptr_tdata   <= (others =>'0');
        s_axis_mm2s_updtptr_tvalid  <= '0';
        s_axis_mm2s_updtptr_tlast   <= '0';
        s_axis_mm2s_updtsts_tdata   <= (others =>'0');
        s_axis_mm2s_updtsts_tvalid  <= '0';
        s_axis_mm2s_updtsts_tlast   <= '0';
        mm2s_new_curdesc            <= (others =>'0');
        mm2s_new_curdesc_wren       <= '0';
        s_axis_mm2s_cmd_tvalid      <= '0';
        s_axis_mm2s_cmd_tdata       <= (others =>'0');
        m_axis_mm2s_sts_tready      <= '0';
        mm2s_halted_clr             <= '0';
        mm2s_halted_set             <= '0';
        mm2s_idle_set               <= '0';
        mm2s_idle_clr               <= '0';
        m_axis_mm2s_cntrl_tdata     <= (others => '0');
        m_axis_mm2s_cntrl_tkeep     <= (others => '0');
        m_axis_mm2s_cntrl_tvalid    <= '0';
        m_axis_mm2s_cntrl_tlast     <= '0';
        mm2s_stop                   <= '0';
        mm2s_desc_flush             <= '0';
        mm2s_all_idle               <= '1';
        mm2s_error                  <= '0'; -- CR#570587
        mm2s_interr_set             <= '0';
        mm2s_slverr_set             <= '0';
        mm2s_decerr_set             <= '0';
        mm2s_smple_done             <= '0';
        cntrl_strm_stop             <= '0';

end generate GEN_NO_MM2S_DMA_CONTROL;

TDEST_FIFO : if (C_ENABLE_MULTI_CHANNEL = 1) generate

    process (m_axi_sg_aclk)
    begin
         if (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
           if (m_axi_sg_aresetn = '0') then
                 desc_fetch_done_del <= '0';
           else --if (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
                 desc_fetch_done_del <= desc_fetch_done;
           end if;
         end if;
    end process; 


    process (m_axis_mm2s_aclk)
    begin
         if (m_axis_mm2s_aclk'event and m_axis_mm2s_aclk = '1') then
           if (m_axi_sg_aresetn = '0') then
                 fifo_empty <= '0';
           else 
                 fifo_empty <= info_fifo_empty;
           end if;
         end if;
    end process; 
    process (m_axis_mm2s_aclk)
    begin
         if (m_axis_mm2s_aclk'event and m_axis_mm2s_aclk = '1') then
           if (m_axi_sg_aresetn = '0') then
                 fifo_empty_first <= '0';
                 fifo_empty_first1 <= '0';
           else
                if (fifo_empty_first = '0' and (info_fifo_empty = '0' and fifo_empty = '1')) then
                 fifo_empty_first <= '1';
                end if;
                 fifo_empty_first1 <= fifo_empty_first;
           end if;
         end if;
    end process; 
              
    first_read_pulse <= fifo_empty_first and (not fifo_empty_first1);

    fifo_read <= first_read_pulse or rd_en_hold; 

    mm2s_desc_info_int <= mm2s_desc_info (19 downto 16) & mm2s_desc_info (12 downto 8) & mm2s_desc_info (4 downto 0);

--    mm2s_strm_tlast_int <= mm2s_strm_tlast and (not info_fifo_empty); 
--    process (m_axis_mm2s_aclk)
--    begin
--       if (m_axis_mm2s_aclk'event and m_axis_mm2s_aclk = '1') then
--         if (p_reset_n = '0') then
--             rd_en_hold <= '0';
--             rd_en_hold_int <= '0';
--         else
--            if (rd_en_hold = '1') then
--              rd_en_hold <= '0';
--            elsif (info_fifo_empty = '0' and mm2s_strm_tlast = '1' and mm2s_strm_tready = '1') then
--              rd_en_hold <= '1';
--              rd_en_hold_int <= '0';
--            else
--              rd_en_hold <= rd_en_hold; 
--              rd_en_hold_int <= rd_en_hold_int;
--            end if;        
--         end if;
--       end if;
--    end process; 


    process (m_axis_mm2s_aclk)
    begin
       if (m_axis_mm2s_aclk'event and m_axis_mm2s_aclk = '1') then
         if (p_reset_n = '0') then
             rd_en_hold <= '0';
             rd_en_hold_int <= '0';
         else
            if (info_fifo_empty = '1' and mm2s_strm_tlast = '1' and mm2s_strm_tready = '1') then
              rd_en_hold <= '1';
              rd_en_hold_int <= '0';
            elsif (info_fifo_empty = '0') then
              rd_en_hold <= mm2s_strm_tlast and mm2s_strm_tready; 
              rd_en_hold_int <= rd_en_hold;
            else
              rd_en_hold <= rd_en_hold; 
              rd_en_hold_int <= rd_en_hold_int;
            end if;        
         end if;
       end if;
    end process; 

    fifo_rst <= not (m_axi_sg_aresetn);

    -- Following FIFO is used to store the Tuser, Tid and xCache info
    I_INFO_FIFO : entity axi_dma_v7_1_13.axi_dma_afifo_autord
      generic map(
         C_DWIDTH        => 14,
         C_DEPTH         => 31                                  ,
         C_CNT_WIDTH     => 5                                   ,
         C_USE_BLKMEM    => 0, 
         C_USE_AUTORD    => 1,
         C_PRMRY_IS_ACLK_ASYNC       => C_PRMRY_IS_ACLK_ASYNC            ,
         C_FAMILY        => C_FAMILY
        )
      port map(
        -- Inputs
         AFIFO_Ainit                => fifo_rst         ,
         AFIFO_Wr_clk               => m_axi_sg_aclk            ,
         AFIFO_Wr_en                => desc_fetch_done_del      ,
         AFIFO_Din                  => mm2s_desc_info_int       ,
         AFIFO_Rd_clk               => m_axis_mm2s_aclk           ,
         AFIFO_Rd_en                => rd_en_hold_int, --fifo_read, --mm2s_strm_tlast_int          ,
         AFIFO_Clr_Rd_Data_Valid    => '0'                      ,

        -- Outputs
         AFIFO_DValid               => open        ,
         AFIFO_Dout                 => mm2s_axis_info          ,
         AFIFO_Full                 => info_fifo_full      ,
         AFIFO_Empty                => info_fifo_empty     ,
         AFIFO_Almost_full          => open                     ,
         AFIFO_Almost_empty         => open                     ,
         AFIFO_Wr_count             => open                     ,
         AFIFO_Rd_count             => open                     ,
         AFIFO_Corr_Rd_count        => open                     ,
         AFIFO_Corr_Rd_count_minus1 => open                     ,
         AFIFO_Rd_ack               => open
        );

end generate TDEST_FIFO;


NO_TDEST_FIFO : if (C_ENABLE_MULTI_CHANNEL = 0) generate
    mm2s_axis_info <= (others => '0');

end generate NO_TDEST_FIFO;

end implementation;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:          axi_dma_s2mm_sg_if.vhd
-- Description: This entity is the S2MM Scatter Gather Interface for Descriptor
--              Fetches and Updates.
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;

library lib_cdc_v1_0_2;
library lib_srl_fifo_v1_0_2;
use lib_srl_fifo_v1_0_2.srl_fifo_f;

-------------------------------------------------------------------------------
entity  axi_dma_s2mm_sg_if is
    generic (
        C_PRMRY_IS_ACLK_ASYNC        : integer range 0 to 1          := 0       ;
            -- Primary MM2S/S2MM sync/async mode
            -- 0 = synchronous mode     - all clocks are synchronous
            -- 1 = asynchronous mode    - Any one of the 4 clock inputs is not
            --                            synchronous to the other
        -----------------------------------------------------------------------
        -- Scatter Gather Parameters
        -----------------------------------------------------------------------
        C_SG_INCLUDE_STSCNTRL_STRM      : integer range 0 to 1          := 1    ;
            -- Include or Exclude AXI Status and AXI Control Streams
            -- 0 = Exclude Status and Control Streams
            -- 1 = Include Status and Control Streams

        C_SG_INCLUDE_DESC_QUEUE         : integer range 0 to 1          := 0    ;
            -- Include or Exclude Scatter Gather Descriptor Queuing
            -- 0 = Exclude SG Descriptor Queuing
            -- 1 = Include SG Descriptor Queuing

        C_SG_USE_STSAPP_LENGTH      : integer range 0 to 1          := 1;
            -- Enable or Disable use of Status Stream Rx Length.  Only valid
            -- if C_SG_INCLUDE_STSCNTRL_STRM = 1
            -- 0 = Don't use Rx Length
            -- 1 = Use Rx Length

        C_SG_LENGTH_WIDTH               : integer range 8 to 23         := 14   ;
            -- Descriptor Buffer Length, Transferred Bytes, and Status Stream
            -- Rx Length Width.  Indicates the least significant valid bits of
            -- descriptor buffer length, transferred bytes, or Rx Length value
            -- in the status word coincident with tlast.

        C_M_AXIS_SG_TDATA_WIDTH          : integer range 32 to 32        := 32  ;
            -- AXI Master Stream in for descriptor fetch

        C_S_AXIS_UPDPTR_TDATA_WIDTH : integer range 32 to 32            := 32   ;
            -- 32 Update Status Bits

        C_S_AXIS_UPDSTS_TDATA_WIDTH : integer range 33 to 33            := 33   ;
            -- 1 IOC bit + 32 Update Status Bits

        C_M_AXI_SG_ADDR_WIDTH           : integer range 32 to 64        := 32   ;
            -- Master AXI Memory Map Data Width for Scatter Gather R/W Port

        C_M_AXI_S2MM_ADDR_WIDTH         : integer range 32 to 64        := 32   ;
            -- Master AXI Memory Map Address Width for S2MM Write Port

        C_S_AXIS_S2MM_STS_TDATA_WIDTH   : integer range 32 to 32        := 32   ;
            -- Slave AXI Status Stream Data Width
        C_NUM_S2MM_CHANNELS             : integer range 1 to 16         := 1 ;

        C_ENABLE_MULTI_CHANNEL                 : integer range 0 to 1          := 0;

        C_MICRO_DMA                     : integer range 0 to 1          := 0;

        C_FAMILY                        : string                        := "virtex5"
            -- Target FPGA Device Family
    );
    port (

        m_axi_sg_aclk               : in  std_logic                         ;                     --
        m_axi_sg_aresetn            : in  std_logic                         ;                     --

        s2mm_desc_info_in              : in std_logic_vector (13 downto 0)     ;
                                                                                                  --
        -- SG S2MM Descriptor Fetch AXI Stream In                                                 --
        m_axis_s2mm_ftch_tdata      : in  std_logic_vector                                        --
                                        (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0);                     --
        m_axis_s2mm_ftch_tvalid     : in  std_logic                         ;                     --
        m_axis_s2mm_ftch_tready     : out std_logic                         ;                     --
        m_axis_s2mm_ftch_tlast      : in  std_logic                         ;                     --

        m_axis_s2mm_ftch_tdata_new      : in  std_logic_vector                                        --
                                        (96+31*0+(0+2)*(C_M_AXI_SG_ADDR_WIDTH-32) downto 0);                     --
        m_axis_s2mm_ftch_tdata_mcdma_new      : in  std_logic_vector                                        --
                                        (63 downto 0);                     --
        m_axis_s2mm_ftch_tdata_mcdma_nxt      : in  std_logic_vector                                        --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0);                     --
        m_axis_s2mm_ftch_tvalid_new     : in  std_logic                         ;                     --
        m_axis_ftch2_desc_available  : in std_logic;
                                                                                                  --
                                                                                                  --
        -- SG S2MM Descriptor Update AXI Stream Out                                               --
        s_axis_s2mm_updtptr_tdata   : out std_logic_vector                                        --
                                     (C_M_AXI_SG_ADDR_WIDTH-1 downto 0) ;                   --
        s_axis_s2mm_updtptr_tvalid  : out std_logic                         ;                     --
        s_axis_s2mm_updtptr_tready  : in  std_logic                         ;                     --
        s_axis_s2mm_updtptr_tlast   : out std_logic                         ;                     --
                                                                                                  --
        s_axis_s2mm_updtsts_tdata   : out std_logic_vector                                        --
                                     (C_S_AXIS_UPDSTS_TDATA_WIDTH-1 downto 0)  ;                  --
        s_axis_s2mm_updtsts_tvalid  : out std_logic                         ;                     --
        s_axis_s2mm_updtsts_tready  : in  std_logic                         ;                     --
        s_axis_s2mm_updtsts_tlast   : out std_logic                         ;                     --
                                                                                                  --
        -- S2MM Descriptor Fetch Request (from s2mm_sm)                                           --
        desc_available              : out std_logic                         ;                     --
        desc_fetch_req              : in  std_logic                         ;                     --
        updt_pending                : out std_logic                         ;
        desc_fetch_done             : out std_logic                         ;                     --
                                                                                                  --
        -- S2MM Descriptor Update Request (from s2mm_sm)                                          --
        desc_update_done            : out std_logic                         ;                     --
        s2mm_sts_received_clr       : out std_logic                         ;                     --
        s2mm_sts_received           : in  std_logic                         ;                     --
                                                                                                  --
        -- Scatter Gather Update Status                                                           --
        s2mm_done                   : in  std_logic                         ;                     --
        s2mm_interr                 : in  std_logic                         ;                     --
        s2mm_slverr                 : in  std_logic                         ;                     --
        s2mm_decerr                 : in  std_logic                         ;                     --
        s2mm_tag                    : in  std_logic_vector(3 downto 0)      ;                     --
        s2mm_brcvd                  : in  std_logic_vector                                        --
                                        (C_SG_LENGTH_WIDTH-1 downto 0)      ;                     --
        s2mm_eof_set                : in  std_logic                         ;                     --
        s2mm_packet_eof             : in  std_logic                         ;                     --
        s2mm_halt                   : in  std_logic                         ;                     --
                                                                                                  --
        -- S2MM Status Stream Interface                                                           --
        stsstrm_fifo_rden           : out std_logic                         ;                     --
        stsstrm_fifo_empty          : in  std_logic                         ;                     --
        stsstrm_fifo_dout           : in  std_logic_vector                                        --
                                        (C_S_AXIS_S2MM_STS_TDATA_WIDTH downto 0);                 --
                                                                                                  --
        -- DataMover Command                                                                      --
        s2mm_cmnd_wr                : in  std_logic                         ;                     --
        s2mm_cmnd_data              : in  std_logic_vector                                        --
                                        (((1+C_ENABLE_MULTI_CHANNEL)*C_M_AXI_S2MM_ADDR_WIDTH+CMD_BASE_WIDTH)-1 downto 0);    --
                                                                                                  --
        -- S2MM Descriptor Field Output                                                           --
        s2mm_new_curdesc            : out std_logic_vector                                        --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                     --
        s2mm_new_curdesc_wren       : out std_logic                         ;                     --
                                                                                                  --
        s2mm_desc_info          : out std_logic_vector                                        --
                                        (31 downto 0);                     --
        s2mm_desc_baddress          : out std_logic_vector                                        --
                                        (C_M_AXI_S2MM_ADDR_WIDTH-1 downto 0);                     --
        s2mm_desc_blength           : out std_logic_vector                                        --
                                        (BUFFER_LENGTH_WIDTH-1 downto 0)  ;                       --
        s2mm_desc_blength_v           : out std_logic_vector                                        --
                                        (BUFFER_LENGTH_WIDTH-1 downto 0)  ;                       --
        s2mm_desc_blength_s           : out std_logic_vector                                        --
                                        (BUFFER_LENGTH_WIDTH-1 downto 0)  ;                       --
        s2mm_desc_cmplt             : out std_logic                         ;                     --
        s2mm_eof_micro              : out std_logic ;
        s2mm_sof_micro              : out std_logic ;
        s2mm_desc_app0              : out std_logic_vector                                        --
                                        (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0)  ;                   --
        s2mm_desc_app1              : out std_logic_vector                                        --
                                        (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0)  ;                   --
        s2mm_desc_app2              : out std_logic_vector                                        --
                                        (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0)  ;                   --
        s2mm_desc_app3              : out std_logic_vector                                        --
                                        (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0)  ;                   --
        s2mm_desc_app4              : out std_logic_vector                                        --
                                        (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0)                      --
    );

end axi_dma_s2mm_sg_if;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_dma_s2mm_sg_if is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";

  ATTRIBUTE async_reg                      : STRING;

-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------
-- Status reserved bits
constant RESERVED_STS           : std_logic_vector(2 downto 0)
                                    := (others => '0');
-- Zero value constant
constant ZERO_VALUE             : std_logic_vector(31 downto 0)
                                    := (others => '0');
-- Zero length constant
constant ZERO_LENGTH            : std_logic_vector(C_SG_LENGTH_WIDTH-1 downto 0)
                                    := (others => '0');

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
signal ftch_shftenbl            : std_logic := '0';

-- fetch descriptor holding registers
signal desc_reg12               : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg11               : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg10               : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg9                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg8                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg7                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg6                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg5                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg4                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg3                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg2                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg1                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal desc_reg0                : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');

signal s2mm_desc_curdesc_lsb    : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal s2mm_desc_curdesc_lsb_nxt    : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal s2mm_desc_curdesc_msb    : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal s2mm_desc_curdesc_msb_nxt    : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal s2mm_desc_baddr_lsb      : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal s2mm_desc_baddr_msb      : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH - 1 downto 0) := (others => '0');
signal s2mm_pending_update      : std_logic := '0';
signal s2mm_new_curdesc_wren_i  : std_logic := '0';
signal s2mm_ioc                 : std_logic := '0';
signal s2mm_pending_pntr_updt   : std_logic := '0';

-- Descriptor Update Signals
signal s2mm_complete            : std_logic := '0';
signal s2mm_xferd_bytes         : std_logic_vector(BUFFER_LENGTH_WIDTH-1 downto 0)      := (others => '0');
signal s2mm_desc_blength_i      : std_logic_vector(BUFFER_LENGTH_WIDTH - 1 downto 0)    := (others => '0');
signal s2mm_desc_blength_v_i      : std_logic_vector(BUFFER_LENGTH_WIDTH - 1 downto 0)    := (others => '0');
signal s2mm_desc_blength_s_i      : std_logic_vector(BUFFER_LENGTH_WIDTH - 1 downto 0)    := (others => '0');

-- Signals for pointer support
-- Make 1 bit wider to allow tagging of LAST for use in generating tlast
signal updt_desc_reg0           : std_logic_vector(C_M_AXI_SG_ADDR_WIDTH-1 downto 0)     := (others => '0');
signal updt_desc_reg1           : std_logic_vector(C_S_AXIS_UPDPTR_TDATA_WIDTH downto 0)     := (others => '0');

signal updt_shftenbl            : std_logic := '0';

signal updtptr_tvalid           : std_logic := '0';
signal updtptr_tlast            : std_logic := '0';
signal updtptr_tdata            : std_logic_vector(C_M_AXI_SG_ADDR_WIDTH-1 downto 0) := (others => '0');

-- Signals for Status Stream Support
signal updt_desc_sts            : std_logic_vector(C_S_AXIS_UPDSTS_TDATA_WIDTH downto 0)     := (others => '0');
signal updt_desc_reg3           : std_logic_vector(C_S_AXIS_UPDSTS_TDATA_WIDTH downto 0)     := (others => '0');
signal updt_zero_reg3           : std_logic_vector(C_S_AXIS_UPDSTS_TDATA_WIDTH downto 0)     := (others => '0');
signal updt_zero_reg4           : std_logic_vector(C_S_AXIS_UPDSTS_TDATA_WIDTH downto 0)     := (others => '0');
signal updt_zero_reg5           : std_logic_vector(C_S_AXIS_UPDSTS_TDATA_WIDTH downto 0)     := (others => '0');
signal updt_zero_reg6           : std_logic_vector(C_S_AXIS_UPDSTS_TDATA_WIDTH downto 0)     := (others => '0');
signal updt_zero_reg7           : std_logic_vector(C_S_AXIS_UPDSTS_TDATA_WIDTH downto 0)     := (others => '0');

signal writing_app_fields       : std_logic := '0';
signal stsstrm_fifo_rden_i      : std_logic := '0';

signal sts_shftenbl             : std_logic := '0';

signal sts_received             : std_logic := '0';
signal sts_received_d1          : std_logic := '0';
signal sts_received_re          : std_logic := '0';

-- Queued Update signals
signal updt_data_clr            : std_logic := '0';
signal updt_sts_clr             : std_logic := '0';
signal updt_data                : std_logic := '0';
signal updt_sts                 : std_logic := '0';

signal ioc_tag                  : std_logic := '0';
signal s2mm_sof_set             : std_logic := '0';
signal s2mm_in_progress         : std_logic := '0';
signal eof_received             : std_logic := '0';
signal sof_received             : std_logic := '0';

signal updtsts_tvalid           : std_logic := '0';
signal updtsts_tlast            : std_logic := '0';
signal updtsts_tdata            : std_logic_vector(C_S_AXIS_UPDSTS_TDATA_WIDTH-1 downto 0) := (others => '0');

signal s2mm_halt_d1_cdc_tig             : std_logic := '0';
signal s2mm_halt_cdc_d2             : std_logic := '0';
signal s2mm_halt_d2             : std_logic := '0';
  --ATTRIBUTE async_reg OF s2mm_halt_d1_cdc_tig  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF s2mm_halt_cdc_d2  : SIGNAL IS "true";

signal desc_fetch_done_i        : std_logic;

-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin


-- Drive buffer length out
s2mm_desc_blength <= s2mm_desc_blength_i;
s2mm_desc_blength_v <= s2mm_desc_blength_v_i;
s2mm_desc_blength_s <= s2mm_desc_blength_s_i;

updt_pending <= s2mm_pending_update;
-- Drive ready if descriptor fetch request is being made
m_axis_s2mm_ftch_tready     <= desc_fetch_req                   -- Request descriptor fetch
                                and not s2mm_pending_update;    -- No pending pointer updates


                            
desc_fetch_done <= desc_fetch_done_i;

-- Shift in data from SG engine if tvalid and fetch request
ftch_shftenbl           <= m_axis_s2mm_ftch_tvalid_new
                            and desc_fetch_req
                            and not s2mm_pending_update;

-- Passed curdes write out to register module
s2mm_new_curdesc_wren   <= s2mm_new_curdesc_wren_i;

-- tvalid asserted means descriptor availble
desc_available          <= m_axis_ftch2_desc_available; --m_axis_s2mm_ftch_tvalid_new;


--***************************************************************************--
--** Register DataMover Halt to secondary if needed
--***************************************************************************--
GEN_FOR_ASYNC : if C_PRMRY_IS_ACLK_ASYNC = 1 generate
begin
    -- Double register to secondary clock domain.  This is sufficient
    -- because halt will remain asserted until halt_cmplt detected in
    -- reset module in secondary clock domain.
REG_TO_SECONDARY : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => s2mm_halt,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => m_axi_sg_aclk,
        scndry_resetn              => '0',
        scndry_out                 => s2mm_halt_cdc_d2,
        scndry_vect_out            => open
    );

--    REG_TO_SECONDARY : process(m_axi_sg_aclk)
--        begin
--            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
--            --    if(m_axi_sg_aresetn = '0')then
--            --        s2mm_halt_d1_cdc_tig <= '0';
--            --        s2mm_halt_d2 <= '0';
--            --    else
--                    s2mm_halt_d1_cdc_tig <= s2mm_halt;
--                    s2mm_halt_cdc_d2 <= s2mm_halt_d1_cdc_tig;
--          --      end if;
--            end if;
--        end process REG_TO_SECONDARY;

                    s2mm_halt_d2 <= s2mm_halt_cdc_d2;

end generate GEN_FOR_ASYNC;

GEN_FOR_SYNC : if C_PRMRY_IS_ACLK_ASYNC = 0 generate
begin
    -- No clock crossing required therefore simple pass through
    s2mm_halt_d2 <= s2mm_halt;

end generate GEN_FOR_SYNC;


--***************************************************************************--
--**                        Descriptor Fetch Logic                         **--
--***************************************************************************--


    s2mm_desc_curdesc_lsb   <= desc_reg0;
--s2mm_desc_curdesc_lsb_nxt   <= desc_reg2;
--s2mm_desc_curdesc_msb_nxt   <= desc_reg3;
    s2mm_desc_baddr_lsb     <= desc_reg4;




    GEN_NO_MCDMA : if C_ENABLE_MULTI_CHANNEL = 0 generate


            desc_fetch_done_i         <= m_axis_s2mm_ftch_tvalid_new; 
                            desc_reg0     <= m_axis_s2mm_ftch_tdata_new (96 downto 65); 
                            desc_reg4     <= m_axis_s2mm_ftch_tdata_new (31 downto 0);
                            desc_reg8     <= m_axis_s2mm_ftch_tdata_new (63 downto 32);
                            desc_reg9( DESC_STS_CMPLTD_BIT)     <= m_axis_s2mm_ftch_tdata_new (64);
                            desc_reg9(30 downto 0)     <= (others => '0');




       s2mm_desc_curdesc_lsb_nxt   <= desc_reg0;
    --   s2mm_desc_curdesc_msb_nxt   <= (others => '0'); --desc_reg1;
       s2mm_desc_info     <= (others => '0'); 
       -- desc 4 and desc 5 are reserved and thus don't care
       s2mm_sof_micro <= desc_reg8 (DESC_SOF_BIT);
       s2mm_eof_micro <= desc_reg8 (DESC_EOF_BIT);
       s2mm_desc_blength_i     <= desc_reg8(DESC_BLENGTH_MSB_BIT downto DESC_BLENGTH_LSB_BIT);
       s2mm_desc_blength_v_i     <= (others => '0'); 
       s2mm_desc_blength_s_i     <= (others => '0') ;


ADDR_64BIT : if C_M_AXI_SG_ADDR_WIDTH > 32 generate
begin

    s2mm_desc_baddr_msb   <= m_axis_s2mm_ftch_tdata_new (128 downto 97);
    s2mm_desc_curdesc_msb     <= m_axis_s2mm_ftch_tdata_new (160 downto 129);

end generate ADDR_64BIT;

ADDR_32BIT : if C_M_AXI_SG_ADDR_WIDTH = 32 generate
begin

    s2mm_desc_curdesc_msb   <= (others => '0');
    s2mm_desc_baddr_msb     <= (others => '0');

end generate ADDR_32BIT;



ADDR_64BIT_DMA : if C_M_AXI_SG_ADDR_WIDTH > 32 generate
begin

       s2mm_desc_curdesc_lsb_nxt   <= desc_reg0;
    s2mm_desc_curdesc_msb_nxt   <= m_axis_s2mm_ftch_tdata_new (160 downto 129);

end generate ADDR_64BIT_DMA;

ADDR_32BIT_DMA : if C_M_AXI_SG_ADDR_WIDTH = 32 generate
begin

       s2mm_desc_curdesc_lsb_nxt   <= desc_reg0;
       s2mm_desc_curdesc_msb_nxt   <= (others => '0');

end generate ADDR_32BIT_DMA;

    end generate GEN_NO_MCDMA;

    GEN_MCDMA : if C_ENABLE_MULTI_CHANNEL = 1 generate


                            desc_fetch_done_i         <= m_axis_s2mm_ftch_tvalid_new; --ftch_shftenbl;

                            desc_reg0     <= m_axis_s2mm_ftch_tdata_new (96 downto 65); --127 downto 96);
                            desc_reg4     <= m_axis_s2mm_ftch_tdata_new (31 downto 0);
                            desc_reg8     <= m_axis_s2mm_ftch_tdata_new (63 downto 32);
                            desc_reg9(DESC_STS_CMPLTD_BIT)     <= m_axis_s2mm_ftch_tdata_new (64); --95 downto 64);
                            desc_reg9(30 downto 0)     <= (others => '0');


                            desc_reg2     <= m_axis_s2mm_ftch_tdata_mcdma_nxt (31 downto 0);
                            desc_reg6     <= m_axis_s2mm_ftch_tdata_mcdma_new (31 downto 0);
                            desc_reg7     <= m_axis_s2mm_ftch_tdata_mcdma_new (63 downto 32);


       s2mm_desc_info     <= desc_reg6 (31 downto 24) & desc_reg9 (23 downto 0);
-- desc 4 and desc 5 are reserved and thus don't care
       s2mm_desc_blength_i     <= "0000000" & desc_reg8(15 downto 0);
       s2mm_desc_blength_v_i     <= "0000000000" & desc_reg7(31 downto 19); 
       s2mm_desc_blength_s_i     <= "0000000" & desc_reg7(15 downto 0);


ADDR_64BIT_1 : if C_M_AXI_SG_ADDR_WIDTH > 32 generate
begin

    s2mm_desc_curdesc_msb   <= m_axis_s2mm_ftch_tdata_new (128 downto 97);
    s2mm_desc_baddr_msb     <= m_axis_s2mm_ftch_tdata_new (160 downto 129);

end generate ADDR_64BIT_1;

ADDR_32BIT_1 : if C_M_AXI_SG_ADDR_WIDTH = 32 generate
begin

    s2mm_desc_curdesc_msb   <= (others => '0');
    s2mm_desc_baddr_msb     <= (others => '0');

end generate ADDR_32BIT_1;


ADDR_64BIT_MCDMA : if C_M_AXI_SG_ADDR_WIDTH > 32 generate
begin

       s2mm_desc_curdesc_lsb_nxt   <= desc_reg2;
       s2mm_desc_curdesc_msb_nxt   <= m_axis_s2mm_ftch_tdata_mcdma_nxt (63 downto 32);

end generate ADDR_64BIT_MCDMA;

ADDR_32BIT_MCDMA : if C_M_AXI_SG_ADDR_WIDTH = 32 generate
begin

       s2mm_desc_curdesc_lsb_nxt   <= desc_reg2;
       s2mm_desc_curdesc_msb_nxt   <= (others => '0');

end generate ADDR_32BIT_MCDMA;


    end generate GEN_MCDMA;

s2mm_desc_cmplt         <= desc_reg9(DESC_STS_CMPLTD_BIT);
s2mm_desc_app0          <= (others => '0');
s2mm_desc_app1          <= (others => '0');
s2mm_desc_app2          <= (others => '0');
s2mm_desc_app3          <= (others => '0');
s2mm_desc_app4          <= (others => '0');

-------------------------------------------------------------------------------
-- BUFFER ADDRESS
-------------------------------------------------------------------------------
-- If 64 bit addressing then concatinate msb to lsb
GEN_NEW_64BIT_BUFADDR : if C_M_AXI_S2MM_ADDR_WIDTH = 64 generate
    s2mm_desc_baddress <= s2mm_desc_baddr_msb & s2mm_desc_baddr_lsb;
--    s2mm_desc_baddr_msb     <= m_axis_s2mm_ftch_tdata_new (128 downto 97);
end generate GEN_NEW_64BIT_BUFADDR;

-- If 32 bit addressing then simply pass lsb out
GEN_NEW_32BIT_BUFADDR : if C_M_AXI_S2MM_ADDR_WIDTH = 32 generate
    s2mm_desc_baddress <= s2mm_desc_baddr_lsb;
end generate GEN_NEW_32BIT_BUFADDR;

-------------------------------------------------------------------------------
-- NEW CURRENT DESCRIPTOR
-------------------------------------------------------------------------------
-- If 64 bit addressing then concatinate msb to lsb
GEN_NEW_64BIT_CURDESC : if C_M_AXI_SG_ADDR_WIDTH = 64 generate
    s2mm_new_curdesc <= s2mm_desc_curdesc_msb_nxt & s2mm_desc_curdesc_lsb_nxt;
end generate GEN_NEW_64BIT_CURDESC;

-- If 32 bit addressing then simply pass lsb out
GEN_NEW_32BIT_CURDESC : if C_M_AXI_SG_ADDR_WIDTH = 32 generate
    s2mm_new_curdesc <= s2mm_desc_curdesc_lsb_nxt;
end generate GEN_NEW_32BIT_CURDESC;

                s2mm_new_curdesc_wren_i <= desc_fetch_done_i; --ftch_shftenbl;

--***************************************************************************--
--**                       Descriptor Update Logic                         **--
--***************************************************************************--
-- SOF Flagging logic for when descriptor queues are enabled in SG Engine
GEN_SOF_QUEUE_MODE : if C_SG_INCLUDE_DESC_QUEUE = 1 generate

-- SOF Queued one count value
constant ONE_COUNT          : std_logic_vector(2 downto 0) := "001";

signal incr_sof_count       : std_logic := '0';
signal decr_sof_count       : std_logic := '0';
signal sof_count            : std_logic_vector(2 downto 0) := (others => '0');
signal sof_received_set     : std_logic := '0';
signal sof_received_clr     : std_logic := '0';
signal cmd_wr_mask          : std_logic := '0';

begin

    -- Keep track of number of commands queued up in data mover to
    -- allow proper setting of SOF's and EOF's when associated
    -- descriptor is updated.
    REG_SOF_COUNT : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    sof_count <= (others => '0');
                elsif(incr_sof_count = '1')then
                    sof_count <= std_logic_vector(unsigned(sof_count(2 downto 0)) + 1);
                elsif(decr_sof_count = '1')then
                    sof_count <= std_logic_vector(unsigned(sof_count(2 downto 0)) - 1);
                end if;
            end if;
        end process REG_SOF_COUNT;

    -- Increment count on each command write that does NOT occur
    -- coincident with a status received
    incr_sof_count  <= s2mm_cmnd_wr and not sts_received_re;

    -- Decrement count on each status received that does NOT
    -- occur coincident with a command write
    decr_sof_count  <= sts_received_re and not s2mm_cmnd_wr;


    -- Drive sof and eof setting to interrupt module for delay interrupt
    --s2mm_packet_sof  <= s2mm_sof_set;
    REG_SOF_STATUS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    sof_received <= '0';
                elsif(sof_received_set = '1')then
                    sof_received <= '1';
                elsif(sof_received_clr = '1')then
                    sof_received <= '0';
                end if;
            end if;
        end process REG_SOF_STATUS;

    -- SOF Received
    -- Case 1 (i.e. already running): EOF received therefore next has to be SOF
    -- Case 2 (i.e. initial command): No commands in queue (count=0) therefore this must be an SOF command
    sof_received_set <= '1' when (sts_received_re = '1'                 -- Status back from Datamover
                              and eof_received = '1')                   -- End of packet received
                                                                        -- OR...
                              or (s2mm_cmnd_wr = '1'                    -- Command written to datamover
                              and cmd_wr_mask = '0'                     -- Not inner-packet command
                              and sof_count = ZERO_VALUE(2 downto 0))   -- No Queued SOF cmnds
                   else '0';

    -- Done with SOF's
    -- Status received and EOF received flag not set
    -- Or status received and EOF received flag set and last SOF
    sof_received_clr <= '1' when (sts_received_re = '1' and eof_received = '0')
                              or (sts_received_re = '1' and eof_received = '1' and sof_count = ONE_COUNT)
                   else '0';

    -- Mask command writes if inner-packet command written.  An inner packet
    -- command is one where status if received and eof_received is not asserted.
    -- This mask is only used for when a cmd_wr occurs and sof_count is zero, meaning
    -- no commands happen to be queued in datamover.
    WR_MASK : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    cmd_wr_mask <= '0';
                -- received data mover status, mask if EOF not set
                -- clear mask if EOF set.
                elsif(sts_received_re = '1')then
                    cmd_wr_mask <= not eof_received;
                end if;
            end if;
        end process WR_MASK;

end generate GEN_SOF_QUEUE_MODE;

-- SOF Flagging logic for when descriptor queues are disabled in SG Engine
GEN_SOF_NO_QUEUE_MODE : if C_SG_INCLUDE_DESC_QUEUE = 0 generate
begin
    -----------------------------------------------------------------------
    -- Assert window around receive packet in order to properly set
    -- SOF and EOF bits in descriptor
    --
    -- SOF for S2MM determined by new command write to datamover, i.e.
    -- command write receive packet not already in progress.
    -----------------------------------------------------------------------
    RX_IN_PROG_PROCESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0' or s2mm_packet_eof = '1')then
                    s2mm_in_progress <= '0';
                    s2mm_sof_set     <= '0';
                elsif(s2mm_in_progress = '0' and s2mm_cmnd_wr = '1')then
                    s2mm_in_progress <= '1';
                    s2mm_sof_set     <= '1';
                else
                    s2mm_in_progress <= s2mm_in_progress;
                    s2mm_sof_set     <= '0';
                end if;
            end if;
        end process RX_IN_PROG_PROCESS;

    -- Drive sof and eof setting to interrupt module for delay interrupt
    --s2mm_packet_sof  <= s2mm_sof_set;
    REG_SOF_STATUS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0' or sts_received_re = '1')then
                    sof_received <= '0';
                elsif(s2mm_sof_set = '1')then
                    sof_received <= '1';
                end if;
            end if;
        end process REG_SOF_STATUS;


end generate GEN_SOF_NO_QUEUE_MODE;

-- IOC and EOF bits in desc update both set via packet eof flag from
-- command/status interface.
eof_received <= s2mm_packet_eof;
s2mm_ioc     <= s2mm_packet_eof;


--***************************************************************************--
--**            Descriptor Update Logic                                    **--
--***************************************************************************--


--*****************************************************************************
--** Pointer Update Logic
--*****************************************************************************

    -----------------------------------------------------------------------
    -- Capture LSB cur descriptor on write for use on descriptor update.
    -- This will be the address the descriptor is updated to
    -----------------------------------------------------------------------
    UPDT_DESC_WRD0: process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    updt_desc_reg0 (31 downto 0) <= (others => '0');
                elsif(s2mm_new_curdesc_wren_i = '1')then

                    updt_desc_reg0 (31 downto 0) <= s2mm_desc_curdesc_lsb;


                end if;
            end if;
        end process UPDT_DESC_WRD0;

    ---------------------------------------------------------------------------
    -- Capture MSB cur descriptor on write for use on descriptor update.
    -- This will be the address the descriptor is updated to
    ---------------------------------------------------------------------------
PTR_64BIT_CURDESC : if C_M_AXI_SG_ADDR_WIDTH = 64 generate
begin
    UPDT_DESC_WRD1: process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    updt_desc_reg0 (C_M_AXI_SG_ADDR_WIDTH-1 downto 32) <= (others => '0');
                elsif(s2mm_new_curdesc_wren_i = '1')then
                    updt_desc_reg0 (C_M_AXI_SG_ADDR_WIDTH-1 downto 32) <= s2mm_desc_curdesc_msb;

                end if;
            end if;
        end process UPDT_DESC_WRD1;
end generate PTR_64BIT_CURDESC;

    -- Shift in pointer to SG engine if tvalid, tready, and not on last word
    updt_shftenbl <=  updt_data and updtptr_tvalid and s_axis_s2mm_updtptr_tready;

    -- Update data done when updating data and tlast received and target
    -- (i.e. SG Engine) is ready
    updt_data_clr <= '1' when updtptr_tvalid = '1'
                          and updtptr_tlast = '1'
                          and s_axis_s2mm_updtptr_tready = '1'
                else '0';

    ---------------------------------------------------------------------------
    -- When desc data ready for update set and hold flag until
    -- data can be updated to queue.  Note it may
    -- be held off due to update of status
    ---------------------------------------------------------------------------
    UPDT_DATA_PROCESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0' or updt_data_clr = '1')then
                    updt_data   <= '0';
                -- clear flag when data update complete
        --        elsif(updt_data_clr = '1')then
        --            updt_data <= '0';
        --        -- set flag when desc fetched as indicated
        --        -- by curdesc wren
                elsif(s2mm_new_curdesc_wren_i = '1')then
                    updt_data <= '1';
                end if;
            end if;
        end process UPDT_DATA_PROCESS;

    updtptr_tvalid  <= updt_data;
    updtptr_tlast   <= DESC_LAST; --updt_desc_reg0(C_S_AXIS_UPDPTR_TDATA_WIDTH);
    updtptr_tdata   <= updt_desc_reg0;



    -- Pass out to sg engine
    s_axis_s2mm_updtptr_tdata    <= updtptr_tdata;
    s_axis_s2mm_updtptr_tlast    <= updtptr_tlast and updtptr_tvalid;
    s_axis_s2mm_updtptr_tvalid   <= updtptr_tvalid;


--*****************************************************************************
--** Status Update Logic - DESCRIPTOR QUEUES INCLUDED                        **
--*****************************************************************************
GEN_DESC_UPDT_QUEUE : if C_SG_INCLUDE_DESC_QUEUE = 1 generate
signal xb_fifo_reset    : std_logic := '0';
signal xb_fifo_full     : std_logic := '0';
begin
    s2mm_complete       <= '1';     -- Fixed at '1'

    -----------------------------------------------------------------------
    -- Need to flag a pending point update to prevent subsequent fetch of
    -- descriptor from stepping on the stored pointer, and buffer length
    -----------------------------------------------------------------------
    REG_PENDING_UPDT : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0' or updt_data_clr = '1')then
                    s2mm_pending_pntr_updt <= '0';
                elsif(s2mm_new_curdesc_wren_i = '1')then
                    s2mm_pending_pntr_updt <= '1';
                end if;
            end if;
        end process REG_PENDING_UPDT;

    -- Pending update on pointer not updated yet or xfer'ed bytes fifo full
    s2mm_pending_update <= s2mm_pending_pntr_updt or xb_fifo_full;

    -- Clear status received flag in cmdsts_if to
    -- allow more status to be received from datamover
    s2mm_sts_received_clr <= updt_sts_clr;

    -- Generate a rising edge off status received in order to
    -- flag status update
    REG_STATUS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    sts_received_d1 <= '0';
                else
                    sts_received_d1 <= s2mm_sts_received;
                end if;
            end if;
        end process REG_STATUS;

    -- CR 566306 Status invalid during halt
    --  sts_received_re <= s2mm_sts_received and not sts_received_d1;
    sts_received_re <= s2mm_sts_received and not sts_received_d1 and not s2mm_halt_d2;

    ---------------------------------------------------------------------------
    -- When status received set and hold flag until
    -- status can be updated to queue.  Note it may
    -- be held off due to update of data
    ---------------------------------------------------------------------------
    UPDT_STS_PROCESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0' or updt_sts_clr = '1')then
                    updt_sts                <= '0';
                -- clear flag when status update done or
                -- datamover halted
      --          elsif(updt_sts_clr = '1')then
      --              updt_sts                <= '0';
                -- set flag when status received
                elsif(sts_received_re = '1')then
                    updt_sts                <= '1';
                end if;
            end if;
        end process UPDT_STS_PROCESS;


    updt_sts_clr <= '1' when updt_sts = '1'
                         and updtsts_tvalid = '1'
                         and updtsts_tlast = '1'
                         and s_axis_s2mm_updtsts_tready = '1'
               else '0';


    -- for queue case used to keep track of number of datamover queued cmnds
    UPDT_DONE_PROCESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    desc_update_done <= '0';
                else
                    desc_update_done <= updt_sts_clr;
                end if;
            end if;
       end process UPDT_DONE_PROCESS;

    --***********************************************************************--
    --**       Descriptor Update Logic - DESCRIPTOR QUEUES - NO STS APP    **--
    --***********************************************************************--

    ---------------------------------------------------------------------------
    -- Generate Descriptor Update Signaling for NO Status App Stream
    ---------------------------------------------------------------------------
    GEN_DESC_UPDT_NO_STSAPP : if C_SG_INCLUDE_STSCNTRL_STRM = 0 generate
    begin

        stsstrm_fifo_rden   <= '0'; -- Not used in the NO sts stream configuration
        xb_fifo_full        <= '0'; -- Not used for indeterminate BTT mode


        -- Transferred byte length from status is equal to bytes transferred field
        -- in descriptor status
        GEN_EQ_23BIT_BYTE_XFERED : if C_SG_LENGTH_WIDTH = 23 generate
        begin

            s2mm_xferd_bytes <= s2mm_brcvd;

        end generate GEN_EQ_23BIT_BYTE_XFERED;

        -- Transferred byte length from status is less than bytes transferred field
        -- in descriptor status therefore need to pad value.
        GEN_LESSTHN_23BIT_BYTE_XFERED : if C_SG_LENGTH_WIDTH < 23 generate
        constant PAD_VALUE : std_logic_vector(22 - C_SG_LENGTH_WIDTH downto 0)
                                := (others => '0');
        begin
            s2mm_xferd_bytes <= PAD_VALUE & s2mm_brcvd;

        end generate GEN_LESSTHN_23BIT_BYTE_XFERED;


        -----------------------------------------------------------------------
        -- Catpure Status.  Status is built from status word from DataMover
        -- and from transferred bytes value.
        -----------------------------------------------------------------------
        UPDT_DESC_STATUS : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        updt_desc_sts  <= (others => '0');

                    elsif(sts_received_re = '1')then
                        updt_desc_sts  <= DESC_LAST
                                         & s2mm_ioc
                                         & s2mm_complete
                                         & s2mm_decerr
                                         & s2mm_slverr
                                         & s2mm_interr
                                         & sof_received -- If asserted also set SOF
                                         & eof_received -- If asserted also set EOF
                                         & RESERVED_STS
                                         & s2mm_xferd_bytes;
                    end if;
                end if;
            end process UPDT_DESC_STATUS;

        -- Drive TVALID
        updtsts_tvalid <= updt_sts;
        -- Drive TLast
        updtsts_tlast  <= updt_desc_sts(C_S_AXIS_UPDSTS_TDATA_WIDTH);
        -- Drive TData
   GEN_DESC_UPDT_MCDMA : if C_ENABLE_MULTI_CHANNEL = 1 generate 
        updtsts_tdata  <= updt_desc_sts(C_S_AXIS_UPDSTS_TDATA_WIDTH-1 downto 20) & 
                          s2mm_desc_info_in (13 downto 10) & "000" & 
                          s2mm_desc_info_in (9 downto 5) & "000" & 
                          s2mm_desc_info_in (4 downto 0);
   end generate GEN_DESC_UPDT_MCDMA;


   GEN_DESC_UPDT_DMA : if C_ENABLE_MULTI_CHANNEL = 0 generate 
        updtsts_tdata  <= updt_desc_sts(C_S_AXIS_UPDSTS_TDATA_WIDTH-1 downto 0);
   end generate GEN_DESC_UPDT_DMA;

    end generate GEN_DESC_UPDT_NO_STSAPP;


    --***********************************************************************--
    --**       Descriptor Update Logic - DESCRIPTOR QUEUES - STS APP       **--
    --***********************************************************************--
    ---------------------------------------------------------------------------
    -- Generate Descriptor Update Signaling for Status App Stream
    ---------------------------------------------------------------------------
    GEN_DESC_UPDT_STSAPP : if C_SG_INCLUDE_STSCNTRL_STRM = 1 generate
    begin


        -- Get rx length is identical to command written, therefor store
        -- the BTT value from the command written to be used as the xferd bytes.
        GEN_USING_STSAPP_LENGTH : if C_SG_USE_STSAPP_LENGTH = 1 generate
        begin
            -----------------------------------------------------------------------
            -- On S2MM transferred bytes equals buffer length.  Capture length
            -- on curdesc write.
            -----------------------------------------------------------------------
            XFERRED_BYTE_FIFO : entity lib_srl_fifo_v1_0_2.srl_fifo_f
              generic map(
                C_DWIDTH        => BUFFER_LENGTH_WIDTH                              ,
                C_DEPTH         => 16                                               ,
                C_FAMILY        => C_FAMILY
                )
              port map(
                Clk             => m_axi_sg_aclk                                  ,
                Reset           => xb_fifo_reset                                    ,
                FIFO_Write      => s2mm_cmnd_wr                                     ,
                Data_In         => s2mm_cmnd_data(BUFFER_LENGTH_WIDTH-1 downto 0)   ,
                FIFO_Read       => sts_received_re                                  ,
                Data_Out        => s2mm_xferd_bytes                                 ,
                FIFO_Empty      => open                                             ,
                FIFO_Full       => xb_fifo_full                                     ,
                Addr            => open
                );

            xb_fifo_reset      <= not m_axi_sg_aresetn;

        end generate GEN_USING_STSAPP_LENGTH;

        -- Not using status app length field therefore primary S2MM DataMover is
        -- configured as a store and forward channel (i.e. indeterminate BTT mode)
        -- Receive length will be reported in datamover status.
        GEN_NOT_USING_STSAPP_LENGTH : if C_SG_USE_STSAPP_LENGTH = 0 generate
        begin
            xb_fifo_full        <= '0';         -- Not used in Indeterminate BTT mode

            -- Transferred byte length from status is equal to bytes transferred field
            -- in descriptor status
            GEN_EQ_23BIT_BYTE_XFERED : if C_SG_LENGTH_WIDTH = 23 generate
            begin

                s2mm_xferd_bytes <= s2mm_brcvd;

            end generate GEN_EQ_23BIT_BYTE_XFERED;

            -- Transferred byte length from status is less than bytes transferred field
            -- in descriptor status therefore need to pad value.
            GEN_LESSTHN_23BIT_BYTE_XFERED : if C_SG_LENGTH_WIDTH < 23 generate
            constant PAD_VALUE : std_logic_vector(22 - C_SG_LENGTH_WIDTH downto 0)
                                    := (others => '0');
            begin
                s2mm_xferd_bytes <= PAD_VALUE & s2mm_brcvd;

            end generate GEN_LESSTHN_23BIT_BYTE_XFERED;

        end generate GEN_NOT_USING_STSAPP_LENGTH;

        -----------------------------------------------------------------------
        -- For EOF Descriptor then need to update APP fields from Status
        -- Stream FIFO
        -----------------------------------------------------------------------
        WRITE_APP_PROCESS : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0' )then

                        writing_app_fields <= '0';
                    -- If writing app fields and reach LAST then stop writing
                    -- app fields
                    elsif(writing_app_fields = '1'                              -- Writing app fields
                    and stsstrm_fifo_dout (C_S_AXIS_S2MM_STS_TDATA_WIDTH) = '1'  -- Last app word (tlast=1)
                    and stsstrm_fifo_rden_i = '1')then                          -- Fifo read
                        writing_app_fields <= '0';

                    -- ON EOF Descriptor, then need to write application fields on desc
                    -- update
                    elsif(s2mm_packet_eof = '1'
                    and s2mm_xferd_bytes /= ZERO_LENGTH) then

                        writing_app_fields <= '1';
                    end if;
                end if;
            end process WRITE_APP_PROCESS;


        -- Shift in apps to SG engine if tvalid, tready, and not on last word
        sts_shftenbl  <=  updt_sts and updtsts_tvalid and s_axis_s2mm_updtsts_tready;

        -----------------------------------------------------------------------
        -- Catpure Status.  Status is built from status word from DataMover
        -- and from transferred bytes value.
        -----------------------------------------------------------------------
        UPDT_DESC_STATUS : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        updt_desc_sts  <= (others => '0');
                    elsif(sts_received_re = '1')then
                        updt_desc_sts  <= DESC_NOT_LAST
                                         & s2mm_ioc
                                         & s2mm_complete
                                         & s2mm_decerr
                                         & s2mm_slverr
                                         & s2mm_interr
                                         & sof_received -- If asserted also set SOF
                                         & eof_received -- If asserted also set EOF
                                         & RESERVED_STS
                                         & s2mm_xferd_bytes;

                    elsif(sts_shftenbl='1')then
                        updt_desc_sts <= updt_desc_reg3;

                    end if;
                end if;
            end process UPDT_DESC_STATUS;


        -----------------------------------------------------------------------
        -- If EOF Descriptor (writing_app_fields=1) then pass data from
        -- status stream FIFO into descriptor update shift registers
        -- Else pass zeros
        -----------------------------------------------------------------------
        UPDT_REG3_MUX : process(writing_app_fields,
                                stsstrm_fifo_dout,
                                updt_zero_reg3,
                                sts_shftenbl)
                begin
                    if(writing_app_fields = '1')then
                        updt_desc_reg3      <= stsstrm_fifo_dout(C_S_AXIS_S2MM_STS_TDATA_WIDTH)              -- Update LAST setting
                                             & '0'
                                             & stsstrm_fifo_dout(C_S_AXIS_S2MM_STS_TDATA_WIDTH-1 downto 0);  -- Update Word
                        stsstrm_fifo_rden_i <= sts_shftenbl;
                    else
                        updt_desc_reg3      <= updt_zero_reg3;
                        stsstrm_fifo_rden_i <= '0';
                    end if;
                end process UPDT_REG3_MUX;

        stsstrm_fifo_rden <= stsstrm_fifo_rden_i;

        -----------------------------------------------------------------------
        -- APP 0 Register (Set to Zero for Non-EOF Descriptor)
        -----------------------------------------------------------------------
        UPDT_ZERO_WRD3  : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0' or sts_received_re = '1')then
                        updt_zero_reg3  <= DESC_NOT_LAST                -- Not last word of stream
                                         & '0'                          -- Don't set IOC
                                         & ZERO_VALUE;                  -- Remainder is zero

                    -- Shift data out on shift enable
                    elsif(sts_shftenbl = '1')then
                        updt_zero_reg3  <= updt_zero_reg4;
                    end if;
                end if;
            end process UPDT_ZERO_WRD3;

        -----------------------------------------------------------------------
        -- APP 1 Register (Set to Zero for Non-EOF Descriptor)
        -----------------------------------------------------------------------
        UPDT_ZERO_WRD4  : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0' or sts_received_re = '1')then
                        updt_zero_reg4  <= DESC_NOT_LAST                -- Not last word of stream
                                         & '0'                          -- Don't set IOC
                                         & ZERO_VALUE;                  -- Remainder is zero
                    -- Shift data out on shift enable
                    elsif(sts_shftenbl = '1')then
                        updt_zero_reg4  <= updt_zero_reg5;
                    end if;
                end if;
            end process UPDT_ZERO_WRD4;

        -----------------------------------------------------------------------
        -- APP 2 Register (Set to Zero for Non-EOF Descriptor)
        -----------------------------------------------------------------------
        UPDT_ZERO_WRD5  : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0' or sts_received_re = '1')then
                        updt_zero_reg5  <= DESC_NOT_LAST                -- Not last word of stream
                                         & '0'                          -- Don't set IOC
                                         & ZERO_VALUE;                  -- Remainder is zero

                    -- Shift data out on shift enable
                    elsif(sts_shftenbl = '1')then
                        updt_zero_reg5  <= updt_zero_reg6;
                    end if;
                end if;
            end process UPDT_ZERO_WRD5;

        -----------------------------------------------------------------------
        -- APP 3 and APP 4 Register (Set to Zero for Non-EOF Descriptor)
        -----------------------------------------------------------------------
        UPDT_ZERO_WRD6  : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0' or sts_received_re = '1')then
                        updt_zero_reg6  <= DESC_NOT_LAST                -- Not last word of stream
                                         & '0'                          -- Don't set IOC
                                         & ZERO_VALUE;                  -- Remainder is zero

                    -- Shift data out on shift enable
                    elsif(sts_shftenbl = '1')then
                        updt_zero_reg6  <= DESC_LAST                    -- Last word of stream
                                         & s2mm_ioc
                                         & ZERO_VALUE;                  -- Remainder is zero
                    end if;
                end if;
            end process UPDT_ZERO_WRD6;

        -----------------------------------------------------------------------
        -- Drive TVALID
        -- If writing app then base on stsstrm fifo empty flag
        -- If writing datamover status then base simply assert on updt_sts
        -----------------------------------------------------------------------
        TVALID_MUX : process(writing_app_fields,updt_sts,stsstrm_fifo_empty)
            begin

                if(updt_sts = '1' and writing_app_fields = '1')then
                    updtsts_tvalid <= not stsstrm_fifo_empty;
                else
                    updtsts_tvalid <= updt_sts;
                end if;

            end process TVALID_MUX;

        -- Drive TLAST
        updtsts_tlast  <= updt_desc_sts(C_S_AXIS_UPDSTS_TDATA_WIDTH);
        -- Drive TDATA
        updtsts_tdata  <= updt_desc_sts(C_S_AXIS_UPDSTS_TDATA_WIDTH-1 downto 0);

    end generate GEN_DESC_UPDT_STSAPP;

    -- Pass out to sg engine
    s_axis_s2mm_updtsts_tdata   <= updtsts_tdata;
    s_axis_s2mm_updtsts_tvalid  <= updtsts_tvalid;
    s_axis_s2mm_updtsts_tlast   <= updtsts_tlast and updtsts_tvalid;


end generate GEN_DESC_UPDT_QUEUE;




--***************************************************************************--
--** Status Update Logic - NO DESCRIPTOR QUEUES                            **--
--***************************************************************************--
GEN_DESC_UPDT_NO_QUEUE : if C_SG_INCLUDE_DESC_QUEUE = 0 generate
begin

    s2mm_sts_received_clr   <= '1'; -- Not needed for the No Queue configuration
    s2mm_complete           <= '1'; -- Fixed at '1' for the No Queue configuration
    s2mm_pending_update     <= '0'; -- Not needed for the No Queue configuration

    -- Status received based on a DONE or an ERROR from DataMover
    sts_received <= s2mm_done or s2mm_interr or s2mm_decerr or s2mm_slverr;

    -- Generate a rising edge off done for use in triggering an
    -- update to the SG engine
    REG_STATUS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    sts_received_d1 <= '0';
                else
                    sts_received_d1 <= sts_received;
                end if;
            end if;
        end process REG_STATUS;

    -- CR 566306 Status invalid during halt
    --  sts_received_re <= sts_received and not sts_received_d1;
    sts_received_re <= sts_received and not sts_received_d1 and not s2mm_halt_d2;


    ---------------------------------------------------------------------------
    -- When status received set and hold flag until
    -- status can be updated to queue.  Note it may
    -- be held off due to update of data
    ---------------------------------------------------------------------------
    UPDT_STS_PROCESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    updt_sts                <= '0';
                -- clear flag when status update done
                elsif(updt_sts_clr = '1')then
                    updt_sts                <= '0';
                -- set flag when status received
                elsif(sts_received_re = '1')then
                    updt_sts                <= '1';
                end if;
            end if;
        end process UPDT_STS_PROCESS;


    -- Clear status update on acceptance of tlast by sg engine
    updt_sts_clr <= '1' when updt_sts = '1'
                         and updtsts_tvalid = '1'
                         and updtsts_tlast = '1'
                         and s_axis_s2mm_updtsts_tready = '1'
               else '0';


    -- for queue case used to keep track of number of datamover queued cmnds
    UPDT_DONE_PROCESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    desc_update_done <= '0';
                else
                    desc_update_done <= updt_sts_clr;
                end if;

            end if;
       end process UPDT_DONE_PROCESS;


    --***********************************************************************--
    --**  Descriptor Update Logic - NO DESCRIPTOR QUEUES - NO STS APP      **--
    --***********************************************************************--
    ---------------------------------------------------------------------------
    -- Generate Descriptor Update Signaling for NO Status App Stream
    ---------------------------------------------------------------------------
    GEN_DESC_UPDT_NO_STSAPP : if C_SG_INCLUDE_STSCNTRL_STRM = 0 generate
    begin

        stsstrm_fifo_rden <= '0';   -- Not used in the NO sts stream configuration

      GEN_NO_MICRO_DMA : if C_MICRO_DMA = 0 generate
        begin

        -- Transferred byte length from status is equal to bytes transferred field
        -- in descriptor status
        GEN_EQ_23BIT_BYTE_XFERED : if C_SG_LENGTH_WIDTH = 23 generate
        begin

            s2mm_xferd_bytes <= s2mm_brcvd;

        end generate GEN_EQ_23BIT_BYTE_XFERED;

        -- Transferred byte length from status is less than bytes transferred field
        -- in descriptor status therefore need to pad value.
        GEN_LESSTHN_23BIT_BYTE_XFERED : if C_SG_LENGTH_WIDTH < 23 generate
        constant PAD_VALUE : std_logic_vector(22 - C_SG_LENGTH_WIDTH downto 0)
                                := (others => '0');
        begin
            s2mm_xferd_bytes <= PAD_VALUE & s2mm_brcvd;

        end generate GEN_LESSTHN_23BIT_BYTE_XFERED;

       end generate GEN_NO_MICRO_DMA;

       GEN_MICRO_DMA : if C_MICRO_DMA = 1 generate
         begin
            s2mm_xferd_bytes <= (others => '0');
       end generate GEN_MICRO_DMA;
        -----------------------------------------------------------------------
        -- Catpure Status.  Status is built from status word from DataMover
        -- and from transferred bytes value.
        -----------------------------------------------------------------------
        UPDT_DESC_WRD2 : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        updt_desc_sts  <= (others => '0');
                    -- Register Status on status received rising edge
                    elsif(sts_received_re = '1')then
                        updt_desc_sts  <= DESC_LAST
                                         & s2mm_ioc
                                         & s2mm_complete
                                         & s2mm_decerr
                                         & s2mm_slverr
                                         & s2mm_interr
                                         & sof_received -- If asserted also set SOF
                                         & eof_received -- If asserted also set EOF
                                         & RESERVED_STS
                                         & s2mm_xferd_bytes;

                    end if;
                end if;
            end process UPDT_DESC_WRD2;

  GEN_DESC_UPDT_MCDMA_NOQUEUE : if C_ENABLE_MULTI_CHANNEL = 1 generate
        updtsts_tdata  <= updt_desc_sts(C_S_AXIS_UPDSTS_TDATA_WIDTH-1 downto 20) & 
                          s2mm_desc_info_in (13 downto 10) & "000" & 
                          s2mm_desc_info_in (9 downto 5) & "000" & 
                          s2mm_desc_info_in (4 downto 0);
   end generate GEN_DESC_UPDT_MCDMA_NOQUEUE;


   GEN_DESC_UPDT_DMA_NOQUEUE : if C_ENABLE_MULTI_CHANNEL = 0 generate
        updtsts_tdata  <= updt_desc_sts(C_S_AXIS_UPDSTS_TDATA_WIDTH-1 downto 0);
   end generate GEN_DESC_UPDT_DMA_NOQUEUE; 
        -- Drive TVALID
        updtsts_tvalid <= updt_sts;
        -- Drive TLAST
        updtsts_tlast  <= updt_desc_sts(C_S_AXIS_UPDSTS_TDATA_WIDTH);
        -- Drive TData
  --      updtsts_tdata  <= updt_desc_sts(C_S_AXIS_UPDSTS_TDATA_WIDTH - 1 downto 0);


    end generate GEN_DESC_UPDT_NO_STSAPP;



    --***********************************************************************--
    --**    Descriptor Update Logic - NO DESCRIPTOR QUEUES - STS APP       **--
    --***********************************************************************--
    ---------------------------------------------------------------------------
    -- Generate Descriptor Update Signaling for NO Status App Stream
    ---------------------------------------------------------------------------
    GEN_DESC_UPDT_STSAPP : if C_SG_INCLUDE_STSCNTRL_STRM = 1 generate
    begin

        -- Rx length is identical to command written, therefore store
        -- the BTT value from the command written to be used as the xferd bytes.
        GEN_USING_STSAPP_LENGTH : if C_SG_USE_STSAPP_LENGTH = 1 generate
        begin
            -----------------------------------------------------------------------
            -- On S2MM transferred bytes equals buffer length.  Capture length
            -- on curdesc write.
            -----------------------------------------------------------------------
            REG_XFERRED_BYTES : process(m_axi_sg_aclk)
                begin
                    if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                        if(m_axi_sg_aresetn = '0')then
                            s2mm_xferd_bytes <= (others => '0');
                        elsif(s2mm_cmnd_wr = '1')then
                            s2mm_xferd_bytes <= s2mm_cmnd_data(BUFFER_LENGTH_WIDTH-1 downto 0);
                        end if;
                    end if;
                end process REG_XFERRED_BYTES;
        end generate GEN_USING_STSAPP_LENGTH;

        -- Configured as a store and forward channel (i.e. indeterminate BTT mode)
        -- Receive length will be reported in datamover status.
        GEN_NOT_USING_STSAPP_LENGTH : if C_SG_USE_STSAPP_LENGTH = 0 generate
        begin

            -- Transferred byte length from status is equal to bytes transferred field
            -- in descriptor status
            GEN_EQ_23BIT_BYTE_XFERED : if C_SG_LENGTH_WIDTH = 23 generate
            begin

                s2mm_xferd_bytes <= s2mm_brcvd;

            end generate GEN_EQ_23BIT_BYTE_XFERED;

            -- Transferred byte length from status is less than bytes transferred field
            -- in descriptor status therefore need to pad value.
            GEN_LESSTHN_23BIT_BYTE_XFERED : if C_SG_LENGTH_WIDTH < 23 generate
            constant PAD_VALUE : std_logic_vector(22 - C_SG_LENGTH_WIDTH downto 0)
                                    := (others => '0');
            begin
                s2mm_xferd_bytes <= PAD_VALUE & s2mm_brcvd;

            end generate GEN_LESSTHN_23BIT_BYTE_XFERED;


        end generate GEN_NOT_USING_STSAPP_LENGTH;

        -----------------------------------------------------------------------
        -- For EOF Descriptor then need to update APP fields from Status
        -- Stream FIFO
        -----------------------------------------------------------------------
        WRITE_APP_PROCESS : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then

                        writing_app_fields <= '0';

                    -- If writing app fields and reach LAST then stop writing
                    -- app fields
                    elsif(writing_app_fields = '1'                              -- Writing app fields
                    and stsstrm_fifo_dout(C_S_AXIS_S2MM_STS_TDATA_WIDTH) = '1'  -- Last app word (tlast=1)
                    and stsstrm_fifo_rden_i = '1')then                          -- Fifo read
                        writing_app_fields <= '0';

                    -- ON EOF Descriptor, then need to write application fields on desc
                    -- update
                    elsif(eof_received = '1'
                    and s2mm_xferd_bytes /= ZERO_LENGTH) then
                        writing_app_fields <= '1';
                    end if;
                end if;
            end process WRITE_APP_PROCESS;

        -- Shift in apps to SG engine if tvalid, tready, and not on last word
        sts_shftenbl  <=  updt_sts and updtsts_tvalid and s_axis_s2mm_updtsts_tready;

        -----------------------------------------------------------------------
        -- Catpure Status.  Status is built from status word from DataMover
        -- and from transferred bytes value.
        -----------------------------------------------------------------------
        UPDT_DESC_WRD2 : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        updt_desc_sts  <= (others => '0');
                    -- Status from Prmry Datamover received
                    elsif(sts_received_re = '1')then
                        updt_desc_sts  <= DESC_NOT_LAST
                                         & s2mm_ioc
                                         & s2mm_complete
                                         & s2mm_decerr
                                         & s2mm_slverr
                                         & s2mm_interr
                                         & sof_received -- If asserted also set SOF
                                         & eof_received -- If asserted also set EOF
                                         & RESERVED_STS
                                         & s2mm_xferd_bytes;
                    -- Shift on descriptor update
                    elsif(sts_shftenbl = '1')then
                        updt_desc_sts <= updt_desc_reg3;

                    end if;
                end if;
            end process UPDT_DESC_WRD2;

        -----------------------------------------------------------------------
        -- If EOF Descriptor (writing_app_fields=1) then pass data from
        -- status stream FIFO into descriptor update shift registers
        -- Else pass zeros
        -----------------------------------------------------------------------
        UPDT_REG3_MUX : process(writing_app_fields,
                                stsstrm_fifo_dout,
                                updt_zero_reg3,
                                sts_shftenbl)
                begin
                    if(writing_app_fields = '1')then
                        updt_desc_reg3      <= stsstrm_fifo_dout(C_S_AXIS_S2MM_STS_TDATA_WIDTH)              -- Update LAST setting
                                             & '0'
                                             & stsstrm_fifo_dout(C_S_AXIS_S2MM_STS_TDATA_WIDTH-1 downto 0);  -- Update Word
                        stsstrm_fifo_rden_i <= sts_shftenbl;
                    else
                        updt_desc_reg3      <= updt_zero_reg3;
                        stsstrm_fifo_rden_i <= '0';
                    end if;
                end process UPDT_REG3_MUX;

        stsstrm_fifo_rden <= stsstrm_fifo_rden_i;

        -----------------------------------------------------------------------
        -- APP 0 Register (Set to Zero for Non-EOF Descriptor)
        -----------------------------------------------------------------------
        UPDT_ZERO_WRD3  : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0' or sts_received_re = '1')then
                        updt_zero_reg3  <= (others => '0');
                    -- Shift data out on shift enable
                    elsif(sts_shftenbl = '1')then
                        updt_zero_reg3  <= updt_zero_reg4;
                    end if;
                end if;
            end process UPDT_ZERO_WRD3;

        -----------------------------------------------------------------------
        -- APP 1 Register (Set to Zero for Non-EOF Descriptor)
        -----------------------------------------------------------------------
        UPDT_ZERO_WRD4  : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0' or sts_received_re = '1')then
                        updt_zero_reg4  <= (others => '0');
                    -- Shift data out on shift enable
                    elsif(sts_shftenbl = '1')then
                        updt_zero_reg4  <= updt_zero_reg5;
                    end if;
                end if;
            end process UPDT_ZERO_WRD4;

        -----------------------------------------------------------------------
        -- APP 2 Register (Set to Zero for Non-EOF Descriptor)
        -----------------------------------------------------------------------
        UPDT_ZERO_WRD5  : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0' or sts_received_re = '1')then
                        updt_zero_reg5  <= (others => '0');
                    -- Shift data out on shift enable
                    elsif(sts_shftenbl = '1')then
                        updt_zero_reg5  <= updt_zero_reg6;
                    end if;
                end if;
            end process UPDT_ZERO_WRD5;

        -----------------------------------------------------------------------
        -- APP 3 Register (Set to Zero for Non-EOF Descriptor)
        -----------------------------------------------------------------------
        UPDT_ZERO_WRD6  : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0' or sts_received_re = '1')then
                        updt_zero_reg6  <= (others => '0');
                    -- Shift data out on shift enable
                    elsif(sts_shftenbl = '1')then
                        updt_zero_reg6  <= updt_zero_reg7;
                    end if;
                end if;
            end process UPDT_ZERO_WRD6;

        -----------------------------------------------------------------------
        -- APP 4 Register (Set to Zero for Non-EOF Descriptor)
        -----------------------------------------------------------------------
        UPDT_ZERO_WRD7  : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        updt_zero_reg7  <= (others => '0');
                    elsif(sts_received_re = '1')then
                        updt_zero_reg7 <=  DESC_LAST
                                         & '0'
                                         & ZERO_VALUE;
                    end if;
                end if;
            end process UPDT_ZERO_WRD7;

        -----------------------------------------------------------------------
        -- Drive TVALID
        -- If writing app then base on stsstrm fifo empty flag
        -- If writing datamover status then base simply assert on updt_sts
        -----------------------------------------------------------------------
        TVALID_MUX : process(writing_app_fields,updt_sts,stsstrm_fifo_empty)
            begin

                if(updt_sts = '1' and writing_app_fields = '1')then
                    updtsts_tvalid <= not stsstrm_fifo_empty;
                else
                    updtsts_tvalid <= updt_sts;
                end if;

            end process TVALID_MUX;


        -- Drive TDATA
        updtsts_tdata <= updt_desc_sts(C_S_AXIS_UPDSTS_TDATA_WIDTH-1 downto 0);

        -- DRIVE TLAST
        updtsts_tlast <= updt_desc_sts(C_S_AXIS_UPDSTS_TDATA_WIDTH);



    end generate GEN_DESC_UPDT_STSAPP;


    -- Pass out to sg engine
    s_axis_s2mm_updtsts_tdata   <= updtsts_tdata;
    s_axis_s2mm_updtsts_tvalid  <= updtsts_tvalid;
    s_axis_s2mm_updtsts_tlast   <= updtsts_tlast and updtsts_tvalid;



end generate GEN_DESC_UPDT_NO_QUEUE;




end implementation;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:          axi_dma_s2mm_sm.vhd
-- Description: This entity contains the S2MM DMA Controller State Machine
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;

library lib_pkg_v1_0_2;
use lib_pkg_v1_0_2.lib_pkg.clog2;


-------------------------------------------------------------------------------
entity  axi_dma_s2mm_sm is
    generic (
        C_M_AXI_S2MM_ADDR_WIDTH     : integer range 32 to 64    := 32;
            -- Master AXI Memory Map Address Width for S2MM Write Port

        C_SG_INCLUDE_STSCNTRL_STRM  : integer range 0 to 1      := 1;
            -- Include or Exclude AXI Status and AXI Control Streams
            -- 0 = Exclude Status and Control Streams
            -- 1 = Include Status and Control Streams

        C_SG_USE_STSAPP_LENGTH      : integer range 0 to 1      := 1;
            -- Enable or Disable use of Status Stream Rx Length.  Only valid
            -- if C_SG_INCLUDE_STSCNTRL_STRM = 1
            -- 0 = Don't use Rx Length
            -- 1 = Use Rx Length

        C_SG_LENGTH_WIDTH           : integer range 8 to 23     := 14;
            -- Width of Buffer Length, Transferred Bytes, and BTT fields

        C_SG_INCLUDE_DESC_QUEUE     : integer range 0 to 1      := 0;
            -- Include or Exclude Scatter Gather Descriptor Queuing
            -- 0 = Exclude SG Descriptor Queuing
            -- 1 = Include SG Descriptor Queuing
        C_ENABLE_MULTI_CHANNEL             : integer range 0 to 1 := 0;

        C_MICRO_DMA      : integer range 0 to 1 := 0;

        C_PRMY_CMDFIFO_DEPTH        : integer range 1 to 16     := 1
            -- Depth of DataMover command FIFO
    );
    port (
        m_axi_sg_aclk               : in  std_logic                         ;                   --
        m_axi_sg_aresetn            : in  std_logic                         ;                   --
                                                                                                --
        s2mm_stop                   : in  std_logic                         ;                   --
                                                                                                --
        -- S2MM Control and Status                                                              --
        s2mm_run_stop               : in  std_logic                         ;                   --
        s2mm_keyhole                : in  std_logic                         ;                   --
        s2mm_ftch_idle              : in  std_logic                         ;                   --
        s2mm_desc_flush             : in  std_logic                         ;                   --
        s2mm_cmnd_idle              : out std_logic                         ;                   --
        s2mm_sts_idle               : out std_logic                         ;                   --
        s2mm_eof_set                : out std_logic                         ;                   --
        s2mm_eof_micro              : in std_logic                         ;                   --
        s2mm_sof_micro              : in std_logic                         ;                   --
                                                                                                --
        -- S2MM Descriptor Fetch Request                                                        --
        desc_fetch_req              : out std_logic                         ;                   --
        desc_fetch_done             : in  std_logic                         ;                   --
        desc_update_done            : in  std_logic                         ;                   --
        updt_pending                : in  std_logic                         ;
        desc_available              : in  std_logic                         ;                   --
                                                                                                --
        -- S2MM Status Stream RX Length                                                         --
        s2mm_rxlength_valid         : in  std_logic                         ;                   --
        s2mm_rxlength_clr           : out std_logic                         ;                   --
        s2mm_rxlength               : in  std_logic_vector                                      --
                                        (C_SG_LENGTH_WIDTH - 1 downto 0)    ;                   --
                                                                                                --
        -- DataMover Command                                                                    --
        s2mm_cmnd_wr                : out std_logic                         ;                   --
        s2mm_cmnd_data              : out std_logic_vector                                      --
                                        ((C_M_AXI_S2MM_ADDR_WIDTH-32+2*32+CMD_BASE_WIDTH+46)-1 downto 0);  --
        s2mm_cmnd_pending           : in  std_logic                         ;                   --
                                                                                                --
        -- Descriptor Fields                                                                    --
        s2mm_desc_info          : in  std_logic_vector                                      --
                                        (31 downto 0);                   --
        s2mm_desc_baddress          : in  std_logic_vector                                      --
                                        (C_M_AXI_S2MM_ADDR_WIDTH-1 downto 0);                   --
        s2mm_desc_blength           : in  std_logic_vector                                      --
                                        (BUFFER_LENGTH_WIDTH-1 downto 0);                        --
        s2mm_desc_blength_v           : in  std_logic_vector                                      --
                                        (BUFFER_LENGTH_WIDTH-1 downto 0);                        --
        s2mm_desc_blength_s           : in  std_logic_vector                                      --
                                        (BUFFER_LENGTH_WIDTH-1 downto 0)                        --

    );

end axi_dma_s2mm_sm;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_dma_s2mm_sm is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------
-- DataMover Commmand TAG
constant S2MM_CMD_TAG       : std_logic_vector(2 downto 0)  := (others => '0');
-- DataMover Command Destination Stream Offset
constant S2MM_CMD_DSA       : std_logic_vector(5 downto 0)  := (others => '0');
-- DataMover Cmnd Reserved Bits
constant S2MM_CMD_RSVD      : std_logic_vector(
                                DATAMOVER_CMD_RSVMSB_BOFST + C_M_AXI_S2MM_ADDR_WIDTH downto
                                DATAMOVER_CMD_RSVLSB_BOFST + C_M_AXI_S2MM_ADDR_WIDTH)
                                := (others => '0');
-- Queued commands counter width
constant COUNTER_WIDTH      : integer := clog2(C_PRMY_CMDFIFO_DEPTH+1);

-- Queued commands zero count
constant ZERO_COUNT         : std_logic_vector(COUNTER_WIDTH - 1 downto 0)
                                := (others => '0');
-- Zero buffer length error - compare value
constant ZERO_LENGTH        : std_logic_vector(C_SG_LENGTH_WIDTH-1 downto 0)
                                := (others => '0');
constant ZERO_BUFFER        : std_logic_vector(BUFFER_LENGTH_WIDTH-1 downto 0)
                                := (others => '0');
-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
-- State Machine Signals
signal desc_fetch_req_cmb       : std_logic := '0';
signal write_cmnd_cmb           : std_logic := '0';
signal s2mm_rxlength_clr_cmb    : std_logic := '0';

signal rxlength                 : std_logic_vector(C_SG_LENGTH_WIDTH-1 downto 0) := (others => '0');
signal s2mm_rxlength_set        : std_logic := '0';
signal blength_grtr_rxlength    : std_logic := '0';
signal rxlength_fetched         : std_logic := '0';

signal cmnds_queued             : std_logic_vector(COUNTER_WIDTH - 1 downto 0) := (others => '0');
signal cmnds_queued_shift             : std_logic_vector(C_PRMY_CMDFIFO_DEPTH - 1 downto 0) := (others => '0');
signal count_incr               : std_logic := '0';
signal count_decr               : std_logic := '0';

signal desc_fetch_done_d1       : std_logic := '0';
signal zero_length_error        : std_logic := '0';
signal s2mm_eof_set_i           : std_logic := '0';

signal queue_more               : std_logic := '0';

signal burst_type               : std_logic;
signal eof_micro                : std_logic;

-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin

EN_MICRO_DMA : if C_MICRO_DMA = 1 generate
begin
 eof_micro <= s2mm_eof_micro;
end generate EN_MICRO_DMA;


NO_MICRO_DMA : if C_MICRO_DMA = 0 generate
begin
 eof_micro <= '0';
end generate NO_MICRO_DMA;

s2mm_eof_set <= s2mm_eof_set_i;

burst_type <= '1' and (not s2mm_keyhole);
-- A 0 s2mm_keyhole means incremental burst
-- a 1 s2mm_keyhole means fixed burst

-------------------------------------------------------------------------------
-- Not using rx length from status stream - (indeterminate length mode)
-------------------------------------------------------------------------------
GEN_SM_FOR_NO_LENGTH : if (C_SG_USE_STSAPP_LENGTH = 0 or C_SG_INCLUDE_STSCNTRL_STRM = 0 or C_ENABLE_MULTI_CHANNEL = 1) generate
type SG_S2MM_STATE_TYPE      is (
                                IDLE,
                                FETCH_DESCRIPTOR,
                           --     EXECUTE_XFER,
                                WAIT_STATUS
                                );

signal s2mm_cs                  : SG_S2MM_STATE_TYPE;
signal s2mm_ns                  : SG_S2MM_STATE_TYPE;


begin
    -- For no status stream or not using length in status app field then eof set is
    -- generated from datamover status (see axi_dma_s2mm_cmdsts_if.vhd)
    s2mm_eof_set_i        <= '0';

    -------------------------------------------------------------------------------
    -- S2MM Transfer State Machine
    -------------------------------------------------------------------------------
    S2MM_MACHINE : process(s2mm_cs,
                           s2mm_run_stop,
                           desc_available,
                           desc_fetch_done,
                           desc_update_done,
                           s2mm_cmnd_pending,
                           s2mm_stop,
                           s2mm_desc_flush,
                           updt_pending 
                        --   queue_more
                           )
        begin

            -- Default signal assignment
            desc_fetch_req_cmb      <= '0';
            write_cmnd_cmb          <= '0';
            s2mm_cmnd_idle          <= '0';
            s2mm_ns                 <= s2mm_cs;

            case s2mm_cs is

                -------------------------------------------------------------------
                when IDLE =>
                    -- fetch descriptor if desc available, not stopped and running
    --                if (updt_pending = '1') then
    --                      s2mm_ns <= WAIT_STATUS;
                    if(s2mm_run_stop = '1' and desc_available = '1'
                --    and s2mm_stop = '0' and queue_more = '1' and updt_pending = '0')then
                    and s2mm_stop = '0' and updt_pending = '0')then
                       if (C_SG_INCLUDE_DESC_QUEUE = 1) then
                          s2mm_ns <= FETCH_DESCRIPTOR;
                          desc_fetch_req_cmb  <= '1';
                       else
                          s2mm_ns <= WAIT_STATUS;
                          write_cmnd_cmb  <= '1';
                       end if;
                    else
                        s2mm_cmnd_idle <= '1';
                        s2mm_ns         <= IDLE;
                    end if;

                -------------------------------------------------------------------
                when FETCH_DESCRIPTOR =>
                    -- exit if error or descriptor flushed
                    if(s2mm_desc_flush = '1' or s2mm_stop = '1')then
                        s2mm_ns         <= IDLE;
                    -- wait until fetch complete then execute
               --     elsif(desc_fetch_done = '1')then
               --         desc_fetch_req_cmb  <= '0';
               --         s2mm_ns             <= EXECUTE_XFER;
                    elsif (s2mm_cmnd_pending = '0')then
                        desc_fetch_req_cmb  <= '0';
                        if (updt_pending = '0') then
                            if(C_SG_INCLUDE_DESC_QUEUE = 1)then
                              s2mm_ns         <= IDLE;
                              write_cmnd_cmb  <= '1';
                            else
--              coverage off
                               s2mm_ns         <= WAIT_STATUS;
--              coverage on
                            end if;
                        end if; 
                    else
                          s2mm_ns <= FETCH_DESCRIPTOR;
                    end if;

                -------------------------------------------------------------------
--                when EXECUTE_XFER =>
--                    -- if error exit
--                    if(s2mm_stop = '1')then
--                        s2mm_ns         <= IDLE;
--                    -- Write another command if there is not one already pending
--                    elsif(s2mm_cmnd_pending = '0')then
--                        if (updt_pending = '0') then
--                          write_cmnd_cmb  <= '1';
--                        end if;
--                        if(C_SG_INCLUDE_DESC_QUEUE = 1)then
--                            s2mm_ns         <= IDLE;
--                        else
--                            s2mm_ns         <= WAIT_STATUS;
--                        end if;
--                    else
--                        s2mm_ns <= EXECUTE_XFER;
--                    end if;

                -------------------------------------------------------------------
                when WAIT_STATUS =>
                    -- for no Q wait until desc updated
                    if(desc_update_done = '1' or s2mm_stop = '1')then
                        s2mm_ns <= IDLE;
                    else
                        s2mm_ns <= WAIT_STATUS;
                    end if;

                -------------------------------------------------------------------
--              coverage off
                when others =>
                    s2mm_ns <= IDLE;
--              coverage on

            end case;
        end process S2MM_MACHINE;

    -------------------------------------------------------------------------------
    -- Register State Machine Statues
    -------------------------------------------------------------------------------
    REGISTER_STATE : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    s2mm_cs     <= IDLE;
                else
                    s2mm_cs     <= s2mm_ns;
                end if;
            end if;
        end process REGISTER_STATE;

    -------------------------------------------------------------------------------
    -- Register State Machine Signalse
    -------------------------------------------------------------------------------
--    SM_SIG_REGISTER : process(m_axi_sg_aclk)
--        begin
--            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
--                if(m_axi_sg_aresetn = '0')then
--                    desc_fetch_req      <= '0'      ;
--                else
--                    if (C_SG_INCLUDE_DESC_QUEUE = 0) then
--                       desc_fetch_req      <= '1';
--                    else
--                       desc_fetch_req      <= desc_fetch_req_cmb   ;
--                    end if;
--                end if;
--            end if;
--        end process SM_SIG_REGISTER;
           desc_fetch_req <= '1' when (C_SG_INCLUDE_DESC_QUEUE = 0) else 
                             desc_fetch_req_cmb ;

    -------------------------------------------------------------------------------
    -- Build DataMover command
    -------------------------------------------------------------------------------
    -- If Bytes To Transfer (BTT) width less than 23, need to add pad
    GEN_CMD_BTT_LESS_23 : if C_SG_LENGTH_WIDTH < 23 generate
    constant PAD_VALUE : std_logic_vector(22 - C_SG_LENGTH_WIDTH downto 0)
                            := (others => '0');
    begin
        -- When command by sm, drive command to s2mm_cmdsts_if
        GEN_DATAMOVER_CMND : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        s2mm_cmnd_wr    <= '0';
               --         s2mm_cmnd_data  <= (others => '0');

                    -- Fetch SM issued a command write
                    elsif(write_cmnd_cmb = '1')then
                        s2mm_cmnd_wr    <= '1';
               --         s2mm_cmnd_data  <=  s2mm_desc_info
               --                             & s2mm_desc_blength_v
               --                             & s2mm_desc_blength_s
               --                             & S2MM_CMD_RSVD
               --                             & "0000"  -- Cat IOC to CMD TAG
               --                             & s2mm_desc_baddress
               --                             & '1'           -- Always reset DRE
               --                             & '0'           -- For Indeterminate BTT mode do not set EOF
               --                             & S2MM_CMD_DSA
               --                             & burst_type  -- Key Hole '1'           -- s2mm_desc_type -- IR# 545697
               --                             & PAD_VALUE
               --                             & s2mm_desc_blength(C_SG_LENGTH_WIDTH-1 downto 0);
                    else
                        s2mm_cmnd_wr    <= '0';

                    end if;
                end if;
            end process GEN_DATAMOVER_CMND;

                        s2mm_cmnd_data  <=  s2mm_desc_info
                                            & s2mm_desc_blength_v
                                            & s2mm_desc_blength_s
                                            & S2MM_CMD_RSVD
                                            & "00" & eof_micro & eof_micro --00"  -- Cat IOC to CMD TAG
                                            & s2mm_desc_baddress
                                            & '1'           -- Always reset DRE
                                            & eof_micro --'0'           -- For Indeterminate BTT mode do not set EOF
                                            & S2MM_CMD_DSA
                                            & burst_type  -- Key Hole '1'           -- s2mm_desc_type -- IR# 545697
                                            & PAD_VALUE
                                            & s2mm_desc_blength(C_SG_LENGTH_WIDTH-1 downto 0);

    end generate GEN_CMD_BTT_LESS_23;

    -- If Bytes To Transfer (BTT) width equal 23, no required pad
    GEN_CMD_BTT_EQL_23 : if C_SG_LENGTH_WIDTH = 23 generate
    begin
        -- When command by sm, drive command to s2mm_cmdsts_if
        GEN_DATAMOVER_CMND : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        s2mm_cmnd_wr    <= '0';
               --         s2mm_cmnd_data  <= (others => '0');

                    -- Fetch SM issued a command write
                    elsif(write_cmnd_cmb = '1')then
                        s2mm_cmnd_wr    <= '1';
               --         s2mm_cmnd_data  <=  s2mm_desc_info 
               --                             & s2mm_desc_blength_v
               --                             & s2mm_desc_blength_s 
               --                             & S2MM_CMD_RSVD
               --                             & "0000"  -- Cat IOC to CMD TAG
               --                             & s2mm_desc_baddress
               --                             & '1'           -- Always reset DRE
               --                             & '0'           -- For indeterminate BTT mode do not set EOF
               --                             & S2MM_CMD_DSA
               --                             & burst_type -- Key Hole '1'           -- s2mm_desc_type -- IR# 545697
               --                             & s2mm_desc_blength;

                    else
                        s2mm_cmnd_wr    <= '0';

                    end if;
                end if;
            end process GEN_DATAMOVER_CMND;

                        s2mm_cmnd_data  <=  s2mm_desc_info 
                                            & s2mm_desc_blength_v
                                            & s2mm_desc_blength_s 
                                            & S2MM_CMD_RSVD
                                            & "00" & eof_micro & eof_micro -- "0000"  -- Cat IOC to CMD TAG
                                            & s2mm_desc_baddress
                                            & '1'           -- Always reset DRE
                                            & eof_micro           -- For indeterminate BTT mode do not set EOF
                                            & S2MM_CMD_DSA
                                            & burst_type -- Key Hole '1'           -- s2mm_desc_type -- IR# 545697
                                            & s2mm_desc_blength;

    end generate GEN_CMD_BTT_EQL_23;



    -- Drive unused output to zero
    s2mm_rxlength_clr   <= '0';

end generate GEN_SM_FOR_NO_LENGTH;



-------------------------------------------------------------------------------
-- Generate state machine and support logic for Using RX Length from Status
-- Stream
-------------------------------------------------------------------------------
-- this would not hold good for MCDMA
GEN_SM_FOR_LENGTH : if (C_SG_USE_STSAPP_LENGTH = 1 and C_SG_INCLUDE_STSCNTRL_STRM = 1 and C_ENABLE_MULTI_CHANNEL = 0) generate
type SG_S2MM_STATE_TYPE      is (
                                IDLE,
                                FETCH_DESCRIPTOR,
                                GET_RXLENGTH,
                                CMPR_LENGTH,
                                EXECUTE_XFER,
                                WAIT_STATUS
                                );

signal s2mm_cs                  : SG_S2MM_STATE_TYPE;
signal s2mm_ns                  : SG_S2MM_STATE_TYPE;

begin

    -------------------------------------------------------------------------------
    -- S2MM Transfer State Machine
    -------------------------------------------------------------------------------
    S2MM_MACHINE : process(s2mm_cs,
                           s2mm_run_stop,
                           desc_available,
                           desc_update_done,
                       --    desc_fetch_done,
                           updt_pending,
                           s2mm_rxlength_valid,
                           rxlength_fetched,
                           s2mm_cmnd_pending,
                           zero_length_error,
                           s2mm_stop,
                           s2mm_desc_flush
                        --   queue_more
                           )
        begin

            -- Default signal assignment
            desc_fetch_req_cmb      <= '0';
            s2mm_rxlength_clr_cmb   <= '0';
            write_cmnd_cmb          <= '0';
            s2mm_cmnd_idle          <= '0';
            s2mm_rxlength_set       <= '0';
            --rxlength_fetched_clr    <= '0';
            s2mm_ns                 <= s2mm_cs;

            case s2mm_cs is

                -------------------------------------------------------------------
                when IDLE =>
                    if(s2mm_run_stop = '1' and desc_available = '1'
                 --   and s2mm_stop = '0' and queue_more = '1' and updt_pending = '0')then
                    and s2mm_stop = '0' and updt_pending = '0')then
                      if (C_SG_INCLUDE_DESC_QUEUE = 0) then
                        if(rxlength_fetched = '0')then
                            s2mm_ns             <= GET_RXLENGTH;
                        else
                            s2mm_ns             <= CMPR_LENGTH;
                        end if;
                      else  
                        s2mm_ns <= FETCH_DESCRIPTOR;
                        desc_fetch_req_cmb  <= '1';
                      end if;
                    else
                        s2mm_cmnd_idle <= '1';
                        s2mm_ns <= IDLE; --FETCH_DESCRIPTOR;
                    end if;

                -------------------------------------------------------------------
                when FETCH_DESCRIPTOR =>
                        desc_fetch_req_cmb  <= '0';
                    -- exit if error or descriptor flushed
                    if(s2mm_desc_flush = '1')then
                        s2mm_ns         <= IDLE;
                    -- Descriptor fetch complete
                    else --if(desc_fetch_done = '1')then
                   --     desc_fetch_req_cmb  <= '0';
                        if(rxlength_fetched = '0')then
                            s2mm_ns             <= GET_RXLENGTH;
                        else
                            s2mm_ns             <= CMPR_LENGTH;
                        end if;

                  --  else
                    --    desc_fetch_req_cmb  <= '1';
                    end if;

                -------------------------------------------------------------------
                WHEN GET_RXLENGTH =>
                    if(s2mm_stop = '1')then
                        s2mm_ns         <= IDLE;
                    -- Buffer length zero, do not compare lengths, execute
                    -- command to force datamover to issue interror
                    elsif(zero_length_error = '1')then
                        s2mm_ns                 <= EXECUTE_XFER;
                    elsif(s2mm_rxlength_valid = '1')then
                        s2mm_rxlength_set       <= '1';
                        s2mm_rxlength_clr_cmb   <= '1';
                        s2mm_ns                 <= CMPR_LENGTH;
                    else
                        s2mm_ns <= GET_RXLENGTH;
                    end if;

                -------------------------------------------------------------------
                WHEN CMPR_LENGTH    =>
                        s2mm_ns                 <= EXECUTE_XFER;

                -------------------------------------------------------------------
                when EXECUTE_XFER =>
                    if(s2mm_stop = '1')then
                        s2mm_ns         <= IDLE;
                    -- write new command if one is not already pending
                    elsif(s2mm_cmnd_pending = '0')then
                        write_cmnd_cmb  <= '1';

                        -- If descriptor queuing enabled then
                        -- do NOT need to wait for status
                        if(C_SG_INCLUDE_DESC_QUEUE = 1)then
                            s2mm_ns         <= IDLE;

                        -- No queuing therefore must wait for
                        -- status before issuing next command
                        else
                            s2mm_ns         <= WAIT_STATUS;
                        end if;
                    else
                            s2mm_ns         <= EXECUTE_XFER;
                    end if;
                -------------------------------------------------------------------
--              coverage off
                when WAIT_STATUS =>
                    if(desc_update_done = '1' or s2mm_stop = '1')then
                        s2mm_ns <= IDLE;
                    else
                        s2mm_ns <= WAIT_STATUS;
                    end if;
--              coverage on

                -------------------------------------------------------------------
--              coverage off
                when others =>
                    s2mm_ns <= IDLE;
--              coverage on

            end case;
        end process S2MM_MACHINE;

    -------------------------------------------------------------------------------
    -- Register state machine states
    -------------------------------------------------------------------------------
    REGISTER_STATE : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    s2mm_cs     <= IDLE;
                else
                    s2mm_cs     <= s2mm_ns;
                end if;
            end if;
        end process REGISTER_STATE;

    -------------------------------------------------------------------------------
    -- Register state machine signals
    -------------------------------------------------------------------------------
    SM_SIG_REGISTER : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    desc_fetch_req      <= '0'      ;
                    s2mm_rxlength_clr   <= '0'      ;
                else
                    if (C_SG_INCLUDE_DESC_QUEUE = 0) then
                       desc_fetch_req      <= '1';
                    else
                       desc_fetch_req      <= desc_fetch_req_cmb   ;
                    end if;
                    s2mm_rxlength_clr   <= s2mm_rxlength_clr_cmb;
                end if;
            end if;
        end process SM_SIG_REGISTER;


    -------------------------------------------------------------------------------
    -- Check for a ZERO value in descriptor buffer length.  If there is
    -- then flag an error and skip waiting for valid rxlength.  cmnd will
    -- get written to datamover with BTT=0 and datamover will flag dmaint error
    -- which will be logged in desc, reset required to clear error
    -------------------------------------------------------------------------------
    REG_ALIGN_DONE : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    desc_fetch_done_d1 <= '0';
                else
                    desc_fetch_done_d1 <= desc_fetch_done;
                end if;
            end if;
        end process REG_ALIGN_DONE;



    -------------------------------------------------------------------------------
    -- Zero length error detection - for determinate mode, detect early to prevent
    -- rxlength calcuation from first taking place.  This will force a 0 BTT
    -- command to be issued to the datamover causing an internal error.
    -------------------------------------------------------------------------------
    REG_ZERO_LNGTH_ERR : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    zero_length_error <= '0';
                elsif(desc_fetch_done_d1 = '1'
                and s2mm_desc_blength(C_SG_LENGTH_WIDTH-1 downto 0) = ZERO_LENGTH)then
                    zero_length_error <= '1';
                end if;
            end if;
        end process REG_ZERO_LNGTH_ERR;


    -------------------------------------------------------------------------------
    -- Capture/Hold receive length from status stream.  Also decrement length
    -- based on if received length is greater than descriptor buffer size. (i.e. is
    -- the case where multiple descriptors/buffers are used to describe one packet)
    -------------------------------------------------------------------------------
    REG_RXLENGTH : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    rxlength            <= (others => '0');
                -- If command register rxlength from status stream fifo
                elsif(s2mm_rxlength_set = '1')then
                    rxlength            <= s2mm_rxlength;

                -- On command write if current desc buffer size not greater
                -- than current rxlength then decrement rxlength in preperations
                -- for subsequent commands
                elsif(write_cmnd_cmb = '1' and blength_grtr_rxlength = '0')then

                    rxlength <= std_logic_vector(unsigned(rxlength(C_SG_LENGTH_WIDTH-1 downto 0))
                                               - unsigned(s2mm_desc_blength(C_SG_LENGTH_WIDTH-1 downto 0)));

                end if;
            end if;
        end process REG_RXLENGTH;

    -------------------------------------------------------------------------------
    -- Calculate if Descriptor Buffer Length is 'Greater Than' or 'Equal To'
    -- Received Length value
    -------------------------------------------------------------------------------
    REG_BLENGTH_GRTR_RXLNGTH : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    blength_grtr_rxlength <= '0';
                elsif(s2mm_desc_blength(C_SG_LENGTH_WIDTH-1 downto 0) >= rxlength)then
                    blength_grtr_rxlength <= '1';
                else
                    blength_grtr_rxlength <= '0';
                end if;
            end if;
        end process REG_BLENGTH_GRTR_RXLNGTH;

    -------------------------------------------------------------------------------
    -- On command assert rxlength fetched flag indicating length grabbed from
    -- status stream fifo
    -------------------------------------------------------------------------------
    RXLENGTH_FTCHED_PROCESS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0' or s2mm_eof_set_i = '1')then
                    rxlength_fetched    <= '0';
                elsif(s2mm_rxlength_set = '1')then
                    rxlength_fetched    <= '1';
                end if;
             end if;
         end process RXLENGTH_FTCHED_PROCESS;

    -------------------------------------------------------------------------------
    -- Build DataMover command
    -------------------------------------------------------------------------------
    -- If Bytes To Transfer (BTT) width less than 23, need to add pad
    GEN_CMD_BTT_LESS_23 : if C_SG_LENGTH_WIDTH < 23 generate
    constant PAD_VALUE : std_logic_vector(22 - C_SG_LENGTH_WIDTH downto 0)
                            := (others => '0');
    begin
        -- When command by sm, drive command to s2mm_cmdsts_if
        GEN_DATAMOVER_CMND : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        s2mm_cmnd_wr    <= '0';
                        s2mm_cmnd_data  <= (others => '0');
                        s2mm_eof_set_i  <= '0';

                    -- Current Desc Buffer will NOT hold entire rxlength of data therefore
                    -- set EOF = based on Desc.EOF and pass buffer length for BTT
                    elsif(write_cmnd_cmb = '1' and blength_grtr_rxlength = '0')then
                        s2mm_cmnd_wr    <= '1';
                        s2mm_cmnd_data  <=  s2mm_desc_info
                                            & ZERO_BUFFER
                                            & ZERO_BUFFER
                                            & S2MM_CMD_RSVD
                                            -- Command Tag
                                            & '0'
                                            & '0'
                                            & '0'  -- Cat. EOF=0 to CMD Tag
                                            & '0'  -- Cat. IOC to CMD TAG
                                            -- Command
                                            & s2mm_desc_baddress
                                            & '1'           -- Always reset DRE
                                            & '0'           -- Not End of Frame
                                            & S2MM_CMD_DSA
                                            & burst_type -- Key Hole '1'           -- s2mm_desc_type -- IR# 545697
                                            & PAD_VALUE
                                            & s2mm_desc_blength(C_SG_LENGTH_WIDTH-1 downto 0);
                        s2mm_eof_set_i  <= '0';


                    -- Current Desc Buffer will hold entire rxlength of data therefore
                    -- set EOF = 1 and pass rxlength for BTT
                    --
                    -- Note: change to mode where EOF generates IOC interrupt as
                    -- opposed to a IOC bit in the descriptor negated need for an
                    -- EOF and IOC tag.  Given time, these two bits could be combined
                    -- into 1.  Associated logic in SG engine would also need to be
                    -- modified as well as in s2mm_sg_if.
                    elsif(write_cmnd_cmb = '1' and blength_grtr_rxlength = '1')then
                        s2mm_cmnd_wr    <= '1';
                        s2mm_cmnd_data  <=  s2mm_desc_info
                                            & ZERO_BUFFER
                                            & ZERO_BUFFER
                                            & S2MM_CMD_RSVD
                                            -- Command Tag
                                            & '0'
                                            & '0'
                                            & '1'  -- Cat. EOF=1 to CMD Tag
                                            & '1'  -- Cat. IOC to CMD TAG
                                            -- Command
                                            & s2mm_desc_baddress
                                            & '1'           -- Always reset DRE
                                            & '1'           -- Set EOF=1
                                            & S2MM_CMD_DSA
                                            & burst_type -- Key Hole '1'           -- s2mm_desc_type -- IR# 545697
                                            & PAD_VALUE
                                            & rxlength;
                        s2mm_eof_set_i    <= '1';

                    else
                 --       s2mm_cmnd_data  <= (others => '0');
                        s2mm_cmnd_wr    <= '0';
                        s2mm_eof_set_i  <= '0';

                    end if;
                end if;
            end process GEN_DATAMOVER_CMND;

    end generate GEN_CMD_BTT_LESS_23;

    -- If Bytes To Transfer (BTT) width equal 23, no required pad
    GEN_CMD_BTT_EQL_23 : if C_SG_LENGTH_WIDTH = 23 generate
    begin
        -- When command by sm, drive command to s2mm_cmdsts_if
        GEN_DATAMOVER_CMND : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        s2mm_cmnd_wr    <= '0';
                        s2mm_cmnd_data  <= (others => '0');
                        s2mm_eof_set_i    <= '0';
                    -- Current Desc Buffer will NOT hold entire rxlength of data therefore
                    -- set EOF = based on Desc.EOF and pass buffer length for BTT
                    elsif(write_cmnd_cmb = '1' and blength_grtr_rxlength = '0')then
                        s2mm_cmnd_wr    <= '1';
                        s2mm_cmnd_data  <=  s2mm_desc_info
                                            & ZERO_BUFFER
                                            & ZERO_BUFFER
                                            & S2MM_CMD_RSVD
                                            --& S2MM_CMD_TAG & s2mm_desc_ioc  -- Cat IOC to CMD TAG
                                            -- Command Tag
                                            & '0'
                                            & '0'
                                            & '0'  -- Cat. EOF='0' to CMD Tag
                                            & '0'  -- Cat. IOC='0' to CMD TAG
                                            -- Command
                                            & s2mm_desc_baddress
                                            & '1'           -- Always reset DRE
                                            & '0'           -- Not End of Frame
                                            & S2MM_CMD_DSA
                                            & burst_type    -- Key Hole '1' -- s2mm_desc_type -- IR# 545697
                                            & s2mm_desc_blength;

                        s2mm_eof_set_i    <= '0';

                    -- Current Desc Buffer will hold entire rxlength of data therefore
                    -- set EOF = 1 and pass rxlength for BTT
                    --
                    -- Note: change to mode where EOF generates IOC interrupt as
                    -- opposed to a IOC bit in the descriptor negated need for an
                    -- EOF and IOC tag.  Given time, these two bits could be combined
                    -- into 1.  Associated logic in SG engine would also need to be
                    -- modified as well as in s2mm_sg_if.
                    elsif(write_cmnd_cmb = '1' and blength_grtr_rxlength = '1')then
                        s2mm_cmnd_wr    <= '1';
                        s2mm_cmnd_data  <=  s2mm_desc_info
                                            & ZERO_BUFFER
                                            & ZERO_BUFFER
                                            & S2MM_CMD_RSVD
                                            --& S2MM_CMD_TAG & s2mm_desc_ioc  -- Cat IOC to CMD TAG
                                            -- Command Tag
                                            & '0'
                                            & '0'
                                            & '1'  -- Cat. EOF='1' to CMD Tag
                                            & '1'  -- Cat. IOC='1' to CMD TAG
                                            -- Command
                                            & s2mm_desc_baddress
                                            & '1'           -- Always reset DRE
                                            & '1'           -- End of Frame
                                            & S2MM_CMD_DSA
                                            & burst_type    -- Key Hole '1' -- s2mm_desc_type -- IR# 545697
                                            & rxlength;
                        s2mm_eof_set_i    <= '1';
                    else
                  --      s2mm_cmnd_data  <= (others => '0');
                        s2mm_cmnd_wr    <= '0';
                        s2mm_eof_set_i    <= '0';

                    end if;
                end if;
            end process GEN_DATAMOVER_CMND;

    end generate GEN_CMD_BTT_EQL_23;

end generate GEN_SM_FOR_LENGTH;


-------------------------------------------------------------------------------
-- Counter for keepting track of pending commands/status in primary datamover
-- Use this to determine if primary datamover for s2mm is Idle.
-------------------------------------------------------------------------------
-- Increment queue count for each command written if not occuring at
-- same time a status from DM being updated to SG engine
count_incr  <= '1' when write_cmnd_cmb = '1' and desc_update_done = '0'
          else '0';

-- Decrement queue count for each status update to SG engine if not occuring
-- at same time as command being written to DM
count_decr  <= '1' when write_cmnd_cmb = '0' and desc_update_done = '1'
          else '0';

-- keep track of number queue commands
--CMD2STS_COUNTER : process(m_axi_sg_aclk)
--    begin
--        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
--            if(m_axi_sg_aresetn = '0' or s2mm_stop = '1')then
--                cmnds_queued <= (others => '0');
--            elsif(count_incr = '1')then
--                cmnds_queued <= std_logic_vector(unsigned(cmnds_queued(COUNTER_WIDTH - 1 downto 0)) + 1);
--            elsif(count_decr = '1')then
--                cmnds_queued <= std_logic_vector(unsigned(cmnds_queued(COUNTER_WIDTH - 1 downto 0)) - 1);
--            end if;
--        end if;
--    end process CMD2STS_COUNTER;

QUEUE_COUNT : if C_SG_INCLUDE_DESC_QUEUE = 1 generate
begin

CMD2STS_COUNTER1 : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0' or s2mm_stop = '1')then
                cmnds_queued_shift <= (others => '0');
            elsif(count_incr = '1')then
                cmnds_queued_shift <= cmnds_queued_shift (2 downto 0) & '1';
            elsif(count_decr = '1')then
                cmnds_queued_shift <= '0' & cmnds_queued_shift (3 downto 1);
            end if;
        end if;
    end process CMD2STS_COUNTER1;

end generate QUEUE_COUNT;


NOQUEUE_COUNT : if C_SG_INCLUDE_DESC_QUEUE = 0 generate
begin

CMD2STS_COUNTER1 : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0' or s2mm_stop = '1')then
                cmnds_queued_shift (0) <= '0';
            elsif(count_incr = '1')then
                cmnds_queued_shift (0) <= '1';
            elsif(count_decr = '1')then
                cmnds_queued_shift (0) <= '0';
            end if;
        end if;
    end process CMD2STS_COUNTER1;

end generate NOQUEUE_COUNT;

-- indicate idle when no more queued commands
--s2mm_sts_idle <= '1' when  cmnds_queued_shift = "0000"
--            else '0';

s2mm_sts_idle <= not cmnds_queued_shift(0);

-------------------------------------------------------------------------------
-- Queue only the amount of commands that can be queued on descriptor update
-- else lock up can occur. Note datamover command fifo depth is set to number
-- of descriptors to queue.
-------------------------------------------------------------------------------
--QUEUE_MORE_PROCESS : process(m_axi_sg_aclk)
--    begin
--        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
--            if(m_axi_sg_aresetn = '0')then
--                queue_more <= '0';
--            elsif(cmnds_queued < std_logic_vector(to_unsigned(C_PRMY_CMDFIFO_DEPTH,COUNTER_WIDTH)))then
--                queue_more <= '1';
--            else
--                queue_more <= '0';
--            end if;
--        end if;
--    end process QUEUE_MORE_PROCESS;

QUEUE_MORE_PROCESS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                queue_more <= '0';
 --           elsif(cmnds_queued < std_logic_vector(to_unsigned(C_PRMY_CMDFIFO_DEPTH,COUNTER_WIDTH)))then
 --               queue_more <= '1';
            else
                queue_more <= not (cmnds_queued_shift (C_PRMY_CMDFIFO_DEPTH-1)); --'0';
            end if;
        end if;
    end process QUEUE_MORE_PROCESS;


end implementation;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:    axi_dma_s2mm_cmdsts_if.vhd
-- Description: This entity is the descriptor fetch command and status inteface
--              for the Scatter Gather Engine AXI DataMover.
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;

-------------------------------------------------------------------------------
entity  axi_dma_s2mm_cmdsts_if is
    generic (
        C_M_AXI_S2MM_ADDR_WIDTH       : integer range 32 to 64          := 32;
            -- Master AXI Memory Map Address Width for S2MM Write Port

        C_DM_STATUS_WIDTH               : integer range 8 to 32         := 8;
            -- Width of DataMover status word
            -- 8  for Determinate BTT Mode
            -- 32 for Indterminate BTT Mode

        C_INCLUDE_SG                : integer range 0 to 1          := 1;
            -- Include or Exclude the Scatter Gather Engine
            -- 0 = Exclude SG Engine - Enables Simple DMA Mode
            -- 1 = Include SG Engine - Enables Scatter Gather Mode

        C_SG_INCLUDE_STSCNTRL_STRM      : integer range 0 to 1          := 1;
            -- Include or Exclude AXI Status and AXI Control Streams
            -- 0 = Exclude Status and Control Streams
            -- 1 = Include Status and Control Streams

        C_SG_USE_STSAPP_LENGTH      : integer range 0 to 1              := 1;
            -- Enable or Disable use of Status Stream Rx Length.  Only valid
            -- if C_SG_INCLUDE_STSCNTRL_STRM = 1
            -- 0 = Don't use Rx Length
            -- 1 = Use Rx Length

        C_SG_LENGTH_WIDTH           : integer range 8 to 23             := 14;
            -- Descriptor Buffer Length, Transferred Bytes, and Status Stream
            -- Rx Length Width.  Indicates the least significant valid bits of
            -- descriptor buffer length, transferred bytes, or Rx Length value
            -- in the status word coincident with tlast.

        C_ENABLE_MULTI_CHANNEL             : integer range 0 to 1              := 0;
        C_MICRO_DMA                        : integer range 0 to 1              := 0;
        C_ENABLE_QUEUE                     : integer range 0 to 1              := 1
    );
    port (
        -----------------------------------------------------------------------
        -- AXI Scatter Gather Interface
        -----------------------------------------------------------------------
        m_axi_sg_aclk               : in  std_logic                         ;                    --
        m_axi_sg_aresetn            : in  std_logic                         ;                    --
                                                                                                 --
        -- Command write interface from mm2s sm                                                  --
        s2mm_cmnd_wr                : in  std_logic                         ;                    --
        s2mm_cmnd_data              : in  std_logic_vector                                       --
                                        ((C_M_AXI_S2MM_ADDR_WIDTH-32+2*32+CMD_BASE_WIDTH+46)-1 downto 0);   --
        s2mm_cmnd_pending           : out std_logic                         ;                    --
                                                                                                 --
        s2mm_packet_eof             : out std_logic                         ;                    --
                                                                                                 --
        s2mm_sts_received_clr       : in  std_logic                         ;                    --
        s2mm_sts_received           : out std_logic                         ;                    --
        s2mm_tailpntr_enble         : in  std_logic                         ;                    --
        s2mm_desc_cmplt             : in  std_logic                         ;                    --
                                                                                                 --
        -- User Command Interface Ports (AXI Stream)                                             --
        s_axis_s2mm_cmd_tvalid      : out std_logic                         ;                    --
        s_axis_s2mm_cmd_tready      : in  std_logic                         ;                    --
        s_axis_s2mm_cmd_tdata       : out std_logic_vector                                       --
                                        ((C_M_AXI_S2MM_ADDR_WIDTH-32+2*32+CMD_BASE_WIDTH+46)-1 downto 0);   --
                                                                                                 --
        -- User Status Interface Ports (AXI Stream)                                              --
        m_axis_s2mm_sts_tvalid      : in  std_logic                         ;                    --
        m_axis_s2mm_sts_tready      : out std_logic                         ;                    --
        m_axis_s2mm_sts_tdata       : in  std_logic_vector                                       --
                                        (C_DM_STATUS_WIDTH - 1 downto 0)    ;                    --
        m_axis_s2mm_sts_tkeep       : in  std_logic_vector((C_DM_STATUS_WIDTH/8)-1 downto 0);    --
                                                                                                 --
        -- Scatter Gather Fetch Status                                                           --
        s2mm_err                    : in  std_logic                         ;                    --
        s2mm_brcvd                  : out std_logic_vector                                       --
                                        (C_SG_LENGTH_WIDTH-1 downto 0)      ;                    --
        s2mm_done                   : out std_logic                         ;                    --
        s2mm_error                  : out std_logic                         ;                    --
        s2mm_interr                 : out std_logic                         ;                    --
        s2mm_slverr                 : out std_logic                         ;                    --
        s2mm_decerr                 : out std_logic                         ;                    --
        s2mm_tag                    : out std_logic_vector(3 downto 0)                           --
    );

end axi_dma_s2mm_cmdsts_if;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_dma_s2mm_cmdsts_if is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------

-- No Constants Declared

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
signal sts_tready           : std_logic := '0';
signal sts_received_i       : std_logic := '0';
signal stale_desc           : std_logic := '0';
signal log_status           : std_logic := '0';

signal s2mm_slverr_i        : std_logic := '0';
signal s2mm_decerr_i        : std_logic := '0';
signal s2mm_interr_i        : std_logic := '0';
signal s2mm_error_or        : std_logic := '0';

signal s2mm_packet_eof_i    : std_logic := '0';
signal smpl_dma_overflow    : std_logic := '0';

-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin
s2mm_slverr     <= s2mm_slverr_i;
s2mm_decerr     <= s2mm_decerr_i;
s2mm_interr     <= s2mm_interr_i or smpl_dma_overflow;


s2mm_packet_eof <= s2mm_packet_eof_i;

-- Stale descriptor if complete bit already set and in tail pointer mode.
stale_desc <= '1' when s2mm_desc_cmplt = '1' and s2mm_tailpntr_enble = '1'
         else '0';

-------------------------------------------------------------------------------
-- DataMover Command Interface
-------------------------------------------------------------------------------


-------------------------------------------------------------------------------
-- When command by fetch sm, drive descriptor fetch command to data mover.
-- Hold until data mover indicates ready.
-------------------------------------------------------------------------------
GEN_HOLD_NO_DATA : if C_ENABLE_QUEUE = 1 generate
begin
GEN_DATAMOVER_CMND : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                s_axis_s2mm_cmd_tvalid  <= '0';
        --        s_axis_s2mm_cmd_tdata   <= (others => '0');
                s2mm_cmnd_pending       <= '0';
            -- new command and descriptor not flagged as stale
            elsif(s2mm_cmnd_wr = '1' and stale_desc = '0')then
                s_axis_s2mm_cmd_tvalid  <= '1';
        --        s_axis_s2mm_cmd_tdata   <= s2mm_cmnd_data;
                s2mm_cmnd_pending       <= '1';
            -- clear flag on datamover acceptance of command
            elsif(s_axis_s2mm_cmd_tready = '1')then
                s_axis_s2mm_cmd_tvalid  <= '0';
        --        s_axis_s2mm_cmd_tdata   <= (others => '0');
                s2mm_cmnd_pending       <= '0';
            end if;
        end if;
    end process GEN_DATAMOVER_CMND;

                s_axis_s2mm_cmd_tdata   <= s2mm_cmnd_data;

end generate GEN_HOLD_NO_DATA;


GEN_HOLD_DATA : if C_ENABLE_QUEUE = 0 generate
begin
GEN_DATAMOVER_CMND : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                s_axis_s2mm_cmd_tvalid  <= '0';
                s_axis_s2mm_cmd_tdata   <= (others => '0');
                s2mm_cmnd_pending       <= '0';
            -- new command and descriptor not flagged as stale
            elsif(s2mm_cmnd_wr = '1' and stale_desc = '0')then
                s_axis_s2mm_cmd_tvalid  <= '1';
                s_axis_s2mm_cmd_tdata   <= s2mm_cmnd_data;
                s2mm_cmnd_pending       <= '1';
            -- clear flag on datamover acceptance of command
            elsif(s_axis_s2mm_cmd_tready = '1')then
                s_axis_s2mm_cmd_tvalid  <= '0';
                s_axis_s2mm_cmd_tdata   <= (others => '0');
                s2mm_cmnd_pending       <= '0';
            end if;
        end if;
    end process GEN_DATAMOVER_CMND;


end generate GEN_HOLD_DATA;

-------------------------------------------------------------------------------
-- DataMover Status Interface
-------------------------------------------------------------------------------
-- Drive ready low during reset to indicate not ready
REG_STS_READY : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                sts_tready <= '0';
            elsif(sts_tready = '1' and m_axis_s2mm_sts_tvalid = '1')then
                sts_tready <= '0';
            elsif(sts_received_i = '0') then
                sts_tready <= '1';
            end if;
        end if;
    end process REG_STS_READY;

-- Pass to DataMover
m_axis_s2mm_sts_tready <= sts_tready;

log_status <= '1' when m_axis_s2mm_sts_tvalid = '1' and sts_received_i = '0'
         else '0';


-- Status stream is included, and using the rxlength from the status stream and in Scatter Gather Mode
DETERMINATE_BTT_MODE : if (C_SG_INCLUDE_STSCNTRL_STRM = 1 and C_SG_USE_STSAPP_LENGTH = 1
                       and C_INCLUDE_SG = 1) or (C_MICRO_DMA = 1) generate
begin
    -- Bytes received not available in determinate byte mode
    s2mm_brcvd          <= (others => '0');
    -- Simple DMA overflow not used in Scatter Gather Mode
    smpl_dma_overflow   <= '0';

    -------------------------------------------------------------------------------
    -- Log status bits out of data mover.
    -------------------------------------------------------------------------------
    DATAMOVER_STS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    s2mm_done       <= '0';
                    s2mm_slverr_i   <= '0';
                    s2mm_decerr_i   <= '0';
                    s2mm_interr_i   <= '0';
                    s2mm_tag        <= (others => '0');
                -- Status valid, therefore capture status
                elsif(m_axis_s2mm_sts_tvalid = '1' and sts_received_i = '0')then
                    s2mm_done       <= m_axis_s2mm_sts_tdata(DATAMOVER_STS_CMDDONE_BIT);
                    s2mm_slverr_i   <= m_axis_s2mm_sts_tdata(DATAMOVER_STS_SLVERR_BIT);
                    s2mm_decerr_i   <= m_axis_s2mm_sts_tdata(DATAMOVER_STS_DECERR_BIT);
                    s2mm_interr_i   <= m_axis_s2mm_sts_tdata(DATAMOVER_STS_INTERR_BIT);
                    s2mm_tag        <= m_axis_s2mm_sts_tdata(DATAMOVER_STS_TAGMSB_BIT downto DATAMOVER_STS_TAGLSB_BIT);
                -- Only assert when valid
                else
                    s2mm_done      <= '0';
                    s2mm_slverr_i  <= '0';
                    s2mm_decerr_i  <= '0';
                    s2mm_interr_i  <= '0';
                    s2mm_tag       <= (others => '0');
                end if;
            end if;
        end process DATAMOVER_STS;

    -- End Of Frame (EOF = 1) detected on status received. Used
    -- for interrupt delay timer
    REG_RX_EOF : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    s2mm_packet_eof_i <= '0';
                elsif(log_status = '1')then
                    s2mm_packet_eof_i <=  m_axis_s2mm_sts_tdata(DATAMOVER_STS_TAGEOF_BIT)
                                       or m_axis_s2mm_sts_tdata(DATAMOVER_STS_INTERR_BIT);
                else
                    s2mm_packet_eof_i <= '0';
                end if;
            end if;
        end process REG_RX_EOF;


end generate DETERMINATE_BTT_MODE;

-- No Status Stream or not using rxlength from status stream or in Simple DMA Mode
INDETERMINATE_BTT_MODE : if (C_SG_INCLUDE_STSCNTRL_STRM = 0 or C_SG_USE_STSAPP_LENGTH = 0
                         or C_INCLUDE_SG = 0) and (C_MICRO_DMA = 0) generate

-- Bytes received MSB index bit
constant BRCVD_MSB_BIT : integer := (C_DM_STATUS_WIDTH - 2) - (BUFFER_LENGTH_WIDTH - C_SG_LENGTH_WIDTH);
-- Bytes received LSB index bit
constant BRCVD_LSB_BIT : integer := (C_DM_STATUS_WIDTH - 2) - (BUFFER_LENGTH_WIDTH - 1);

begin

    -------------------------------------------------------------------------------
    -- Log status bits out of data mover.
    -------------------------------------------------------------------------------
    DATAMOVER_STS : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    s2mm_brcvd      <= (others => '0');
                    s2mm_done       <= '0';
                    s2mm_slverr_i   <= '0';
                    s2mm_decerr_i   <= '0';
                    s2mm_interr_i   <= '0';
                    s2mm_tag        <= (others => '0');
                -- Status valid, therefore capture status
                elsif(m_axis_s2mm_sts_tvalid = '1' and sts_received_i = '0')then
                    s2mm_brcvd      <= m_axis_s2mm_sts_tdata(BRCVD_MSB_BIT downto BRCVD_LSB_BIT);
                    s2mm_done       <= m_axis_s2mm_sts_tdata(DATAMOVER_STS_CMDDONE_BIT);
                    s2mm_slverr_i   <= m_axis_s2mm_sts_tdata(DATAMOVER_STS_SLVERR_BIT);
                    s2mm_decerr_i   <= m_axis_s2mm_sts_tdata(DATAMOVER_STS_DECERR_BIT);
                    s2mm_interr_i   <= m_axis_s2mm_sts_tdata(DATAMOVER_STS_INTERR_BIT);
                    s2mm_tag        <= m_axis_s2mm_sts_tdata(DATAMOVER_STS_TAGMSB_BIT downto DATAMOVER_STS_TAGLSB_BIT);
                -- Only assert when valid
                else
                    s2mm_brcvd     <= (others => '0');
                    s2mm_done      <= '0';
                    s2mm_slverr_i  <= '0';
                    s2mm_decerr_i  <= '0';
                    s2mm_interr_i  <= '0';
                    s2mm_tag       <= (others => '0');
                end if;
            end if;
        end process DATAMOVER_STS;

    -- End Of Frame (EOF = 1) detected on statis received. Used
    -- for interrupt delay timer
    REG_RX_EOF : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    s2mm_packet_eof_i <= '0';
                elsif(log_status = '1')then
                    s2mm_packet_eof_i <=  m_axis_s2mm_sts_tdata(DATAMOVER_STS_TLAST_BIT)
                                       or m_axis_s2mm_sts_tdata(DATAMOVER_STS_INTERR_BIT);
                else
                    s2mm_packet_eof_i <= '0';
                end if;
            end if;
        end process REG_RX_EOF;

    -- If in Simple DMA mode then generate overflow flag
    GEN_OVERFLOW_SMPL_DMA : if C_INCLUDE_SG = 0 generate
        REG_OVERFLOW : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        smpl_dma_overflow <= '0';
                    -- If status received and TLAST bit is NOT set then packet is bigger than
                    -- BTT value commanded which is an invalid command
                    elsif(log_status = '1' and m_axis_s2mm_sts_tdata(DATAMOVER_STS_TLAST_BIT) = '0')then
                        smpl_dma_overflow <= '1';
                    end if;
                end if;
            end process REG_OVERFLOW;
    end generate GEN_OVERFLOW_SMPL_DMA;

    -- If in Scatter Gather Mode then do NOT generate simple dma mode overflow flag
    GEN_NO_OVERFLOW_SMPL_DMA : if C_INCLUDE_SG = 1 generate
    begin
        smpl_dma_overflow <= '0';
    end generate GEN_NO_OVERFLOW_SMPL_DMA;

end generate INDETERMINATE_BTT_MODE;





-- Flag when status is received.  Used to hold status until sg if
-- can use status.  This only has meaning when SG Engine Queues are turned
-- on
STS_RCVD_FLAG : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0' or s2mm_sts_received_clr = '1')then
                sts_received_i  <= '0';
            -- Status valid, therefore capture status
            elsif(m_axis_s2mm_sts_tvalid = '1' and sts_received_i = '0')then
                sts_received_i  <= '1';
            end if;
        end if;
    end process STS_RCVD_FLAG;

s2mm_sts_received    <= sts_received_i;

-------------------------------------------------------------------------------
-- Register global error from data mover.
-------------------------------------------------------------------------------
s2mm_error_or <= s2mm_slverr_i or s2mm_decerr_i or s2mm_interr_i or smpl_dma_overflow;

-- Log errors into a global error output
S2MM_ERROR_PROCESS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                s2mm_error <= '0';
            -- If Datamover issues error on the transfer or if a stale descriptor is
            -- detected when in tailpointer mode then issue an error
            elsif((s2mm_error_or = '1')
               or (stale_desc = '1' and s2mm_cmnd_wr='1'))then
                s2mm_error <= '1';
            end if;
        end if;
    end process S2MM_ERROR_PROCESS;



end implementation;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:    axi_dma_s2mm_sts_mngr.vhd
-- Description: This entity mangages 'halt' and 'idle' status for the S2MM
--              channel
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library lib_cdc_v1_0_2;
library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;

-------------------------------------------------------------------------------
entity  axi_dma_s2mm_sts_mngr is
    generic (
        C_PRMRY_IS_ACLK_ASYNC        : integer range 0 to 1          := 0
            -- Primary MM2S/S2MM sync/async mode
            -- 0 = synchronous mode     - all clocks are synchronous
            -- 1 = asynchronous mode    - Any one of the 4 clock inputs is not
            --                            synchronous to the other
    );
    port (
        -----------------------------------------------------------------------
        -- AXI Scatter Gather Interface
        -----------------------------------------------------------------------
        m_axi_sg_aclk               : in  std_logic                         ;          --
        m_axi_sg_aresetn            : in  std_logic                         ;          --
                                                                                       --
        -- system state                                                                --
        s2mm_run_stop               : in  std_logic                         ;          --
        s2mm_ftch_idle              : in  std_logic                         ;          --
        s2mm_updt_idle              : in  std_logic                         ;          --
        s2mm_cmnd_idle              : in  std_logic                         ;          --
        s2mm_sts_idle               : in  std_logic                         ;          --
                                                                                       --
        -- stop and halt control/status                                                --
        s2mm_stop                   : in  std_logic                         ;          --
        s2mm_halt_cmplt             : in  std_logic                         ;          --
                                                                                       --
        -- system control                                                              --
        s2mm_all_idle               : out std_logic                         ;          --
        s2mm_halted_clr             : out std_logic                         ;          --
        s2mm_halted_set             : out std_logic                         ;          --
        s2mm_idle_set               : out std_logic                         ;          --
        s2mm_idle_clr               : out std_logic                                    --

    );

end axi_dma_s2mm_sts_mngr;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_dma_s2mm_sts_mngr is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


  ATTRIBUTE async_reg                      : STRING;

-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------

-- No Constants Declared

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------

signal all_is_idle          : std_logic := '0';
signal all_is_idle_d1       : std_logic := '0';
signal all_is_idle_re       : std_logic := '0';
signal all_is_idle_fe       : std_logic := '0';
signal s2mm_datamover_idle  : std_logic := '0';

signal s2mm_halt_cmpt_d1_cdc_tig    : std_logic := '0';
signal s2mm_halt_cmpt_cdc_d2    : std_logic := '0';
signal s2mm_halt_cmpt_d2    : std_logic := '0';
  --ATTRIBUTE async_reg OF s2mm_halt_cmpt_d1_cdc_tig  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF s2mm_halt_cmpt_cdc_d2  : SIGNAL IS "true";

-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin
-- all is idle when all is idle
all_is_idle <=  s2mm_ftch_idle
            and s2mm_updt_idle
            and s2mm_cmnd_idle
            and s2mm_sts_idle;


s2mm_all_idle   <= all_is_idle;

-------------------------------------------------------------------------------
-- For data mover halting look at halt complete to determine when halt
-- is done and datamover has completly halted.  If datamover not being
-- halted then can ignore flag thus simply flag as idle.
-------------------------------------------------------------------------------
GEN_FOR_ASYNC : if C_PRMRY_IS_ACLK_ASYNC = 1 generate
begin
    -- Double register to secondary clock domain.  This is sufficient
    -- because halt_cmplt will remain asserted until detected in
    -- reset module in secondary clock domain.
REG_TO_SECONDARY : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => s2mm_halt_cmplt,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => m_axi_sg_aclk,
        scndry_resetn              => '0',
        scndry_out                 => s2mm_halt_cmpt_cdc_d2,
        scndry_vect_out            => open
    );

--    REG_TO_SECONDARY : process(m_axi_sg_aclk)
--        begin
--            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
----                if(m_axi_sg_aresetn = '0')then
----                    s2mm_halt_cmpt_d1_cdc_tig <= '0';
----                    s2mm_halt_cmpt_d2 <= '0';
----                else
--                    s2mm_halt_cmpt_d1_cdc_tig <= s2mm_halt_cmplt;
--                    s2mm_halt_cmpt_cdc_d2 <= s2mm_halt_cmpt_d1_cdc_tig;
----                end if;
--            end if;
--        end process REG_TO_SECONDARY;

                    s2mm_halt_cmpt_d2 <= s2mm_halt_cmpt_cdc_d2;

end generate GEN_FOR_ASYNC;

GEN_FOR_SYNC : if C_PRMRY_IS_ACLK_ASYNC = 0 generate
begin
    -- No clock crossing required therefore simple pass through
    s2mm_halt_cmpt_d2 <= s2mm_halt_cmplt;

end generate GEN_FOR_SYNC;

s2mm_datamover_idle  <= '1' when (s2mm_stop = '1' and s2mm_halt_cmpt_d2 = '1')
                              or (s2mm_stop = '0')
                   else '0';

-------------------------------------------------------------------------------
-- Set halt bit if run/stop cleared and all processes are idle
-------------------------------------------------------------------------------
HALT_PROCESS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                s2mm_halted_set <= '0';
            elsif(s2mm_run_stop = '0' and all_is_idle = '1' and s2mm_datamover_idle = '1')then
                s2mm_halted_set <= '1';
            else
                s2mm_halted_set <=  '0';
            end if;
        end if;
    end process HALT_PROCESS;

-------------------------------------------------------------------------------
-- Clear halt bit if run/stop is set and SG engine begins to fetch descriptors
-------------------------------------------------------------------------------
NOT_HALTED_PROCESS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                s2mm_halted_clr <= '0';
            elsif(s2mm_run_stop = '1')then
                s2mm_halted_clr <= '1';
            else
                s2mm_halted_clr <= '0';
            end if;
        end if;
    end process NOT_HALTED_PROCESS;

-------------------------------------------------------------------------------
-- Register ALL is Idle to create rising and falling edges on idle flag
-------------------------------------------------------------------------------
IDLE_REG_PROCESS : process(m_axi_sg_aclk)
    begin
        if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
            if(m_axi_sg_aresetn = '0')then
                all_is_idle_d1 <= '0';
            else
                all_is_idle_d1 <= all_is_idle;
            end if;
        end if;
    end process IDLE_REG_PROCESS;

all_is_idle_re  <= all_is_idle and not all_is_idle_d1;
all_is_idle_fe  <= not all_is_idle and all_is_idle_d1;

-- Set or Clear IDLE bit in DMASR
s2mm_idle_set <= all_is_idle_re and s2mm_run_stop;
s2mm_idle_clr <= all_is_idle_fe;


end implementation;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:          axi_dma_s2mm_sts_strm.vhd.vhd
-- Description: This entity is the AXI Status Stream Interface
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;

library lib_srl_fifo_v1_0_2;
library lib_cdc_v1_0_2;
library lib_pkg_v1_0_2;
use lib_pkg_v1_0_2.lib_pkg.all;

-------------------------------------------------------------------------------
entity  axi_dma_s2mm_sts_strm is
    generic (

        C_PRMRY_IS_ACLK_ASYNC           : integer range 0 to 1         := 0;
            -- Primary MM2S/S2MM sync/async mode
            -- 0 = synchronous mode     - all clocks are synchronous
            -- 1 = asynchronous mode    - Primary data path channels (MM2S and S2MM)
            --                            run asynchronous to AXI Lite, DMA Control,
            --                            and SG.

        -----------------------------------------------------------------------
        -- Scatter Gather Parameters
        -----------------------------------------------------------------------
        C_S_AXIS_S2MM_STS_TDATA_WIDTH   : integer range 32 to 32        := 32;
            -- Slave AXI Status Stream Data Width

        C_SG_USE_STSAPP_LENGTH          : integer range 0 to 1          := 1;
            -- Enable or Disable use of Status Stream Rx Length.  Only valid
            -- if C_SG_INCLUDE_STSCNTRL_STRM = 1
            -- 0 = Don't use Rx Length
            -- 1 = Use Rx Length

        C_SG_LENGTH_WIDTH               : integer range 8 to 23         := 14;
            -- Descriptor Buffer Length, Transferred Bytes, and Status Stream
            -- Rx Length Width.  Indicates the least significant valid bits of
            -- descriptor buffer length, transferred bytes, or Rx Length value
            -- in the status word coincident with tlast.

        C_ENABLE_SKID                   : integer range 0 to 1          := 0;

        C_FAMILY                        : string            := "virtex5"
            -- Target FPGA Device Family

    );
    port (

        m_axi_sg_aclk               : in  std_logic                         ;                  --
        m_axi_sg_aresetn            : in  std_logic                         ;                  --
                                                                                               --
        axi_prmry_aclk              : in  std_logic                         ;                  --
        p_reset_n                   : in  std_logic                         ;                  --
                                                                                               --
        s2mm_stop                   : in  std_logic                         ;                  --
                                                                                               --
        s2mm_rxlength_valid         : out std_logic                         ;                  --
        s2mm_rxlength_clr           : in  std_logic                         ;                  --
        s2mm_rxlength               : out std_logic_vector                                     --
                                        (C_SG_LENGTH_WIDTH - 1 downto 0)    ;                  --
                                                                                               --
        stsstrm_fifo_rden           : in  std_logic                         ;                  --
        stsstrm_fifo_empty          : out std_logic                         ;                  --
        stsstrm_fifo_dout           : out std_logic_vector                                     --
                                        (C_S_AXIS_S2MM_STS_TDATA_WIDTH downto 0);              --
                                                                                               --
        -- Stream to Memory Map Status Stream Interface                                        --
        s_axis_s2mm_sts_tdata       : in  std_logic_vector                                     --
                                        (C_S_AXIS_S2MM_STS_TDATA_WIDTH-1 downto 0);            --
        s_axis_s2mm_sts_tkeep       : in  std_logic_vector                                     --
                                        ((C_S_AXIS_S2MM_STS_TDATA_WIDTH/8)-1 downto 0);        --
        s_axis_s2mm_sts_tvalid      : in  std_logic                         ;                  --
        s_axis_s2mm_sts_tready      : out std_logic                         ;                  --
        s_axis_s2mm_sts_tlast       : in  std_logic                                            --
    );

end axi_dma_s2mm_sts_strm;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_dma_s2mm_sts_strm is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------
-- Status Stream FIFO Depth
constant STSSTRM_FIFO_DEPTH     : integer := 16;
-- Status Stream FIFO Data Count Width (Unsused)
constant STSSTRM_FIFO_CNT_WIDTH : integer := clog2(STSSTRM_FIFO_DEPTH+1);

constant USE_LOGIC_FIFOS        : integer   := 0; -- Use Logic FIFOs
constant USE_BRAM_FIFOS         : integer   := 1; -- Use BRAM FIFOs

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
signal fifo_full        : std_logic := '0';
signal fifo_din         : std_logic_vector(C_S_AXIS_S2MM_STS_TDATA_WIDTH downto 0) := (others => '0');
signal fifo_wren        : std_logic := '0';
signal fifo_sinit       : std_logic := '0';

signal rxlength_cdc_from         : std_logic_vector(C_SG_LENGTH_WIDTH-1 downto 0) := (others => '0');
signal rxlength_valid_cdc_from   : std_logic := '0';

    signal rxlength_valid_trdy : std_logic := '0';
--signal sts_tvalid_re    : std_logic := '0';-- CR565502
--signal sts_tvalid_d1    : std_logic := '0';-- CR565502

signal sts_tvalid       : std_logic := '0';
signal sts_tready       : std_logic := '0';
signal sts_tdata        : std_logic_vector(C_S_AXIS_S2MM_STS_TDATA_WIDTH-1 downto 0) := (others => '0');
signal sts_tkeep        : std_logic_vector((C_S_AXIS_S2MM_STS_TDATA_WIDTH/8)-1 downto 0) := (others => '0');
signal sts_tlast        : std_logic := '0';

signal m_tvalid         : std_logic := '0';
signal m_tready         : std_logic := '0';
signal m_tdata          : std_logic_vector(C_S_AXIS_S2MM_STS_TDATA_WIDTH-1 downto 0) := (others => '0');
signal m_tkeep          : std_logic_vector((C_S_AXIS_S2MM_STS_TDATA_WIDTH/8)-1 downto 0) := (others => '0');
signal m_tlast          : std_logic := '0';



signal tag_stripped     : std_logic := '0';
signal mask_tag_write   : std_logic := '0';
--signal mask_tag_hold    : std_logic := '0';-- CR565502

signal skid_rst         : std_logic := '0';

-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin


-- Primary Clock is synchronous to Secondary Clock therfore
-- instantiate a sync fifo.
GEN_SYNC_FIFO : if C_PRMRY_IS_ACLK_ASYNC = 0 generate
signal s2mm_stop_d1 : std_logic := '0';
signal s2mm_stop_re : std_logic := '0';
signal sts_rden : std_logic := '0';
signal follower_empty : std_logic := '0';
signal fifo_empty : std_logic := '0';
signal fifo_out : std_logic_vector (C_S_AXIS_S2MM_STS_TDATA_WIDTH downto 0) := (others => '0');

begin
        -- Generate Synchronous FIFO
--    I_STSSTRM_FIFO : entity lib_srl_fifo_v1_0_2.sync_fifo_fg
--        generic map (
--            C_FAMILY                =>  C_FAMILY                ,
--            C_MEMORY_TYPE           =>  USE_LOGIC_FIFOS,
--            C_WRITE_DATA_WIDTH      =>  C_S_AXIS_S2MM_STS_TDATA_WIDTH + 1,
--            C_WRITE_DEPTH           =>  STSSTRM_FIFO_DEPTH       ,
--            C_READ_DATA_WIDTH       =>  C_S_AXIS_S2MM_STS_TDATA_WIDTH + 1,
--            C_READ_DEPTH            =>  STSSTRM_FIFO_DEPTH       ,
--            C_PORTS_DIFFER          =>  0,
--            C_HAS_DCOUNT            =>  1, --req for proper fifo operation
--            C_DCOUNT_WIDTH          =>  STSSTRM_FIFO_CNT_WIDTH,
--            C_HAS_ALMOST_FULL       =>  0,
--            C_HAS_RD_ACK            =>  0,
--            C_HAS_RD_ERR            =>  0,
--            C_HAS_WR_ACK            =>  0,
--            C_HAS_WR_ERR            =>  0,
--            C_RD_ACK_LOW            =>  0,
--            C_RD_ERR_LOW            =>  0,
--            C_WR_ACK_LOW            =>  0,
--            C_WR_ERR_LOW            =>  0,
--            C_PRELOAD_REGS          =>  1,-- 1 = first word fall through
--            C_PRELOAD_LATENCY       =>  0 -- 0 = first word fall through
--  --          C_USE_EMBEDDED_REG      =>  1 -- 0 ;
--        )
--        port map (
--
--            Clk             =>  m_axi_sg_aclk       ,
--            Sinit           =>  fifo_sinit          ,
--            Din             =>  fifo_din            ,
--            Wr_en           =>  fifo_wren           ,
--            Rd_en           =>  stsstrm_fifo_rden   ,
--            Dout            =>  stsstrm_fifo_dout   ,
--            Full            =>  fifo_full           ,
--            Empty           =>  stsstrm_fifo_empty  ,
--            Almost_full     =>  open                ,
--            Data_count      =>  open                ,
--            Rd_ack          =>  open                ,
--            Rd_err          =>  open                ,
--            Wr_ack          =>  open                ,
--            Wr_err          =>  open
--
--        );

       I_UPDT_STS_FIFO : entity lib_srl_fifo_v1_0_2.srl_fifo_f
       generic map (
         C_DWIDTH            =>  C_S_AXIS_S2MM_STS_TDATA_WIDTH + 1,
         C_DEPTH             =>  16    ,
         C_FAMILY            =>  C_FAMILY
         )
       port map (
         Clk           =>  m_axi_sg_aclk       ,
         Reset         =>  fifo_sinit              ,
         FIFO_Write    =>  fifo_wren       ,
         Data_In       =>  fifo_din      ,
         FIFO_Read     =>  sts_rden, --sts_queue_rden      ,
         Data_Out      =>  fifo_out, --sts_queue_dout      ,
         FIFO_Empty    =>  fifo_empty, --sts_queue_empty      ,
         FIFO_Full     =>  fifo_full    ,
         Addr          =>  open
         );

   sts_rden <= (not fifo_empty) and follower_empty;

   stsstrm_fifo_empty <= follower_empty;

process (m_axi_sg_aclk)
begin
      if (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
           if (fifo_sinit = '1' or stsstrm_fifo_rden = '1') then
              follower_empty <= '1';     
           elsif (sts_rden = '1') then
              follower_empty <= '0';     
           end if;
      end if;
end process;   


process (m_axi_sg_aclk)
begin
      if (m_axi_sg_aclk'event and m_axi_sg_aclk = '1') then
           if (fifo_sinit = '1') then
              stsstrm_fifo_dout <= (others => '0');     
           elsif (sts_rden = '1') then
              stsstrm_fifo_dout <= fifo_out;     
           end if;
      end if;
end process;   

    fifo_sinit              <= not m_axi_sg_aresetn;
    fifo_din                <= sts_tlast & sts_tdata;
    fifo_wren               <= sts_tvalid and not fifo_full and not rxlength_valid_cdc_from and not mask_tag_write;
    sts_tready              <= not fifo_sinit and not fifo_full and not rxlength_valid_cdc_from;


-- CR565502 - particular throttle condition caused masking of tag write to not occur
-- simplified logic will provide more robust handling of tag write mask
--    -- Create register delay of status tvalid in order to create a
--    -- rising edge pulse.  note xx_re signal will hold at 1 if
--    -- fifo full on rising edge of tvalid.
--    REG_TVALID : process(axi_prmry_aclk)
--        begin
--            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
--                if(m_axi_sg_aresetn = '0')then
--                    sts_tvalid_d1 <= '0';
--                elsif(fifo_full = '0')then
--                    sts_tvalid_d1 <= sts_tvalid;
--                end if;
--            end if;
--        end process REG_TVALID;
--
--    -- rising edge on tvalid used to gate off status tag from being
--    -- writen into fifo.
--    sts_tvalid_re <= sts_tvalid and not sts_tvalid_d1;

    REG_TAG_STRIPPED : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    tag_stripped <= '0';
                -- Reset on write of last word
                elsif(fifo_wren = '1' and sts_tlast = '1')then
                    tag_stripped <= '0';
                -- Set on beginning of new status stream
                elsif(sts_tready = '1' and sts_tvalid = '1')then
                    tag_stripped <= '1';
                end if;
            end if;
        end process REG_TAG_STRIPPED;

-- CR565502 - particular throttle condition caused masking of tag write to not occur
-- simplified logic will provide more robust handling of tag write mask
--    REG_MASK_TAG : process(m_axi_sg_aclk)
--        begin
--            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
--                if(m_axi_sg_aresetn = '0')then
--                    mask_tag_hold <= '0';
--                elsif((sts_tvalid_re = '1' and tag_stripped = '0')
--                   or (fifo_wren = '1' and sts_tlast = '1'))then
--                    mask_tag_hold <= '1';
--                elsif(tag_stripped = '1')then
--                    mask_tag_hold <= '0';
--                end if;
--            end if;
--        end process;
--
--    -- Mask TAG if not already masked and rising edge of tvalid
--    mask_tag_write <= not tag_stripped and (sts_tvalid_re or mask_tag_hold);
    mask_tag_write <= not tag_stripped and sts_tready and sts_tvalid;

    -- Generate logic to capture receive length when Use Receive Length is
    -- enabled
    GEN_STS_APP_LENGTH : if C_SG_USE_STSAPP_LENGTH = 1 generate
    begin
        -- Register receive length on assertion of last and valid
        -- Mark rxlength as valid for higher processes
        REG_RXLENGTH : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0' or s2mm_rxlength_clr = '1')then
                        rxlength_cdc_from       <= (others => '0');
                        rxlength_valid_cdc_from <= '0';
                    elsif(sts_tlast = '1' and sts_tvalid = '1' and sts_tready = '1')then
                        rxlength_cdc_from       <= sts_tdata(C_SG_LENGTH_WIDTH-1 downto 0);
                        rxlength_valid_cdc_from <= '1';
                    end if;
                end if;
            end process REG_RXLENGTH;

        s2mm_rxlength_valid <= rxlength_valid_cdc_from;
        s2mm_rxlength       <= rxlength_cdc_from;

    end generate GEN_STS_APP_LENGTH;

    -- Do NOT generate logic to capture receive length when option disabled
    GEN_NO_STS_APP_LENGTH : if C_SG_USE_STSAPP_LENGTH = 0 generate
    begin
        s2mm_rxlength_valid <= '0';
        s2mm_rxlength       <= (others => '0');
    end generate GEN_NO_STS_APP_LENGTH;

    -- register stop to create re pulse
    REG_STOP : process(axi_prmry_aclk)
        begin
            if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
                if(p_reset_n = '0')then
                    s2mm_stop_d1 <= '0';
                else
                    s2mm_stop_d1 <= s2mm_stop;
                end if;
            end if;
        end process REG_STOP;

    s2mm_stop_re <= s2mm_stop and not s2mm_stop_d1;

    skid_rst   <= not m_axi_sg_aresetn;

ENABLE_SKID : if C_ENABLE_SKID = 1 generate
begin

    ---------------------------------------------------------------------------
    -- Buffer AXI Signals
    ---------------------------------------------------------------------------
    STS_SKID_BUF_I : entity axi_dma_v7_1_13.axi_dma_skid_buf
        generic map(
            C_WDATA_WIDTH       => C_S_AXIS_S2MM_STS_TDATA_WIDTH
        )
        port map(
            -- System Ports
            ACLK                => m_axi_sg_aclk                            ,
            ARST                => skid_rst                                 ,

            skid_stop           => s2mm_stop_re                             ,

            -- Slave Side (Stream Data Input)
            S_VALID             => s_axis_s2mm_sts_tvalid                   ,
            S_READY             => s_axis_s2mm_sts_tready                   ,
            S_Data              => s_axis_s2mm_sts_tdata                    ,
            S_STRB              => s_axis_s2mm_sts_tkeep                    ,
            S_Last              => s_axis_s2mm_sts_tlast                    ,

            -- Master Side (Stream Data Output
            M_VALID             => sts_tvalid                               ,
            M_READY             => sts_tready                               ,
            M_Data              => sts_tdata                                ,
            M_STRB              => sts_tkeep                                ,
            M_Last              => sts_tlast
        );

end generate ENABLE_SKID;


DISABLE_SKID : if C_ENABLE_SKID = 0 generate
begin

     sts_tvalid <= s_axis_s2mm_sts_tvalid;
     s_axis_s2mm_sts_tready <= sts_tready;
     sts_tdata <= s_axis_s2mm_sts_tdata;
     sts_tkeep <= s_axis_s2mm_sts_tkeep;
     sts_tlast <= s_axis_s2mm_sts_tlast;

end generate DISABLE_SKID;



end generate GEN_SYNC_FIFO;


-- Primary Clock is asynchronous to Secondary Clock therfore
-- instantiate an async fifo.
GEN_ASYNC_FIFO : if C_PRMRY_IS_ACLK_ASYNC = 1 generate
  ATTRIBUTE async_reg                      : STRING;

signal s2mm_stop_reg   : std_logic := '0'; -- CR605883
signal p_s2mm_stop_d1_cdc_tig  : std_logic := '0';
signal p_s2mm_stop_d2  : std_logic := '0';
signal p_s2mm_stop_d3  : std_logic := '0';
signal p_s2mm_stop_re  : std_logic := '0';
  --ATTRIBUTE async_reg OF p_s2mm_stop_d1_cdc_tig  : SIGNAL IS "true";
  --ATTRIBUTE async_reg OF p_s2mm_stop_d2  : SIGNAL IS "true";
begin



    -- Generate Asynchronous FIFO
    I_STSSTRM_FIFO : entity axi_dma_v7_1_13.axi_dma_afifo_autord
      generic map(
         C_DWIDTH        => C_S_AXIS_S2MM_STS_TDATA_WIDTH + 1    ,
--         C_DEPTH         => STSSTRM_FIFO_DEPTH                  ,
--         C_CNT_WIDTH     => STSSTRM_FIFO_CNT_WIDTH              ,
         C_DEPTH         => 15                  ,
         C_CNT_WIDTH     => 4                   ,
         C_USE_BLKMEM    => USE_LOGIC_FIFOS                     ,
         C_FAMILY        => C_FAMILY
        )
      port map(
        -- Inputs
         AFIFO_Ainit                => fifo_sinit               ,
         AFIFO_Wr_clk               => axi_prmry_aclk           ,
         AFIFO_Wr_en                => fifo_wren                ,
         AFIFO_Din                  => fifo_din                 ,
         AFIFO_Rd_clk               => m_axi_sg_aclk            ,
         AFIFO_Rd_en                => stsstrm_fifo_rden        ,
         AFIFO_Clr_Rd_Data_Valid    => '0'                      ,

        -- Outputs
         AFIFO_DValid               => open                     ,
         AFIFO_Dout                 => stsstrm_fifo_dout        ,
         AFIFO_Full                 => fifo_full                ,
         AFIFO_Empty                => stsstrm_fifo_empty       ,
         AFIFO_Almost_full          => open                     ,
         AFIFO_Almost_empty         => open                     ,
         AFIFO_Wr_count             => open                     ,
         AFIFO_Rd_count             => open                     ,
         AFIFO_Corr_Rd_count        => open                     ,
         AFIFO_Corr_Rd_count_minus1 => open                     ,
         AFIFO_Rd_ack               => open
        );

    fifo_sinit              <= not p_reset_n;

    fifo_din                <= sts_tlast & sts_tdata;
    fifo_wren               <= sts_tvalid               -- valid data
                                and not fifo_full       -- fifo has room
                                and not rxlength_valid_trdy --rxlength_valid_cdc_from  -- not holding a valid length
                                and not mask_tag_write; -- not masking off tag word

    sts_tready              <= not fifo_sinit and not fifo_full and not rxlength_valid_trdy; --rxlength_valid_cdc_from;
-- CR565502 - particular throttle condition caused masking of tag write to not occur
-- simplified logic will provide more robust handling of tag write mask
--    -- Create register delay of status tvalid in order to create a
--    -- rising edge pulse.  note xx_re signal will hold at 1 if
--    -- fifo full on rising edge of tvalid.
--    REG_TVALID : process(axi_prmry_aclk)
--        begin
--            if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
--                if(m_axi_sg_aresetn = '0')then
--                    sts_tvalid_d1 <= '0';
--                elsif(fifo_full = '0')then
--                    sts_tvalid_d1 <= sts_tvalid;
--                end if;
--            end if;
--        end process REG_TVALID;
--    -- rising edge on tvalid used to gate off status tag from being
--    -- writen into fifo.
--    sts_tvalid_re <= sts_tvalid and not sts_tvalid_d1;

    REG_TAG_STRIPPED : process(axi_prmry_aclk)
        begin
            if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
                if(p_reset_n = '0')then
                    tag_stripped <= '0';
                -- Reset on write of last word
                elsif(fifo_wren = '1' and sts_tlast = '1')then
                    tag_stripped <= '0';
                -- Set on beginning of new status stream
                elsif(sts_tready = '1' and sts_tvalid = '1')then
                    tag_stripped <= '1';
                end if;
            end if;
        end process REG_TAG_STRIPPED;

-- CR565502 - particular throttle condition caused masking of tag write to not occur
-- simplified logic will provide more robust handling of tag write mask
--    REG_MASK_TAG : process(axi_prmry_aclk)
--        begin
--            if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
--                if(m_axi_sg_aresetn = '0')then
--                    mask_tag_hold <= '0';
--                elsif(tag_stripped = '1')then
--                    mask_tag_hold <= '0';
--
--                elsif(sts_tvalid_re = '1'
--                or (fifo_wren = '1' and sts_tlast = '1'))then
--                    mask_tag_hold <= '1';
--                end if;
--            end if;
--        end process;
--
--    -- Mask TAG if not already masked and rising edge of tvalid
--    mask_tag_write <= not tag_stripped and (sts_tvalid_re or mask_tag_hold);

    mask_tag_write <= not tag_stripped and sts_tready and sts_tvalid;

    -- Generate logic to capture receive length when Use Receive Length is
    -- enabled
    GEN_STS_APP_LENGTH : if C_SG_USE_STSAPP_LENGTH = 1 generate
    signal rxlength_clr_d1_cdc_tig      : std_logic := '0';
    signal rxlength_clr_d2      : std_logic := '0';

    signal rxlength_d1_cdc_to          : std_logic_vector(C_SG_LENGTH_WIDTH-1 downto 0) := (others => '0');
    signal rxlength_d2          : std_logic_vector(C_SG_LENGTH_WIDTH-1 downto 0) := (others => '0');
    signal rxlength_valid_d1_cdc_to    : std_logic := '0';
    signal rxlength_valid_d2_cdc_from    : std_logic := '0';
    signal rxlength_valid_d3    : std_logic := '0';
    signal rxlength_valid_d4    : std_logic := '0';
    signal rxlength_valid_d1_back_cdc_to, rxlength_valid_d2_back : std_logic := '0';
      ATTRIBUTE async_reg                      : STRING;
      --ATTRIBUTE async_reg OF rxlength_d1_cdc_to  : SIGNAL IS "true";
      --ATTRIBUTE async_reg OF rxlength_d2  : SIGNAL IS "true";
      --ATTRIBUTE async_reg OF rxlength_valid_d1_cdc_to  : SIGNAL IS "true";
      --ATTRIBUTE async_reg OF rxlength_valid_d1_back_cdc_to  : SIGNAL IS "true";
      --ATTRIBUTE async_reg OF rxlength_valid_d2_back  : SIGNAL IS "true";
    

    begin
        -- Double register from secondary clock domain to primary
S2P_CLK_CROSS : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => s2mm_rxlength_clr,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => axi_prmry_aclk,
        scndry_resetn              => '0',
        scndry_out                 => rxlength_clr_d2,
        scndry_vect_out            => open
    );

--        S2P_CLK_CROSS : process(axi_prmry_aclk)
--            begin
--                if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
--                    if(p_reset_n = '0')then
--                        rxlength_clr_d1_cdc_tig <= '0';
--                        rxlength_clr_d2 <= '0';
--                    else
--                        rxlength_clr_d1_cdc_tig <= s2mm_rxlength_clr;
--                        rxlength_clr_d2 <= rxlength_clr_d1_cdc_tig;
--                    end if;
--                end if;
--            end process S2P_CLK_CROSS;

        -- Register receive length on assertion of last and valid
        -- Mark rxlength as valid for higher processes

        TRDY_RXLENGTH : process(axi_prmry_aclk)
            begin
                if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
                    if(p_reset_n = '0' or rxlength_clr_d2 = '1')then
                        rxlength_valid_trdy <= '0';
                    elsif(sts_tlast = '1' and sts_tvalid = '1' and sts_tready = '1')then
                        rxlength_valid_trdy <= '1';
                    end if;
                end if;
            end process TRDY_RXLENGTH;


        REG_RXLENGTH : process(axi_prmry_aclk)
            begin
                if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
                    if(p_reset_n = '0') then -- or rxlength_clr_d2 = '1')then
                        rxlength_cdc_from       <= (others => '0');
                        rxlength_valid_cdc_from <= '0';
                    elsif(sts_tlast = '1' and sts_tvalid = '1' and sts_tready = '1')then
                        rxlength_cdc_from       <= sts_tdata(C_SG_LENGTH_WIDTH-1 downto 0);
                        rxlength_valid_cdc_from <= '1';
                    elsif (rxlength_valid_d2_back = '1') then
                        rxlength_valid_cdc_from <= '0';
                    end if;
                end if;
            end process REG_RXLENGTH;


SYNC_RXLENGTH : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => rxlength_valid_d2_cdc_from,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => axi_prmry_aclk,
        scndry_resetn              => '0',
        scndry_out                 => rxlength_valid_d2_back,
        scndry_vect_out            => open
    );

--        SYNC_RXLENGTH : process(axi_prmry_aclk)
--            begin
--                if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
--                    if(p_reset_n = '0') then -- or rxlength_clr_d2 = '1')then
--
--                        rxlength_valid_d1_back_cdc_to   <= '0';
--                        rxlength_valid_d2_back   <= '0';
--                    else 
--                        rxlength_valid_d1_back_cdc_to   <= rxlength_valid_d2_cdc_from;
--                        rxlength_valid_d2_back   <= rxlength_valid_d1_back_cdc_to;
--                    
--                    end if;
--                end if;
--            end process SYNC_RXLENGTH;


        -- Double register from primary clock domain to secondary

P2S_CLK_CROSS : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => rxlength_valid_cdc_from,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => m_axi_sg_aclk,
        scndry_resetn              => '0',
        scndry_out                 => rxlength_valid_d2_cdc_from,
        scndry_vect_out            => open
    );


P2S_CLK_CROSS2 : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 0,
        C_VECTOR_WIDTH             => C_SG_LENGTH_WIDTH,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => '0',
        prmry_vect_in              => rxlength_cdc_from,

        scndry_aclk                => m_axi_sg_aclk,
        scndry_resetn              => '0',
        scndry_out                 => open, 
        scndry_vect_out            => rxlength_d2
    );


        P2S_CLK_CROSS1 : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0') then -- or s2mm_rxlength_clr = '1') then
--                        rxlength_d1_cdc_to         <= (others => '0');
--                        rxlength_d2         <= (others => '0');
--                        rxlength_valid_d1_cdc_to   <= '0';
--                        rxlength_valid_d2_cdc_from   <= '0';
                        rxlength_valid_d3   <= '0';
                    else
--                        rxlength_d1_cdc_to         <= rxlength_cdc_from;
--                        rxlength_d2         <= rxlength_d1_cdc_to;
--                        rxlength_valid_d1_cdc_to   <= rxlength_valid_cdc_from;
--                        rxlength_valid_d2_cdc_from   <= rxlength_valid_d1_cdc_to;
                        rxlength_valid_d3   <= rxlength_valid_d2_cdc_from;
                    end if;
                end if;
            end process P2S_CLK_CROSS1;

           process (m_axi_sg_aclk)
           begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0' or s2mm_rxlength_clr = '1')then
                        rxlength_valid_d4 <= '0';    
                    elsif (rxlength_valid_d3 = '1' and rxlength_valid_d2_cdc_from = '0') then
                        rxlength_valid_d4 <= '1';    
                    end if;
                 end if;
           end process;

        s2mm_rxlength       <= rxlength_d2;
       -- s2mm_rxlength_valid <= rxlength_valid_d2;
        s2mm_rxlength_valid <= rxlength_valid_d4;

    end generate GEN_STS_APP_LENGTH;

    -- Do NOT generate logic to capture receive length when option disabled
    GEN_NO_STS_APP_LENGTH : if C_SG_USE_STSAPP_LENGTH = 0 generate
        s2mm_rxlength_valid <= '0';
        s2mm_rxlength       <= (others => '0');
    end generate GEN_NO_STS_APP_LENGTH;

    -- CR605883
    -- Register stop to provide pure FF output for synchronizer
    REG_STOP : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    s2mm_stop_reg <= '0';
                else
                    s2mm_stop_reg <= s2mm_stop;
                end if;
            end if;
        end process REG_STOP;


    -- double register s2mm error into primary clock domain

REG_ERR2PRMRY : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => s2mm_stop_reg,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => axi_prmry_aclk,
        scndry_resetn              => '0',
        scndry_out                 => p_s2mm_stop_d2,
        scndry_vect_out            => open
    );


    REG_ERR2PRMRY1 : process(axi_prmry_aclk)
        begin
            if(axi_prmry_aclk'EVENT and axi_prmry_aclk = '1')then
                if(p_reset_n = '0')then
--                    p_s2mm_stop_d1_cdc_tig <= '0';
--                    p_s2mm_stop_d2 <= '0';
                    p_s2mm_stop_d3 <= '0';
                else
                    --p_s2mm_stop_d1_cdc_tig <= s2mm_stop;      -- CR605883
--                    p_s2mm_stop_d1_cdc_tig <= s2mm_stop_reg;
--                    p_s2mm_stop_d2 <= p_s2mm_stop_d1_cdc_tig;
                    p_s2mm_stop_d3 <= p_s2mm_stop_d2;
                end if;
            end if;
        end process REG_ERR2PRMRY1;

    p_s2mm_stop_re <= p_s2mm_stop_d2 and not p_s2mm_stop_d3;

    skid_rst   <= not p_reset_n;

    ---------------------------------------------------------------------------
    -- Buffer AXI Signals
    ---------------------------------------------------------------------------
    STS_SKID_BUF_I : entity axi_dma_v7_1_13.axi_dma_skid_buf
        generic map(
            C_WDATA_WIDTH       => C_S_AXIS_S2MM_STS_TDATA_WIDTH
        )
        port map(
            -- System Ports
            ACLK                => axi_prmry_aclk                           ,
            ARST                => skid_rst                                 ,

            skid_stop           => p_s2mm_stop_re                           ,

            -- Slave Side (Stream Data Input)
            S_VALID             => s_axis_s2mm_sts_tvalid                   ,
            S_READY             => s_axis_s2mm_sts_tready                   ,
            S_Data              => s_axis_s2mm_sts_tdata                    ,
            S_STRB              => s_axis_s2mm_sts_tkeep                    ,
            S_Last              => s_axis_s2mm_sts_tlast                    ,

            -- Master Side (Stream Data Output
            M_VALID             => sts_tvalid                               ,
            M_READY             => sts_tready                               ,
            M_Data              => sts_tdata                                ,
            M_STRB              => sts_tkeep                                ,
            M_Last              => sts_tlast
        );


end generate GEN_ASYNC_FIFO;




end implementation;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------
-------------------------------------------------------------------------------
-- Filename:          axi_dma_s2mm_mngr.vhd
-- Description: This entity is the top level entity for the AXI DMA S2MM
--              manager.
--
-- VHDL-Standard:   VHDL'93
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;

library unisim;
use unisim.vcomponents.all;

library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;

-------------------------------------------------------------------------------
entity  axi_dma_s2mm_mngr is
    generic(

        C_PRMRY_IS_ACLK_ASYNC           : integer range 0 to 1         := 0;
            -- Primary MM2S/S2MM sync/async mode
            -- 0 = synchronous mode     - all clocks are synchronous
            -- 1 = asynchronous mode    - Primary data path channels (MM2S and S2MM)
            --                            run asynchronous to AXI Lite, DMA Control,
            --                            and SG.

        C_PRMY_CMDFIFO_DEPTH            : integer range 1 to 16         := 1;
            -- Depth of DataMover command FIFO

        C_DM_STATUS_WIDTH               : integer range 8 to 32         := 8;
            -- Width of DataMover status word
            -- 8  for Determinate BTT Mode
            -- 32 for Indterminate BTT Mode

        -----------------------------------------------------------------------
        -- Scatter Gather Parameters
        -----------------------------------------------------------------------
        C_INCLUDE_SG                : integer range 0 to 1          := 1;
            -- Include or Exclude the Scatter Gather Engine
            -- 0 = Exclude SG Engine - Enables Simple DMA Mode
            -- 1 = Include SG Engine - Enables Scatter Gather Mode

        C_SG_INCLUDE_STSCNTRL_STRM      : integer range 0 to 1      := 1;
            -- Include or Exclude AXI Status and AXI Control Streams
            -- 0 = Exclude Status and Control Streams
            -- 1 = Include Status and Control Streams

        C_SG_INCLUDE_DESC_QUEUE     : integer range 0 to 1          := 0;
            -- Include or Exclude Scatter Gather Descriptor Queuing
            -- 0 = Exclude SG Descriptor Queuing
            -- 1 = Include SG Descriptor Queuing

        C_SG_USE_STSAPP_LENGTH      : integer range 0 to 1          := 1;
            -- Enable or Disable use of Status Stream Rx Length.  Only valid
            -- if C_SG_INCLUDE_STSCNTRL_STRM = 1
            -- 0 = Don't use Rx Length
            -- 1 = Use Rx Length

        C_SG_LENGTH_WIDTH               : integer range 8 to 23     := 14;
            -- Descriptor Buffer Length, Transferred Bytes, and Status Stream
            -- Rx Length Width.  Indicates the least significant valid bits of
            -- descriptor buffer length, transferred bytes, or Rx Length value
            -- in the status word coincident with tlast.


        C_M_AXI_SG_ADDR_WIDTH           : integer range 32 to 64    := 32;
            -- Master AXI Memory Map Address Width for Scatter Gather R/W Port

        C_M_AXIS_SG_TDATA_WIDTH          : integer range 32 to 32    := 32;
            -- AXI Master Stream in for descriptor fetch

        C_S_AXIS_UPDPTR_TDATA_WIDTH : integer range 32 to 32        := 32;
            -- 32 Update Status Bits

        C_S_AXIS_UPDSTS_TDATA_WIDTH : integer range 33 to 33        := 33;
            -- 1 IOC bit + 32 Update Status Bits

        C_S_AXIS_S2MM_STS_TDATA_WIDTH : integer range 32 to 32    := 32;
            -- Slave AXI Status Stream Data Width

        -----------------------------------------------------------------------
        -- Stream to Memory Map (S2MM) Parameters
        -----------------------------------------------------------------------
        C_INCLUDE_S2MM                  : integer range 0 to 1      := 1;
            -- Include or exclude S2MM primary data path
            -- 0 = Exclude S2MM primary data path
            -- 1 = Include S2MM primary data path

        C_M_AXI_S2MM_ADDR_WIDTH         : integer range 32 to 64    := 32;
            -- Master AXI Memory Map Address Width for S2MM Write Port

        C_NUM_S2MM_CHANNELS             : integer range 1 to 16     := 1;

        C_ENABLE_MULTI_CHANNEL                 : integer range 0 to 1    := 0;     
        C_MICRO_DMA                     : integer range 0 to 1 := 0;
 
        C_FAMILY                        : string            := "virtex5"
            -- Target FPGA Device Family
    );
    port (

        -- Secondary Clock and Reset
        m_axi_sg_aclk               : in  std_logic                         ;                      --
        m_axi_sg_aresetn            : in  std_logic                         ;                      --
                                                                                                   --
        -- Primary Clock and Reset                                                                 --
        axi_prmry_aclk              : in  std_logic                         ;                      --
        p_reset_n                   : in  std_logic                         ;                      --
                                                                                                   --
        soft_reset                  : in  std_logic                         ;                      --
        -- MM2S Control and Status                                                                 --
        s2mm_run_stop               : in  std_logic                         ;                      --
        s2mm_keyhole                : in  std_logic                         ;
        s2mm_halted                 : in  std_logic                         ;                      --
        s2mm_ftch_idle              : in  std_logic                         ;                      --
        s2mm_updt_idle              : in  std_logic                         ;                      --
        s2mm_tailpntr_enble         : in  std_logic                         ;                      --
        s2mm_ftch_err_early         : in  std_logic                         ;                      --
        s2mm_ftch_stale_desc        : in  std_logic                         ;                      --
        s2mm_halt                   : in  std_logic                         ;                      --
        s2mm_halt_cmplt             : in  std_logic                         ;                      --
        s2mm_packet_eof_out         : out std_logic                         ;
        s2mm_halted_clr             : out std_logic                         ;                      --
        s2mm_halted_set             : out std_logic                         ;                      --
        s2mm_idle_set               : out std_logic                         ;                      --
        s2mm_idle_clr               : out std_logic                         ;                      --
        s2mm_new_curdesc            : out std_logic_vector                                         --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0)  ;                      --
        s2mm_new_curdesc_wren       : out std_logic                         ;                      --
        s2mm_stop                   : out std_logic                         ;                      --
        s2mm_desc_flush             : out std_logic                         ;                      --
        s2mm_all_idle               : out std_logic                         ;                      --
        s2mm_error                  : out std_logic                         ;                      --
        mm2s_error                  : in  std_logic                         ;                      --
        s2mm_desc_info_in              : in  std_logic_vector (13 downto 0)    ;

        -- Simple DMA Mode Signals
        s2mm_da                     : in  std_logic_vector                                         --
                                        (C_M_AXI_S2MM_ADDR_WIDTH-1 downto 0);                      --
        s2mm_length                 : in  std_logic_vector                                         --
                                        (C_SG_LENGTH_WIDTH-1 downto 0)      ;                      --
        s2mm_length_wren            : in  std_logic                         ;                      --
        s2mm_smple_done             : out std_logic                         ;                      --
        s2mm_interr_set             : out std_logic                         ;                      --
        s2mm_slverr_set             : out std_logic                         ;                      --
        s2mm_decerr_set             : out std_logic                         ;                      --
        s2mm_bytes_rcvd             : out std_logic_vector                                         --
                                        (C_SG_LENGTH_WIDTH-1 downto 0)      ;                      --
        s2mm_bytes_rcvd_wren        : out std_logic                         ;                      --
                                                                                                   --
        -- SG S2MM Descriptor Fetch AXI Stream In                                                  --
        m_axis_s2mm_ftch_tdata      : in  std_logic_vector                                         --
                                        (C_M_AXIS_SG_TDATA_WIDTH-1 downto 0);                      --
        m_axis_s2mm_ftch_tvalid     : in  std_logic                         ;                      --
        m_axis_s2mm_ftch_tready     : out std_logic                         ;                      --
        m_axis_s2mm_ftch_tlast      : in  std_logic                         ;                      --

        m_axis_s2mm_ftch_tdata_new      : in  std_logic_vector                                         --
                                        (96+31*0+(0+2)*(C_M_AXI_SG_ADDR_WIDTH-32) downto 0);                      --
        m_axis_s2mm_ftch_tdata_mcdma_new      : in  std_logic_vector                                         --
                                        (63 downto 0);                      --
        m_axis_s2mm_ftch_tdata_mcdma_nxt      : in  std_logic_vector                                         --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0);                      --
        m_axis_s2mm_ftch_tvalid_new     : in  std_logic                         ;                      --
        m_axis_ftch2_desc_available     : in std_logic;
                                                                                                   --
                                                                                                   --
        -- SG S2MM Descriptor Update AXI Stream Out                                                --
        s_axis_s2mm_updtptr_tdata   : out std_logic_vector                                         --
                                        (C_M_AXI_SG_ADDR_WIDTH-1 downto 0) ;                 --
        s_axis_s2mm_updtptr_tvalid  : out std_logic                         ;                      --
        s_axis_s2mm_updtptr_tready  : in  std_logic                         ;                      --
        s_axis_s2mm_updtptr_tlast   : out std_logic                         ;                      --
                                                                                                   --
        s_axis_s2mm_updtsts_tdata   : out std_logic_vector                                         --
                                        (C_S_AXIS_UPDSTS_TDATA_WIDTH-1 downto 0) ;                 --
        s_axis_s2mm_updtsts_tvalid  : out std_logic                         ;                      --
        s_axis_s2mm_updtsts_tready  : in  std_logic                         ;                      --
        s_axis_s2mm_updtsts_tlast   : out std_logic                         ;                      --
                                                                                                   --
        -- User Command Interface Ports (AXI Stream)                                               --
        s_axis_s2mm_cmd_tvalid      : out std_logic                         ;                      --
        s_axis_s2mm_cmd_tready      : in  std_logic                         ;                      --
        s_axis_s2mm_cmd_tdata       : out std_logic_vector                                         --
                                        ((C_M_AXI_S2MM_ADDR_WIDTH-32+2*32+CMD_BASE_WIDTH+46)-1 downto 0);     --
                                                                                                   --
        -- User Status Interface Ports (AXI Stream)                                                --
        m_axis_s2mm_sts_tvalid      : in  std_logic                         ;                      --
        m_axis_s2mm_sts_tready      : out std_logic                         ;                      --
        m_axis_s2mm_sts_tdata       : in  std_logic_vector                                         --
                                        (C_DM_STATUS_WIDTH - 1 downto 0)    ;                      --
        m_axis_s2mm_sts_tkeep       : in  std_logic_vector((C_DM_STATUS_WIDTH/8-1) downto 0);      --
        s2mm_err                    : in  std_logic                         ;                      --
        updt_error                  : in  std_logic                         ;                      --
        ftch_error                  : in  std_logic                         ;                      --
                                                                                                   --
        -- Stream to Memory Map Status Stream Interface                                            --
        s_axis_s2mm_sts_tdata       : in  std_logic_vector                                         --
                                        (C_S_AXIS_S2MM_STS_TDATA_WIDTH-1 downto 0);                --
        s_axis_s2mm_sts_tkeep       : in  std_logic_vector                                         --
                                        ((C_S_AXIS_S2MM_STS_TDATA_WIDTH/8)-1 downto 0);            --
        s_axis_s2mm_sts_tvalid      : in  std_logic                         ;                      --
        s_axis_s2mm_sts_tready      : out std_logic                         ;                      --
        s_axis_s2mm_sts_tlast       : in  std_logic                                                --
    );

end axi_dma_s2mm_mngr;

-------------------------------------------------------------------------------
-- Architecture
-------------------------------------------------------------------------------
architecture implementation of axi_dma_s2mm_mngr is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


-------------------------------------------------------------------------------
-- Functions
-------------------------------------------------------------------------------

-- No Functions Declared

-------------------------------------------------------------------------------
-- Constants Declarations
-------------------------------------------------------------------------------

-- No Constants Declared

-------------------------------------------------------------------------------
-- Signal / Type Declarations
-------------------------------------------------------------------------------
-- Primary DataMover Command signals
signal s2mm_cmnd_wr                 : std_logic := '0';
signal s2mm_cmnd_data               : std_logic_vector
                                        ((C_M_AXI_S2MM_ADDR_WIDTH-32+2*32+CMD_BASE_WIDTH+46)-1 downto 0) := (others => '0');
signal s2mm_cmnd_pending            : std_logic := '0';
-- Primary DataMover Status signals
signal s2mm_done                    : std_logic := '0';
signal s2mm_stop_i                  : std_logic := '0';
signal s2mm_interr                  : std_logic := '0';
signal s2mm_slverr                  : std_logic := '0';
signal s2mm_decerr                  : std_logic := '0';
signal s2mm_tag                     : std_logic_vector(3 downto 0) := (others => '0');
signal s2mm_brcvd                   : std_logic_vector(C_SG_LENGTH_WIDTH-1 downto 0) := (others => '0');
signal dma_s2mm_error               : std_logic := '0';
signal soft_reset_d1                : std_logic := '0';
signal soft_reset_d2                : std_logic := '0';
signal soft_reset_re                : std_logic := '0';
signal s2mm_error_i                 : std_logic := '0';
signal sts_strm_stop                : std_logic := '0';
signal s2mm_halted_set_i            : std_logic := '0';

signal s2mm_sts_received_clr        : std_logic := '0';
signal s2mm_sts_received            : std_logic := '0';

signal s2mm_cmnd_idle               : std_logic := '0';
signal s2mm_sts_idle                : std_logic := '0';
signal s2mm_eof_set                 : std_logic := '0';
signal s2mm_packet_eof              : std_logic := '0';

-- Scatter Gather Interface signals
signal desc_fetch_req               : std_logic := '0';
signal desc_fetch_done              : std_logic := '0';
signal desc_update_req              : std_logic := '0';
signal desc_update_done             : std_logic := '0';
signal desc_available               : std_logic := '0';

signal s2mm_desc_baddress           : std_logic_vector(C_M_AXI_S2MM_ADDR_WIDTH-1 downto 0)  := (others => '0');
signal s2mm_desc_info           : std_logic_vector(31 downto 0)  := (others => '0');
signal s2mm_desc_blength            : std_logic_vector(BUFFER_LENGTH_WIDTH-1 downto 0)    := (others => '0');
signal s2mm_desc_blength_v            : std_logic_vector(BUFFER_LENGTH_WIDTH-1 downto 0)    := (others => '0');
signal s2mm_desc_blength_s            : std_logic_vector(BUFFER_LENGTH_WIDTH-1 downto 0)    := (others => '0');
signal s2mm_desc_cmplt              : std_logic := '0';
signal s2mm_desc_app0               : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH-1 downto 0)    := (others => '0');
signal s2mm_desc_app1               : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH-1 downto 0)    := (others => '0');
signal s2mm_desc_app2               : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH-1 downto 0)    := (others => '0');
signal s2mm_desc_app3               : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH-1 downto 0)    := (others => '0');
signal s2mm_desc_app4               : std_logic_vector(C_M_AXIS_SG_TDATA_WIDTH-1 downto 0)    := (others => '0');


-- S2MM Status Stream Signals
signal s2mm_rxlength_valid          : std_logic := '0';
signal s2mm_rxlength_clr            : std_logic := '0';
signal s2mm_rxlength                : std_logic_vector(C_SG_LENGTH_WIDTH - 1 downto 0) := (others => '0');
signal stsstrm_fifo_rden            : std_logic := '0';
signal stsstrm_fifo_empty           : std_logic := '0';
signal stsstrm_fifo_dout            : std_logic_vector(C_S_AXIS_S2MM_STS_TDATA_WIDTH downto 0) := (others => '0');
signal s2mm_desc_flush_i            : std_logic := '0';

signal updt_pending                 : std_logic := '0';
signal s2mm_cmnd_wr_1               : std_logic := '0';
signal s2mm_eof_micro, s2mm_sof_micro : std_logic := '0';
-------------------------------------------------------------------------------
-- Begin architecture logic
-------------------------------------------------------------------------------
begin

-------------------------------------------------------------------------------
-- Include S2MM (Received) Channel
-------------------------------------------------------------------------------
GEN_S2MM_DMA_CONTROL : if C_INCLUDE_S2MM = 1 generate
begin

    -- pass out to register module
    s2mm_halted_set <= s2mm_halted_set_i;

    -------------------------------------------------------------------------------
    -- Graceful shut down logic
    -------------------------------------------------------------------------------

    -- Error from DataMover (DMAIntErr, DMADecErr, or DMASlvErr) or SG Update error
    -- or SG Fetch error, or Stale Descriptor Error
    s2mm_error_i    <= dma_s2mm_error                -- Primary data mover reports error
                        or updt_error                -- SG Update engine reports error
                        or ftch_error                -- SG Fetch engine reports error
                        or s2mm_ftch_err_early       -- SG Fetch engine reports early error on S2MM
                        or s2mm_ftch_stale_desc;     -- SG Fetch stale descriptor error

    -- pass out to shut down mm2s
    s2mm_error <= s2mm_error_i;

    -- Clear run/stop and stop state machines due to errors or soft reset
    -- Error based on datamover error report or sg update error or sg fetch error
    -- SG update error and fetch error included because need to shut down, no way
    -- to update descriptors on sg update error and on fetch error descriptor
    -- data is corrupt therefor do not want to issue the xfer command to primary datamover

--CR#566306 status for both mm2s and s2mm datamover are masked during shutdown therefore
-- need to stop all processes regardless of the source of the error.
--    s2mm_stop_i    <= s2mm_error                -- Error
--                   or soft_reset;               -- Soft Reset issued
    s2mm_stop_i    <= s2mm_error_i              -- Error on s2mm
                   or mm2s_error                -- Error on mm2s
                   or soft_reset;               -- Soft Reset issued


    -- Register signals out
    REG_OUT : process(m_axi_sg_aclk)
        begin
            if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                if(m_axi_sg_aresetn = '0')then
                    s2mm_stop           <= '0';
                    s2mm_desc_flush_i   <= '0';
                else
                    s2mm_stop           <= s2mm_stop_i;
                    -- Flush any fetch descriptors if error or if run stop cleared
                    s2mm_desc_flush_i   <= s2mm_stop_i or not s2mm_run_stop;
                end if;
            end if;
        end process REG_OUT;


    -- Generate DMA Controller For Scatter Gather Mode
    GEN_SCATTER_GATHER_MODE : if C_INCLUDE_SG = 1 generate
    begin
        -- Not used in Scatter Gather mode
        s2mm_smple_done <= '0';
        s2mm_interr_set <= '0';
        s2mm_slverr_set <= '0';
        s2mm_decerr_set <= '0';
        s2mm_bytes_rcvd             <= (others => '0');
        s2mm_bytes_rcvd_wren        <= '0';

        -- Flush descriptors
        s2mm_desc_flush <= s2mm_desc_flush_i;

OLD_CMD_WR : if (C_SG_USE_STSAPP_LENGTH = 1 and C_SG_INCLUDE_STSCNTRL_STRM = 1 and C_ENABLE_MULTI_CHANNEL = 0) generate
begin
     s2mm_cmnd_wr <=  s2mm_cmnd_wr_1;
end generate OLD_CMD_WR;

NEW_CMD_WR : if (C_SG_USE_STSAPP_LENGTH = 0 or C_SG_INCLUDE_STSCNTRL_STRM = 0 or C_ENABLE_MULTI_CHANNEL = 1) generate
begin
     s2mm_cmnd_wr <=  m_axis_s2mm_ftch_tvalid_new;
end generate NEW_CMD_WR;

        ---------------------------------------------------------------------------
        -- S2MM Primary DMA Controller State Machine
        ---------------------------------------------------------------------------
        I_S2MM_SM : entity  axi_dma_v7_1_13.axi_dma_s2mm_sm
            generic map(
                C_M_AXI_S2MM_ADDR_WIDTH     => C_M_AXI_S2MM_ADDR_WIDTH          ,
                C_SG_LENGTH_WIDTH           => C_SG_LENGTH_WIDTH                ,
                C_SG_INCLUDE_DESC_QUEUE     => C_SG_INCLUDE_DESC_QUEUE          ,
                C_SG_INCLUDE_STSCNTRL_STRM  => C_SG_INCLUDE_STSCNTRL_STRM       ,
                C_SG_USE_STSAPP_LENGTH      => C_SG_USE_STSAPP_LENGTH           ,
                C_ENABLE_MULTI_CHANNEL      => C_ENABLE_MULTI_CHANNEL                  ,
                C_MICRO_DMA                 => C_MICRO_DMA                      ,
                C_PRMY_CMDFIFO_DEPTH        => C_PRMY_CMDFIFO_DEPTH
            )
            port map(
                m_axi_sg_aclk               => m_axi_sg_aclk                    ,
                m_axi_sg_aresetn            => m_axi_sg_aresetn                 ,

                s2mm_stop                   => s2mm_stop_i                      ,

                -- Channel 1 Control and Status
                s2mm_run_stop               => s2mm_run_stop                    ,
                s2mm_keyhole                => s2mm_keyhole                     ,
                s2mm_ftch_idle              => s2mm_ftch_idle                   ,
                s2mm_desc_flush             => s2mm_desc_flush_i                ,
                s2mm_cmnd_idle              => s2mm_cmnd_idle                   ,
                s2mm_sts_idle               => s2mm_sts_idle                    ,
                s2mm_eof_set                => s2mm_eof_set                     ,
                s2mm_eof_micro              => s2mm_eof_micro,
                s2mm_sof_micro              => s2mm_sof_micro,

                -- S2MM Status Stream RX Length
                s2mm_rxlength_valid         => s2mm_rxlength_valid              ,
                s2mm_rxlength_clr           => s2mm_rxlength_clr                ,
                s2mm_rxlength               => s2mm_rxlength                    ,

                -- S2MM Descriptor Fetch Request (from s2mm_sm)
                desc_fetch_req              => desc_fetch_req                   ,
                desc_fetch_done             => desc_fetch_done                  ,
                desc_update_done            => desc_update_done                 ,
                updt_pending                => updt_pending                     ,
                desc_available              => desc_available                   ,

                -- DataMover Command
                s2mm_cmnd_wr                => s2mm_cmnd_wr_1                     ,
                s2mm_cmnd_data              => s2mm_cmnd_data                   ,
                s2mm_cmnd_pending           => s2mm_cmnd_pending                ,

                -- Descriptor Fields
                s2mm_desc_baddress          => s2mm_desc_baddress               ,
                s2mm_desc_info          => s2mm_desc_info               ,
                s2mm_desc_blength           => s2mm_desc_blength,
                s2mm_desc_blength_v           => s2mm_desc_blength_v,
                s2mm_desc_blength_s           => s2mm_desc_blength_s
            );

        ---------------------------------------------------------------------------
        -- S2MM Scatter Gather State Machine
        ---------------------------------------------------------------------------
        I_S2MM_SG_IF : entity  axi_dma_v7_1_13.axi_dma_s2mm_sg_if
            generic map(

                -------------------------------------------------------------------
                -- Scatter Gather Parameters
                -------------------------------------------------------------------
                C_PRMRY_IS_ACLK_ASYNC       => C_PRMRY_IS_ACLK_ASYNC            ,
                C_SG_INCLUDE_STSCNTRL_STRM  => C_SG_INCLUDE_STSCNTRL_STRM       ,
                C_SG_INCLUDE_DESC_QUEUE     => C_SG_INCLUDE_DESC_QUEUE          ,
                C_SG_USE_STSAPP_LENGTH      => C_SG_USE_STSAPP_LENGTH           ,
                C_SG_LENGTH_WIDTH           => C_SG_LENGTH_WIDTH                ,
                C_M_AXIS_SG_TDATA_WIDTH     => C_M_AXIS_SG_TDATA_WIDTH          ,
                C_S_AXIS_UPDPTR_TDATA_WIDTH => C_S_AXIS_UPDPTR_TDATA_WIDTH      ,
                C_S_AXIS_UPDSTS_TDATA_WIDTH => C_S_AXIS_UPDSTS_TDATA_WIDTH      ,
                C_M_AXI_SG_ADDR_WIDTH       => C_M_AXI_SG_ADDR_WIDTH            ,
                C_M_AXI_S2MM_ADDR_WIDTH     => C_M_AXI_S2MM_ADDR_WIDTH          ,
                C_S_AXIS_S2MM_STS_TDATA_WIDTH=> C_S_AXIS_S2MM_STS_TDATA_WIDTH   ,
                C_NUM_S2MM_CHANNELS         => C_NUM_S2MM_CHANNELS              ,
                C_ENABLE_MULTI_CHANNEL             => C_ENABLE_MULTI_CHANNEL    ,
                C_MICRO_DMA                 => C_MICRO_DMA                      ,
                C_FAMILY                    => C_FAMILY
            )
            port map(

                m_axi_sg_aclk               => m_axi_sg_aclk                    ,
                m_axi_sg_aresetn            => m_axi_sg_aresetn                 ,
                s2mm_desc_info_in              => s2mm_desc_info_in                   ,

                -- SG S2MM Descriptor Fetch AXI Stream In
                m_axis_s2mm_ftch_tdata      => m_axis_s2mm_ftch_tdata           ,
                m_axis_s2mm_ftch_tvalid     => m_axis_s2mm_ftch_tvalid          ,
                m_axis_s2mm_ftch_tready     => m_axis_s2mm_ftch_tready          ,
                m_axis_s2mm_ftch_tlast      => m_axis_s2mm_ftch_tlast           ,

                m_axis_s2mm_ftch_tdata_new      => m_axis_s2mm_ftch_tdata_new           ,
                m_axis_s2mm_ftch_tdata_mcdma_new      => m_axis_s2mm_ftch_tdata_mcdma_new           ,
                m_axis_s2mm_ftch_tdata_mcdma_nxt      => m_axis_s2mm_ftch_tdata_mcdma_nxt           ,
                m_axis_s2mm_ftch_tvalid_new     => m_axis_s2mm_ftch_tvalid_new          ,
                m_axis_ftch2_desc_available     => m_axis_ftch2_desc_available ,

                -- SG S2MM Descriptor Update AXI Stream Out
                s_axis_s2mm_updtptr_tdata   => s_axis_s2mm_updtptr_tdata        ,
                s_axis_s2mm_updtptr_tvalid  => s_axis_s2mm_updtptr_tvalid       ,
                s_axis_s2mm_updtptr_tready  => s_axis_s2mm_updtptr_tready       ,
                s_axis_s2mm_updtptr_tlast   => s_axis_s2mm_updtptr_tlast        ,

                s_axis_s2mm_updtsts_tdata   => s_axis_s2mm_updtsts_tdata        ,
                s_axis_s2mm_updtsts_tvalid  => s_axis_s2mm_updtsts_tvalid       ,
                s_axis_s2mm_updtsts_tready  => s_axis_s2mm_updtsts_tready       ,
                s_axis_s2mm_updtsts_tlast   => s_axis_s2mm_updtsts_tlast        ,

                -- S2MM Descriptor Fetch Request (from s2mm_sm)
                desc_available              => desc_available                   ,
                desc_fetch_req              => desc_fetch_req                   ,
                desc_fetch_done             => desc_fetch_done                  ,
                updt_pending                => updt_pending                     ,

                -- S2MM Status Stream Interface
                stsstrm_fifo_rden           => stsstrm_fifo_rden                ,
                stsstrm_fifo_empty          => stsstrm_fifo_empty               ,
                stsstrm_fifo_dout           => stsstrm_fifo_dout                ,

                -- Update command write interface from s2mm sm
                s2mm_cmnd_wr                => s2mm_cmnd_wr                     ,
                s2mm_cmnd_data              => s2mm_cmnd_data (
                                                 ((1+C_ENABLE_MULTI_CHANNEL)*
                                                   C_M_AXI_S2MM_ADDR_WIDTH+
                                                   CMD_BASE_WIDTH)-1 downto 0)  ,


                -- S2MM Descriptor Update Request (from s2mm_sm)
                desc_update_done            => desc_update_done                 ,

                s2mm_sts_received_clr       => s2mm_sts_received_clr            ,
                s2mm_sts_received           => s2mm_sts_received                ,
                s2mm_desc_cmplt             => s2mm_desc_cmplt                  ,
                s2mm_done                   => s2mm_done                        ,
                s2mm_interr                 => s2mm_interr                      ,
                s2mm_slverr                 => s2mm_slverr                      ,
                s2mm_decerr                 => s2mm_decerr                      ,
                s2mm_tag                    => s2mm_tag                         ,
                s2mm_brcvd                  => s2mm_brcvd                       ,
                s2mm_eof_set                => s2mm_eof_set                     ,
                s2mm_packet_eof             => s2mm_packet_eof                  ,
                s2mm_halt                   => s2mm_halt                        ,
                s2mm_eof_micro              => s2mm_eof_micro,
                s2mm_sof_micro              => s2mm_sof_micro,

                -- S2MM Descriptor Field Output
                s2mm_new_curdesc            => s2mm_new_curdesc                 ,
                s2mm_new_curdesc_wren       => s2mm_new_curdesc_wren            ,
                s2mm_desc_baddress          => s2mm_desc_baddress               ,
                s2mm_desc_blength           => s2mm_desc_blength                ,
                s2mm_desc_blength_v           => s2mm_desc_blength_v                ,
                s2mm_desc_blength_s           => s2mm_desc_blength_s                ,
                s2mm_desc_info              => s2mm_desc_info                   ,
                s2mm_desc_app0              => s2mm_desc_app0                   ,
                s2mm_desc_app1              => s2mm_desc_app1                   ,
                s2mm_desc_app2              => s2mm_desc_app2                   ,
                s2mm_desc_app3              => s2mm_desc_app3                   ,
                s2mm_desc_app4              => s2mm_desc_app4
            );
    end generate GEN_SCATTER_GATHER_MODE;

      s2mm_packet_eof_out <= s2mm_packet_eof;

    -- Generate DMA Controller for Simple DMA Mode
    GEN_SIMPLE_DMA_MODE : if C_INCLUDE_SG = 0 generate
    begin


        -- Scatter Gather signals not used in Simple DMA Mode
        s2mm_desc_flush <= '0';
        m_axis_s2mm_ftch_tready     <= '0';
        s_axis_s2mm_updtptr_tdata   <= (others => '0');
        s_axis_s2mm_updtptr_tvalid  <= '0';
        s_axis_s2mm_updtptr_tlast   <= '0';
        s_axis_s2mm_updtsts_tdata   <= (others => '0');
        s_axis_s2mm_updtsts_tvalid  <= '0';
        s_axis_s2mm_updtsts_tlast   <= '0';
        desc_fetch_req              <= '0';
        desc_available              <= '0';
        desc_fetch_done             <= '0';
        desc_update_done            <= '0';
        s2mm_rxlength_clr           <= '0';
        stsstrm_fifo_rden           <= '0';
        s2mm_new_curdesc            <= (others => '0');
        s2mm_new_curdesc_wren       <= '0';
        s2mm_desc_baddress          <= (others => '0');
        s2mm_desc_info          <= (others => '0');
        s2mm_desc_blength           <= (others => '0');
        s2mm_desc_blength_v           <= (others => '0');
        s2mm_desc_blength_s           <= (others => '0');
        s2mm_desc_cmplt             <= '0';
        s2mm_desc_app0              <= (others => '0');
        s2mm_desc_app1              <= (others => '0');
        s2mm_desc_app2              <= (others => '0');
        s2mm_desc_app3              <= (others => '0');
        s2mm_desc_app4              <= (others => '0');

        -- Simple DMA State Machine
        I_S2MM_SMPL_SM : entity axi_dma_v7_1_13.axi_dma_smple_sm
            generic map(
                C_M_AXI_ADDR_WIDTH          => C_M_AXI_S2MM_ADDR_WIDTH  ,
                C_MICRO_DMA                 => C_MICRO_DMA ,
                C_SG_LENGTH_WIDTH           => C_SG_LENGTH_WIDTH
            )
            port map(
                m_axi_sg_aclk               => m_axi_sg_aclk            ,
                m_axi_sg_aresetn            => m_axi_sg_aresetn         ,

                -- Channel 1 Control and Status
                run_stop                    => s2mm_run_stop            ,
                keyhole                     => s2mm_keyhole              ,
                stop                        => s2mm_stop_i              ,
                cmnd_idle                   => s2mm_cmnd_idle           ,
                sts_idle                    => s2mm_sts_idle            ,

                -- DataMover Status
                sts_received                => s2mm_sts_received        ,
                sts_received_clr            => s2mm_sts_received_clr    ,

                -- DataMover Command
                cmnd_wr                     => s2mm_cmnd_wr             ,
                cmnd_data                   => s2mm_cmnd_data           ,
                cmnd_pending                => s2mm_cmnd_pending        ,

                -- Trasnfer Qualifiers
                xfer_length_wren            => s2mm_length_wren         ,
                xfer_address                => s2mm_da                  ,
                xfer_length                 => s2mm_length
            );

        -- Pass Done/Error Status out to DMASR
        s2mm_interr_set                 <= s2mm_interr;
        s2mm_slverr_set                 <= s2mm_slverr;
        s2mm_decerr_set                 <= s2mm_decerr;
        s2mm_bytes_rcvd                 <= s2mm_brcvd;
        s2mm_bytes_rcvd_wren            <= s2mm_done;

        -- S2MM Simple DMA Transfer Done - used to assert IOC bit in DMASR.
                         -- Receive clear when not shutting down
        s2mm_smple_done    <= s2mm_sts_received_clr when s2mm_stop_i = '0'
                         -- Else halt set prior to halted being set
                         else s2mm_halted_set_i when s2mm_halted = '0'
                         else '0';

    end generate GEN_SIMPLE_DMA_MODE;

    -------------------------------------------------------------------------------
    -- S2MM DataMover Command / Status Interface
    -------------------------------------------------------------------------------
    I_S2MM_CMDSTS : entity  axi_dma_v7_1_13.axi_dma_s2mm_cmdsts_if
        generic map(
            C_M_AXI_S2MM_ADDR_WIDTH     => C_M_AXI_S2MM_ADDR_WIDTH          ,
            C_DM_STATUS_WIDTH           => C_DM_STATUS_WIDTH                ,
            C_SG_INCLUDE_STSCNTRL_STRM  => C_SG_INCLUDE_STSCNTRL_STRM       ,
            C_SG_USE_STSAPP_LENGTH      => C_SG_USE_STSAPP_LENGTH           ,
            C_SG_LENGTH_WIDTH           => C_SG_LENGTH_WIDTH                ,
            C_INCLUDE_SG                => C_INCLUDE_SG                     ,
            C_ENABLE_MULTI_CHANNEL             => C_ENABLE_MULTI_CHANNEL   ,
                C_MICRO_DMA                 => C_MICRO_DMA                      ,
            C_ENABLE_QUEUE                  => C_SG_INCLUDE_DESC_QUEUE 
        )
        port map(
            m_axi_sg_aclk               => m_axi_sg_aclk                    ,
            m_axi_sg_aresetn            => m_axi_sg_aresetn                 ,

            -- Update command write interface from s2mm sm
            s2mm_cmnd_wr                => s2mm_cmnd_wr                     ,
            s2mm_cmnd_data              => s2mm_cmnd_data                   ,
            s2mm_cmnd_pending           => s2mm_cmnd_pending                ,
            s2mm_packet_eof             => s2mm_packet_eof                  , -- EOF Detected
            s2mm_sts_received_clr       => s2mm_sts_received_clr            ,
            s2mm_sts_received           => s2mm_sts_received                ,
            s2mm_tailpntr_enble         => s2mm_tailpntr_enble              ,
            s2mm_desc_cmplt             => s2mm_desc_cmplt                  ,

            -- User Command Interface Ports (AXI Stream)
            s_axis_s2mm_cmd_tvalid      => s_axis_s2mm_cmd_tvalid           ,
            s_axis_s2mm_cmd_tready      => s_axis_s2mm_cmd_tready           ,
            s_axis_s2mm_cmd_tdata       => s_axis_s2mm_cmd_tdata            ,

            -- User Status Interface Ports (AXI Stream)
            m_axis_s2mm_sts_tvalid      => m_axis_s2mm_sts_tvalid           ,
            m_axis_s2mm_sts_tready      => m_axis_s2mm_sts_tready           ,
            m_axis_s2mm_sts_tdata       => m_axis_s2mm_sts_tdata            ,
            m_axis_s2mm_sts_tkeep       => m_axis_s2mm_sts_tkeep            ,

            -- S2MM Primary DataMover Status
            s2mm_brcvd                  => s2mm_brcvd                       ,
            s2mm_err                    => s2mm_err                         ,
            s2mm_done                   => s2mm_done                        ,
            s2mm_error                  => dma_s2mm_error                   ,
            s2mm_interr                 => s2mm_interr                      ,
            s2mm_slverr                 => s2mm_slverr                      ,
            s2mm_decerr                 => s2mm_decerr                      ,
            s2mm_tag                    => s2mm_tag
        );


    ---------------------------------------------------------------------------
    -- Halt / Idle Status Manager
    ---------------------------------------------------------------------------
    I_S2MM_STS_MNGR : entity  axi_dma_v7_1_13.axi_dma_s2mm_sts_mngr
        generic map(
            C_PRMRY_IS_ACLK_ASYNC       => C_PRMRY_IS_ACLK_ASYNC
        )
        port map(
            m_axi_sg_aclk               => m_axi_sg_aclk                    ,
            m_axi_sg_aresetn            => m_axi_sg_aresetn                 ,

            -- dma control and sg engine status signals
            s2mm_run_stop               => s2mm_run_stop                    ,
            s2mm_ftch_idle              => s2mm_ftch_idle                   ,
            s2mm_updt_idle              => s2mm_updt_idle                   ,
            s2mm_cmnd_idle              => s2mm_cmnd_idle                   ,
            s2mm_sts_idle               => s2mm_sts_idle                    ,

            -- stop and halt control/status
            s2mm_stop                   => s2mm_stop_i                      ,
            s2mm_halt_cmplt             => s2mm_halt_cmplt                  ,

            -- system state and control
            s2mm_all_idle               => s2mm_all_idle                    ,
            s2mm_halted_clr             => s2mm_halted_clr                  ,
            s2mm_halted_set             => s2mm_halted_set_i                ,
            s2mm_idle_set               => s2mm_idle_set                    ,
            s2mm_idle_clr               => s2mm_idle_clr
        );


    -- S2MM Status Stream Included
    GEN_STS_STREAM : if C_SG_INCLUDE_STSCNTRL_STRM = 1 and C_INCLUDE_SG = 1 generate
    begin
        -- Register soft reset to create rising edge pulse to use for shut down.
        -- soft_reset from DMACR does not clear until after all reset processes
        -- are done.  This causes stop to assert too long causing issue with
        -- status stream skid buffer.
        REG_SFT_RST : process(m_axi_sg_aclk)
            begin
                if(m_axi_sg_aclk'EVENT and m_axi_sg_aclk = '1')then
                    if(m_axi_sg_aresetn = '0')then
                        soft_reset_d1   <= '0';
                        soft_reset_d2   <= '0';
                    else
                        soft_reset_d1   <= soft_reset;
                        soft_reset_d2   <= soft_reset_d1;
                    end if;
                end if;
            end process REG_SFT_RST;

        -- Rising edge soft reset pulse
        soft_reset_re <= soft_reset_d1 and not soft_reset_d2;

        -- Status Stream module stop requires rising edge of soft reset to
        -- shut down due to DMACR.SoftReset does not deassert on internal hard reset
        -- It clears after therefore do not want to issue another stop to sts strm
        -- skid buffer.
        sts_strm_stop <= s2mm_error_i               -- Error
                      or soft_reset_re;             -- Soft Reset issued

        I_S2MM_STS_STREAM : entity axi_dma_v7_1_13.axi_dma_s2mm_sts_strm
            generic map(

                C_PRMRY_IS_ACLK_ASYNC       => C_PRMRY_IS_ACLK_ASYNC            ,
                C_S_AXIS_S2MM_STS_TDATA_WIDTH=> C_S_AXIS_S2MM_STS_TDATA_WIDTH   ,
                C_SG_USE_STSAPP_LENGTH      => C_SG_USE_STSAPP_LENGTH           ,
                C_SG_LENGTH_WIDTH           => C_SG_LENGTH_WIDTH                ,
                C_FAMILY                    => C_FAMILY
            )
            port map(

                m_axi_sg_aclk               => m_axi_sg_aclk                    ,
                m_axi_sg_aresetn            => m_axi_sg_aresetn                 ,

                axi_prmry_aclk              => axi_prmry_aclk                   ,
                p_reset_n                   => p_reset_n                        ,

                s2mm_stop                   => sts_strm_stop                    ,

                s2mm_rxlength_valid         => s2mm_rxlength_valid              ,
                s2mm_rxlength_clr           => s2mm_rxlength_clr                ,
                s2mm_rxlength               => s2mm_rxlength                    ,
                stsstrm_fifo_rden           => stsstrm_fifo_rden                ,
                stsstrm_fifo_empty          => stsstrm_fifo_empty               ,
                stsstrm_fifo_dout           => stsstrm_fifo_dout                ,

                -- Stream to Memory Map Status Stream Interface                 ,
                s_axis_s2mm_sts_tdata       => s_axis_s2mm_sts_tdata            ,
                s_axis_s2mm_sts_tkeep       => s_axis_s2mm_sts_tkeep            ,
                s_axis_s2mm_sts_tvalid      => s_axis_s2mm_sts_tvalid           ,
                s_axis_s2mm_sts_tready      => s_axis_s2mm_sts_tready           ,
                s_axis_s2mm_sts_tlast       => s_axis_s2mm_sts_tlast
            );
    end generate GEN_STS_STREAM;

    -- S2MM Status Stream Not Included
    GEN_NO_STS_STREAM : if C_SG_INCLUDE_STSCNTRL_STRM = 0 or C_INCLUDE_SG = 0 generate
    begin
        s2mm_rxlength_valid     <= '0';
        s2mm_rxlength           <= (others => '0');
        stsstrm_fifo_empty      <= '1';
        stsstrm_fifo_dout       <= (others => '0');
        s_axis_s2mm_sts_tready  <= '0';
    end generate GEN_NO_STS_STREAM;


end generate GEN_S2MM_DMA_CONTROL;



-------------------------------------------------------------------------------
-- Do Not Include S2MM Channel
-------------------------------------------------------------------------------
GEN_NO_S2MM_DMA_CONTROL : if C_INCLUDE_S2MM = 0 generate
begin
        m_axis_s2mm_ftch_tready     <= '0';
        s_axis_s2mm_updtptr_tdata   <= (others =>'0');
        s_axis_s2mm_updtptr_tvalid  <= '0';
        s_axis_s2mm_updtptr_tlast   <= '0';
        s_axis_s2mm_updtsts_tdata   <= (others =>'0');
        s_axis_s2mm_updtsts_tvalid  <= '0';
        s_axis_s2mm_updtsts_tlast   <= '0';
        s2mm_new_curdesc            <= (others =>'0');
        s2mm_new_curdesc_wren       <= '0';
        s_axis_s2mm_cmd_tvalid      <= '0';
        s_axis_s2mm_cmd_tdata       <= (others =>'0');
        m_axis_s2mm_sts_tready      <= '0';
        s2mm_halted_clr             <= '0';
        s2mm_halted_set             <= '0';
        s2mm_idle_set               <= '0';
        s2mm_idle_clr               <= '0';
        s_axis_s2mm_sts_tready      <= '0';
        s2mm_stop                   <= '0';
        s2mm_desc_flush             <= '0';
        s2mm_all_idle               <= '1';
        s2mm_error                  <= '0'; -- CR#570587
        s2mm_packet_eof_out         <= '0';
        s2mm_smple_done             <= '0';
        s2mm_interr_set             <= '0';
        s2mm_slverr_set             <= '0';
        s2mm_decerr_set             <= '0';
        s2mm_bytes_rcvd             <= (others => '0');
        s2mm_bytes_rcvd_wren        <= '0';

        

end generate GEN_NO_S2MM_DMA_CONTROL;

end implementation;


--  (c) Copyright 2012 Xilinx, Inc. All rights reserved.
--
--  This file contains confidential and proprietary information
--  of Xilinx, Inc. and is protected under U.S. and
--  international copyright and other intellectual property
--  laws.
--
--  DISCLAIMER
--  This disclaimer is not a license and does not grant any
--  rights to the materials distributed herewith. Except as
--  otherwise provided in a valid license issued to you by
--  Xilinx, and to the maximum extent permitted by applicable
--  law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
--  WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
--  AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
--  BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
--  INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
--  (2) Xilinx shall not be liable (whether in contract or tort,
--  including negligence, or under any other theory of
--  liability) for any loss or damage of any kind or nature
--  related to, arising under or in connection with these
--  materials, including for any direct, or any indirect,
--  special, incidental, or consequential loss or damage
--  (including loss of data, profits, goodwill, or any type of
--  loss or damage suffered as a result of any action brought
--  by a third party) even if such damage or loss was
--  reasonably foreseeable or Xilinx had been advised of the
--  possibility of the same.
--
--  CRITICAL APPLICATIONS
--  Xilinx products are not designed or intended to be fail-
--  safe, or for use in any application requiring fail-safe
--  performance, such as life-support or safety devices or
--  systems, Class III medical devices, nuclear facilities,
--  applications related to the deployment of airbags, or any
--  other applications that could lead to death, personal
--  injury, or severe property or environmental damage
--  (individually and collectively, "Critical
--  Applications"). Customer assumes the sole risk and
--  liability of any use of Xilinx products in Critical
--  Applications, subject only to applicable laws and
--  regulations governing limitations on product liability.
--
--  THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
--  PART OF THIS FILE AT ALL TIMES. 
------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_misc.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;


library unisim;
use unisim.vcomponents.all;

library lib_cdc_v1_0_2;

library axi_dma_v7_1_13;
use axi_dma_v7_1_13.axi_dma_pkg.all;


 
entity axi_dma_cmd_split is
     generic (
             C_ADDR_WIDTH  : integer range 32 to 64    := 32;
             C_DM_STATUS_WIDTH               : integer range 8 to 32         := 8;
             C_INCLUDE_S2MM : integer range 0 to 1     := 0 
             );
     port (
           clock : in std_logic;
           sgresetn : in std_logic;
           clock_sec : in std_logic;
           aresetn : in std_logic;

   -- command coming from _MNGR 
           s_axis_cmd_tvalid : in std_logic;
           s_axis_cmd_tready : out std_logic;
           s_axis_cmd_tdata  : in std_logic_vector ((C_ADDR_WIDTH-32+2*32+CMD_BASE_WIDTH+46)-1 downto 0);

   -- split command to DM
           s_axis_cmd_tvalid_s : out std_logic;
           s_axis_cmd_tready_s : in std_logic;
           s_axis_cmd_tdata_s  : out std_logic_vector ((C_ADDR_WIDTH+CMD_BASE_WIDTH+8)-1 downto 0);
   -- Tvalid from Datamover
           tvalid_from_datamover    : in std_logic;
           status_in                : in std_logic_vector (C_DM_STATUS_WIDTH-1 downto 0);
           tvalid_unsplit           : out std_logic;
           status_out               : out std_logic_vector (C_DM_STATUS_WIDTH-1 downto 0);

   -- Tlast of stream data from Datamover
           tlast_stream_data        : in std_logic;
           tready_stream_data        : in std_logic;
           tlast_unsplit            : out std_logic;  
           tlast_unsplit_user       : out std_logic  

          );
end entity axi_dma_cmd_split;

architecture implementation of axi_dma_cmd_split is
  attribute DowngradeIPIdentifiedWarnings: string;
  attribute DowngradeIPIdentifiedWarnings of implementation : architecture is "yes";


type SPLIT_MM2S_STATE_TYPE      is (
                                IDLE,
                                SEND,
                                SPLIT
                                );

signal mm2s_cs                  : SPLIT_MM2S_STATE_TYPE;
signal mm2s_ns                  : SPLIT_MM2S_STATE_TYPE;

signal mm2s_cmd    : std_logic_vector (C_ADDR_WIDTH-32+2*32+CMD_BASE_WIDTH+46-1 downto 0);
signal command_ns    : std_logic_vector (C_ADDR_WIDTH-32+2*32+CMD_BASE_WIDTH-1 downto 0);
signal command    : std_logic_vector (C_ADDR_WIDTH-32+2*32+CMD_BASE_WIDTH-1 downto 0);

signal cache_info  : std_logic_vector (31 downto 0);
signal vsize_data  : std_logic_vector (22 downto 0);
signal vsize_data_int  : std_logic_vector (22 downto 0);
signal vsize       : std_logic_vector (22 downto 0);
signal counter     : std_logic_vector (22 downto 0);
signal counter_tlast     : std_logic_vector (22 downto 0);
signal split_cmd   : std_logic_vector (31+(C_ADDR_WIDTH-32) downto 0);
signal stride_data : std_logic_vector (22 downto 0);
signal vsize_over   : std_logic;

signal cmd_proc_cdc_from    : std_logic;
signal cmd_proc_cdc_to    : std_logic;
signal cmd_proc_cdc    : std_logic;
signal cmd_proc_ns    : std_logic;
  ATTRIBUTE async_reg                      : STRING;
--  ATTRIBUTE async_reg OF cmd_proc_cdc_to  : SIGNAL IS "true";
--  ATTRIBUTE async_reg OF cmd_proc_cdc  : SIGNAL IS "true";


signal cmd_out    : std_logic;
signal cmd_out_ns    : std_logic;

signal split_out    : std_logic;
signal split_out_ns    : std_logic;

signal command_valid : std_logic;
signal command_valid_ns : std_logic;
signal command_ready : std_logic;
signal reset_lock : std_logic;
signal reset_lock_tlast : std_logic;


signal tvalid_unsplit_int : std_logic;
signal tlast_stream_data_int : std_logic;

signal ready_for_next_cmd : std_logic;
signal ready_for_next_cmd_tlast : std_logic;
signal ready_for_next_cmd_tlast_cdc_from : std_logic;
signal ready_for_next_cmd_tlast_cdc_to : std_logic;
signal ready_for_next_cmd_tlast_cdc : std_logic;

--  ATTRIBUTE async_reg OF ready_for_next_cmd_tlast_cdc_to  : SIGNAL IS "true";
--  ATTRIBUTE async_reg OF ready_for_next_cmd_tlast_cdc  : SIGNAL IS "true";

signal tmp1, tmp2, tmp3, tmp4 : std_logic;
signal tlast_int : std_logic;

signal eof_bit : std_logic;
signal eof_bit_cdc_from : std_logic;
signal eof_bit_cdc_to : std_logic;
signal eof_bit_cdc : std_logic;
signal eof_set : std_logic;
signal over_ns, over : std_logic;

signal cmd_in : std_logic;

signal status_out_int : std_logic_vector (C_DM_STATUS_WIDTH-1 downto 0);

begin

s_axis_cmd_tvalid_s <= command_valid;
command_ready <= s_axis_cmd_tready_s;
s_axis_cmd_tdata_s <= command (103+(C_ADDR_WIDTH-32) downto 96+(C_ADDR_WIDTH-32)) & command (71+(C_ADDR_WIDTH-32) downto 0);


REGISTER_STATE_MM2S : process(clock)
    begin
        if(clock'EVENT and clock = '1')then
            if(sgresetn = '0')then
                mm2s_cs     <= IDLE;
                cmd_proc_cdc_from <= '0';
                cmd_out <= '0';
                command <= (others => '0');
                command_valid <= '0';
                split_out <= '0';
                over <= '0';
            else
                mm2s_cs     <= mm2s_ns;
                cmd_proc_cdc_from <= cmd_proc_ns;
                cmd_out <= cmd_out_ns;
                command <= command_ns;
                command_valid <= command_valid_ns;
                split_out <= split_out_ns;
                over <= over_ns;
            end if;
        end if;
    end process REGISTER_STATE_MM2S;


-- grab the MM2S command coming from MM2S_mngr
REGISTER_MM2S_CMD : process(clock)
    begin
        if(clock'EVENT and clock = '1')then
            if(sgresetn = '0')then
                mm2s_cmd <= (others => '0');
                s_axis_cmd_tready <= '0';
                cache_info <= (others => '0');
                vsize_data <= (others => '0');
                vsize_data_int <= (others => '0');
                stride_data <= (others => '0');
                eof_bit_cdc_from <= '0';
                cmd_in <= '0';
            elsif (s_axis_cmd_tvalid = '1' and ready_for_next_cmd = '1' and cmd_proc_cdc_from = '0' and ready_for_next_cmd_tlast_cdc = '1') then  -- when there is no processing being done, means it is ready to accept
                mm2s_cmd     <= s_axis_cmd_tdata;
                s_axis_cmd_tready <= '1';
                cache_info <= s_axis_cmd_tdata (149+(C_ADDR_WIDTH-32) downto 118+(C_ADDR_WIDTH-32));
                vsize_data <= s_axis_cmd_tdata (117+(C_ADDR_WIDTH-32) downto 95+(C_ADDR_WIDTH-32));
                vsize_data_int <= s_axis_cmd_tdata (117+(C_ADDR_WIDTH-32) downto 95+(C_ADDR_WIDTH-32)) - '1';
                stride_data <= s_axis_cmd_tdata (94+(C_ADDR_WIDTH-32) downto 72+(C_ADDR_WIDTH-32));
                eof_bit_cdc_from <= s_axis_cmd_tdata (30);
                cmd_in <= '1';
            else
                mm2s_cmd     <= mm2s_cmd; --split_cmd;
                vsize_data   <= vsize_data;
                vsize_data_int   <= vsize_data_int;
                stride_data   <= stride_data;
                cache_info <= cache_info;
                s_axis_cmd_tready <= '0';
                eof_bit_cdc_from <= eof_bit_cdc_from;
                cmd_in <= '0';
            end if;
        end if;
    end process REGISTER_MM2S_CMD;


REGISTER_DECR_VSIZE : process(clock)
    begin
        if(clock'EVENT and clock = '1')then
            if(sgresetn = '0')then
                vsize <= "00000000000000000000000";
            elsif (command_valid = '1' and command_ready = '1' and (vsize < vsize_data_int)) then  -- sending a cmd out to DM
                vsize <= vsize + '1';
            elsif (cmd_proc_cdc_from = '0') then  -- idle or when all cmd are sent to DM
                vsize <= "00000000000000000000000";
            else 
                vsize <= vsize;    
            end if;
        end if;
    end process REGISTER_DECR_VSIZE;

    vsize_over <= '1' when (vsize = vsize_data_int) else '0';
  --  eof_set <= eof_bit when (vsize = vsize_data_int) else '0';


 REGISTER_SPLIT : process(clock)
     begin
         if(clock'EVENT and clock = '1')then
             if(sgresetn = '0')then
                 split_cmd <= (others => '0');
             elsif (s_axis_cmd_tvalid = '1' and cmd_proc_cdc_from = '0' and ready_for_next_cmd = '1' and ready_for_next_cmd_tlast_cdc = '1') then
                 split_cmd <= s_axis_cmd_tdata (63+(C_ADDR_WIDTH-32) downto 32);          -- capture the ba when a new cmd arrives
             elsif (split_out = '1') then  -- add stride to previous ba
                 split_cmd <= split_cmd + stride_data;
             else 
                 split_cmd <= split_cmd;
             end if;

         end if;
     end process REGISTER_SPLIT;



MM2S_MACHINE : process(mm2s_cs,
                       s_axis_cmd_tvalid,
                       cmd_proc_cdc_from, 
                       vsize_over, command_ready,
                       cache_info, mm2s_cmd,
                       split_cmd, eof_set,
                       cmd_in, command
                       )
    begin
         over_ns <= '0'; 
                       cmd_proc_ns <= '0';      -- ready to receive new command 
                       split_out_ns <= '0';
                       command_valid_ns <= '0';
         mm2s_ns <= mm2s_cs;
         command_ns <= command;  
        -- Default signal assignment
        case mm2s_cs is

            -------------------------------------------------------------------
            when IDLE => 
                       command_ns <=  cache_info & mm2s_cmd (72+(C_ADDR_WIDTH-32) downto 65+(C_ADDR_WIDTH-32)) & split_cmd & mm2s_cmd (31) & eof_set & mm2s_cmd (29 downto 0); -- buf length remains the same
                  --     command_ns <=  cache_info & mm2s_cmd (72 downto 65) & split_cmd & mm2s_cmd (31 downto 0); -- buf length remains the same
                   if (cmd_in = '1' and cmd_proc_cdc_from = '0') then
                       cmd_proc_ns <= '1';      -- new command has come in and i need to start processing
                       mm2s_ns <= SEND;
                       over_ns <= '0'; 
                       split_out_ns <= '1'; 
                       command_valid_ns <= '1';
                   else 
                       mm2s_ns <= IDLE; 
                       over_ns <= '0'; 
                       cmd_proc_ns <= '0';      -- ready to receive new command 
                       split_out_ns <= '0'; 
                       command_valid_ns <= '0';
                   end if;

            -------------------------------------------------------------------
            when SEND =>
                       cmd_out_ns <= '1';
                       command_ns <=  command;

                       if (vsize_over = '1' and command_ready = '1') then
                         mm2s_ns <= IDLE; 
                         cmd_proc_ns <= '1';
                         command_valid_ns <= '0';
                         split_out_ns <= '0'; 
                         over_ns <= '1'; 
                       elsif  (command_ready = '0') then --(command_valid = '1' and command_ready = '0') then
                         mm2s_ns <= SEND;
                         command_valid_ns <= '1';
                         cmd_proc_ns <= '1'; 
                         split_out_ns <= '0'; 
                         over_ns <= '0';
                       else 
                         mm2s_ns <= SPLIT;
                         command_valid_ns <= '0';
                         cmd_proc_ns <= '1';
                         over_ns <= '0'; 
                         split_out_ns <= '0'; 
                       end if;
                  
            -------------------------------------------------------------------
            when SPLIT =>
                         cmd_proc_ns <= '1';
                         mm2s_ns <= SEND; 
                         command_ns <=  cache_info & mm2s_cmd (72+(C_ADDR_WIDTH-32) downto 65+(C_ADDR_WIDTH-32)) & split_cmd & mm2s_cmd (31) & eof_set & mm2s_cmd (29 downto 0); -- buf length remains the same
        --                 command_ns <=  cache_info & mm2s_cmd (72 downto 65) & split_cmd & mm2s_cmd (31 downto 0); -- buf length remains the same
                         cmd_out_ns <= '0';
                         split_out_ns <= '1'; 
                         command_valid_ns <= '1';

            -------------------------------------------------------------------
          -- coverage off
            when others =>
                mm2s_ns <= IDLE;
          -- coverage on

        end case;
    end process MM2S_MACHINE;


SWALLOW_TVALID : process(clock)
    begin
        if(clock'EVENT and clock = '1')then
            if(sgresetn = '0')then
                counter <= (others => '0');
           --     tvalid_unsplit_int <= '0';
                reset_lock <= '1';
                ready_for_next_cmd <= '0';
            elsif (vsize_data_int = "00000000000000000000000") then
           --     tvalid_unsplit_int <= '0';
                ready_for_next_cmd <= '1';
                reset_lock <= '0';
            elsif ((tvalid_from_datamover = '1') and (counter < vsize_data_int)) then
                counter <= counter + '1';
           --     tvalid_unsplit_int <= '0';
                ready_for_next_cmd <= '0';
                reset_lock <= '0';
            elsif ((counter = vsize_data_int) and (reset_lock = '0') and (tvalid_from_datamover = '1')) then
                counter <= (others => '0');
          --      tvalid_unsplit_int <= '1';
                ready_for_next_cmd <= '1';
            else
                counter <= counter;
           --     tvalid_unsplit_int <= '0';
                if (cmd_proc_cdc_from = '1') then
                   ready_for_next_cmd <= '0';
                else
                   ready_for_next_cmd <= ready_for_next_cmd;
                end if;
            end if;
        end if;
    end process SWALLOW_TVALID;

                tvalid_unsplit_int <= tvalid_from_datamover when (counter = vsize_data_int) else '0'; --tvalid_unsplit_int;

SWALLOW_TDATA : process(clock)
    begin
        if(clock'EVENT and clock = '1')then
            if (sgresetn = '0' or cmd_in = '1') then
               tvalid_unsplit <= '0';
               status_out_int <= (others => '0');
            else
               tvalid_unsplit <= tvalid_unsplit_int;
               if (tvalid_from_datamover = '1') then 
                  status_out_int (C_DM_STATUS_WIDTH-2 downto 0) <= status_in (C_DM_STATUS_WIDTH-2 downto 0) or status_out_int (C_DM_STATUS_WIDTH-2 downto 0); 
               else
                  status_out_int <= status_out_int;
               end if;

               if (tvalid_unsplit_int = '1') then
                  status_out_int (C_DM_STATUS_WIDTH-1) <= status_in (C_DM_STATUS_WIDTH-1);
               end if;
            end if;
        end if;
    end process SWALLOW_TDATA;


        status_out <= status_out_int;

SWALLOW_TLAST_GEN : if C_INCLUDE_S2MM = 0 generate
begin


    eof_set <= '1'; --eof_bit when (vsize = vsize_data_int) else '0';

CDC_CMD_PROC1 : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
    )
    port map (
        prmry_aclk                 => '0',
        prmry_resetn               => '0',
        prmry_in                   => cmd_proc_cdc_from,
        prmry_vect_in              => (others => '0'),

        scndry_aclk                => clock_sec,
        scndry_resetn              => '0',
        scndry_out                 => cmd_proc_cdc,
        scndry_vect_out            => open
    );

CDC_CMD_PROC2 : entity  lib_cdc_v1_0_2.cdc_sync
    generic map (
        C_CDC_TYPE                 => 1,
        C_RESET_STATE              => 0,
        C_SINGLE_BIT               => 1,
        C_VECTOR_WIDTH             => 32,
        C_MTBF_STAGES              => MTBF_STAGES
  